/* Copyright 2025 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "main.h"

/* objcoll.c: object-collision functions */
extern void objcoll(void);

/* settings.c: load and save system-settings */
extern void settings_load(struct s_game *);
extern void settings_save(struct s_game *);
extern void settings_data_default(struct s_game *);

/* leveldata.c: create level-data */
extern VG_BOOL leveldata(int, struct g_level *, int);

/* gamelevel.c: do actions according to game-level */
extern VG_BOOL gamelevel_before(struct s_game *);
extern VG_BOOL gamelevel_start(struct s_game *, struct g_level *);
extern VG_BOOL gamelevel_done(int);
extern void gamelevel_stop(struct s_game *);
extern void gamelevel_after(struct s_game *);

static VG_BOOL cvexec_main_menu(struct s_game *);
static VG_BOOL show_help(void);
static VG_BOOL show_cover(void);
static VG_BOOL dgexec_sysmenu(void);
static VG_BOOL level_clear(int, int);
static VG_BOOL game_over(int);
static void draw_bottom(struct s_game *, int);


/* load and execute canvas main-menu; return: VG_TRUE=start-game, VG_FALSE=exit-request */
static VG_BOOL
cvexec_main_menu(struct s_game *sgame)
{
  const char *selname;
  struct VG_Canvas *cvas;
  int retw = VG_TRUE;

cvmm_start:
  vg4->window->clear();

  cvas = vg4->canvas->load("files/canvas/main-menu/top.cvas", NULL);
  if (cvas == NULL) { return VG_FALSE; }

  if (sgame->iset_old.idx_level == 0) {
    vg4->canvas->disable(cvas, "bt_lastgame", VG_TRUE);
  }

cvmm_redo:
  if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; goto cvmm_end; }
  if (selname == NULL || *selname == '\0') { goto cvmm_redo; }

  if (strcmp(selname, "bt_help") == 0) {
    if (!show_help()) { retw = VG_FALSE; goto cvmm_end; }
  } else if (strcmp(selname, "bt_sysmenu") == 0) {
    if (!dgexec_sysmenu()) { retw = VG_FALSE; goto cvmm_end; }
  } else if (strcmp(selname, "bt_newgame") == 0) {
    struct VG_Canvas *cvas_sub;
    VG_BOOL filmskip;
    settings_data_default(sgame);
    cvas_sub = vg4->canvas->load("files/canvas/difficulty/top.cvas", NULL);
    if (cvas_sub != NULL) {
      struct VG_Position possub;
      struct VG_ImagecopyAttrPixel iattr_pixel;
      VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&iattr_pixel);
      iattr_pixel.brightness = 60;
      possub = vg4->canvas->subcanvas(cvas, &iattr_pixel);
cvmm_redo_sub:
      if (!vg4->canvas->exec(cvas_sub, &possub, &selname)) { vg4->canvas->destroy(cvas_sub); retw = VG_FALSE; goto cvmm_end; }
      if (selname == NULL || *selname == '\0') { goto cvmm_redo_sub; }
      if (strcmp(selname, "easy") == 0) {
        sgame->iset_old.difficulty = sgame->iset_new.difficulty = 0;
      } else {
        sgame->iset_old.difficulty = sgame->iset_new.difficulty = 1;
      }
      vg4->canvas->destroy(cvas_sub);
    }
    if (!vg4->film->play("files/film/intro", NULL, &filmskip, NULL)) { retw = VG_FALSE; goto cvmm_end; }
    retw = VG_TRUE;
    goto cvmm_end;
  } else if (strcmp(selname, "bt_lastgame") == 0) {
    retw = VG_TRUE;
    goto cvmm_end;
  } else if (strcmp(selname, "bt_exit") == 0) {
    retw = VG_FALSE; 
    goto cvmm_end;
  }

  vg4->canvas->destroy(cvas);
  goto cvmm_start;

cvmm_end:
  vg4->canvas->destroy(cvas);
  return retw;
}


/* show help */
static VG_BOOL
show_help(void)
{
  struct VG_Canvas *cvas;
  const char *selname;
  struct VG_Hash *hvar;
  VG_BOOL retw, asusp;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "title", vg4->mlang->get("menu", "help-title"));
  vg4->hash->setstr(hvar, "text", vg4->mlang->get("menu", "help-text"));

  /* load canvas */
  cvas = vg4->canvas->load("files/canvas/help/top.cvas", hvar);
  if (cvas == NULL) { return VG_FALSE; }

  asusp = vg4->audio->suspend(VG_TRUE);
  retw = VG_TRUE;

  for (;;) {
    /* execute canvas */
    if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; break; }
    if (selname == NULL) { break; }
    if (*selname == '\0' || strcmp(selname, "close") == 0) { break; }
  }

  vg4->canvas->destroy(cvas);
  vg4->hash->destroy(hvar);
  vg4->audio->suspend(asusp);

  return retw;
}


/* show cover image */
static VG_BOOL
show_cover(void)
{
  struct VG_Image *bgimg, *tximg;
  int k_quit1, k_quit2, ilauf, txw, txh;
  VG_BOOL retw;
  struct VG_ImagecopyAttr iattr;
  struct VG_ImagecopyAttrPixel wattr_pixel;

  bgimg = vg4->image->load("files/images/cover.bmp");
  if (bgimg == NULL) { return VG_FALSE; }
  tximg = vg4->font->totext("Snakes", "[fgcolor=0xffff00]", NULL, NULL, NULL);
  if (tximg == NULL) { return VG_FALSE; }
  vg4->image->getsize(tximg, NULL, &txw, &txh);

  if ((k_quit1 = vg4->input->key_insert("Quit", VG_FALSE, VG_FALSE)) == 0) { return VG_FALSE; }
  vg4->input->key_setkbd(k_quit1, VG_INPUT_KBDCODE_RETURN);
  if ((k_quit2 = vg4->input->key_insert("Quit", VG_FALSE, VG_FALSE)) == 0) { vg4->input->key_remove(k_quit1); return VG_FALSE; }
  vg4->input->key_setkbd(k_quit2, VG_INPUT_KBDCODE_SPACE);

  retw = VG_TRUE;
  for (ilauf = 0; ilauf < 250; ilauf++) {
    if (!vg4->input->update(VG_FALSE)) { retw = VG_FALSE; break; }
    if (vg4->input->key_newpressed(k_quit1)) { break; }
    if (vg4->input->key_newpressed(k_quit2)) { break; }

    vg4->window->clear();
    vg4->window->copy(bgimg, NULL, NULL);

    VG_IMAGECOPY_ATTR_DEFAULT(&iattr);

    if (ilauf < 36) {
      iattr.image.rotate = ilauf * 10;
      iattr.image.zoom_width = txw * (100 * ilauf * 10 / 360) / 100;
      iattr.image.zoom_height = txh * (100 * ilauf * 10 / 360) / 100;
      iattr.image.zoom_ispercent = VG_FALSE;
    } else if (ilauf < 130) {
      iattr.image.zoom_width = txw * (100 * ilauf * 10 / 360) / 100;
      iattr.image.zoom_height = txh * (100 * ilauf * 10 / 360) / 100;
      iattr.image.zoom_ispercent = VG_FALSE;
    } else {
      iattr.image.zoom_width = txw * (100 * 130 * 10 / 360) / 100;
      iattr.image.zoom_height = txh * (100 * 130 * 10 / 360) / 100;
      iattr.image.zoom_ispercent = VG_FALSE;
    }

    if (ilauf == 160 || ilauf == 163) {
      VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&wattr_pixel);
      wattr_pixel.pixelcolor = VG_PIXELCOLOR_INVERT;
      vg4->window->setattr(&wattr_pixel);
    } else if (ilauf == 161 || ilauf == 164) {
      VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&wattr_pixel);
      vg4->window->setattr(&wattr_pixel);
    }

    vg4->window->copy(tximg, NULL, &iattr);

    vg4->window->flush();
    vg4->misc->wait_time(40);
  }

  vg4->input->key_remove(k_quit1);
  vg4->input->key_remove(k_quit2);

  vg4->image->destroy(tximg);
  vg4->image->destroy(bgimg);
  VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&wattr_pixel);
  vg4->window->setattr(&wattr_pixel);
  vg4->window->clear();
  vg4->window->flush();
  if (retw == VG_TRUE) { vg4->misc->wait_time(1000); }

  return retw;
}


/* execute system-menu dialog */
static VG_BOOL
dgexec_sysmenu(void)
{
  struct VG_Hash *hvar;
  VG_BOOL retw;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "top:title", vg4->mlang->get("menu", "System-menu"));
  vg4->hash->setstr(hvar, "exit:top:title", vg4->mlang->get("menu", "Quit game?"));
  vg4->hash->setstr(hvar, "volume:top:title", vg4->mlang->get("menu", "Set audio volumes"));
  vg4->hash->setstr(hvar, "keydef:top:title", vg4->mlang->get("menu", "Key Redefinition"));
  vg4->hash->setstr(hvar, "keydef:press:title", vg4->mlang->get("menu", "Press key"));
  vg4->hash->setstr(hvar, "windowsize:top:title", vg4->mlang->get("menu", "Resize window"));
  vg4->hash->setstr(hvar, "winbright:top:title", vg4->mlang->get("menu", "Brightness"));

  vg4->audio->suspend(VG_TRUE);
  retw = vg4->dialog->sysmenu(NULL, NULL, hvar, NULL, NULL, NULL, NULL);
  vg4->audio->suspend(VG_FALSE);
  vg4->hash->destroy(hvar);

  return retw;
}


/* show level ending */
static VG_BOOL
level_clear(int level, int audc)
{
  struct VG_Image *tximg;
  int ilauf;
  char buf[64];

  while (vg4->audio->is_playing(audc, NULL)) {
    if (!vg4->input->update(VG_FALSE)) { return VG_FALSE; }
    vg4->window->flush();
    vg4->misc->wait_time(100);
  }

  snprintf(buf, sizeof(buf), "Level %d clear", level);
  tximg = vg4->font->totext(buf, "[fgcolor=0xffff00]", NULL, NULL, NULL);
  if (tximg == NULL) { return VG_FALSE; }
  vg4->window->clear();
  vg4->window->copy(tximg, NULL, NULL);
  vg4->image->destroy(tximg);

  for (ilauf = 0; ilauf < 30; ilauf++) {
    if (!vg4->input->update(VG_FALSE)) { return VG_FALSE; }
    vg4->window->flush();
    vg4->misc->wait_time(100);
  }

  vg4->window->clear();
  vg4->window->flush();

  return VG_TRUE;
}


/* game over */
static VG_BOOL
game_over(int audc)
{
  struct VG_Image *tximg;

  if (!vg4->misc->fadeout(audc)) { return VG_FALSE; }

  tximg = vg4->font->totext("Game over", "[fgcolor=0xffff00]", NULL, NULL, NULL);
  if (tximg == NULL) { return VG_FALSE; }
  vg4->window->clear();
  vg4->window->copy(tximg, NULL, NULL);
  vg4->image->destroy(tximg);

  audc = vg4->audio->load("files/audio/gamov.wav", 100, VG_AUDIO_VOLUME_MUSIC);
  if (audc == 0) { return VG_FALSE; }
  vg4->audio->play(audc, VG_FALSE, VG_FALSE);

  while (vg4->audio->is_playing(audc, NULL)) {
    if (!vg4->input->update(VG_FALSE)) { return VG_FALSE; }
    vg4->window->flush();
    vg4->misc->wait_time(100);
  }

  vg4->window->clear();
  vg4->window->flush();

  return VG_TRUE;
}


/* draw info onto the bottom */
static void
draw_bottom(struct s_game *sgame, int esnake_remain)
{
  struct VG_Image *imgtxt;
  char buf[512];
  struct VG_Position posi;
  int gapbottom;

  if (sgame == NULL) { return; }

  snprintf(buf, sizeof(buf), "%%{table[cells=50,50 boxwidth=%d fgcolor=0xffff00]: %%{cell&left: %s=%02d%%%%}%%{cell&right: %s=%d%%}%%}",
           sgame->winw,
           vg4->mlang->get("game", "health"), sgame->iset_new.isnake_health,
           vg4->mlang->get("game", "snakes"), esnake_remain);

  imgtxt = vg4->font->totext(buf, NULL, NULL, NULL, NULL);
  if (imgtxt == NULL) { return; }

  vg4->font->gaps(NULL, NULL, &gapbottom);
  posi.pos = VG_POS_LOWER_LEFT;
  posi.x = 0;
  posi.y = sgame->winh - 1 + gapbottom;
  vg4->window->copy(imgtxt, &posi, NULL);

  vg4->image->destroy(imgtxt);
}


int main(int argc, char **argv) {
  struct s_game sgame;
  struct g_level glevel;
  char buf[256];
  int audc_bgmusic, esnake_remain;
  VG_BOOL filmskip;
  enum { GAMEPLAY_NOOP = 0, GAMEPLAY_DONE, GAMEPLAY_DEAD, GAMEPLAY_MENU } gplay;

  (void)argc; (void)argv;

  /* initializations */

  memset(&sgame, 0, sizeof(sgame));

  if (!VG_init("Snakes")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_LOW, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&sgame.winw, &sgame.winh);
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_FALSE)) { VG_dest(); exit(1); }
  vg4->input->mouse_grabbing(VG_FALSE);

  vg4->mlang->fb_locale("en");
  vg4->mlang->add("files/mlang");

  objcoll();

  /* setting keys */

  /* quit with ALT+Q, not changeable, local key */
  if ((sgame.kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((sgame.kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* system-menu with ESC, not changeable, local key */
  if ((sgame.kref.k_sysmenu = vg4->input->key_insert("System-menu", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_sysmenu, VG_INPUT_KBDCODE_ESCAPE);
  /* help with H, changeable, local key */
  if ((sgame.kref.k_help = vg4->input->key_insert(vg4->mlang->get("menu", "Help"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_help, VG_INPUT_KBDCODE_H);
  /* pause with P, changeable, local key */
  if ((sgame.kref.k_pause = vg4->input->key_insert(vg4->mlang->get("menu", "Pause"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_pause, VG_INPUT_KBDCODE_P);
  /* move-up with cursor-key UP, changeable, local key */
  if ((sgame.kref.k_up = vg4->input->key_insert(vg4->mlang->get("menu", "Move up"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_up, VG_INPUT_KBDCODE_UCURS);
  vg4->input->key_setgc(sgame.kref.k_up, 0, VG_INPUT_GCAXIS_RIGHTY_UP);
  /* move-down with cursor-key DOWN, changeable, local key */
  if ((sgame.kref.k_down = vg4->input->key_insert(vg4->mlang->get("menu", "Move down"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_down, VG_INPUT_KBDCODE_DCURS);
  vg4->input->key_setgc(sgame.kref.k_down, 0, VG_INPUT_GCAXIS_RIGHTY_DOWN);
  /* move-left with cursor-key LEFT, changeable, local key */
  if ((sgame.kref.k_left = vg4->input->key_insert(vg4->mlang->get("menu", "Move left"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_left, VG_INPUT_KBDCODE_LCURS);
  vg4->input->key_setgc(sgame.kref.k_left, 0, VG_INPUT_GCAXIS_LEFTX_LEFT);
  /* move-right with cursor-key RIGHT, changeable, local key */
  if ((sgame.kref.k_right = vg4->input->key_insert(vg4->mlang->get("menu", "Move right"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_right, VG_INPUT_KBDCODE_RCURS);
  vg4->input->key_setgc(sgame.kref.k_right, 0, VG_INPUT_GCAXIS_LEFTX_RIGHT);
  /* shoot with spacebar, changeable, local key */
  if ((sgame.kref.k_shoot = vg4->input->key_insert(vg4->mlang->get("menu", "Shoot"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_shoot, VG_INPUT_KBDCODE_SPACE);
  vg4->input->key_setgc(sgame.kref.k_shoot, 0, VG_INPUT_GCBUTTON_RIGHTSHOULDER);

  /* create collision-tag with size of window */
  sgame.coll_tag = vg4->collision->create(NULL, 0, 0);

  /* load settings */
  settings_load(&sgame);

  /* show cover image */
  if (!show_cover()) { VG_dest(); exit(0); }

main_menu:
  if (!cvexec_main_menu(&sgame)) { goto endgame; }

  /* create level-global object-instances */
  if (!gamelevel_before(&sgame)) { goto endgame; }

  /* level loop */
  gplay = GAMEPLAY_NOOP;
  for (;; sgame.iset_new.idx_level++) {
    /* actualize settings */
    sgame.iset_old = sgame.iset_new;

    /* get level data */
    if (!leveldata(sgame.iset_new.idx_level + 1, &glevel, sgame.iset_new.difficulty)) { gplay = GAMEPLAY_DONE; break; }

    /* show intersection */
    snprintf(buf, sizeof(buf), "files/film/zw_seq/%d", sgame.iset_new.idx_level + 1);
    if (!vg4->film->play(buf, NULL, &filmskip, NULL)) { goto endgame; }

    /* create level-local object-instances */
    if (!gamelevel_start(&sgame, &glevel)) { goto endgame; }

    /* load and play background music looping */
    snprintf(buf, sizeof(buf), "files/audio/m%d.mid", sgame.iset_new.idx_level / 2 % 5 + 1);
    audc_bgmusic = vg4->audio->load(buf, 100, VG_AUDIO_VOLUME_MUSIC);
    vg4->audio->play(audc_bgmusic, VG_TRUE, VG_FALSE);

    /* game loop */
    for (;;) {
      if (!vg4->input->update(VG_TRUE)) { goto endgame; }

      /* quit to main-menu? */
      if (vg4->input->key_newpressed(sgame.kref.k_quit_q) && vg4->input->key_pressed(sgame.kref.k_quit_lalt)) {
        gplay = GAMEPLAY_MENU;
        break;
      }

      /* help? */
      if (vg4->input->key_newpressed(sgame.kref.k_help)) {
        if (!show_help()) { goto endgame; }
      }

      /* pause? */
      if (vg4->input->key_newpressed(sgame.kref.k_pause)) {
        if (!vg4->misc->pause()) { goto endgame; }
      }

      /* system-menu? */
      if (vg4->input->key_newpressed(sgame.kref.k_sysmenu)) {
        if (!dgexec_sysmenu()) { goto endgame; }
      }

      /* call object-management-function f_run() for all object-managements */
      vg4->object->call_mgmt_run(&sgame);

      /* call f_run() of all object-instances */
      if (!vg4->object->call_run(&sgame)) {
        /* player is dead */
        gplay = GAMEPLAY_DEAD;
      }

      /* call f_draw() of all object-instances */
      vg4->window->clear();
      vg4->object->call_draw(&sgame);

      /* draw bottom-info */
      vg4->object->call_mgmt_data(&sgame, OBJID_ESNAKE, &esnake_remain);
      draw_bottom(&sgame, esnake_remain);

      /* flush contents to window and wait */
      vg4->window->flush();
      vg4->misc->wait_time(40);

      /* player is dead? */
      if (gplay == GAMEPLAY_DEAD) { break; }

      /* level clear? */
      if (gamelevel_done(esnake_remain)) { break; }
    }

    if (gplay == GAMEPLAY_NOOP) {  /* show level ending */
      vg4->audio->stop(audc_bgmusic, VG_TRUE);
      if (!level_clear(sgame.iset_new.idx_level + 1, audc_bgmusic)) { goto endgame; }
    } else if (gplay == GAMEPLAY_DEAD) {  /* show game over */
      vg4->audio->stop(audc_bgmusic, VG_TRUE);
      if (!game_over(audc_bgmusic)) { goto endgame; }
    }

    /* stop and unload background music */
    vg4->audio->stop(audc_bgmusic, VG_FALSE);
    vg4->audio->unload(audc_bgmusic);

    /* destroy level-local object-instances */
    gamelevel_stop(&sgame);

    if (gplay != GAMEPLAY_NOOP) { break; }
  }

  /* reset window parameters */
  vg4->window->setattr(NULL);

  /* actualize settings */
  sgame.iset_new = sgame.iset_old;

  /* destroy level-global object-instances */
  gamelevel_after(&sgame);

  if (gplay == GAMEPLAY_DONE) {  /* winner */
    if (!vg4->film->play("files/film/extro", NULL, NULL, NULL)) { goto endgame; }
    settings_data_default(&sgame);
  }

  goto main_menu;

endgame:
  /* save settings */
  settings_save(&sgame);

  /* destroy and exit */
  VG_dest();
  exit(0);
}
