/* object OBJID_BOTTLE */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-bottle.h"

unsigned int objnew_BOTTLE(void *, int, int);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static void f_draw(void *, struct VG_Object *);


/* export-function to create a new object-instance of OBJID_BOTTLE
 * @param vgame       private structure of the game
 * @param bottle_no   number of loops poison-bottle is inactive
 * @param bottle_yes  number of loops poison-bottle is active
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_BOTTLE(void *vgame, int bottle_no, int bottle_yes)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_bottle *objvars;

  if (sgame == NULL || bottle_no <= 0 || bottle_yes < 0) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  objvars->bottle_no = bottle_no;
  objvars->bottle_yes = bottle_yes;
  objvars->bottle_loop = -bottle_no;

  /* load sprite */
  objvars->sprt = vg4->sprite->load("files/sprites/giftfl.sprite");
  if (objvars->sprt == NULL) { f_free(vgame, objvars); return 0; }
  vg4->sprite->imagesize(objvars->sprt, "avg", &objvars->rect.w, &objvars->rect.h);

  /* load sound */
  objvars->snd_bottle = vg4->audio->load("files/audio/giftflasche.wav", 100, VG_AUDIO_VOLUME_SOUND);

  /* create object-instance */
  objp = vg4->object->create(OBJID_BOTTLE, 0, 0, 2, objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_draw = f_draw;

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct sobj_bottle *objvars = (struct sobj_bottle *)opriv;

  (void)vgame;

  /* free private struct */
  if (objvars->sprt != NULL) { vg4->sprite->destroy(objvars->sprt); }
  if (objvars->snd_bottle > 0) { vg4->audio->unload(objvars->snd_bottle); }
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_bottle *objvars = (struct sobj_bottle *)objp->opriv;

  if (sgame == NULL) { return VG_TRUE; }
  if (objvars->bottle_yes < 1) { return VG_TRUE; }

  objvars->bottle_loop++;

  if (objvars->bottle_loop == 0) {  /* activate bottle */
    /* insert bottle with actual position into collision-tag */
    vg4->collision->insert(sgame->coll_tag, objp->instanceid, &objvars->rect, 100);
    /* find an position without collisions */
    for (;;) {
      /* get random position */
      objvars->rect.x = vg4->random->get(OBJID_BOTTLE, sgame->grect.x, sgame->grect.x + sgame->grect.w - objvars->rect.w);
      objvars->rect.y = vg4->random->get(OBJID_BOTTLE, sgame->grect.y, sgame->grect.y + sgame->grect.h - objvars->rect.h);
      /* check for collisions: if none, use position */
      if (vg4->collision->setpos(sgame->coll_tag, objp->instanceid, &objvars->rect, NULL) == 0) { break; }
    }
    vg4->sprite->rewind(objvars->sprt);

  } else if (objvars->bottle_loop == objvars->bottle_yes) {  /* deactivate bottle */
    /* remove bottle from collision-tag */
    vg4->collision->remove(sgame->coll_tag, objp->instanceid);
    /* set number of loops to reactivate */
    objvars->bottle_loop = -objvars->bottle_no;
  }

  return VG_TRUE;
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct sobj_bottle *objvars = (struct sobj_bottle *)objp->opriv;
  struct VG_Image *imgp;
  struct VG_ImagecopyAttr iattr;

  (void)vgame;

  if (objvars->bottle_loop < 0) { return; }

  if (vg4->sprite->next(objvars->sprt, &imgp, &iattr) && imgp != NULL) {
    struct VG_Position posi;
    vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &objvars->rect), &iattr);
  }
}
