/* object OBJID_ESNAKE */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-esnake.h"

unsigned int objnew_ESNAKE(void *, int);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static void f_draw(void *, struct VG_Object *);
static void f_childexit(void *, struct VG_Object *, struct VG_Object *);

/* kinds of enemy snakes */
static struct es_kind es_kinds[] = {
  { 5, 1, 1, 6 },     /* Buschnatter */
  { 6, 1, 1, 8 },     /* gelbe Ringelotter */
  { 6, 2, 1, 9 },     /* gruene Buschotter */
  { 7, 1, 1, 12 },    /* Dschungelviper */
  { 8, 2, 1, 10 },    /* Bergnatter */
  { 7, 2, 1, 18 },    /* Nebelnatter */
  { 8, 3, 1, 11 },    /* gemeine Grabviper */
  { 9, 1, 2, 13 },    /* gescheckte Felsotter */
  { 8, 1, 2, 15 },    /* Nachtviper */
  { 12, 2, 2, 15 },   /* schmale Nadelviper */
};


/* export-function to create a new object-instance of OBJID_ESNAKE
 * @param vgame       private structure of the game
 * @param kind        kind of esnake
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_ESNAKE(void *vgame, int kind)
{     
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_esnake *objvars;
  struct VG_Rect rect;
  char buf[256];
      
  if (sgame == NULL) { return 0; }

  if (kind < 0 || kind >= (int)(sizeof(es_kinds) / sizeof(*es_kinds))) {
    fprintf(stderr, "Kind %d of esnake invalid\n", kind);
    return 0;
  }
  
  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  objvars->kind = es_kinds[kind];
  objvars->limb_anz = objvars->kind.limb_max;
  if (objvars->limb_anz > ESNAKE_LIMB_MAX) { objvars->limb_anz = ESNAKE_LIMB_MAX; }

  /* load images */
  objvars->img_head_short = vg4->image->load("files/images/ekopfshort.bmp");
  if (objvars->img_head_short == NULL) { f_free(vgame, objvars); return 0; }
  objvars->img_head_long = vg4->image->load("files/images/ekopflong.bmp");
  if (objvars->img_head_long == NULL) { f_free(vgame, objvars); return 0; }
  objvars->img_eye = vg4->image->load("files/images/eauge.bmp");
  if (objvars->img_eye == NULL) { f_free(vgame, objvars); return 0; }
  snprintf(buf, sizeof(buf), "files/images/eglied%d.bmp", kind);
  objvars->img_limb = vg4->image->load(buf);
  if (objvars->img_limb == NULL) { f_free(vgame, objvars); return 0; }

  /* create tongue images */
  /* up */
  objvars->img_tg[ES_MOVING_DIR_UP] = vg4->image->create(4, 5);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_UP], 0, 0, 0, 2, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_UP], 3, 0, 3, 2, VG_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 2; rect.h = 3;
  vg4->image->draw_rect(objvars->img_tg[ES_MOVING_DIR_UP], &rect, VG_COLOR_RED, VG_TRUE);
  /* right */
  objvars->img_tg[ES_MOVING_DIR_RIGHT] = vg4->image->create(5, 4);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_RIGHT], 2, 0, 4, 0, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_RIGHT], 2, 3, 4, 3, VG_COLOR_RED);
  rect.x = 0; rect.w = 3; rect.y = 1; rect.h = 2;
  vg4->image->draw_rect(objvars->img_tg[ES_MOVING_DIR_RIGHT], &rect, VG_COLOR_RED, VG_TRUE);
  /* down */
  objvars->img_tg[ES_MOVING_DIR_DOWN] = vg4->image->create(4, 5);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_DOWN], 0, 2, 0, 4, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_DOWN], 3, 2, 3, 4, VG_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 0; rect.h = 3;
  vg4->image->draw_rect(objvars->img_tg[ES_MOVING_DIR_DOWN], &rect, VG_COLOR_RED, VG_TRUE);
  /* left */
  objvars->img_tg[ES_MOVING_DIR_LEFT] = vg4->image->create(5, 4);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_LEFT], 0, 0, 2, 0, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[ES_MOVING_DIR_LEFT], 0, 3, 2, 3, VG_COLOR_RED);
  rect.x = 2; rect.w = 3; rect.y = 1; rect.h = 2;
  vg4->image->draw_rect(objvars->img_tg[ES_MOVING_DIR_LEFT], &rect, VG_COLOR_RED, VG_TRUE);

  /* create limbstream and set limbidx */
  { int wks, hks, wkl, hkl, wk, wl, hl, lidx;
    vg4->image->getsize(objvars->img_head_short, NULL, &wks, &hks);
    wks = (wks > hks ? wks : hks);
    vg4->image->getsize(objvars->img_head_long, NULL, &wkl, &hkl);
    wkl = (wkl > hkl ? wkl : hkl);
    wk = (wkl > wks ? wkl : wks);
    vg4->image->getsize(objvars->img_limb, NULL, &wl, &hl);
    wl = (wl > hl ? wl : hl);
    objvars->limbstream_size = objvars->kind.limb_max * wl - wl / 2 + 1 + wk / 2 + 1;
    objvars->limbstream = calloc(objvars->limbstream_size, sizeof(*objvars->limbstream));
    if (objvars->limbstream == NULL) { f_free(vgame, objvars); return 0; }
    objvars->limbidx[0] = wk / 2 + wk % 2 + wl / 2;
    for (lidx = 1; lidx < objvars->kind.limb_max; lidx++) {
      objvars->limbidx[lidx] = objvars->limbidx[lidx - 1] + wl;
    }
  }

  /* load audios */
  objvars->snd_hit = vg4->audio->load("files/audio/es_giftgetroffen.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->snd_dead = vg4->audio->load("files/audio/es_gefressen.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->snd_come = vg4->audio->load("files/audio/es_kommt.wav", 100, VG_AUDIO_VOLUME_SOUND);

  /* create object-instance */
  objp = vg4->object->create(OBJID_ESNAKE, 0, 0, 3, objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_draw = f_draw;
  objp->f_childexit = f_childexit;

  /* set position */
  { int wks, hks, wkl, hkl, ipos, iz1, iz2;
    vg4->image->getsize(objvars->img_head_short, NULL, &wks, &hks);
    vg4->image->getsize(objvars->img_head_long, NULL, &wkl, &hkl);
    objvars->rectc.rect.w = (wkl < wks ? wkl : wks);
    objvars->rectc.rect.h = (hkl < hks ? hkl : hks);
    iz1 = vg4->random->get(OBJID_ESNAKE, 1, 4);
    iz2 = vg4->random->get(OBJID_ESNAKE, 0, 1);
    if (iz1 == 1) {
      objvars->rectc.rect.x = sgame->grect.x + 1;
      objvars->rectc.rect.y = sgame->grect.y + 1;
      objvars->movdir = (iz2 ? ES_MOVING_DIR_RIGHT : ES_MOVING_DIR_DOWN);
    } else if (iz1 == 2) {
      objvars->rectc.rect.x = sgame->grect.x + sgame->grect.w - objvars->rectc.rect.w - 1;
      objvars->rectc.rect.y = sgame->grect.y + 1;
      objvars->movdir = iz2 ? ES_MOVING_DIR_LEFT : ES_MOVING_DIR_DOWN;
    } else if (iz1 == 3) {
      objvars->rectc.rect.x = sgame->grect.x + 1;
      objvars->rectc.rect.y = sgame->grect.y + sgame->grect.h - objvars->rectc.rect.h - 1;
      objvars->movdir = iz2 ? ES_MOVING_DIR_RIGHT : ES_MOVING_DIR_UP;
    } else {
      objvars->rectc.rect.x = sgame->grect.x + sgame->grect.w - objvars->rectc.rect.w - 1;
      objvars->rectc.rect.y = sgame->grect.y + sgame->grect.h - objvars->rectc.rect.h - 1;
      objvars->movdir = iz2 ? ES_MOVING_DIR_LEFT : ES_MOVING_DIR_UP;
    }
    objvars->keep_moving = -90;
    objvars->rand_poison = 100 / objvars->kind.poison_max;

    for (ipos = 0; ipos < objvars->limbstream_size; ipos++) {
      objvars->limbstream[ipos].x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
      objvars->limbstream[ipos].y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      objvars->limbstream[ipos].movdir = objvars->movdir;
    }
  }

  /* don't insert object-instance into collision-tag now */

  /* play coming-sound */
  vg4->audio->play(objvars->snd_come, VG_FALSE, VG_FALSE);

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{ 
  struct sobj_esnake *objvars = (struct sobj_esnake *)opriv;
  
  (void)vgame;

  /* free private struct */
  if (objvars->limbstream != NULL) { free(objvars->limbstream); }
  if (objvars->snd_hit > 0) { vg4->audio->unload(objvars->snd_hit); }
  if (objvars->snd_dead > 0) { vg4->audio->unload(objvars->snd_dead); }
  if (objvars->snd_come > 0) { vg4->audio->unload(objvars->snd_come); }
  if (objvars->img_head_short != NULL) { vg4->image->destroy(objvars->img_head_short); }
  if (objvars->img_head_long != NULL) { vg4->image->destroy(objvars->img_head_long); }
  if (objvars->img_eye != NULL) { vg4->image->destroy(objvars->img_eye); }
  if (objvars->img_limb != NULL) { vg4->image->destroy(objvars->img_limb); }
  if (objvars->img_tg[ES_MOVING_DIR_UP] != NULL) { vg4->image->destroy(objvars->img_tg[ES_MOVING_DIR_UP]); }
  if (objvars->img_tg[ES_MOVING_DIR_RIGHT] != NULL) { vg4->image->destroy(objvars->img_tg[ES_MOVING_DIR_RIGHT]); }
  if (objvars->img_tg[ES_MOVING_DIR_DOWN] != NULL) { vg4->image->destroy(objvars->img_tg[ES_MOVING_DIR_DOWN]); }
  if (objvars->img_tg[ES_MOVING_DIR_LEFT] != NULL) { vg4->image->destroy(objvars->img_tg[ES_MOVING_DIR_LEFT]); }
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  const int coll_percent = 80;
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_esnake *objvars = (struct sobj_esnake *)objp->opriv;
  int angle, i1;

  if (sgame == NULL) { return VG_TRUE; }

  /* did not yet come out */
  if (objvars->keep_moving < 0) {
    if (++objvars->keep_moving == 0) {
      /* now insert object-instance into collision-tag */
      vg4->collision->insert(sgame->coll_tag, objp->instanceid, &objvars->rectc.rect, coll_percent);
    }
    return VG_TRUE;
  }

  /* destroy esnake, if it has no more limbs and snd_dead-sound is over */
  if (objvars->limb_anz == 0) {
    if (!vg4->audio->is_playing(objvars->snd_dead, NULL)) { vg4->object->destroy(vgame, objp->instanceid); }
    return VG_TRUE;
  }

  /* been eaten: no moving */
  if (objvars->dying > 0) {
    if (--objvars->dying == 0) {
      vg4->object->destroy(vgame, objp->instanceid);
    } else {
      objvars->limb_anz = objvars->dying / 10 + 1;
    }
    return VG_TRUE;
  }

  /* decrement camhat */
  if (objvars->camhat > 0) { objvars->camhat--; }

  /* change moving direction? */
  if (--objvars->keep_moving <= 0) {
    for (;;) {
      i1 = vg4->random->get(OBJID_ESNAKE, 0, 3);
      if (i1 != objvars->movdir) { objvars->movdir = i1; break; }
    }
    objvars->keep_moving = vg4->random->get(OBJID_ESNAKE, 30, 200);
  }
  angle = objvars->movdir * 90;

  /* spit poison? */
  if (objvars->rand_poison > 0) { objvars->rand_poison--; }
  if (vg4->random->get(OBJID_ESNAKE, 0, 30 + objvars->rand_poison * 5) <= 2) {
    if (objvars->poison_shots < objvars->kind.poison_max) {
      int shot_x, shot_y, shot_angle;
      shot_x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
      shot_y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      shot_angle = objvars->movdir * 90;
      if (objnew_SHOT(vgame, objp->instanceid, shot_x, shot_y, shot_angle, objvars->kind.poison_strength) > 0) {
        objvars->poison_shots++;
        objvars->rand_poison = 100 / objvars->kind.poison_max;
      }
    }
  }

  /* move and check for collisions */
  { struct VG_RectCent *rectcp;
    int rectc_nr;

    rectc_nr = vg4->misc->moving_one_step(&objvars->rectc, angle, objvars->kind.moving_factor, &rectcp);

    if (!vg4->misc->move_and_check_collisions(vgame, objp->instanceid, &objvars->rectc, sgame->coll_tag, rectcp, rectc_nr)) {
      ; /* object-instance is dead: ignore it here */
    }

    if (rectcp != NULL) { free(rectcp); }
  }

  /* move limbs */
  i1 = objvars->rectc.rect.x + objvars->rectc.rect.w / 2 - objvars->limbstream[0].x;
  if (i1 == 0) { i1 = objvars->rectc.rect.y + objvars->rectc.rect.h / 2 - objvars->limbstream[0].y; }
  if (i1 < 0) { i1 = -i1; }
  for (; i1 > 0; i1--) {
    memmove(&objvars->limbstream[1], &objvars->limbstream[0], sizeof(*objvars->limbstream) * (objvars->limbstream_size - 1));
    objvars->limbstream[0].x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
    objvars->limbstream[0].y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
    objvars->limbstream[0].movdir = objvars->movdir;
  }

  return VG_TRUE;
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_esnake *objvars = (struct sobj_esnake *)objp->opriv;
  struct VG_ImagecopyAttr iattr;
  struct VG_Position posi;

  /* esnake is coming? */
  if (objvars->keep_moving < 0) {
    struct VG_Rect rect;
    int iprz = (-(objvars->keep_moving % 30) / 3 + 1) * 10;
    rect.w = objvars->rectc.rect.w * iprz / 100;
    rect.h = objvars->rectc.rect.h * iprz / 100;
    rect.x = objvars->rectc.rect.x + (objvars->rectc.rect.w - rect.w) / 2;
    rect.y = objvars->rectc.rect.y + (objvars->rectc.rect.h - rect.h) / 2;
    vg4->window->draw_rect(&rect, VG_COLOR_BLACK, VG_TRUE);
    return;
  }

  /* draw limbs */
  if (objvars->camhat == 0 || (objvars->camhat < 100 && objvars->camhat % 20 < 5)) {
    int lidx;
    for (lidx = objvars->limb_anz - 1; lidx >= 0; lidx--) {
      VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
      iattr.image.rotate = objvars->limbstream[objvars->limbidx[lidx]].movdir * 90;
      posi.x = objvars->limbstream[objvars->limbidx[lidx]].x;
      posi.y = objvars->limbstream[objvars->limbidx[lidx]].y;
      posi.pos = VG_POS_CENTERED;
      vg4->window->copy(objvars->img_limb, &posi, &iattr);
    }
  }

  /* draw head */
  VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
  iattr.image.rotate = objvars->movdir * 90;
  if (objvars->camhat == 0) {  /* draw head */
    struct data_exchange dtex;
    memset(&dtex, 0, sizeof(dtex));
    dtex.type = DTEX_LIMBS;
    vg4->object->call_data(sgame, sgame->isnake_instanceid, &dtex);
    if (dtex.u.limb_anz < objvars->limb_anz) {  /* draw red head */
      vg4->window->copy(objvars->img_head_long, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &iattr);
    } else {  /* draw green head */
      vg4->window->copy(objvars->img_head_short, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &iattr);
    }
  } else {  /* draw eyes */
    vg4->window->copy(objvars->img_eye, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &iattr);
  }

  /* draw tongue */
  if (objvars->camhat == 0 && objvars->limb_anz > 0) {
    if (vg4->random->get(OBJID_ESNAKE, 1, 12) == 1) {
      int isize;
      posi.pos = VG_POS_CENTERED;
      posi.x = posi.y = 0;
      if (objvars->movdir == ES_MOVING_DIR_UP) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, NULL, &isize);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
        posi.y = objvars->rectc.rect.y - isize / 2;
      } else if (objvars->movdir == ES_MOVING_DIR_RIGHT) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, &isize, NULL);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w + isize / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      } else if (objvars->movdir == ES_MOVING_DIR_DOWN) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, NULL, &isize);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h + isize / 2;
      } else if (objvars->movdir == ES_MOVING_DIR_LEFT) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, &isize, NULL);
        posi.x = objvars->rectc.rect.x - isize / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      }
      vg4->window->copy(objvars->img_tg[objvars->movdir], &posi, NULL);
    }
  }
}


/* a child is being destroyed, called from child's vg4->object->destroy() */
static void
f_childexit(void *vgame, struct VG_Object *objp, struct VG_Object *child_objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_esnake *objvars = (struct sobj_esnake *)objp->opriv;
  
  if (sgame == NULL) { return; }

  if (strcmp(child_objp->objid, OBJID_SHOT) != 0) { return; }

  /* decrement esnake's running shots */
  objvars->poison_shots--;
}
