/* object OBJID_GROUND */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-ground.h"

unsigned int objnew_GROUND(void *, struct g_level *);

static void f_free(void *, void *);
static void f_draw(void *, struct VG_Object *);

static struct VG_Image * create_ground(struct s_game *, struct g_level *);


/* export-function to create a new object-instance of OBJID_GROUND
 * @param vgame     private structure of the game
 * @param glevel    level-data of actual level
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_GROUND(void *vgame, struct g_level *glevel)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_ground *objvars;

  if (sgame == NULL || glevel == NULL) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  objvars->imgp = create_ground(sgame, glevel);
  if (objvars->imgp == NULL) { f_free(vgame, objvars); return 0; }

  /* create object-instance */
  objp = vg4->object->create(OBJID_GROUND, 0, 0, 1, objvars);
  objp->f_free = f_free;
  objp->f_draw = f_draw;

  return objp->instanceid;
}


/* create ground image, create wall-objects and draw them on the ground image, return it */
static struct VG_Image *
create_ground(struct s_game *sgame, struct g_level *glevel)
{
  const int wall_x = 7;  /* number of wall-columns in x-direction */
  const int wall_y = 4;  /* number of wall-columns in y-direction */
  const int nowall_x = 2;  /* space-wall-columns between wall-columns in x-direction */
  const int nowall_y = 2;  /* space-wall-columns between wall-columns in y-direction */
  const int wall_conn = wall_x * wall_y / 2;  /* number of connections between wall-columns = walls */
  int xwall[wall_x], ywall[wall_y], xpos, ypos, i1, i2;
  char buf[256];
  struct VG_ImagecopyAttr iattr;
  struct VG_Image *wallimg, *bgimg, *groundimg;
  int wallsize;
  struct VG_Rect rect, wallrect;
  struct VG_Position posi;

  if (sgame == NULL || glevel == NULL) { return NULL; }

  wallimg = bgimg = groundimg = NULL;

  /* background image */
  snprintf(buf, sizeof(buf), "files/images/bgimage%d.bmp", glevel->bgno);
  bgimg = vg4->image->load(buf);
  if (bgimg == NULL) { goto endgr; }

  /* wall image */
  snprintf(buf, sizeof(buf), "files/images/wall%d.bmp", sgame->iset_new.idx_level / 2 % 5 + 1);
  wallimg = vg4->image->load(buf);
  if (wallimg == NULL) { goto endgr; }
  vg4->image->getsize(wallimg, NULL, &wallrect.w, &wallrect.h);
  if (wallrect.w != wallrect.h) { fprintf(stderr, "Wall must be a square\n"); goto endgr; }
  wallsize = (wallrect.w > wallrect.h ? wallrect.w : wallrect.h);

  /* ground image */
  groundimg = vg4->image->create(sgame->winw, sgame->winh);
  if (groundimg == NULL) { goto endgr; }
  vg4->image->fill(groundimg, glevel->bgcolor);

  VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
  iattr.pixel.brightness = glevel->bgbright;
  vg4->image->copy(groundimg, bgimg, NULL, &iattr);

  /* +++ create walls and draw them on the ground image +++ */

  /* x-position of walls */
  for (xpos = 0; xpos < wall_x; xpos++) {
    xwall[xpos] = (sgame->winw - ((wall_x + 1) * (wallsize * (nowall_x + 1)) + wallsize)) / 2
                   + (xpos + 1) * (wallsize * (nowall_x + 1)) + wallsize / 2;
  }

  /* y-position of walls */
  for (ypos = 0; ypos < wall_y; ypos++) {
    ywall[ypos] = (ypos + 1) * (wallsize * (nowall_y + 1)) + wallsize / 2;
  }

  /* create and draw wall-columns */
  for (ypos = 0; ypos < wall_y; ypos++) {
    for (xpos = 0; xpos < wall_x; xpos++) {
      wallrect.x = xwall[xpos] - wallsize / 2;
      wallrect.y = ywall[ypos] - wallsize / 2;
      if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
      vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
    }
  }

  /* create and draw horizontal walls */
  for (i1 = 0; i1 < wall_conn / 2; i1++) {
    xpos = vg4->random->get("WALL_ON_GROUND", 0, wall_x - 2);
    ypos = vg4->random->get("WALL_ON_GROUND", 0, wall_y - 1);
    for (i2 = 1; i2 <= nowall_x; i2++) {
      wallrect.x = xwall[xpos] + (i2 * wallsize) - wallsize / 2;
      wallrect.y = ywall[ypos] - wallsize / 2;
      if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
      vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
    }
  }

  /* create and draw vertical walls */
  for (i1 = 0; i1 < wall_conn / 2; i1++) {
    xpos = vg4->random->get("WALL_ON_GROUND", 0, wall_x - 1);
    ypos = vg4->random->get("WALL_ON_GROUND", 0, wall_y - 2);
    for (i2 = 1; i2 <= nowall_y; i2++) {
      wallrect.x = xwall[xpos] - wallsize / 2;
      wallrect.y = ywall[ypos] + (i2 * wallsize) - wallsize / 2;
      if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
      vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
    }
  }

  /* draw horizontal borders */
  for (i1 = 0; i1 < (wall_x * (nowall_x + 1)) + nowall_x + 2; i1++) {
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - ((nowall_y + 1) * wallsize) - wallsize / 2;
    if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
    vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    wallrect.y = ywall[wall_y - 1] + (nowall_y + 1) * wallsize - wallsize / 2;
    if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
    vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
  }

  /* draw vertical borders */
  for (i1 = 0; i1 < (wall_y * (nowall_y + 1)) + nowall_y + 2; i1++) {
    wallrect.x = xwall[0] - ((nowall_x + 1) * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - ((nowall_y + 1) * wallsize) + (i1 * wallsize) - wallsize / 2;
    if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
    vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
    wallrect.x = xwall[wall_x - 1] + ((nowall_x + 1) * wallsize) - wallsize / 2;
    wallrect.y = ywall[0] - (nowall_y + 1) * wallsize + (i1 * wallsize) - wallsize / 2;
    if (objnew_WALL(sgame, &wallrect) == 0) { goto endgr; }
    vg4->image->copy(groundimg, wallimg, vg4->misc->rect2position(&posi, &wallrect), NULL);
  }

  /* +++ make outside of ground black +++ */

  rect.x = 0;
  rect.w = xwall[0] - ((nowall_x + 1) * wallsize) - (wallsize / 2);
  rect.y = 0;
  rect.h = sgame->winh;
  vg4->image->draw_rect(groundimg, &rect, VG_COLOR_BLACK, VG_TRUE);

  rect.x = xwall[0] - ((nowall_x + 1) * wallsize) + (((wall_x * (nowall_x + 1)) + nowall_x + 2) * wallsize) - (wallsize / 2);
  rect.w = sgame->winw - rect.x;
  rect.y = 0;
  rect.h = sgame->winh;
  vg4->image->draw_rect(groundimg, &rect, VG_COLOR_BLACK, VG_TRUE);

  rect.x = 0;
  rect.w = sgame->winw;
  rect.y = ywall[0] - (nowall_y + 1) * wallsize + (((wall_y * (nowall_y + 1)) + nowall_y + 2) * wallsize) - (wallsize / 2);
  rect.h = sgame->winh - rect.y;
  vg4->image->draw_rect(groundimg, &rect, VG_COLOR_BLACK, VG_TRUE);

  /* rectangle of ground */
  sgame->grect.x = xwall[0] - ((nowall_x + 1) * wallsize) - (wallsize / 2) + wallsize;
  sgame->grect.w = (xwall[0] - ((nowall_x + 1) * wallsize) + (((wall_x * (nowall_x + 1)) + nowall_x + 2) * wallsize) - (wallsize / 2)) - sgame->grect.x - wallsize;
  sgame->grect.y = ywall[0] - ((nowall_y + 1) * wallsize - (wallsize / 2));
  sgame->grect.h = (ywall[0] - ((nowall_y + 1) * wallsize) + (((wall_y * (nowall_y + 1)) + nowall_y + 2) * wallsize) - (wallsize / 2)) - sgame->grect.y - wallsize;

  /* done */
  vg4->image->destroy(bgimg);
  vg4->image->destroy(wallimg);
  return groundimg;

endgr:
  if (bgimg != NULL) { vg4->image->destroy(bgimg); }
  if (wallimg != NULL) { vg4->image->destroy(wallimg); }
  if (groundimg != NULL) { vg4->image->destroy(groundimg); }

  return NULL;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct sobj_ground *objvars = (struct sobj_ground *)opriv;

  /* destroy all walls */
  vg4->object->destroy_objid(vgame, OBJID_WALL, 0);

  /* free private struct */
  if (objvars->imgp != NULL) { vg4->image->destroy(objvars->imgp); }
  free(objvars);
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct sobj_ground *objvars = (struct sobj_ground *)objp->opriv;

  (void)vgame;

  vg4->window->copy(objvars->imgp, NULL, NULL);
}
