/* object OBJID_ISNAKE */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-isnake.h"

unsigned int objnew_ISNAKE(void *);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static void f_draw(void *, struct VG_Object *);
static int f_data(void *, struct VG_Object *, void *);
static void f_childexit(void *, struct VG_Object *, struct VG_Object *);


/* export-function to create a new object-instance of OBJID_ISNAKE
 * @param vgame       private structure of the game
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_ISNAKE(void *vgame)
{
  const int coll_percent = 80;
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_isnake *objvars;
  struct VG_Rect rect;

  if (sgame == NULL) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  objvars->limb_anz = (ISNAKE_LIMB_MAX * sgame->iset_new.isnake_health + 50) / 100;
  if (objvars->limb_anz > ISNAKE_LIMB_MAX) { objvars->limb_anz = ISNAKE_LIMB_MAX; }

  /* load images */
  objvars->img_head = vg4->image->load("files/images/ikopf.bmp");
  if (objvars->img_head == NULL) { f_free(vgame, objvars); return 0; }
  objvars->img_dead = vg4->image->load("files/images/tkopf.bmp");
  if (objvars->img_dead == NULL) { f_free(vgame, objvars); return 0; }
  objvars->img_eye = vg4->image->load("files/images/iauge.bmp");
  if (objvars->img_eye == NULL) { f_free(vgame, objvars); return 0; }
  objvars->img_limb = vg4->image->load("files/images/iglied.bmp");
  if (objvars->img_limb == NULL) { f_free(vgame, objvars); return 0; }

  /* create tongue images */
  /* up */
  objvars->img_tg[IS_MOVING_DIR_UP] = vg4->image->create(4, 5);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_UP], 0, 0, 0, 2, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_UP], 3, 0, 3, 2, VG_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 2; rect.h = 3;
  vg4->image->draw_rect(objvars->img_tg[IS_MOVING_DIR_UP], &rect, VG_COLOR_RED, VG_TRUE);
  /* right */
  objvars->img_tg[IS_MOVING_DIR_RIGHT] = vg4->image->create(5, 4);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_RIGHT], 2, 0, 4, 0, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_RIGHT], 2, 3, 4, 3, VG_COLOR_RED);
  rect.x = 0; rect.w = 3; rect.y = 1; rect.h = 2;
  vg4->image->draw_rect(objvars->img_tg[IS_MOVING_DIR_RIGHT], &rect, VG_COLOR_RED, VG_TRUE);
  /* down */
  objvars->img_tg[IS_MOVING_DIR_DOWN] = vg4->image->create(4, 5);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_DOWN], 0, 2, 0, 4, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_DOWN], 3, 2, 3, 4, VG_COLOR_RED);
  rect.x = 1; rect.w = 2; rect.y = 0; rect.h = 3;
  vg4->image->draw_rect(objvars->img_tg[IS_MOVING_DIR_DOWN], &rect, VG_COLOR_RED, VG_TRUE);
  /* left */
  objvars->img_tg[IS_MOVING_DIR_LEFT] = vg4->image->create(5, 4);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_LEFT], 0, 0, 2, 0, VG_COLOR_RED);
  vg4->image->draw_line(objvars->img_tg[IS_MOVING_DIR_LEFT], 0, 3, 2, 3, VG_COLOR_RED);
  rect.x = 2; rect.w = 3; rect.y = 1; rect.h = 2;
  vg4->image->draw_rect(objvars->img_tg[IS_MOVING_DIR_LEFT], &rect, VG_COLOR_RED, VG_TRUE);

  /* create limbstream and set limbidx */
  { int wk, hk, wl, hl, lidx;
    vg4->image->getsize(objvars->img_head, NULL, &wk, &hk);
    wk = (wk > hk ? wk : hk);
    vg4->image->getsize(objvars->img_limb, NULL, &wl, &hl);
    wl = (wl > hl ? wl : hl);
    objvars->limbstream_size = ISNAKE_LIMB_MAX * wl - wl / 2 + 1 + wk / 2 + 1;
    objvars->limbstream = calloc(objvars->limbstream_size, sizeof(*objvars->limbstream));
    if (objvars->limbstream == NULL) { f_free(vgame, objvars); return 0; }
    objvars->limbidx[0] = wk / 2 + wk % 2 + wl / 2;
    for (lidx = 1; lidx < ISNAKE_LIMB_MAX; lidx++) {
      objvars->limbidx[lidx] = objvars->limbidx[lidx - 1] + wl;
    }
  }

  /* load audios */
  objvars->snd_hit = vg4->audio->load("files/audio/is_giftgetroffen.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->snd_spit = vg4->audio->load("files/audio/is_giftspritzen.wav", 100, VG_AUDIO_VOLUME_SOUND);

  /* create object-instance */
  objp = vg4->object->create(OBJID_ISNAKE, 0, 0, 4, objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_draw = f_draw;
  objp->f_data = f_data;
  objp->f_childexit = f_childexit;

  /* set position */
  { struct data_exchange dtex;
    memset(&dtex, 0, sizeof(dtex));
    dtex.type = DTEX_RESET;
    f_data(vgame, objp, &dtex);
  }

  /* insert object-instance into collision-tag */
  vg4->collision->insert(sgame->coll_tag, objp->instanceid, &objvars->rectc.rect, coll_percent);

  sgame->isnake_instanceid = objp->instanceid;

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct sobj_isnake *objvars = (struct sobj_isnake *)opriv;

  (void)vgame;

  /* free private struct */
  if (objvars->limbstream != NULL) { free(objvars->limbstream); }
  if (objvars->snd_hit > 0) { vg4->audio->unload(objvars->snd_hit); }
  if (objvars->snd_spit > 0) { vg4->audio->unload(objvars->snd_spit); }
  if (objvars->img_head != NULL) { vg4->image->destroy(objvars->img_head); }
  if (objvars->img_dead != NULL) { vg4->image->destroy(objvars->img_dead); }
  if (objvars->img_eye != NULL) { vg4->image->destroy(objvars->img_eye); }
  if (objvars->img_limb != NULL) { vg4->image->destroy(objvars->img_limb); }
  if (objvars->img_tg[IS_MOVING_DIR_UP] != NULL) { vg4->image->destroy(objvars->img_tg[IS_MOVING_DIR_UP]); }
  if (objvars->img_tg[IS_MOVING_DIR_RIGHT] != NULL) { vg4->image->destroy(objvars->img_tg[IS_MOVING_DIR_RIGHT]); }
  if (objvars->img_tg[IS_MOVING_DIR_DOWN] != NULL) { vg4->image->destroy(objvars->img_tg[IS_MOVING_DIR_DOWN]); }
  if (objvars->img_tg[IS_MOVING_DIR_LEFT] != NULL) { vg4->image->destroy(objvars->img_tg[IS_MOVING_DIR_LEFT]); }
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  const int moving_factor = 10;
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_isnake *objvars = (struct sobj_isnake *)objp->opriv;
  int angle;

  if (sgame == NULL) { return VG_TRUE; }

  /* game over, if isnake has no more limbs and snd_hit-sound is over */
  if (objvars->limb_anz == 0) {
    if (!vg4->audio->is_playing(objvars->snd_hit, NULL)) { return VG_FALSE; }
    return VG_TRUE;
  }

  /* eating an enemy snake: no moving */
  if (objvars->eating > 0) { objvars->eating--; return VG_TRUE; }

  /* decrement camhat */
  if (objvars->camhat > 0) { objvars->camhat--; }

  /* check for key-strokes */
  angle = -1;
  if (angle < 0 && vg4->input->key_pressed(sgame->kref.k_up)) {
    objvars->movdir = IS_MOVING_DIR_UP;
    angle = objvars->movdir * 90;
  }
  if (angle < 0 && vg4->input->key_pressed(sgame->kref.k_right)) {
    objvars->movdir = IS_MOVING_DIR_RIGHT;
    angle = objvars->movdir * 90;
  }
  if (angle < 0 && vg4->input->key_pressed(sgame->kref.k_down)) {
    objvars->movdir = IS_MOVING_DIR_DOWN;
    angle = objvars->movdir * 90;
  }
  if (angle < 0 && vg4->input->key_pressed(sgame->kref.k_left)) {
    objvars->movdir = IS_MOVING_DIR_LEFT;
    angle = objvars->movdir * 90;
  }
  if (vg4->input->key_newpressed(sgame->kref.k_shoot) && !objvars->poison_shot) {
    int shot_x, shot_y, shot_angle;
    shot_x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
    shot_y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
    shot_angle = objvars->movdir * 90;
    if (objnew_SHOT(vgame, objp->instanceid, shot_x, shot_y, shot_angle, 0) > 0) {
      objvars->poison_shot = VG_TRUE;
      vg4->audio->play(objvars->snd_spit, VG_FALSE, VG_FALSE);
    }
  }

  /* move and check for collisions */
  if (angle >= 0) {
    struct VG_RectCent *rectcp;
    int rectc_nr;

    rectc_nr = vg4->misc->moving_one_step(&objvars->rectc, angle, moving_factor, &rectcp);

    if (!vg4->misc->move_and_check_collisions(vgame, objp->instanceid, &objvars->rectc, sgame->coll_tag, rectcp, rectc_nr)) {
      ; /* object-instance is dead: ignore it here */
    }

    if (rectcp != NULL) { free(rectcp); }
  }

  /* move limbs */
  if (objvars->rectc.rect.x + objvars->rectc.rect.w / 2 != objvars->limbstream[0].x
      || objvars->rectc.rect.y + objvars->rectc.rect.h / 2 != objvars->limbstream[0].y) {
    memmove(&objvars->limbstream[1], &objvars->limbstream[0], sizeof(*objvars->limbstream) * (objvars->limbstream_size - 1));
    objvars->limbstream[0].x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
    objvars->limbstream[0].y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
    objvars->limbstream[0].movdir = objvars->movdir;
  }

  return VG_TRUE;
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct sobj_isnake *objvars = (struct sobj_isnake *)objp->opriv;
  struct VG_ImagecopyAttr iattr;
  struct VG_Position posi;

  (void)vgame;

  /* draw limbs */
  if (objvars->camhat == 0 || (objvars->camhat < 100 && objvars->camhat % 20 < 5)) {
    int lidx;
    for (lidx = objvars->limb_anz - 1; lidx >= 0; lidx--) {
      VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
      iattr.image.rotate = objvars->limbstream[objvars->limbidx[lidx]].movdir * 90;
      posi.x = objvars->limbstream[objvars->limbidx[lidx]].x;
      posi.y = objvars->limbstream[objvars->limbidx[lidx]].y;
      posi.pos = VG_POS_CENTERED;
      vg4->window->copy(objvars->img_limb, &posi, &iattr);
    }
  }

  /* draw head */
  VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
  iattr.image.rotate = objvars->movdir * 90;
  if (objvars->limb_anz == 0) {  /* draw dead-head */
    vg4->window->copy(objvars->img_dead, vg4->misc->rect2position(&posi, &objvars->rectc.rect), NULL);
  } else if (objvars->camhat == 0) {  /* draw head */ 
    vg4->window->copy(objvars->img_head, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &iattr);
  } else {  /* draw eyes */
    vg4->window->copy(objvars->img_eye, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &iattr);
  }

  /* draw tongue */
  if (objvars->camhat == 0) {
    if (vg4->random->get(OBJID_ISNAKE, 1, 12) == 1) {
      int isize;
      posi.pos = VG_POS_CENTERED;
      posi.x = posi.y = 0;
      if (objvars->movdir == IS_MOVING_DIR_UP) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, NULL, &isize);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
        posi.y = objvars->rectc.rect.y - isize / 2;
      } else if (objvars->movdir == IS_MOVING_DIR_RIGHT) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, &isize, NULL);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w + isize / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      } else if (objvars->movdir == IS_MOVING_DIR_DOWN) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, NULL, &isize);
        posi.x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h + isize / 2;
      } else if (objvars->movdir == IS_MOVING_DIR_LEFT) {
        vg4->image->getsize(objvars->img_tg[objvars->movdir], NULL, &isize, NULL);
        posi.x = objvars->rectc.rect.x - isize / 2;
        posi.y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
      }
      vg4->window->copy(objvars->img_tg[objvars->movdir], &posi, NULL);
    }
  }
}


/* get/set data from/for object-instance, called from vg4->object->call_data() */
static int
f_data(void *vgame, struct VG_Object *objp, void *vptr)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_isnake *objvars = (struct sobj_isnake *)objp->opriv;
  struct data_exchange *dtexp = (struct data_exchange *)vptr;

  if (sgame == NULL) { return 0; }

  if (dtexp != NULL) {
    if (dtexp->type == DTEX_RESET) {
      /* set snake to origin place */
      int lidx;
      memset(&objvars->rectc, 0, sizeof(objvars->rectc));
      vg4->image->getsize(objvars->img_head, NULL, &objvars->rectc.rect.w, &objvars->rectc.rect.h);
      objvars->rectc.rect.x = sgame->grect.x + 2;
      objvars->rectc.rect.y = (sgame->winh - objvars->rectc.rect.h) / 2;
      objvars->movdir = IS_MOVING_DIR_RIGHT;
      objvars->camhat = 0;
      objvars->eating = 0;
      vg4->collision->setpos(sgame->coll_tag, objp->instanceid, &objvars->rectc.rect, NULL);

      for (lidx = 0; lidx < objvars->limbstream_size; lidx++) {
        objvars->limbstream[lidx].x = objvars->rectc.rect.x + objvars->rectc.rect.w / 2;
        objvars->limbstream[lidx].y = objvars->rectc.rect.y + objvars->rectc.rect.h / 2;
        objvars->limbstream[lidx].movdir = objvars->movdir;
      }

    } else if (dtexp->type == DTEX_LIMBS) {
      /* set actual number of limbs */
      dtexp->u.limb_anz = objvars->limb_anz;

    } else {
      memset(&dtexp->u, 0, sizeof(dtexp->u));
    }
  }

  return 0;
}


/* a child is being destroyed, called from child's vg4->object->destroy() */
static void
f_childexit(void *vgame, struct VG_Object *objp, struct VG_Object *child_objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_isnake *objvars = (struct sobj_isnake *)objp->opriv;

  if (sgame == NULL) { return; }

  if (strcmp(child_objp->objid, OBJID_SHOT) != 0) { return; }

  /* set snake's running shot to false */
  objvars->poison_shot = VG_FALSE;
}
