/* object OBJID_SHOT */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-shot.h"

unsigned int objnew_SHOT(void *, unsigned int, int, int, int, int);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static void f_draw(void *, struct VG_Object *);


/* export-function to create a new object-instance of OBJID_SHOT
 * @param vgame              private structure of the game
 * @param parent_instanceid  instance-ID of parent
 * @param xm                 x-position (centered)
 * @param ym                 y-position (centered)
 * @param angle              angle in degrees (0, 90, 180, 270)
 * @param shotimg            0=ishot, 1=eshot1, 2=eshot2
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_SHOT(void *vgame, unsigned int parent_instanceid, int xm, int ym, int angle, int shotimg)
{     
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_shot *objvars;

  if (sgame == NULL) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  objvars->angle = angle;
  VG_IMAGECOPY_ATTR_DEFAULT(&objvars->iattr);

  /* load image */
  if (shotimg == 0) {
    objvars->imgp = vg4->image->load("files/images/ishot.bmp");
    objvars->poison_strength = 1;
  } else if (shotimg == 1) {
    objvars->imgp = vg4->image->load("files/images/eshot1.bmp");
    objvars->poison_strength = 1;
  } else if (shotimg == 2) {
    objvars->imgp = vg4->image->load("files/images/eshot2.bmp");
    objvars->poison_strength = 2;
  }
  objvars->shotimg = shotimg;
  if (objvars->imgp == NULL) { f_free(vgame, objvars); return 0; }

  /* clone rotated image */
  { struct VG_Image *imgp;
    struct VG_ImagecopyAttr iattr;
    VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
    iattr.image.rotate = angle;
    imgp = objvars->imgp;
    objvars->imgp = vg4->image->clone(imgp, NULL, &iattr);
    vg4->image->destroy(imgp);
  }

  /* set position */
  vg4->image->getsize(objvars->imgp, NULL, &objvars->rectc.rect.w, &objvars->rectc.rect.h);
  objvars->rectc.rect.x = xm - objvars->rectc.rect.w / 2;
  objvars->rectc.rect.y = ym - objvars->rectc.rect.h / 2;

  /* create object-instance */
  objp = vg4->object->create(OBJID_SHOT, 0, parent_instanceid, 5, objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_draw = f_draw;

  /* insert object-instance into collision-tag */
  vg4->collision->insert(sgame->coll_tag, objp->instanceid, &objvars->rectc.rect, 100);

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{ 
  struct sobj_shot *objvars = (struct sobj_shot *)opriv;
  
  (void)vgame;

  /* free private struct */
  if (objvars->imgp != NULL) { vg4->image->destroy(objvars->imgp); }
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  const int moving_factor = 40;
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_shot *objvars = (struct sobj_shot *)objp->opriv;

  if (sgame == NULL) { return VG_TRUE; }

  /* move and check for collisions */
  { struct VG_RectCent *rectcp;
    int rectc_nr;

    rectc_nr = vg4->misc->moving_one_step(&objvars->rectc, objvars->angle, moving_factor, &rectcp);

    if (!vg4->misc->move_and_check_collisions(vgame, objp->instanceid, &objvars->rectc, sgame->coll_tag, rectcp, rectc_nr)) {
      /* object-instance is dead, destroy it */
      vg4->object->destroy(vgame, objp->instanceid);
      if (rectcp != NULL) { free(rectcp); }
      return VG_TRUE;
    }

    if (rectcp != NULL) { free(rectcp); }
  }

  if (objvars->fliploop == 0) {  /* flip at the vertical axe */
    if (objvars->iattr.image.flip == VG_AXE_NONE) {
      objvars->iattr.image.flip = VG_AXE_VERTICAL;
    } else {
      objvars->iattr.image.flip = VG_AXE_NONE;
    }
    objvars->fliploop = 5;
  }
  objvars->fliploop--;

  return VG_TRUE;
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct sobj_shot *objvars = (struct sobj_shot *)objp->opriv;
  struct VG_Position posi;

  (void)vgame;

  vg4->window->copy(objvars->imgp, vg4->misc->rect2position(&posi, &objvars->rectc.rect), &objvars->iattr);
}
