/* management of object OBJID_ESNAKE */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "main.h"

VG_BOOL objmgmt_ESNAKE(void *, int, int, int, int *);

static void f_free(void);
static void f_run(void *);
static int f_data(void *, void *);

/* internal structure for this object-management */
static struct {
  int esnake_aktiv;   /* minimum number of active enemy snakes */
  int esnake_max;     /* number of enemy snakes in this level */
  int *kind;          /* kind of each enemy snake */
  int loop_newsnake;  /* number of loops after that a new enemy snake appears */
  int kind_pos;       /* position in kind */
  int loop_pos;       /* position in loop_newsnake */
} mgmt_esnake;


/* export-function to create the management of OBJID_ESNAKE
 * @param vgame       private structure of the game
 * @param esnake_aktiv   minimum number of active enemy snakes
 * @param esnake_max     number of enemy snakes in this level
 * @param loop_newsnake  number of loops after that a new enemy snake appears
 * @param kind           array: kind of each enemy snake
 * @return  VG_TRUE = OK or VG_FALSE = Error
 */
VG_BOOL
objmgmt_ESNAKE(void *vgame, int esnake_aktiv, int esnake_max, int loop_newsnake, int *kind)
{     
  struct VG_ObjMgmt *omgmt;

  (void)vgame;

  /* set internal structure for this object-management */

  memset(&mgmt_esnake, 0, sizeof(mgmt_esnake));

  mgmt_esnake.esnake_aktiv = esnake_aktiv;
  if (mgmt_esnake.esnake_aktiv <= 0) { mgmt_esnake.esnake_aktiv = 0; }
  mgmt_esnake.esnake_max = esnake_max;
  if (mgmt_esnake.esnake_max <= 0) { fprintf(stderr, "Management of ESNAKE: esnake_max <= 0\n"); return VG_FALSE; }
  mgmt_esnake.loop_newsnake = loop_newsnake;
  if (mgmt_esnake.loop_newsnake <= 10) { mgmt_esnake.loop_newsnake = 10; }
  if (kind == NULL) { fprintf(stderr, "Management of ESNAKE: kind = NULL\n"); return VG_FALSE; }

  mgmt_esnake.kind = calloc(mgmt_esnake.esnake_max, sizeof(int));
  if (mgmt_esnake.kind == NULL) { fprintf(stderr, "calloc: %s\n", strerror(errno)); return VG_FALSE; }
  memcpy(mgmt_esnake.kind, kind, sizeof(int) * mgmt_esnake.esnake_max);

  /* create object-management */
  omgmt = vg4->object->mgmt_create(OBJID_ESNAKE);
  omgmt->f_free = f_free;
  omgmt->f_run = f_run;
  omgmt->f_data = f_data;

  return VG_TRUE;
}


/* reset internal structure for this object-management, called from vg4->object->mgmt_destroy() */
static void
f_free(void)
{ 
  if (mgmt_esnake.kind != NULL) { free(mgmt_esnake.kind); }
  memset(&mgmt_esnake, 0, sizeof(mgmt_esnake));
}


/* execute managing, called from vg4->object->call_mgmt_run() */
static void
f_run(void *vgame)
{
  (void)vgame;

  if (mgmt_esnake.kind_pos == mgmt_esnake.esnake_max) { return; }
  if (mgmt_esnake.loop_pos >= mgmt_esnake.loop_newsnake) { mgmt_esnake.loop_pos--; return; }

  if (++mgmt_esnake.loop_pos == mgmt_esnake.loop_newsnake) {
    /* create a new instance of OBJID_ESNAKE */
    if (objnew_ESNAKE(vgame, mgmt_esnake.kind[mgmt_esnake.kind_pos]) == 0) {
      /* error */
      mgmt_esnake.kind_pos = mgmt_esnake.esnake_max;
      return;
    }
    mgmt_esnake.kind_pos++;
    mgmt_esnake.loop_pos = 0;

  } else {
    int enr = vg4->object->list(OBJID_ESNAKE, 0, NULL);
    if (enr < mgmt_esnake.esnake_aktiv) {
      /* activate coming of the next esnake */
      mgmt_esnake.loop_pos = mgmt_esnake.loop_newsnake + vg4->random->get("MGMT-" OBJID_ESNAKE, 30, 90);
    }
  }
}


/* query remaining number of esnake-instances including active ones, called from vg4->object->call_mgmt_data() */
static int
f_data(void *vgame, void *vptr)
{
  int *enr = (int *)vptr;

  (void)vgame;
  if (enr == NULL) { return 0; }

  *enr = mgmt_esnake.esnake_max - mgmt_esnake.kind_pos;
  *enr += vg4->object->list(OBJID_ESNAKE, 0, NULL);

  return 0;
}
