/* load and save settings */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames4.h>
#include "main.h"

#define SETTINGS_FILENAME "snakes"

void settings_load(struct s_game *);
void settings_save(struct s_game *);
void settings_data_default(struct s_game *);

static char * get_file(void);
static void data_load(struct s_game *, FILE *);
static void data_save(struct s_game *, FILE *);


/* get name of save-file */
static char *
get_file(void)
{
  static char svfile[256] = "";

  if (*svfile == '\0') {
    char *home;
    if ((home = getenv("HOME")) == NULL) { home = "."; }
    snprintf(svfile, sizeof(svfile), "%s/.VgaGames4", home);
    mkdir(svfile, 0755);
    snprintf(svfile, sizeof(svfile), "%s/.VgaGames4/%s", home, SETTINGS_FILENAME);
  }

  return svfile;
}


/* load settings */
void
settings_load(struct s_game *sgame)
{
  char *fname, *pbuf;
  FILE *ffp;
  struct stat sbuf;
  size_t fsize;

  fname = get_file();
  ffp = fopen(fname, "r");
  if (ffp == NULL) { return; }

  if (fstat(fileno(ffp), &sbuf) < 0) {
    fclose(ffp);
    fprintf(stderr, "fstat %s: %s\n", fname, strerror(errno));
    return;
  }
  if (sbuf.st_size == 0) { fclose(ffp); return; }
  fsize = sbuf.st_size;

  pbuf = malloc(fsize);
  if (pbuf == NULL) {
    fclose(ffp);
    fprintf(stderr, "Reading settings-file %s: malloc: %s\n", fname, strerror(errno));
    return;
  }

  if (fgets(pbuf, fsize, ffp) == NULL) {
    fclose(ffp);
    free(pbuf);
    fprintf(stderr, "Cannot read settings-file %s: short read\n", fname);
    return;
  }

  fsize = strlen(pbuf);
  if (fsize > 0 && pbuf[fsize - 1] == '\n') {
    fsize--;
    if (fsize > 0 && pbuf[fsize - 1] == '\r') { fsize--; }
    pbuf[fsize] = '\0';
  }

  vg4->misc->settings_load(pbuf);
  free(pbuf);

  data_load(sgame, ffp);
  fclose(ffp);
}


/* save settings */
void
settings_save(struct s_game *sgame)
{
  char *fname, *sstr;
  FILE *ffp;

  fname = get_file();
  ffp = fopen(fname, "w");
  if (ffp == NULL) { return; }

  sstr = vg4->misc->settings_save(VG_SETTING_WINDOWSCALE | VG_SETTING_WINDOWBRIGHTNESS | VG_SETTING_VOLUME | VG_SETTING_KEYS);

  fprintf(ffp, "%s\n", sstr);
  free(sstr);

  data_save(sgame, ffp);
  fclose(ffp);
}


/* set additional individual data to default */
void
settings_data_default(struct s_game *sgame)
{
  if (sgame == NULL) { return; }
  sgame->iset_old.isnake_health = 100;
  sgame->iset_old.idx_level = 0;
  sgame->iset_old.difficulty = 0;
  sgame->iset_new = sgame->iset_old;
}


/* load additional individual data */
static void
data_load(struct s_game *sgame, FILE *ffp)
{
  char buf[64];

  if (sgame == NULL || ffp == NULL) { return; }

  /* last game state */
  settings_data_default(sgame);
  if (fgets(buf, sizeof(buf), ffp) == NULL) { return; }
  sgame->iset_old.isnake_health = atoi(buf);
  if (fgets(buf, sizeof(buf), ffp) == NULL) { return; }
  sgame->iset_old.idx_level = atoi(buf);
  if (fgets(buf, sizeof(buf), ffp) == NULL) { return; }
  sgame->iset_old.difficulty = atoi(buf);
  sgame->iset_new = sgame->iset_old;
}


/* save additional individual data */
static void
data_save(struct s_game *sgame, FILE *ffp)
{
  if (sgame == NULL || ffp == NULL) { return; }

  fprintf(ffp, "%d\n", sgame->iset_old.isnake_health);
  fprintf(ffp, "%d\n", sgame->iset_old.idx_level);
  fprintf(ffp, "%d\n", sgame->iset_old.difficulty);
}
