/* Copyright 2025 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "main.h"

/* settings.c: load and save system-settings */
extern void settings_load(void);
extern void settings_save(void);

static VG_BOOL cvexec_main_menu(int *);
static VG_BOOL show_help(void);
static VG_BOOL dgexec_sysmenu(void);


/* load and execute canvas main-menu; return: VG_TRUE=start-game, VG_FALSE=exit-request */
static VG_BOOL
cvexec_main_menu(int *maze_size)
{
  const char *selname;
  struct VG_Canvas *cvas;
  int retw = VG_TRUE;

  if (maze_size == NULL) { return VG_FALSE; }

  *maze_size = 1;

cvmm_start:
  vg4->window->clear();

  cvas = vg4->canvas->load("files/canvas/main-menu/top.cvas", NULL);
  if (cvas == NULL) { return VG_FALSE; }

cvmm_redo:
  if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; goto cvmm_end; }
  if (selname == NULL || *selname == '\0') { goto cvmm_redo; }

  if (strcmp(selname, "bt_help") == 0) {
    if (!show_help()) { retw = VG_FALSE; goto cvmm_end; }

  } else if (strcmp(selname, "bt_sysmenu") == 0) {
    if (!dgexec_sysmenu()) { retw = VG_FALSE; goto cvmm_end; }

  } else if (strcmp(selname, "bt_start") == 0) {
    struct VG_Canvas *cvas_sub;

    /* select maze */
    cvas_sub = vg4->canvas->load("files/canvas/maze-size/top.cvas", NULL);
    if (cvas_sub != NULL) {
      struct VG_Position possub;
      struct VG_ImagecopyAttrPixel iattr_pixel;
      VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&iattr_pixel);
      iattr_pixel.brightness = 40;
      possub = vg4->canvas->subcanvas(cvas, &iattr_pixel);
      if (!vg4->canvas->exec(cvas_sub, &possub, &selname)) { vg4->canvas->destroy(cvas_sub); retw = VG_FALSE; goto cvmm_end; }
      if (selname == NULL || *selname == '\0') { vg4->canvas->destroy(cvas_sub); goto cvmm_redo; }
      if (strcmp(selname, "small") == 0) {
        *maze_size = 1;
      } else if (strcmp(selname, "medium") == 0) {
        *maze_size = 2;
      } else {
        *maze_size = 3;
      }
      vg4->canvas->destroy(cvas_sub);
    }

    /* show opening door */
    { struct VG_Sprite *sprt = vg4->sprite->load("files/images/tor.sprite");
      if (sprt != NULL) {
        struct VG_Image *imgp;
        struct VG_ImagecopyAttr iattr;
        while (vg4->sprite->next(sprt, &imgp, &iattr)) {
          vg4->window->clear();
          if (imgp != NULL) { vg4->window->copy(imgp, NULL, &iattr); }
          vg4->window->flush();
          vg4->misc->wait_time(80);
        }
        vg4->sprite->destroy(sprt);
      }
    }
    retw = VG_TRUE;
    goto cvmm_end;

  } else if (strcmp(selname, "bt_exit") == 0) {
    retw = VG_FALSE; 
    goto cvmm_end;
  }

  vg4->canvas->destroy(cvas);
  goto cvmm_start;

cvmm_end:
  vg4->canvas->destroy(cvas);
  return retw;
}


/* show help */
static VG_BOOL
show_help(void)
{
  struct VG_Canvas *cvas;
  const char *selname;
  struct VG_Hash *hvar;
  VG_BOOL retw, asusp;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "title", vg4->mlang->get("menu", "help-title"));
  vg4->hash->setstr(hvar, "text", vg4->mlang->get("menu", "help-text"));

  /* load canvas */
  cvas = vg4->canvas->load("files/canvas/help/top.cvas", hvar);
  if (cvas == NULL) { return VG_FALSE; }

  asusp = vg4->audio->suspend(VG_TRUE);
  retw = VG_TRUE;

  for (;;) {
    /* execute canvas */
    if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; break; }
    if (selname == NULL) { break; }
    if (*selname == '\0' || strcmp(selname, "close") == 0) { break; }
  }

  vg4->canvas->destroy(cvas);
  vg4->hash->destroy(hvar);
  vg4->audio->suspend(asusp);

  return retw;
}


/* execute system-menu dialog */
static VG_BOOL
dgexec_sysmenu(void)
{
  struct VG_Hash *hvar;
  VG_BOOL retw;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "top:title", vg4->mlang->get("menu", "System-menu"));
  vg4->hash->setstr(hvar, "exit:top:title", vg4->mlang->get("menu", "Quit game?"));
  vg4->hash->setstr(hvar, "volume:top:title", vg4->mlang->get("menu", "Set audio volumes"));
  vg4->hash->setstr(hvar, "keydef:top:title", vg4->mlang->get("menu", "Key Redefinition"));
  vg4->hash->setstr(hvar, "keydef:press:title", vg4->mlang->get("menu", "Press key"));
  vg4->hash->setstr(hvar, "windowsize:top:title", vg4->mlang->get("menu", "Resize window"));
  vg4->hash->setstr(hvar, "winbright:top:title", vg4->mlang->get("menu", "Brightness"));

  vg4->audio->suspend(VG_TRUE);
  retw = vg4->dialog->sysmenu(NULL, NULL, hvar, NULL, NULL, NULL, NULL);
  vg4->audio->suspend(VG_FALSE);
  vg4->hash->destroy(hvar);

  return retw;
}


int main(int argc, char **argv) {
  struct s_game sgame;
  int maze_size, clockmax;
  VG_BOOL is_maskul, filmskip;
  char mazebuf[4096];
  enum { GAMEPLAY_NOOP = 0, GAMEPLAY_DONE, GAMEPLAY_DEAD, GAMEPLAY_MENU } gplay;

  (void)argc; (void)argv;

  /* +++ initializations +++ */

  memset(&sgame, 0, sizeof(sgame));

  if (!VG_init("Haunted Castle Maze")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_HIGH, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&sgame.winw, &sgame.winh);
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_TRUE)) { VG_dest(); exit(1); }
  vg4->input->mouse_grabbing(VG_FALSE);

  vg4->mlang->fb_locale("en");
  vg4->mlang->add("files/mlang");

  /* +++ setting keys +++ */

  /* quit with ALT+Q, not changeable, local key */
  if ((sgame.kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((sgame.kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* system-menu with ESC, not changeable, local key */
  if ((sgame.kref.k_sysmenu = vg4->input->key_insert("System-menu", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_sysmenu, VG_INPUT_KBDCODE_ESCAPE);
  /* help with H, changeable, local key */
  if ((sgame.kref.k_help = vg4->input->key_insert(vg4->mlang->get("menu", "Help"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_help, VG_INPUT_KBDCODE_H);
  /* pause with P, changeable, local key */
  if ((sgame.kref.k_pause = vg4->input->key_insert(vg4->mlang->get("menu", "Pause"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_pause, VG_INPUT_KBDCODE_P);

  /* go-forward with cursor-key UP, changeable, local key */
  if ((sgame.kref.k_forward = vg4->input->key_insert(vg4->mlang->get("menu", "Go forward"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_forward, VG_INPUT_KBDCODE_UCURS);
  vg4->input->key_setgc(sgame.kref.k_forward, 0, VG_INPUT_GCAXIS_RIGHTY_UP);
  /* go-backward with cursor-key DOWN, changeable, local key */
  if ((sgame.kref.k_backward = vg4->input->key_insert(vg4->mlang->get("menu", "Go backward"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_backward, VG_INPUT_KBDCODE_DCURS);
  vg4->input->key_setgc(sgame.kref.k_backward, 0, VG_INPUT_GCAXIS_RIGHTY_DOWN);
  /* turn-left with cursor-key LEFT, changeable, local key */
  if ((sgame.kref.k_turn_left = vg4->input->key_insert(vg4->mlang->get("menu", "Turn left"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_turn_left, VG_INPUT_KBDCODE_LCURS);
  vg4->input->key_setgc(sgame.kref.k_turn_left, 0, VG_INPUT_GCAXIS_LEFTX_LEFT);
  /* turn-right with cursor-key RIGHT, changeable, local key */
  if ((sgame.kref.k_turn_right = vg4->input->key_insert(vg4->mlang->get("menu", "Turn right"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_turn_right, VG_INPUT_KBDCODE_RCURS);
  vg4->input->key_setgc(sgame.kref.k_turn_right, 0, VG_INPUT_GCAXIS_LEFTX_RIGHT);

  /* toggle-map with M, changeable, local key */
  if ((sgame.kref.k_toggle_map = vg4->input->key_insert(vg4->mlang->get("menu", "Toggle map"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_toggle_map, VG_INPUT_KBDCODE_M);
  vg4->input->key_setgc(sgame.kref.k_toggle_map, 0, VG_INPUT_GCBUTTON_Y);
  /* toggle-lamp with Spacebar, changeable, local key */
  if ((sgame.kref.k_toggle_lamp = vg4->input->key_insert(vg4->mlang->get("menu", "Toggle lamp"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_toggle_lamp, VG_INPUT_KBDCODE_SPACE);
  vg4->input->key_setgc(sgame.kref.k_toggle_lamp, 0, VG_INPUT_GCBUTTON_A);
  /* use-item with Return, changeable, local key */
  if ((sgame.kref.k_use_item = vg4->input->key_insert(vg4->mlang->get("menu", "Use item"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_use_item, VG_INPUT_KBDCODE_RETURN);
  vg4->input->key_setgc(sgame.kref.k_use_item, 0, VG_INPUT_GCBUTTON_RIGHTSHOULDER);


  /* load settings */
  settings_load();

  /* select character */
  is_maskul = VG_TRUE;
  { struct VG_Canvas *cvas = vg4->canvas->load("files/canvas/select-character/top.cvas", NULL);
    if (cvas != NULL) {
      const char *selname;
sel_char:
      if (!vg4->canvas->exec(cvas, NULL, &selname)) { vg4->canvas->destroy(cvas); VG_dest(); exit(0); }
      if (selname == NULL || *selname == '\0') { goto sel_char; }
      if (strcmp(selname, "bt-m") == 0) {
        is_maskul = VG_TRUE;
      } else {
        is_maskul = VG_FALSE;
      }
      vg4->canvas->destroy(cvas);
    }
  }

  /* play intro */
  { struct VG_Hash *hvar = vg4->hash->create();
    char buf[16];
    snprintf(buf, sizeof(buf), "%s", (is_maskul ? "m" : "w"));
    vg4->hash->setstr(hvar, "ENDG", buf);
    snprintf(buf, sizeof(buf), "%s", (is_maskul ? "0x0088ff" : "0xff69b4"));
    vg4->hash->setstr(hvar, "TEXTCOLOR", buf);
    if (!vg4->film->play("files/film/intro/1", NULL, &filmskip, hvar)) { VG_dest(); exit(0); }
    if (!filmskip && !vg4->film->play("files/film/intro/2", NULL, &filmskip, hvar)) { VG_dest(); exit(0); }
    vg4->hash->destroy(hvar);
  }

main_menu:
  if (!cvexec_main_menu(&maze_size)) { goto endgame; }

  /* create maze data */
  { FILE *pfp;
    char bcom[256];
    size_t blen;
    int alg;
    int msize = maze_size * 9 + 8;

    while ((alg = ZUFALL(0, 4)) == 2) {;}
    snprintf(bcom, sizeof(bcom), "./mazegen -a %d -w %d -h %d", alg, msize, msize);
    if ((pfp = popen(bcom, "r")) == NULL) {
      fprintf(stderr, "popen(%s): %s\n", bcom, strerror(errno));
      goto endgame;
    }

    msize = 0;
    for (;;) {
      if ((fgets(mazebuf + msize, sizeof(mazebuf) - msize, pfp)) == NULL) { break; }
      blen = strlen(mazebuf + msize);
      if (blen > 1 && mazebuf[msize + blen - 1] == '\n' && mazebuf[msize + blen - 2] == '\r') {
        mazebuf[msize + blen - 2] = '\n';
        mazebuf[msize + blen - 1] = '\0';
        blen--;
      }
      msize += (int)blen;
      if (msize >= (int)sizeof(mazebuf) - 1) { break; }
    }

    if (pclose(pfp) == -1) {
      fprintf(stderr, "pclose(%s): %s\n", bcom, strerror(errno));
      goto endgame;
    }
  }

  /* +++ create object-instances +++ */

  /* create maze */
  if (objnew_MAZE(&sgame, mazebuf) == 0) { goto endgame; }

  /* set clock speed */
  clockmax = sgame.maze.wsize * sgame.maze.hsize / 10;
  sgame.clock = sgame.clockdelta = 0;

  /* create player */
  if (objnew_PLAYER(&sgame, is_maskul) == 0) { goto endgame; }

  /* show map */
  { struct VG_KeyList keylist;
    struct VG_Image *imgp = NULL;
    int showit = 0;
    int kanz = vg4->input->keylist(&keylist, 0);
    for (kanz--; kanz >= 0; kanz--) {
      if (keylist.key[kanz].keyref == sgame.kref.k_toggle_map) {
        char mbuf[128];
        snprintf(mbuf, sizeof(mbuf), "Press %%{txt[fgcolor=0xff0000]: %s%%}", keylist.key[kanz].codename);
        imgp = vg4->font->totext(mbuf, "[fgcolor=0xbb0000 bgcolor=0x444444]", NULL, NULL, NULL);
        break;
      }
    }
    sgame.ply.map = 2;
    vg4->window->clear();
    for (;;) {
      if (!vg4->input->update(VG_TRUE)) { goto endgame; }
      if (vg4->input->key_newpressed(sgame.kref.k_toggle_map)) { break; }
      draw_mazemap(&sgame);
      if (imgp != NULL) {
        if (showit > 0) {
          vg4->window->copy(imgp, NULL, NULL);
          if (--showit == 0) { showit = -30; }
        } else {
          if (++showit >= 0) { showit = 10; }
        }
      }
      vg4->window->flush();
      vg4->misc->wait_time(100);
    }
    sgame.ply.map = 0;
    if (imgp != NULL) { vg4->image->destroy(imgp); }
  }

  /* create the management of ghosts */
  if (!objmgmt_GHOST(&sgame)) { goto endgame; }

  /* create minotaur */
  if (objnew_MINOTAUR(&sgame) == 0) { goto endgame; }

  /* create music */
  if (objnew_MUSIC(&sgame) == 0) { goto endgame; }

  /* +++ game loop +++ */

  gplay = GAMEPLAY_NOOP;
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { goto endgame; }

    /* quit to main-menu? */
    if (vg4->input->key_newpressed(sgame.kref.k_quit_q) && vg4->input->key_pressed(sgame.kref.k_quit_lalt)) {
      gplay = GAMEPLAY_MENU;
      break;
    }

    /* help? */
    if (vg4->input->key_newpressed(sgame.kref.k_help)) {
      if (!show_help()) { goto endgame; }
    }

    /* pause? */
    if (vg4->input->key_newpressed(sgame.kref.k_pause)) {
      if (!vg4->misc->pause()) { goto endgame; }
    }

    /* system-menu? */
    if (vg4->input->key_newpressed(sgame.kref.k_sysmenu)) {
      if (!dgexec_sysmenu()) { goto endgame; }
    }

    /* call object-management-function f_run() for all object-managements */
    vg4->object->call_mgmt_run(&sgame);

    /* call f_run() of all object-instances */
    if (!vg4->object->call_run(&sgame)) {
      if (sgame.ply.dead > 0) {  /* player is dead */
        gplay = GAMEPLAY_DEAD;
      } else {  /* player has won */
        gplay = GAMEPLAY_DONE;
      }
    }

    /* call f_draw() of all object-instances */
    vg4->window->clear();
    vg4->object->call_draw(&sgame);

    /* flush contents to window and wait */
    if (sgame.ply.map) {
      int wbright = vg4->window->getbrightness();
      vg4->window->setbrightness(100);
      vg4->window->flush();
      vg4->window->setbrightness(wbright);
    } else {
      vg4->window->flush();
    }
    vg4->misc->wait_time(80);

    /* player dead or escaped? */
    if (gplay != GAMEPLAY_NOOP) { break; }

    /* clock */
    if (++sgame.clockdelta == clockmax) {
      if (++sgame.clock < 0) { sgame.clock = 100; }
      sgame.clockdelta = 0;
    }
  }

  /* reset window parameters */
  vg4->window->setattr(NULL);

  /* destroy object-instances and -management */
  vg4->object->destroy_objid(&sgame, OBJID_MUSIC, 0);
  vg4->object->destroy_objid(&sgame, OBJID_MINOTAUR, 0);
  vg4->object->mgmt_destroy(OBJID_GHOST);
  vg4->object->destroy_objid(&sgame, OBJID_GHOST, 0);
  vg4->object->destroy_objid(&sgame, OBJID_PLAYER, 0);
  vg4->object->destroy_objid(&sgame, OBJID_MAZE, 0);

  if (gplay == GAMEPLAY_DEAD || gplay == GAMEPLAY_DONE) {
    struct VG_Hash *hvar = vg4->hash->create();
    char buf[128];
    snprintf(buf, sizeof(buf), "%s", (is_maskul ? "m" : "w"));
    vg4->hash->setstr(hvar, "ENDG", buf);
    snprintf(buf, sizeof(buf), "%s", (is_maskul ? "0x0088ff" : "0xff69b4"));
    vg4->hash->setstr(hvar, "TEXTCOLOR", buf);
    if (gplay == GAMEPLAY_DEAD) {
      int audc;
      snprintf(buf, sizeof(buf), "files/audio/%s-scream.wav", (is_maskul ? "m" : "w"));
      audc = vg4->audio->load(buf, 100, VG_AUDIO_VOLUME_SOUND);
      vg4->audio->play(audc, VG_FALSE, VG_FALSE);
      if (!vg4->misc->fadeout(audc)) { goto endgame; }
      vg4->audio->unload(audc);
      if (sgame.ply.dead == 2) {
        if (!vg4->film->play("files/film/extro/fear", NULL, &filmskip, hvar)) { goto endgame; }
      } else {
        if (!vg4->film->play("files/film/extro/dead", NULL, &filmskip, hvar)) { goto endgame; }
      }
    } else {
      if (!vg4->film->play("files/film/extro/gone", NULL, &filmskip, hvar)) { goto endgame; }
    }
    vg4->hash->destroy(hvar);
  }

  goto main_menu;

endgame:
  /* save settings */
  settings_save();

  /* destroy and exit */
  VG_dest();
  exit(0);
}
