/* object OBJID_MAZE */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <math.h>
#include <errno.h>
#include "obj-maze.h"

unsigned int objnew_MAZE(void *, const char *);

static void f_free(void *, void *);
static void f_draw(void *, struct VG_Object *);

struct box_obj_pos {
  int distance;
  struct obj_pos opos;
};

static VG_BOOL load_maze(struct s_game *, const char *);
static VG_BOOL load_images(struct sobj_maze *);
static int sort_oposp(const void *, const void *);
static void draw_ghost(struct s_game *, struct obj_pos *, int, int);


/* export-function to create a new object-instance of OBJID_MAZE
 * @param vgame       private structure of the game
 * @param mazebuf     maze description
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_MAZE(void *vgame, const char *mazebuf)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_maze *objvars;

  if (sgame == NULL || mazebuf == NULL || *mazebuf == '\0') { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  if (!load_maze(sgame, mazebuf)) { f_free(vgame, objvars); return 0; }
  if (!load_images(objvars)) { f_free(vgame, objvars); return 0; }
  objvars->w_col = MAZE_WCOL_NORMAL;

  /* create object-instance */
  objp = vg4->object->create(OBJID_MAZE, 0, 0, 1, objvars);
  objp->f_free = f_free;
  objp->f_draw = f_draw;

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_maze *objvars = (struct sobj_maze *)opriv;
  int i1;

  if (sgame == NULL) { return; }

  /* free private struct */

  if (sgame->maze.map != NULL) {
    for (i1 = 0; i1 < sgame->maze.hsize; i1++) {
      if (sgame->maze.map[i1] != NULL) { free(sgame->maze.map[i1]); }
    }
    free(sgame->maze.map);
  }

  for (i1 = 0; i1 < MAZEBLICK_MAX; i1++) {
    if (objvars->mazeimg.s[i1].wand_links.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_links.img); }
    if (objvars->mazeimg.s[i1].wand_links.imgexit != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_links.imgexit); }
    if (objvars->mazeimg.s[i1].wand_links.imgportal != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_links.imgportal); }
    if (objvars->mazeimg.s[i1].wand_rechts.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_rechts.img); }
    if (objvars->mazeimg.s[i1].wand_rechts.imgexit != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_rechts.imgexit); }
    if (objvars->mazeimg.s[i1].wand_rechts.imgportal != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].wand_rechts.imgportal); }
    if (objvars->mazeimg.s[i1].saeule_links.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].saeule_links.img); }
    if (objvars->mazeimg.s[i1].saeule_rechts.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].saeule_rechts.img); }
    if (objvars->mazeimg.s[i1].halbwand_links.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_links.img); }
    if (objvars->mazeimg.s[i1].halbwand_links.noimg != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_links.noimg); }
    if (objvars->mazeimg.s[i1].halbwand_links.imgexit != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_links.imgexit); }
    if (objvars->mazeimg.s[i1].halbwand_links.imgportal != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_links.imgportal); }
    if (objvars->mazeimg.s[i1].halbwand_rechts.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_rechts.img); }
    if (objvars->mazeimg.s[i1].halbwand_rechts.noimg != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_rechts.noimg); }
    if (objvars->mazeimg.s[i1].halbwand_rechts.imgexit != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_rechts.imgexit); }
    if (objvars->mazeimg.s[i1].halbwand_rechts.imgportal != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].halbwand_rechts.imgportal); }
    if (objvars->mazeimg.s[i1].vorne.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].vorne.img); }
    if (objvars->mazeimg.s[i1].vorne.imgexit != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].vorne.imgexit); }
    if (objvars->mazeimg.s[i1].vorne.imgportal != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].vorne.imgportal); }
    if (objvars->mazeimg.s[i1].torch.left.img_torch != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].torch.left.img_torch); }
    if (objvars->mazeimg.s[i1].torch.right.img_torch != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].torch.right.img_torch); }
    if (objvars->mazeimg.s[i1].torch.vorne.img_torch != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].torch.vorne.img_torch); }
    if (objvars->mazeimg.s[i1].torch.left.sprt_flame != NULL) { vg4->sprite->destroy(objvars->mazeimg.s[i1].torch.left.sprt_flame); }
    if (objvars->mazeimg.s[i1].torch.right.sprt_flame != NULL) { vg4->sprite->destroy(objvars->mazeimg.s[i1].torch.right.sprt_flame); }
    if (objvars->mazeimg.s[i1].torch.vorne.sprt_flame != NULL) { vg4->sprite->destroy(objvars->mazeimg.s[i1].torch.vorne.sprt_flame); }
    if (objvars->mazeimg.s[i1].item.garlic.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].item.garlic.img); }
    if (objvars->mazeimg.s[i1].item.valerian.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].item.valerian.img); }
    if (objvars->mazeimg.s[i1].item.jewel.img != NULL) { vg4->image->destroy(objvars->mazeimg.s[i1].item.jewel.img); }
  }

  if (objvars->mazeimg.dreh != NULL) { vg4->image->destroy(objvars->mazeimg.dreh); }
  if (objvars->mazeimg.bg != NULL) { vg4->image->destroy(objvars->mazeimg.bg); }
  if (objvars->img_jewel_klein != NULL) { vg4->image->destroy(objvars->img_jewel_klein); }

  free(objvars);
}


/* create maze from maze description */
static VG_BOOL
load_maze(struct s_game *sgame, const char *mazebuf)
{
  static const char *hexmapping = "0123456789abcdef";
  char buf[256], *ptr;
  size_t slen;
  int i1, i2, i3, wpos, hpos, anz_garlic, anz_valerian, anz_jewel;
  const char *pt1, *pt2;

  if (sgame == NULL || mazebuf == NULL) { return VG_FALSE; }
  sgame->maze.wsize = sgame->maze.hsize = 0;
  sgame->maze.map = NULL;

  /* create maze */
  for (pt1 = mazebuf, pt2 = strchr(pt1, '\n'); ; pt1 = pt2 + 1, pt2 = strchr(pt1, '\n')) {
    /* read next line */
    if (pt2 == NULL) {
      pt2 = pt1 + strlen(pt1);
      if (pt1 == pt2) { break; }
    }
    snprintf(buf, sizeof(buf), "%.*s", (int)(pt2 - pt1), pt1);
    slen = strlen(buf);
    if (slen > 0 && buf[slen - 1] == '\n') { slen--; }
    if (slen > 0 && buf[slen - 1] == '\r') { slen--; }
    buf[slen] = '\0';
    if (slen == 0) { fprintf(stderr, "maze-description invalid(1)\n"); return VG_FALSE; }

    /* set width of maze */
    if (sgame->maze.wsize == 0) {
      sgame->maze.wsize = (int)slen;
    } else if (sgame->maze.wsize != (int)slen) {
      fprintf(stderr, "maze-description invalid(2)\n");
      return VG_FALSE;
    }

    /* increment height of maze */
    if (sgame->maze.hsize == 0) {
      sgame->maze.map = malloc(sizeof(unsigned short *));
    } else {
      sgame->maze.map = realloc(sgame->maze.map, sizeof(unsigned short *) * (sgame->maze.hsize + 1));
    }
    if (sgame->maze.map == NULL) { fprintf(stderr, "malloc/realloc error\n"); return VG_FALSE; }

    /* fill line */
    sgame->maze.map[sgame->maze.hsize] = malloc(sizeof(unsigned short) * sgame->maze.wsize);
    if (sgame->maze.map[sgame->maze.hsize] == NULL) { fprintf(stderr, "malloc error\n"); return VG_FALSE; }
    for (wpos = 0; wpos < sgame->maze.wsize; wpos++) {
      if ((ptr = strchr(hexmapping, buf[wpos])) == NULL) { fprintf(stderr, "maze-description invalid(3)\n"); return VG_FALSE; }
      sgame->maze.map[sgame->maze.hsize][wpos] = (unsigned short)(int)(size_t)(ptr - hexmapping);
    }
    sgame->maze.hsize++;

    if (*pt2 == '\0') { break; }
  }
  if (sgame->maze.hsize == 0) { fprintf(stderr, "maze-description invalid(4)\n"); return VG_FALSE; }

  /* +++ addons +++ */

  for (hpos = 0; hpos < sgame->maze.hsize; hpos++) {
    for (wpos = 0; wpos < sgame->maze.wsize; wpos++) {
      if (sgame->maze.map[hpos][wpos] != 0) {
        int widx, wanz, zfl;
        for (wanz = 0, widx = 0; widx < MAP_SHIFT; widx++) {
          if (sgame->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) { wanz++; }
        }
        if (wanz > 0) {
          /* torch */
          zfl = ZUFALL(0, 11) % wanz;  /* dep. on MAP_SHIFT */
          for (widx = 0; widx < MAP_SHIFT; widx++) {
            if (sgame->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) {
              if (zfl-- == 0) {
                sgame->maze.map[hpos][wpos] |= (unsigned short)(1 << (TORCH_SHIFT + widx));
                break;
              }
            }
          }
          /* secret wall-portal */
          if (ZUFALL(0, 29) < 3) {
            zfl = ZUFALL(0, 11) % wanz;  /* dep. on MAP_SHIFT */
            for (widx = 0; widx < MAP_SHIFT; widx++) {
              if (sgame->maze.map[hpos][wpos] & (unsigned short)(1 << widx)) {
                if (zfl-- == 0) {
                  if ((hpos == 0 && widx == 0)
                      || (hpos == sgame->maze.hsize - 1 && widx == 2)
                      || (wpos == 0 && widx == 3)
                      || (wpos == sgame->maze.wsize - 1 && widx == 1)
                     ) {
                    ;
                  } else {
                    sgame->maze.map[hpos][wpos] |= (unsigned short)(1 << (PORTAL_SHIFT + widx));
                  }
                  break;
                }
              }
            }
          }
        }
      }
    }
  }

  /* items */
  i1 = (sgame->maze.wsize * sgame->maze.hsize) / 100;
  anz_garlic = (sgame->maze.wsize * sgame->maze.hsize) / 85;
  if (i1 > 0) { anz_garlic += (ZUFALL(1, i1) - (i1 / 2)); }
  anz_valerian = (sgame->maze.wsize * sgame->maze.hsize) / 90;
  if (i1 > 0) { anz_valerian += (ZUFALL(1, i1) - (i1 / 2)); }
  anz_jewel = (sgame->maze.wsize + sgame->maze.hsize) / 20;
  if (anz_jewel == 0) { anz_jewel = 1; }

  i1 = sgame->maze.wsize * sgame->maze.hsize;

  /* item:garlic */
  for (i2 = 0; i2 < anz_garlic; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / sgame->maze.hsize;
    wpos = i3 % sgame->maze.hsize;
    if (!(sgame->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      sgame->maze.map[hpos][wpos] |= (unsigned short)ITEM_GARLIC;
    } else {
      i2--;
    }
  }

  /* item:valerian */
  for (i2 = 0; i2 < anz_valerian; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / sgame->maze.hsize;
    wpos = i3 % sgame->maze.hsize;
    if (!(sgame->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      sgame->maze.map[hpos][wpos] |= (unsigned short)ITEM_VALERIAN;
    } else {
      i2--;
    }
  }

  /* item:jewel */
  for (i2 = 0; i2 < anz_jewel; i2++) {
    i3 = ZUFALL(1, i1) - 1;
    hpos = i3 / sgame->maze.hsize;
    wpos = i3 % sgame->maze.hsize;
    if (!(sgame->maze.map[hpos][wpos] & ((unsigned short)ITEM_GARLIC | (unsigned short)ITEM_VALERIAN | (unsigned short)ITEM_JEWEL))) {
      sgame->maze.map[hpos][wpos] |= (unsigned short)ITEM_JEWEL;
    } else {
      i2--;
    }
  }

  /* get start-position and exit */

  sgame->ply.x = sgame->ply.y = sgame->ply.dir = 0;
  sgame->maze.exit_x = sgame->maze.exit_y = sgame->maze.exit_dir = 0;

  for (wpos = 0; wpos < sgame->maze.wsize; wpos++) {
    if (!(sgame->maze.map[0][wpos] & WAND_NORD)) {
      if (sgame->ply.dir == 0) {
        sgame->ply.x = wpos; sgame->ply.y = 0; sgame->ply.dir = WAND_SUED;
      } else {
        sgame->maze.exit_x = wpos; sgame->maze.exit_y = 0; sgame->maze.exit_dir = WAND_NORD;
      }
      sgame->maze.map[0][wpos] |= WAND_NORD;
    }
    if (!(sgame->maze.map[sgame->maze.hsize - 1][wpos] & WAND_SUED)) {
      if (sgame->ply.dir == 0) {
        sgame->ply.x = wpos; sgame->ply.y = sgame->maze.hsize - 1; sgame->ply.dir = WAND_NORD;
      } else {
        sgame->maze.exit_x = wpos; sgame->maze.exit_y = sgame->maze.hsize - 1; sgame->maze.exit_dir = WAND_SUED;
      }
      sgame->maze.map[sgame->maze.hsize - 1][wpos] |= WAND_SUED;
    }
  }

  for (wpos = 0; wpos < sgame->maze.hsize; wpos++) {
    if (!(sgame->maze.map[wpos][0] & WAND_WEST)) {
      if (sgame->ply.dir == 0) {
        sgame->ply.x = 0; sgame->ply.y = wpos; sgame->ply.dir = WAND_OST;
      } else {
        sgame->maze.exit_x = 0; sgame->maze.exit_y = wpos; sgame->maze.exit_dir = WAND_WEST;
      }
      sgame->maze.map[wpos][0] |= WAND_WEST;
    }
    if (!(sgame->maze.map[wpos][sgame->maze.wsize - 1] & WAND_OST)) {
      if (sgame->ply.dir == 0) {
        sgame->ply.x = sgame->maze.wsize - 1; sgame->ply.y = wpos; sgame->ply.dir = WAND_WEST;
      } else {
        sgame->maze.exit_x = sgame->maze.wsize - 1; sgame->maze.exit_y = wpos; sgame->maze.exit_dir = WAND_OST;
      }
      sgame->maze.map[wpos][sgame->maze.wsize - 1] |= WAND_OST;
    }
  }

  if (sgame->ply.dir == 0 || sgame->maze.exit_dir == 0) { fprintf(stderr, "Maze: found no start or no exit\n"); return VG_FALSE; }

  return VG_TRUE;
}


/* load images */
static VG_BOOL
load_images(struct sobj_maze *objvars)
{
  struct i_pos { int xm, ym; };
  struct i_pos pos_wand_links[MAZEBLICK_MAX] = {
    { 81, 240 }, { 211, 238 }, { 256, 241 }, { 276, 241 }
  };
  struct i_pos pos_wand_rechts[MAZEBLICK_MAX] = {
    { 559, 240 }, { 429, 238 }, { 384, 241 }, { 363, 241 }
  };
  struct i_pos pos_saeule_links[MAZEBLICK_MAX] = {
    { 169, 239 }, { 234, 239 }, { 264, 241 }, { 277, 242 }
  };
  struct i_pos pos_saeule_rechts[MAZEBLICK_MAX] = {
    { 471, 239 }, { 406, 239 }, { 376, 241 }, { 363, 242 }
  };
  struct i_pos pos_halbwand_links[MAZEBLICK_MAX] = {
    { 72, 240 }, { 0, 0 }, { 0, 0 }, { 0, 0 }
  };
  struct i_pos pos_halbwand_rechts[MAZEBLICK_MAX] = {
    { 567, 240 }, { 0, 0 }, { 0, 0 }, { 0, 0 }
  };
  struct i_pos pos_vorne[MAZEBLICK_MAX] = {
    { 320, 240 }, { 319, 239 }, { 319, 241 }, { 319, 241 }
  };
  struct i_pos pos_torch_links[MAZEBLICK_MAX] = {
    { 148, 168 }, { 240, 202 }, { 272, 218 }, { 286, 228 }
  };
  struct i_pos pos_torch_rechts[MAZEBLICK_MAX] = {
    { 491, 168 }, { 399, 202 }, { 367, 218 }, { 353, 228 }
  };
  struct i_pos pos_torch_vorne[MAZEBLICK_MAX] = {
    { 320, 170 }, { 320, 200 }, { 320, 215 }, { 320, 223 }
  };
  struct i_pos pos_item_garlic[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  struct i_pos pos_item_valerian[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  struct i_pos pos_item_jewel[MAZEBLICK_MAX] = {
    { 320, 420 }, { 320, 310 }, { 320, 290 }, { 320, 275 }
  };
  int mazeblick_percent[MAZEBLICK_MAX] = { 100, 55, 33, 22 };
  int i1;
  char buf[256];

  if (objvars == NULL) { return VG_FALSE; }

  objvars->img_jewel_klein = vg4->image->load("files/images/jewel-klein.bmp");
  if (objvars->img_jewel_klein  == NULL) { return VG_FALSE; }

  objvars->mazeimg.bg = vg4->image->load("files/images/maze/hintergrund.bmp");
  if (objvars->mazeimg.bg == NULL) { return VG_FALSE; }

  objvars->mazeimg.dreh = vg4->image->load("files/images/maze/dreh.bmp");
  if (objvars->mazeimg.dreh == NULL) { return VG_FALSE; }

  for (i1 = 0; i1 < MAZEBLICK_MAX; i1++) {
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-wand-links.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_links.img = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_links.img == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-ausgang-links.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_links.imgexit = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_links.imgexit == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-wand-links-portal.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_links.imgportal = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_links.imgportal == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].wand_links.xm = pos_wand_links[i1].xm;
    objvars->mazeimg.s[i1].wand_links.ym = pos_wand_links[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/maze/s%d-wand-rechts.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_rechts.img = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_rechts.img == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-ausgang-rechts.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_rechts.imgexit = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_rechts.imgexit == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-wand-rechts-portal.bmp", i1 + 1);
    objvars->mazeimg.s[i1].wand_rechts.imgportal = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].wand_rechts.imgportal == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].wand_rechts.xm = pos_wand_rechts[i1].xm;
    objvars->mazeimg.s[i1].wand_rechts.ym = pos_wand_rechts[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/maze/s%d-saeule-links.bmp", i1 + 1);
    objvars->mazeimg.s[i1].saeule_links.img = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].saeule_links.img == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].saeule_links.xm = pos_saeule_links[i1].xm;
    objvars->mazeimg.s[i1].saeule_links.ym = pos_saeule_links[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/maze/s%d-saeule-rechts.bmp", i1 + 1);
    objvars->mazeimg.s[i1].saeule_rechts.img = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].saeule_rechts.img == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].saeule_rechts.xm = pos_saeule_rechts[i1].xm;
    objvars->mazeimg.s[i1].saeule_rechts.ym = pos_saeule_rechts[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/maze/s%d-halbwand.bmp", i1 + 1);
    if (access(buf, F_OK) == 0) {
      objvars->mazeimg.s[i1].halbwand_links.img = vg4->image->load(buf);
      objvars->mazeimg.s[i1].halbwand_rechts.img = vg4->image->load(buf);
      snprintf(buf, sizeof(buf), "files/images/maze/s%d-ausgang-halbwand.bmp", i1 + 1);
      objvars->mazeimg.s[i1].halbwand_links.imgexit = vg4->image->load(buf);
      objvars->mazeimg.s[i1].halbwand_rechts.imgexit = vg4->image->load(buf);
      if (objvars->mazeimg.s[i1].halbwand_rechts.imgexit == NULL) { return VG_FALSE; }
      snprintf(buf, sizeof(buf), "files/images/maze/s%d-nohalbwand-links.bmp", i1 + 1);
      objvars->mazeimg.s[i1].halbwand_links.noimg = vg4->image->load(buf);
      if (objvars->mazeimg.s[i1].halbwand_links.noimg == NULL) { return VG_FALSE; }
      snprintf(buf, sizeof(buf), "files/images/maze/s%d-nohalbwand-rechts.bmp", i1 + 1);
      objvars->mazeimg.s[i1].halbwand_rechts.noimg = vg4->image->load(buf);
      if (objvars->mazeimg.s[i1].halbwand_rechts.noimg == NULL) { return VG_FALSE; }
      snprintf(buf, sizeof(buf), "files/images/maze/s%d-halbwand-portal.bmp", i1 + 1);
      objvars->mazeimg.s[i1].halbwand_links.imgportal = vg4->image->load(buf);
      objvars->mazeimg.s[i1].halbwand_rechts.imgportal = vg4->image->load(buf);
      if (objvars->mazeimg.s[i1].halbwand_rechts.imgportal == NULL) { return VG_FALSE; }
      objvars->mazeimg.s[i1].halbwand_links.xm = pos_halbwand_links[i1].xm;
      objvars->mazeimg.s[i1].halbwand_links.ym = pos_halbwand_links[i1].ym;
      objvars->mazeimg.s[i1].halbwand_rechts.xm = pos_halbwand_rechts[i1].xm;
      objvars->mazeimg.s[i1].halbwand_rechts.ym = pos_halbwand_rechts[i1].ym;
    }

    snprintf(buf, sizeof(buf), "files/images/maze/s%d-vorne.bmp", i1 + 1);
    objvars->mazeimg.s[i1].vorne.img = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].vorne.img == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-ausgang-vorne.bmp", i1 + 1);
    objvars->mazeimg.s[i1].vorne.imgexit = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].vorne.imgexit == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/maze/s%d-vorne-portal.bmp", i1 + 1);
    objvars->mazeimg.s[i1].vorne.imgportal = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].vorne.imgportal == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].vorne.xm = pos_vorne[i1].xm;
    objvars->mazeimg.s[i1].vorne.ym = pos_vorne[i1].ym;

    objvars->mazeimg.s[i1].torch.percent = mazeblick_percent[i1];

    snprintf(buf, sizeof(buf), "files/images/torch/torch-left.bmp");
    objvars->mazeimg.s[i1].torch.left.img_torch = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].torch.left.img_torch == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/torch/flame-left.sprite");
    objvars->mazeimg.s[i1].torch.left.sprt_flame = vg4->sprite->load(buf);
    if (objvars->mazeimg.s[i1].torch.left.sprt_flame == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].torch.left.xm_torch = pos_torch_links[i1].xm;
    objvars->mazeimg.s[i1].torch.left.ym_torch = pos_torch_links[i1].ym;
    objvars->mazeimg.s[i1].torch.left.xm_flame = pos_torch_links[i1].xm - 1;
    objvars->mazeimg.s[i1].torch.left.ym_flame = pos_torch_links[i1].ym - (16 * mazeblick_percent[i1] / 100);
    vg4->sprite->imagesize(objvars->mazeimg.s[i1].torch.left.sprt_flame, "avg", &objvars->mazeimg.s[i1].torch.left.w_flame, &objvars->mazeimg.s[i1].torch.left.h_flame);

    snprintf(buf, sizeof(buf), "files/images/torch/torch-right.bmp");
    objvars->mazeimg.s[i1].torch.right.img_torch = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].torch.right.img_torch == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/torch/flame-right.sprite");
    objvars->mazeimg.s[i1].torch.right.sprt_flame = vg4->sprite->load(buf);
    if (objvars->mazeimg.s[i1].torch.right.sprt_flame == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].torch.right.xm_torch = pos_torch_rechts[i1].xm;
    objvars->mazeimg.s[i1].torch.right.ym_torch = pos_torch_rechts[i1].ym;
    objvars->mazeimg.s[i1].torch.right.xm_flame = pos_torch_rechts[i1].xm + 1;
    objvars->mazeimg.s[i1].torch.right.ym_flame = pos_torch_rechts[i1].ym - (16 * mazeblick_percent[i1] / 100);
    vg4->sprite->imagesize(objvars->mazeimg.s[i1].torch.right.sprt_flame, "avg", &objvars->mazeimg.s[i1].torch.right.w_flame, &objvars->mazeimg.s[i1].torch.right.h_flame);

    snprintf(buf, sizeof(buf), "files/images/torch/torch-vorne.bmp");
    objvars->mazeimg.s[i1].torch.vorne.img_torch = vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].torch.vorne.img_torch == NULL) { return VG_FALSE; }
    snprintf(buf, sizeof(buf), "files/images/torch/flame-vorne.sprite");
    objvars->mazeimg.s[i1].torch.vorne.sprt_flame = vg4->sprite->load(buf);
    if (objvars->mazeimg.s[i1].torch.vorne.sprt_flame == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].torch.vorne.xm_torch = pos_torch_vorne[i1].xm;
    objvars->mazeimg.s[i1].torch.vorne.ym_torch = pos_torch_vorne[i1].ym;
    objvars->mazeimg.s[i1].torch.vorne.xm_flame = pos_torch_vorne[i1].xm;
    objvars->mazeimg.s[i1].torch.vorne.ym_flame = pos_torch_vorne[i1].ym - (28 * mazeblick_percent[i1] / 100);
    vg4->sprite->imagesize(objvars->mazeimg.s[i1].torch.vorne.sprt_flame, "avg", &objvars->mazeimg.s[i1].torch.vorne.w_flame, &objvars->mazeimg.s[i1].torch.vorne.h_flame);

    objvars->mazeimg.s[i1].item.percent = mazeblick_percent[i1];

    snprintf(buf, sizeof(buf), "files/images/garlic.bmp");
    objvars->mazeimg.s[i1].item.garlic.img= vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].item.garlic.img == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].item.garlic.xm = pos_item_garlic[i1].xm;
    objvars->mazeimg.s[i1].item.garlic.ym = pos_item_garlic[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/valerian.bmp");
    objvars->mazeimg.s[i1].item.valerian.img= vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].item.valerian.img == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].item.valerian.xm = pos_item_valerian[i1].xm;
    objvars->mazeimg.s[i1].item.valerian.ym = pos_item_valerian[i1].ym;

    snprintf(buf, sizeof(buf), "files/images/jewel.bmp");
    objvars->mazeimg.s[i1].item.jewel.img= vg4->image->load(buf);
    if (objvars->mazeimg.s[i1].item.jewel.img == NULL) { return VG_FALSE; }
    objvars->mazeimg.s[i1].item.jewel.xm = pos_item_jewel[i1].xm;
    objvars->mazeimg.s[i1].item.jewel.ym = pos_item_jewel[i1].ym;
  }

  return VG_TRUE;
}


/* sort according to distance */
static int
sort_oposp(const void *vbopos1, const void *vbopos2)
{
  const struct box_obj_pos *bopos1, *bopos2;
        
  bopos1 = (struct box_obj_pos *)vbopos1;
  bopos2 = (struct box_obj_pos *)vbopos2;
  if (bopos1->distance < bopos2->distance) { return 1; }
  if (bopos1->distance > bopos2->distance) { return -1; }
  return 0;
}


/* draw ghost or minotaur */
static void
draw_ghost(struct s_game *sgame, struct obj_pos *opos, int mazeblick, int indirekt)
{ 
  struct g_pos { int xl, xr; };
  struct g_pos pos_ghost[MAZEBLICK_MAX] = {
    { -160, 800 }, { 150, 490 }, { 220, 420 }, { 255, 385 }
  };    
  struct VG_ImagecopyAttr gesattr, iattr; 
  double dpos;
  int xpos, xprz;
  VG_BOOL doagain;
        
  if (sgame == NULL || opos == NULL || opos->img == NULL || mazeblick < 0) { return; }

  if (sgame->ply.lamp && opos->u_pg == 2) { return; }  /* lamp on: don't draw ghost/minotaur */

  VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
  xpos = sgame->winw / 2;
  doagain = VG_FALSE;

  if (opos->dir == 0 && opos->prz == -1) {
    if (indirekt) { return; }
    xpos = opos->x;

  } else if (sgame->ply.dir == WAND_NORD && opos->y == sgame->ply.y - mazeblick) {
    if (opos->x != sgame->ply.x) {
      if ((opos->x == sgame->ply.x + 1 && opos->dir == WAND_WEST && opos->prz > 50)
        || (opos->x == sgame->ply.x - 1 && opos->dir == WAND_OST && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->x == sgame->ply.x) { return; }
    if (!indirekt && opos->x != sgame->ply.x) { return; }
    if (indirekt && (opos->dir == WAND_NORD || opos->dir == WAND_SUED)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_NORD) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_SUED) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = VG_TRUE;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_WEST) { xprz = opos->prz; } else if (opos->dir == WAND_OST) { xprz = -opos->prz; }
      xprz = 100 - 100 * (opos->x - sgame->ply.x) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) {
      iattr.image.zoom_width = iattr.image.zoom_height = -2. * ((dpos + 1.) - 12.);
    } else {
      iattr.image.zoom_width = iattr.image.zoom_height = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2));
    }
    iattr.image.zoom_ispercent = VG_TRUE;

  } else if (sgame->ply.dir == WAND_OST && opos->x == sgame->ply.x + mazeblick) {
    if (opos->y != sgame->ply.y) {
      if ((opos->y == sgame->ply.y + 1 && opos->dir == WAND_NORD && opos->prz > 50)
        || (opos->y == sgame->ply.y - 1 && opos->dir == WAND_SUED && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->y == sgame->ply.y) { return; }
    if (!indirekt && opos->y != sgame->ply.y) { return; }
    if (indirekt && (opos->dir == WAND_OST || opos->dir == WAND_WEST)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_OST) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_WEST) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = VG_TRUE;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_NORD) { xprz = opos->prz; } else if (opos->dir == WAND_SUED) { xprz = -opos->prz; }
      xprz = 100 - 100 * (opos->y - sgame->ply.y) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) {
      iattr.image.zoom_width = iattr.image.zoom_height = -2. * ((dpos + 1.) - 12.);
    } else {
      iattr.image.zoom_width = iattr.image.zoom_height = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2));
    }
    iattr.image.zoom_ispercent = VG_TRUE;

  } else if (sgame->ply.dir == WAND_SUED && opos->y == sgame->ply.y + mazeblick) {
    if (opos->x != sgame->ply.x) {
      if ((opos->x == sgame->ply.x + 1 && opos->dir == WAND_WEST && opos->prz > 50)
        || (opos->x == sgame->ply.x - 1 && opos->dir == WAND_OST && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->x == sgame->ply.x) { return; }
    if (!indirekt && opos->x != sgame->ply.x) { return; }
    if (indirekt && (opos->dir == WAND_SUED || opos->dir == WAND_NORD)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_SUED) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_NORD) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = VG_TRUE;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_WEST) { xprz = -opos->prz; } else if (opos->dir == WAND_OST) { xprz = opos->prz; }
      xprz = 100 + 100 * (opos->x - sgame->ply.x) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) {
      iattr.image.zoom_width = iattr.image.zoom_height = -2. * ((dpos + 1.) - 12.);
    } else {
      iattr.image.zoom_width = iattr.image.zoom_height = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2));
    }
    iattr.image.zoom_ispercent = VG_TRUE;

  } else if (sgame->ply.dir == WAND_WEST && opos->x == sgame->ply.x - mazeblick) {
    if (opos->y != sgame->ply.y) {
      if ((opos->y == sgame->ply.y + 1 && opos->dir == WAND_NORD && opos->prz > 50)
        || (opos->y == sgame->ply.y - 1 && opos->dir == WAND_SUED && opos->prz > 50)) {
        if (indirekt) { return; }
        indirekt = 1;
      }
    }
    if (indirekt && opos->y == sgame->ply.y) { return; }
    if (!indirekt && opos->y != sgame->ply.y) { return; }
    if (indirekt && (opos->dir == WAND_WEST || opos->dir == WAND_OST)) { return; }
    dpos = (double)mazeblick;
    if (opos->dir == WAND_WEST) {
      dpos += ((double)opos->prz / 100.);
    } else if (opos->dir == WAND_OST) {
      dpos -= ((double)opos->prz / 100.);
      if (dpos < .0) { return; }
      doagain = VG_TRUE;
    }
    if (mazeblick < MAZEBLICK_MAX) {
      xprz = 0;
      if (opos->dir == WAND_NORD) { xprz = -opos->prz; } else if (opos->dir == WAND_SUED) { xprz = opos->prz; }
      xprz = 100 + 100 * (opos->y - sgame->ply.y) + xprz;
      xpos = pos_ghost[mazeblick].xr - (pos_ghost[mazeblick].xr - pos_ghost[mazeblick].xl) * xprz / 200;
    }
    if (dpos > 6.) {
      iattr.image.zoom_width = iattr.image.zoom_height = -2. * ((dpos + 1.) - 12.);
    } else {
      iattr.image.zoom_width = iattr.image.zoom_height = 220. / ((dpos + 1.) * sqrt((dpos + 1.) + .2));
    }
    iattr.image.zoom_ispercent = VG_TRUE;

  } else {
    return;
  }

  if (doagain && mazeblick == 0) { doagain = VG_FALSE; }

  vg4->image->attr_sum(&gesattr, &opos->attr, &iattr);
  if (doagain) {  /* draw object-instance not now but at next call */
    opos->x = xpos; opos->attr = gesattr; opos->dir = 0; opos->prz = -1;
  } else {
    struct VG_Position posi;
    posi.x = xpos;
    posi.y = sgame->winh / 2;
    posi.pos = VG_POS_CENTERED;
    vg4->window->copy(opos->img, &posi, &gesattr);
    opos->img = NULL;
  }
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{ 
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_maze *objvars = (struct sobj_maze *)objp->opriv;
  struct VG_Position posi;
  struct VG_ImagecopyAttrPixel wattr_pixel;

  if (sgame == NULL) { return; }

  /* draw maze-instance */
  { const int mazeblick_extend = 10;
    int mazeblick;
    struct VG_Image *imgp;
    int ypos, xpos, imgw, imgh, yexit_links, xexit_links, yexit_rechts, xexit_rechts, wand_links, wand_rechts;

    /* background */
    vg4->window->copy(objvars->mazeimg.bg, NULL, NULL);

    if (sgame->ply.turn == 0) {
      struct obj_pos opos;
      struct box_obj_pos *boposp_ghost = NULL;
      int opospmax_ghost = 0;
      int oidx, icol, distance;
      struct VG_ImagecopyAttr gesattr, iattr, zattr;
      struct VG_Rect rect;
      char tbuf[64], tparm[64];
      int numberof, ipos;
      unsigned int *idlist;

      /* find next front wall */
      for (mazeblick = 0; mazeblick < mazeblick_extend; mazeblick++) {
        if (sgame->ply.dir == WAND_NORD) {
          ypos = sgame->ply.y - mazeblick; xpos = sgame->ply.x;
          if (ypos == 0) { break; }
        } else if (sgame->ply.dir == WAND_OST) {
          ypos = sgame->ply.y; xpos = sgame->ply.x + mazeblick;
          if (xpos == sgame->maze.wsize) { break; }
        } else if (sgame->ply.dir == WAND_SUED) {
          ypos = sgame->ply.y + mazeblick; xpos = sgame->ply.x;
          if (ypos == sgame->maze.hsize) { break; }
        } else if (sgame->ply.dir == WAND_WEST) {
          ypos = sgame->ply.y; xpos = sgame->ply.x - mazeblick;
          if (xpos == 0) { break; }
        } else { break; }
        if ((sgame->maze.map[ypos][xpos] & sgame->ply.dir)) { break; }
      }

      /* set relevant ghosts into boposp_ghost[] */
      numberof = vg4->object->list(OBJID_GHOST, 0, &idlist);
      for (ipos = 0; ipos < numberof; ipos++) {
        opos.dir = 1; 
        if (vg4->object->call_data(sgame, idlist[ipos], &opos) > 0) {
          distance = 0;
          if (sgame->ply.dir == WAND_NORD) {
            if (opos.y > sgame->ply.y || opos.y < sgame->ply.y - mazeblick
                || opos.x < sgame->ply.x - 1 || opos.x > sgame->ply.x + 1) {
              continue;
            } else {
              distance = sgame->ply.y - opos.y;
            }
          }
          if (sgame->ply.dir == WAND_OST) {
            if (opos.x < sgame->ply.x || opos.x > sgame->ply.x + mazeblick
                || opos.y < sgame->ply.y - 1 || opos.y > sgame->ply.y + 1) {
              continue;
            } else {
              distance = opos.x - sgame->ply.x;
            }
          }
          if (sgame->ply.dir == WAND_SUED) {
            if (opos.y < sgame->ply.y || opos.y > sgame->ply.y + mazeblick
                || opos.x < sgame->ply.x - 1 || opos.x > sgame->ply.x + 1) {
              continue;
            } else {
              distance = opos.y - sgame->ply.y;
            }
          }
          if (sgame->ply.dir == WAND_WEST) {
            if (opos.x > sgame->ply.x || opos.x < sgame->ply.x - mazeblick
                || opos.y < sgame->ply.y - 1 || opos.y > sgame->ply.y + 1) {
              continue;
            } else {
              distance = sgame->ply.x - opos.x;
            }
          }
          opospmax_ghost++;
          if (boposp_ghost == NULL) {
            boposp_ghost = malloc(sizeof(*boposp_ghost));
          } else {
            boposp_ghost = realloc(boposp_ghost, sizeof(*boposp_ghost) * opospmax_ghost);
          }
          if (boposp_ghost == NULL) { fprintf(stderr, "malloc/calloc: %s\n", strerror(errno)); opospmax_ghost = 0; break; }
          boposp_ghost[opospmax_ghost - 1].opos = opos;
          boposp_ghost[opospmax_ghost - 1].distance = distance;
        }
      }
      if (idlist != NULL) { free(idlist); }

      /* add minotaur into boposp_ghost[] if relevant */
      numberof = vg4->object->list(OBJID_MINOTAUR, 0, &idlist);
      if (numberof == 1) {
        opos.dir = 1; 
        if (vg4->object->call_data(sgame, idlist[0], &opos) > 0) {
          distance = 0;
          if (sgame->ply.dir == WAND_NORD) {
            if (opos.y > sgame->ply.y || opos.y < sgame->ply.y - mazeblick
                || opos.x < sgame->ply.x - 1 || opos.x > sgame->ply.x + 1) {
              opos.img = NULL;
            } else {
              distance = sgame->ply.y - opos.y;
            }
          }
          if (sgame->ply.dir == WAND_OST) {
            if (opos.x < sgame->ply.x || opos.x > sgame->ply.x + mazeblick
                || opos.y < sgame->ply.y - 1 || opos.y > sgame->ply.y + 1) {
              opos.img = NULL;
            } else {
              distance = opos.x - sgame->ply.x;
            }
          }
          if (sgame->ply.dir == WAND_SUED) {
            if (opos.y < sgame->ply.y || opos.y > sgame->ply.y + mazeblick
                || opos.x < sgame->ply.x - 1 || opos.x > sgame->ply.x + 1) {
              opos.img = NULL;
            } else {
              distance = opos.y - sgame->ply.y;
            }
          }
          if (sgame->ply.dir == WAND_WEST) {
            if (opos.x > sgame->ply.x || opos.x < sgame->ply.x - mazeblick
                || opos.y < sgame->ply.y - 1 || opos.y > sgame->ply.y + 1) {
              opos.img = NULL;
            } else {
              distance = sgame->ply.x - opos.x;
            }
          }
          if (opos.img != NULL) {
            opospmax_ghost++;
            if (boposp_ghost == NULL) {
              boposp_ghost = malloc(sizeof(*boposp_ghost));
            } else {
              boposp_ghost = realloc(boposp_ghost, sizeof(*boposp_ghost) * opospmax_ghost);
            }
            if (boposp_ghost == NULL) { fprintf(stderr, "malloc/calloc: %s\n", strerror(errno)); opospmax_ghost = 0; }
            if (boposp_ghost != NULL) {
              boposp_ghost[opospmax_ghost - 1].opos = opos;
              boposp_ghost[opospmax_ghost - 1].distance = distance;
            }
          }
        }
        if (idlist != NULL) { free(idlist); }
      }

      /* sort according distance */
      qsort(boposp_ghost, opospmax_ghost, sizeof(*boposp_ghost), sort_oposp);

      if (mazeblick >= MAZEBLICK_MAX) {
        for (; mazeblick >= MAZEBLICK_MAX; mazeblick--) {
          for (oidx = 0; oidx < opospmax_ghost; oidx++) {
            draw_ghost(sgame, &boposp_ghost[oidx].opos, mazeblick, 0);
          }
        }
      }

      for (; mazeblick >= 0; mazeblick--) {
        if (sgame->ply.dir == WAND_NORD) {
          ypos = sgame->ply.y - mazeblick; xpos = sgame->ply.x;
          yexit_links = ypos; xexit_links = xpos - 1;
          yexit_rechts = ypos; xexit_rechts = xpos + 1;
          wand_links = WAND_WEST; wand_rechts = WAND_OST;
        } else if (sgame->ply.dir == WAND_OST) {
          ypos = sgame->ply.y; xpos = sgame->ply.x + mazeblick;
          yexit_links = ypos - 1; xexit_links = xpos;
          yexit_rechts = ypos + 1; xexit_rechts = xpos;
          wand_links = WAND_NORD; wand_rechts = WAND_SUED;
        } else if (sgame->ply.dir == WAND_SUED) {
          ypos = sgame->ply.y + mazeblick; xpos = sgame->ply.x;
          yexit_links = ypos; xexit_links = xpos + 1;
          yexit_rechts = ypos; xexit_rechts = xpos - 1;
          wand_links = WAND_OST; wand_rechts = WAND_WEST;
        } else if (sgame->ply.dir == WAND_WEST) {
          ypos = sgame->ply.y; xpos = sgame->ply.x - mazeblick;
          yexit_links = ypos + 1; xexit_links = xpos;
          yexit_rechts = ypos - 1; xexit_rechts = xpos;
          wand_links = WAND_SUED; wand_rechts = WAND_NORD;
        } else { break; }

        /* vorne */
        if ((sgame->maze.map[ypos][xpos] & sgame->ply.dir)) {
          if (sgame->maze.exit_y == ypos && sgame->maze.exit_x == xpos && sgame->maze.exit_dir == sgame->ply.dir) {
            imgp = objvars->mazeimg.s[mazeblick].vorne.imgexit;
          } else {
            imgp = objvars->mazeimg.s[mazeblick].vorne.img;
            if (sgame->maze.map[ypos][xpos] & (sgame->ply.dir << PORTAL_SHIFT)) {
              if (objvars->w_col == MAZE_WCOL_LAMP) { imgp = objvars->mazeimg.s[mazeblick].vorne.imgportal; }
            }
          }
          posi.x = objvars->mazeimg.s[mazeblick].vorne.xm;
          posi.y = objvars->mazeimg.s[mazeblick].vorne.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, NULL);
        }

        /* halbwand */

        if (!(sgame->maze.map[ypos][xpos] & wand_links) && objvars->mazeimg.s[mazeblick].halbwand_links.img != NULL) {
          if (sgame->maze.exit_y == yexit_links && sgame->maze.exit_x == xexit_links && sgame->maze.exit_dir == sgame->ply.dir) {
            imgp = objvars->mazeimg.s[mazeblick].halbwand_links.imgexit;
          } else {
            imgp = objvars->mazeimg.s[mazeblick].halbwand_links.img;
            if (!(sgame->maze.map[yexit_links][xexit_links] & sgame->ply.dir)) {
              imgp = objvars->mazeimg.s[mazeblick].halbwand_links.noimg;
            } else if (sgame->maze.map[yexit_links][xexit_links] & (sgame->ply.dir << PORTAL_SHIFT)) {
              if (objvars->w_col == MAZE_WCOL_LAMP) { imgp = objvars->mazeimg.s[mazeblick].halbwand_links.imgportal; }
            }
          }
          posi.x = objvars->mazeimg.s[mazeblick].halbwand_links.xm;
          posi.y = objvars->mazeimg.s[mazeblick].halbwand_links.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, NULL);
        }

        if (!(sgame->maze.map[ypos][xpos] & wand_rechts) && objvars->mazeimg.s[mazeblick].halbwand_rechts.img != NULL) {
          if (sgame->maze.exit_y == yexit_rechts && sgame->maze.exit_x == xexit_rechts && sgame->maze.exit_dir == sgame->ply.dir) {
            imgp = objvars->mazeimg.s[mazeblick].halbwand_rechts.imgexit;
          } else {
            imgp = objvars->mazeimg.s[mazeblick].halbwand_rechts.img;
            if (!(sgame->maze.map[yexit_rechts][xexit_rechts] & sgame->ply.dir)) {
              imgp = objvars->mazeimg.s[mazeblick].halbwand_rechts.noimg;
            } else if (sgame->maze.map[yexit_rechts][xexit_rechts] & (sgame->ply.dir << PORTAL_SHIFT)) {
              if (objvars->w_col == MAZE_WCOL_LAMP) { imgp = objvars->mazeimg.s[mazeblick].halbwand_rechts.imgportal; }
            }
          }
          posi.x = objvars->mazeimg.s[mazeblick].halbwand_rechts.xm;
          posi.y = objvars->mazeimg.s[mazeblick].halbwand_rechts.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, NULL);
        }

        /* saeule */
        posi.pos = VG_POS_CENTERED;
        posi.x = objvars->mazeimg.s[mazeblick].saeule_links.xm;
        posi.y = objvars->mazeimg.s[mazeblick].saeule_links.ym;
        vg4->window->copy(objvars->mazeimg.s[mazeblick].saeule_links.img, &posi, NULL);
        posi.x = objvars->mazeimg.s[mazeblick].saeule_rechts.xm;
        posi.y = objvars->mazeimg.s[mazeblick].saeule_rechts.ym;
        vg4->window->copy(objvars->mazeimg.s[mazeblick].saeule_rechts.img, &posi, NULL);

        /* players+ghosts+minotaur before wand */
        for (oidx = 0; oidx < opospmax_ghost; oidx++) {
          draw_ghost(sgame, &boposp_ghost[oidx].opos, mazeblick, 1);
        }

        /* wand */

        if (sgame->maze.map[ypos][xpos] & wand_links) {
          if (sgame->maze.exit_y == ypos && sgame->maze.exit_x == xpos && sgame->maze.exit_dir == wand_links) {
            imgp = objvars->mazeimg.s[mazeblick].wand_links.imgexit;
          } else {
            imgp = objvars->mazeimg.s[mazeblick].wand_links.img;
            if (sgame->maze.map[ypos][xpos] & (wand_links << PORTAL_SHIFT)) {
              if (objvars->w_col == MAZE_WCOL_LAMP) { imgp = objvars->mazeimg.s[mazeblick].wand_links.imgportal; }
            }
          }
          posi.x = objvars->mazeimg.s[mazeblick].wand_links.xm;
          posi.y = objvars->mazeimg.s[mazeblick].wand_links.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, NULL);
        }

        if (sgame->maze.map[ypos][xpos] & wand_rechts) {
          if (sgame->maze.exit_y == ypos && sgame->maze.exit_x == xpos && sgame->maze.exit_dir == wand_rechts) {
            imgp = objvars->mazeimg.s[mazeblick].wand_rechts.imgexit;
          } else {
            imgp = objvars->mazeimg.s[mazeblick].wand_rechts.img;
            if (sgame->maze.map[ypos][xpos] & (wand_rechts << PORTAL_SHIFT)) {
              if (objvars->w_col == MAZE_WCOL_LAMP) { imgp = objvars->mazeimg.s[mazeblick].wand_rechts.imgportal; }
            }
          }
          posi.x = objvars->mazeimg.s[mazeblick].wand_rechts.xm;
          posi.y = objvars->mazeimg.s[mazeblick].wand_rechts.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, NULL);
        }

        /* torch */

        if (sgame->maze.map[ypos][xpos] & sgame->ply.dir
          && sgame->maze.map[ypos][xpos] & (sgame->ply.dir << TORCH_SHIFT)
        ) {
          imgp = objvars->mazeimg.s[mazeblick].torch.vorne.img_torch;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent;
          posi.x = objvars->mazeimg.s[mazeblick].torch.vorne.xm_torch;
          posi.y = objvars->mazeimg.s[mazeblick].torch.vorne.ym_torch;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
          if (sgame->clock <= 100) {
            vg4->sprite->next(objvars->mazeimg.s[mazeblick].torch.vorne.sprt_flame, &imgp, &iattr);
            if (imgp != NULL) {
              VG_IMAGECOPY_ATTR_DEFAULT(&zattr);
              zattr.image.zoom_ispercent = VG_TRUE;
              zattr.image.zoom_width = zattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent * (100 - sgame->clock) / 100;
              vg4->image->attr_sum(&gesattr, &iattr, &zattr);
              posi.x = objvars->mazeimg.s[mazeblick].torch.vorne.xm_flame;
              posi.y = objvars->mazeimg.s[mazeblick].torch.vorne.ym_flame + objvars->mazeimg.s[mazeblick].torch.vorne.h_flame / 2 * (100 - zattr.image.zoom_height) / 100 * objvars->mazeimg.s[mazeblick].torch.percent / 100;
              posi.pos = VG_POS_CENTERED;
              vg4->window->copy(imgp, &posi, &gesattr);
            }
          }
        }

        if (sgame->maze.map[ypos][xpos] & wand_links
          && sgame->maze.map[ypos][xpos] & (wand_links << TORCH_SHIFT)
        ) {
          imgp = objvars->mazeimg.s[mazeblick].torch.left.img_torch;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent;
          posi.x = objvars->mazeimg.s[mazeblick].torch.left.xm_torch;
          posi.y = objvars->mazeimg.s[mazeblick].torch.left.ym_torch;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
          if (sgame->clock <= 100) {
            vg4->sprite->next(objvars->mazeimg.s[mazeblick].torch.left.sprt_flame, &imgp, &iattr);
            if (imgp != NULL) {
              VG_IMAGECOPY_ATTR_DEFAULT(&zattr);
              zattr.image.zoom_ispercent = VG_TRUE;
              zattr.image.zoom_width = zattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent * (100 - sgame->clock) / 100;
              vg4->image->attr_sum(&gesattr, &iattr, &zattr);
              posi.x = objvars->mazeimg.s[mazeblick].torch.left.xm_flame - objvars->mazeimg.s[mazeblick].torch.left.w_flame / 4 * (100 - zattr.image.zoom_width) / 100 * objvars->mazeimg.s[mazeblick].torch.percent / 100;
              posi.y = objvars->mazeimg.s[mazeblick].torch.left.ym_flame + objvars->mazeimg.s[mazeblick].torch.left.h_flame / 2 * (100 - zattr.image.zoom_height) / 100 * objvars->mazeimg.s[mazeblick].torch.percent / 100;
              posi.pos = VG_POS_CENTERED;
              vg4->window->copy(imgp, &posi, &gesattr);
            }
          }
        }

        if (sgame->maze.map[ypos][xpos] & wand_rechts
          && sgame->maze.map[ypos][xpos] & (wand_rechts << TORCH_SHIFT)
        ) {
          imgp = objvars->mazeimg.s[mazeblick].torch.right.img_torch;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent;
          posi.x = objvars->mazeimg.s[mazeblick].torch.right.xm_torch;
          posi.y = objvars->mazeimg.s[mazeblick].torch.right.ym_torch;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
          if (sgame->clock <= 100) {
            vg4->sprite->next(objvars->mazeimg.s[mazeblick].torch.right.sprt_flame, &imgp, &iattr);
            if (imgp != NULL) {
              VG_IMAGECOPY_ATTR_DEFAULT(&zattr);
              zattr.image.zoom_ispercent = VG_TRUE;
              zattr.image.zoom_width = zattr.image.zoom_height = objvars->mazeimg.s[mazeblick].torch.percent * (100 - sgame->clock) / 100;
              vg4->image->attr_sum(&gesattr, &iattr, &zattr);
              posi.x = objvars->mazeimg.s[mazeblick].torch.right.xm_flame - objvars->mazeimg.s[mazeblick].torch.right.w_flame / 4 * (100 - zattr.image.zoom_width) / 100 * objvars->mazeimg.s[mazeblick].torch.percent / 100;
              posi.y = objvars->mazeimg.s[mazeblick].torch.right.ym_flame + objvars->mazeimg.s[mazeblick].torch.right.h_flame / 2 * (100 - zattr.image.zoom_height) / 100 * objvars->mazeimg.s[mazeblick].torch.percent / 100;
              posi.pos = VG_POS_CENTERED;
              vg4->window->copy(imgp, &posi, &gesattr);
            }
          }
        }

        /* item: garlic */
        if (sgame->maze.map[ypos][xpos] & ITEM_GARLIC) {
          imgp = objvars->mazeimg.s[mazeblick].item.garlic.img;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].item.percent;
          posi.x = objvars->mazeimg.s[mazeblick].item.garlic.xm;
          posi.y = objvars->mazeimg.s[mazeblick].item.garlic.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
        }

        /* item: valerian */
        if (sgame->maze.map[ypos][xpos] & ITEM_VALERIAN) {
          imgp = objvars->mazeimg.s[mazeblick].item.valerian.img;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].item.percent;
          posi.x = objvars->mazeimg.s[mazeblick].item.valerian.xm;
          posi.y = objvars->mazeimg.s[mazeblick].item.valerian.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
        }

        /* item: jewel */
        if (sgame->maze.map[ypos][xpos] & ITEM_JEWEL) {
          imgp = objvars->mazeimg.s[mazeblick].item.jewel.img;
          VG_IMAGECOPY_ATTR_DEFAULT(&gesattr);
          gesattr.image.zoom_ispercent = VG_TRUE;
          gesattr.image.zoom_width = gesattr.image.zoom_height = objvars->mazeimg.s[mazeblick].item.percent;
          posi.x = objvars->mazeimg.s[mazeblick].item.jewel.xm;
          posi.y = objvars->mazeimg.s[mazeblick].item.jewel.ym;
          posi.pos = VG_POS_CENTERED;
          vg4->window->copy(imgp, &posi, &gesattr);
        }

        /* players+ghosts+minotaur after wand */
        for (oidx = 0; oidx < opospmax_ghost; oidx++) {
          draw_ghost(sgame, &boposp_ghost[oidx].opos, mazeblick, 0);
        }
      }

      if (sgame->ply.go != 0) {
        wattr_pixel = vg4->window->getattr();
        VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
        vg4->window->setattr(&iattr.pixel);
        if (sgame->ply.go < 0) {  /* go backward */
          iattr.image.zoom_ispercent = VG_TRUE;
          iattr.image.zoom_width = iattr.image.zoom_height = 100 - sgame->ply.go / 2;
        } else if (sgame->ply.go > 0) {  /* go forward */
          iattr.image.zoom_ispercent = VG_TRUE;
          iattr.image.zoom_width = iattr.image.zoom_height = 100 + (100 - sgame->ply.go) / 2;
        }
        imgp = vg4->window->clone(NULL, &iattr);
        vg4->window->copy(imgp, NULL, NULL);
        vg4->image->destroy(imgp);
        vg4->window->setattr(&wattr_pixel);
      }

      if (opospmax_ghost > 0) { free(boposp_ghost); }

      /* draw text: direction */
      imgp = NULL;
      if (sgame->ply.dir == WAND_NORD) {
        imgp = vg4->font->totext(vg4->mlang->get("game", "east"), "[font=sys:low fgcolor=0xffff00]", NULL, NULL, NULL);
      } else if (sgame->ply.dir == WAND_OST) {
        imgp = vg4->font->totext(vg4->mlang->get("game", "south"), "[font=sys:low fgcolor=0xffff00]", NULL, NULL, NULL);
      } else if (sgame->ply.dir == WAND_SUED) {
        imgp = vg4->font->totext(vg4->mlang->get("game", "west"), "[font=sys:low fgcolor=0xffff00]", NULL, NULL, NULL);
      } else if (sgame->ply.dir == WAND_WEST) {
        imgp = vg4->font->totext(vg4->mlang->get("game", "north"), "[font=sys:low fgcolor=0xffff00]", NULL, NULL, NULL);
      }
      if (imgp != NULL) {
        vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
        rect.x = (sgame->winw - rect.w) / 2;
        rect.y = 10;
        vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
        vg4->image->destroy(imgp);
      }

      /* draw text: fear */
      if (sgame->ply.fear <= 30) {
        icol = VG_COLOR_GREEN;
      } else if (sgame->ply.fear <= 60) {
        icol = VG_COLOR_ORANGE;
      } else {
        icol = VG_COLOR_RED;
      }
      snprintf(tbuf, sizeof(tbuf), "%s: %d%%", vg4->mlang->get("game", "fear"), sgame->ply.fear);
      snprintf(tparm, sizeof(tparm), "[font=sys:low fgcolor=0x%06x]", vg4->misc->colorbrightness(icol, 100));
      imgp = vg4->font->totext(tbuf, tparm, NULL, NULL, NULL);
      if (imgp != NULL) {
        vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
        rect.x = 10;
        rect.y = 10;
        vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
        vg4->image->destroy(imgp);
      }

      /* draw text: garlic */
      if (sgame->ply.garlic > 0) {
        if (sgame->ply.garlic >= 60) {
          icol = VG_COLOR_GREEN;
        } else if (sgame->ply.garlic >= 30) {
          icol = VG_COLOR_ORANGE;
        } else {
          icol = VG_COLOR_RED;
        }
        snprintf(tbuf, sizeof(tbuf), "%s: %d%%", vg4->mlang->get("game", "garlic"), sgame->ply.garlic);
        snprintf(tparm, sizeof(tparm), "[font=sys:low fgcolor=0x%06x]", vg4->misc->colorbrightness(icol, 100));
        imgp = vg4->font->totext(tbuf, tparm, NULL, NULL, NULL);
        if (imgp != NULL) {
          vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
          rect.x = 10;
          rect.y = 24;
          vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
          vg4->image->destroy(imgp);
        }
      }

      /* draw jewel */
      if (sgame->ply.jewel) {
        vg4->image->getsize(objvars->img_jewel_klein, NULL, NULL, &rect.h);
        posi.x = sgame->winw / 2;
        posi.y = 30 + rect.h / 2;
        posi.pos = VG_POS_CENTERED;
        vg4->window->copy(objvars->img_jewel_klein, &posi, NULL);
      }

      /* draw text: clock */
      if (sgame->clock < 100) {
        snprintf(tbuf, sizeof(tbuf), "%d %s", 60 - (sgame->clock * 60 / 100), vg4->mlang->get("game", "min_to_midnight"));
        icol = VG_COLOR_GREEN;
      } else if (sgame->clock == 100) {
        snprintf(tbuf, sizeof(tbuf), "%s", vg4->mlang->get("game", "midnight"));
        icol = VG_COLOR_ORANGE;
      } else {
        snprintf(tbuf, sizeof(tbuf), "%s", vg4->mlang->get("game", "after_midnight"));
        icol = VG_COLOR_RED;
      }
      snprintf(tparm, sizeof(tparm), "[font=sys:low fgcolor=0x%06x]", vg4->misc->colorbrightness(icol, 100));
      imgp = vg4->font->totext(tbuf, tparm, NULL, NULL, NULL);
      if (imgp != NULL) {
        vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
        rect.x = sgame->winw - rect.w - 10;
        rect.y = 10;
        vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
        vg4->image->destroy(imgp);
      }

      /* draw text: ghosts */
      numberof = vg4->object->list(OBJID_GHOST, 0, NULL);
      snprintf(tbuf, sizeof(tbuf), "%s: %d", vg4->mlang->get("game", "ghosts"), numberof);
      snprintf(tparm, sizeof(tparm), "[font=sys:low fgcolor=0x%06x]", vg4->misc->colorbrightness(VG_COLOR_BROWN, 150));
      imgp = vg4->font->totext(tbuf, tparm, NULL, NULL, NULL);
      if (imgp != NULL) {
        vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
        rect.x = sgame->winw - rect.w - 10;
        rect.y = 24;
        vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
        vg4->image->destroy(imgp);
      }

      /* draw text: radar */
      snprintf(tbuf, sizeof(tbuf), "%s: %d %s",
               vg4->mlang->get("game", "radar"),
               red_distance(sgame, sgame->ply.instanceid),
               vg4->mlang->get("game", "steps"));
      snprintf(tparm, sizeof(tparm), "[font=sys:low fgcolor=0x%06x]", vg4->misc->colorbrightness(VG_COLOR_BROWN, 150));
      imgp = vg4->font->totext(tbuf, tparm, NULL, NULL, NULL);
      if (imgp != NULL) {
        vg4->image->getsize(imgp, NULL, &rect.w, &rect.h);
        rect.x = sgame->winw - rect.w - 10;
        rect.y = 38;
        vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &rect), NULL);
        vg4->image->destroy(imgp);
      }

    } else {
      imgp = objvars->mazeimg.dreh;
      vg4->image->getsize(imgp, NULL, &imgw, &imgh);
      if (sgame->ply.turn < 0) {  /* turn left */
        xpos = sgame->winw - sgame->winw * (-sgame->ply.turn) / 100;
      } else {  /* turn right */
        xpos = sgame->winw * sgame->ply.turn / 100;
      }
      posi.x = xpos;
      posi.y = sgame->winh / 2;
      posi.pos = VG_POS_CENTERED;
      vg4->window->copy(imgp, &posi, NULL);
    }
  }

  /* reduce brightness according to clock */
  wattr_pixel = vg4->window->getattr();
  if (sgame->clock >= 0 && sgame->clock <= 100) {
    wattr_pixel.brightness = 100 - sgame->clock * 6 / 10;
  } else {
    wattr_pixel.brightness = 40;
  }
  vg4->window->setattr(&wattr_pixel);

  if (sgame->ply.map) { draw_mazemap(sgame); }

  if (sgame->ply.lamp) {
    wattr_pixel = vg4->window->getattr();
    wattr_pixel.colorize_color = VG_COLOR_BLUE;
    wattr_pixel.colorize_percent = 20;
    wattr_pixel.brightness = 100;
    vg4->window->setattr(&wattr_pixel);
    objvars->w_col = MAZE_WCOL_LAMP;

  } else {
    wattr_pixel = vg4->window->getattr();
    wattr_pixel.colorize_color = VG_COLOR_TRANSPARENT;
    wattr_pixel.colorize_percent = 0;
    if (sgame->ply.map) { wattr_pixel.brightness = 100; }
    vg4->window->setattr(&wattr_pixel);
    objvars->w_col = MAZE_WCOL_NORMAL;

    if (!sgame->ply.map) {
      const int walk_violet = 10;
      struct walking_way wway;
      struct obj_pos opos;
      unsigned int *idlist;
      int numberof, ipos, walk_red, walk_max;
      int minway = -1, obfollow = 0;

      walk_red = red_distance(sgame, sgame->ply.instanceid);
      if (walk_red < walk_violet) { walk_max = walk_violet; } else { walk_max = walk_red; }

      numberof = vg4->object->list(OBJID_GHOST, 0, &idlist);
      for (ipos = 0; ipos < numberof; ipos++) {
        opos.dir = 0; 
        if (vg4->object->call_data(sgame, idlist[ipos], &opos) > 0) {
          create_way(sgame, &wway, walk_max, opos.x, opos.y, sgame->ply.x, sgame->ply.y);
          if ((wway.anz > 0 && wway.anz < walk_max) || (opos.x == sgame->ply.x && opos.y == sgame->ply.y)) {
            if (minway == -1 || wway.anz < minway) { minway = wway.anz; }
            if (opos.u.gh.do_follow && wway.anz <= walk_red) {
              if (obfollow == 0 || obfollow > 1 + wway.anz) { obfollow = 1 + wway.anz; }
            }
          }
        }
      }
      if (idlist != NULL) { free(idlist); }

      numberof = vg4->object->list(OBJID_MINOTAUR, 0, &idlist);
      if (numberof == 1) {
        opos.dir = 0; 
        if (vg4->object->call_data(sgame, idlist[0], &opos) > 0) {
          create_way(sgame, &wway, walk_max, opos.x, opos.y, sgame->ply.x, sgame->ply.y);
          if ((wway.anz > 0 && wway.anz < walk_max) || (opos.x == sgame->ply.x && opos.y == sgame->ply.y)) {
            if (minway == -1 || wway.anz < minway) { minway = wway.anz; }
            if (opos.u.gh.do_follow && wway.anz <= walk_red) {
              if (obfollow == 0 || obfollow > 1 + wway.anz) { obfollow = 1 + wway.anz; }
            }
          }
        }
      }
      if (idlist != NULL) { free(idlist); }

      if (minway >= 0) {
        if (obfollow > 0) { minway = obfollow - 1; }
        wattr_pixel = vg4->window->getattr();
        wattr_pixel.colorize_color = (obfollow ? VG_COLOR_RED : VG_COLOR_VIOLET);
        wattr_pixel.colorize_percent = (walk_max - minway) * 4;
        vg4->window->setattr(&wattr_pixel);
      }
    }
  }
}
