/* object OBJID_PLAYER */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-player.h"

unsigned int objnew_PLAYER(void *, VG_BOOL);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static int f_data(void *, struct VG_Object *, void *);


/* export-function to create a new object-instance of OBJID_PLAYER
 * @param vgame       private structure of the game
 * @param is_maskul   whether player is masculine
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_PLAYER(void *vgame, VG_BOOL is_maskul)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_player *objvars;
  char buf[256];
  int hpos;

  if (sgame == NULL) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  /* load images */
  snprintf(buf, sizeof(buf), "files/images/player/pl-%s.bmp", (is_maskul ? "m" : "w"));
  objvars->img = vg4->image->load(buf);
  if (objvars->img == NULL) { f_free(vgame, objvars); return 0; }

  objvars->pos.x = sgame->ply.x;
  objvars->pos.y = sgame->ply.y;
  objvars->pos.dir = sgame->ply.dir;
  objvars->go = objvars->turn = 0;
  objvars->fear = 0;
  objvars->fearmulti = 187;
  objvars->dead = 0;
  objvars->map = 0;
  sgame->ply.map = objvars->map;
  objvars->ghost_instanceid = 0;
  objvars->lamp_ontime = 0;
  objvars->garlic_smell = 0;
  objvars->garlic_smellmax = sgame->maze.wsize;
  objvars->has_jewel = VG_FALSE;

  /* create mapvisited */
  objvars->mapvisited = calloc(sgame->maze.hsize, sizeof(char *));
  if (objvars->mapvisited == NULL) { fprintf(stderr, "calloc error\n"); return 0; }
  for (hpos = 0; hpos < sgame->maze.hsize; hpos++) {
    objvars->mapvisited[hpos] = calloc(sgame->maze.wsize, sizeof(char));
    if (objvars->mapvisited[hpos] == NULL) { fprintf(stderr, "calloc error\n"); return 0; }
  }
  objvars->mapvisited[objvars->pos.y][objvars->pos.x] |= 128;
  sgame->ply.mapvisited = objvars->mapvisited;

  /* load audios */
  objvars->sound.footstep[0] = vg4->audio->load("files/audio/footstep1.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->sound.footstep[1] = vg4->audio->load("files/audio/footstep2.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->sound.step = 0;
  objvars->sound.eat = vg4->audio->load("files/audio/eat.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->sound.drink = vg4->audio->load("files/audio/drink.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->sound.wall_portal = vg4->audio->load("files/audio/wall-portal.wav", 100, VG_AUDIO_VOLUME_SOUND);
  objvars->sound.jewel = vg4->audio->load("files/audio/jewel.wav", 100, VG_AUDIO_VOLUME_SOUND);
  snprintf(buf, sizeof(buf), "files/audio/%s-fear.wav", (is_maskul ? "m" : "w"));
  objvars->sound.fear = vg4->audio->load(buf, 100, VG_AUDIO_VOLUME_SOUND);

  /* create object-instance */
  objp = vg4->object->create(OBJID_PLAYER, 0, 0, 1, objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_data = f_data;

  sgame->ply.dead = 0;
  sgame->ply.instanceid = objp->instanceid;

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_player *objvars = (struct sobj_player *)opriv;

  if (sgame == NULL) { return; }

  /* free private struct */
  if (objvars->img != NULL) { vg4->image->destroy(objvars->img); }
  if (objvars->mapvisited != NULL) {
    int hpos;
    for (hpos = 0; hpos < sgame->maze.hsize; hpos++) {
      if (objvars->mapvisited[hpos] != NULL) { free(objvars->mapvisited[hpos]); }
    }
    free(objvars->mapvisited);
    sgame->ply.mapvisited = NULL;
  }
  if (objvars->sound.footstep[0] > 0) { vg4->audio->unload(objvars->sound.footstep[0]); }
  if (objvars->sound.footstep[1] > 0) { vg4->audio->unload(objvars->sound.footstep[1]); }
  if (objvars->sound.eat > 0) { vg4->audio->unload(objvars->sound.eat); }
  if (objvars->sound.drink > 0) { vg4->audio->unload(objvars->sound.drink); }
  if (objvars->sound.wall_portal > 0) { vg4->audio->unload(objvars->sound.wall_portal); }
  if (objvars->sound.jewel > 0) { vg4->audio->unload(objvars->sound.jewel); }
  if (objvars->sound.fear > 0) { vg4->audio->unload(objvars->sound.fear); }
  sgame->ply.instanceid = 0;
  sgame->ply.x = sgame->ply.y = sgame->ply.dir = 0;
  sgame->ply.go = sgame->ply.turn = 0;
  sgame->ply.fear = 0;
  sgame->ply.map = 0;
  sgame->ply.lamp = VG_FALSE;
  sgame->ply.garlic = 0;
  sgame->ply.jewel = VG_FALSE;
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_player *objvars = (struct sobj_player *)objp->opriv;
  int retw;

  if (sgame == NULL) { return VG_TRUE; }

  retw = VG_TRUE;

  /* wait until terrorizing ghost disappears */
  if (objvars->ghost_instanceid > 0) {
    struct VG_Object *objp_ghost = vg4->object->instance_getobj(objvars->ghost_instanceid);
    if (objp_ghost != NULL) { goto endrun; }
    objvars->ghost_instanceid = 0;
    sgame->ply.fear = objvars->fear / objvars->fearmulti;
    if (objvars->fear >= 100 * objvars->fearmulti) {  /* player dead because of ghost */
      objvars->dead = 2;
      sgame->ply.dead = objvars->dead;
      retw = VG_FALSE;
      goto endrun;
    }
  }

  /* check collisions with ghosts */
  { int numberof, ipos;
    unsigned int *idlist;
    struct obj_pos opos;

    numberof = vg4->object->list(OBJID_GHOST, 0, &idlist);
    for (ipos = 0; ipos < numberof; ipos++) {
      opos.dir = 0;
      if (vg4->object->call_data(vgame, idlist[ipos], &opos) > 0) {
        if (objvars->pos.x == opos.x && objvars->pos.y == opos.y) {  /* collided with ghost */
          objcoll_PLAYER_GHOST(sgame, objp->instanceid, idlist[ipos]);
          break;
        }
      }
    }
    if (idlist != NULL) { free(idlist); }
  }

  if (objvars->ghost_instanceid > 0) {  /* collided with ghost */
    objvars->go = objvars->turn = 0;
    sgame->ply.go = sgame->ply.turn = 0;
    sgame->ply.x = objvars->pos.x;
    sgame->ply.y = objvars->pos.y;
    objvars->mapvisited[objvars->pos.y][objvars->pos.x] |= 128;
    goto endrun;
  }

  /* check collisions with minotaur */
  { int numberof, ipos;
    unsigned int *idlist;
    struct obj_pos opos;

    numberof = vg4->object->list(OBJID_MINOTAUR, 0, &idlist);
    for (ipos = 0; ipos < numberof; ipos++) {
      opos.dir = 0;
      if (vg4->object->call_data(vgame, idlist[ipos], &opos) > 0) {
        if (objvars->pos.x == opos.x && objvars->pos.y == opos.y) {  /* collided with minotaur */
          objcoll_PLAYER_MINOTAUR(sgame, objp->instanceid, idlist[ipos]);
          break;
        }
      }
    }
    if (idlist != NULL) { free(idlist); }
  }

  if (objvars->dead == 1) {  /* collided with minotaur */
    sgame->ply.dead = objvars->dead;
    retw = VG_FALSE;
    goto endrun;
  }

  /* update lamp-time, fear and garlic-smell */
  if (objvars->lamp_ontime != 0) { objvars->lamp_ontime++; }
  if (objvars->fear > 0) {
    objvars->fear--;
    sgame->ply.fear = objvars->fear / objvars->fearmulti;
  }
  if (objvars->garlic_smell > 0) {
    if (++objvars->garlic_smellmom == objvars->garlic_smellmax) {
      objvars->garlic_smell -= 5;
      objvars->garlic_smellmom = 0;
      if (objvars->garlic_smell < 0) { objvars->garlic_smell = 0; }
      sgame->ply.garlic = objvars->garlic_smell;
    }
  }

  /* lamp */
  if (vg4->input->key_newpressed(sgame->kref.k_toggle_lamp)) {
    if (objvars->map == 0) {
      if (objvars->lamp_ontime != 0) { objvars->lamp_ontime = -objvars->lamp_ontime; } else { objvars->lamp_ontime = 1; }
      sgame->ply.lamp = (objvars->lamp_ontime > 0 ? VG_TRUE : VG_FALSE);
    }
  }

  /* going */
  if (objvars->go != 0) {
    if (objvars->go < 0) { /* going backward */
      objvars->go += 20;
    } else if (objvars->go > 0) { /* going forward */
      objvars->go -= 20;
      if (objvars->go == 0) {
        sgame->ply.x = objvars->pos.x;
        sgame->ply.y = objvars->pos.y;
      }
    }
    sgame->ply.go = objvars->go;
    if (objvars->go != 0) { goto endrun; }
    objvars->mapvisited[objvars->pos.y][objvars->pos.x] |= 128;
  }

  /* turning */
  if (objvars->turn != 0) {
    if (objvars->turn < 0) { /* turning left */
      objvars->turn += 25;
    } else if (objvars->turn > 0) { /* turning right */
      objvars->turn -= 25;
    }   
    sgame->ply.turn = objvars->turn;
    if (objvars->turn != 0) { goto endrun; }
  }

  /* map */
  if (vg4->input->key_newpressed(sgame->kref.k_toggle_map)) {
    if (objvars->map == 0) {
      objvars->map = 1;
      if (objvars->lamp_ontime > 0) { objvars->lamp_ontime = -objvars->lamp_ontime; }
    } else {
      objvars->map = 0;
    }
    sgame->ply.map = objvars->map;
    sgame->ply.lamp = (objvars->lamp_ontime > 0 ? VG_TRUE : VG_FALSE);
  }

  /* map opened? */
  if (objvars->map > 0) {
    /* use jewel? */
    if (vg4->input->key_newpressed(sgame->kref.k_use_item) && objvars->has_jewel) {
      objvars->map = 4;
      objvars->has_jewel = 0;
      sgame->ply.jewel = objvars->has_jewel;
      sgame->ply.map = objvars->map;
    }
    goto endrun;
  }

  /* use item? */
  if (vg4->input->key_newpressed(sgame->kref.k_use_item)) {

    /* garlic */
    if (sgame->maze.map[objvars->pos.y][objvars->pos.x] & ITEM_GARLIC) {
      if (objvars->garlic_smell == 0) {  /* actually not using garlic */
        /* cancel following ghosts */
        { int numberof, ipos;
          unsigned int *idlist;
          struct obj_pos opos;
          numberof = vg4->object->list(OBJID_GHOST, 0, &idlist);
          for (ipos = 0; ipos < numberof; ipos++) {
            opos.dir = 2;
            vg4->object->call_data(vgame, idlist[ipos], &opos);
          }
          if (idlist != NULL) { free(idlist); }
        }
        objvars->garlic_smell = 100;
        objvars->garlic_smellmom = 0;
        sgame->ply.garlic = objvars->garlic_smell;
        sgame->maze.map[objvars->pos.y][objvars->pos.x] &= ~ITEM_GARLIC;
        vg4->audio->play(objvars->sound.eat, VG_FALSE, VG_FALSE);
      }

    /* valerian */
    } else if (sgame->maze.map[objvars->pos.y][objvars->pos.x] & ITEM_VALERIAN) {
      if (objvars->fear >= 10 * objvars->fearmulti) {
        objvars->fear /= 2;
        sgame->ply.fear = objvars->fear / objvars->fearmulti;
        sgame->maze.map[objvars->pos.y][objvars->pos.x] &= ~ITEM_VALERIAN;
        vg4->audio->play(objvars->sound.drink, VG_FALSE, VG_FALSE);
      }

    /* jewel */
    } else if (sgame->maze.map[objvars->pos.y][objvars->pos.x] & ITEM_JEWEL) {
      if (!objvars->has_jewel) {
        objvars->has_jewel = VG_TRUE;
        sgame->ply.jewel = objvars->has_jewel;
        sgame->maze.map[objvars->pos.y][objvars->pos.x] &= ~ITEM_JEWEL;
        vg4->audio->play(objvars->sound.jewel, VG_FALSE, VG_FALSE);
      }
    }
  }

  /* go forward? */
  if (vg4->input->key_pressed(sgame->kref.k_forward)) {
    if (!(sgame->maze.map[objvars->pos.y][objvars->pos.x] & objvars->pos.dir)
        || sgame->maze.map[objvars->pos.y][objvars->pos.x] & (objvars->pos.dir << PORTAL_SHIFT)
       ) {
      if (sgame->maze.map[objvars->pos.y][objvars->pos.x] & (objvars->pos.dir << PORTAL_SHIFT)) {
        objvars->mapvisited[objvars->pos.y][objvars->pos.x] |= objvars->pos.dir;
        vg4->audio->play(objvars->sound.wall_portal, VG_FALSE, VG_FALSE);
      }
      if (objvars->pos.dir == WAND_NORD) {
        objvars->pos.y--;
      } else if (objvars->pos.dir == WAND_OST) {
        objvars->pos.x++;
      } else if (objvars->pos.dir == WAND_SUED) {
        objvars->pos.y++;
      } else if (objvars->pos.dir == WAND_WEST) {
        objvars->pos.x--;
      }
      objvars->go = 100;
      vg4->audio->play(objvars->sound.footstep[objvars->sound.step], VG_FALSE, VG_FALSE);
      objvars->sound.step = !objvars->sound.step;

    } else if (sgame->maze.exit_x == objvars->pos.x
               && sgame->maze.exit_y == objvars->pos.y
               && sgame->maze.exit_dir == objvars->pos.dir) {
      retw = VG_FALSE;
      goto endrun;
    }
  }

  /* go backward? */
  if (vg4->input->key_pressed(sgame->kref.k_backward)) {
    int idir = 0;
    if (objvars->pos.dir == WAND_NORD) {
      idir = WAND_SUED;
    } else if (objvars->pos.dir == WAND_OST) {
      idir = WAND_WEST;
    } else if (objvars->pos.dir == WAND_SUED) {
      idir = WAND_NORD;
    } else if (objvars->pos.dir == WAND_WEST) {
      idir = WAND_OST;
    }

    if (!(sgame->maze.map[objvars->pos.y][objvars->pos.x] & idir)
        || sgame->maze.map[objvars->pos.y][objvars->pos.x] & (idir << PORTAL_SHIFT)
       ) {
      if (sgame->maze.map[objvars->pos.y][objvars->pos.x] & (idir << PORTAL_SHIFT)) {
        objvars->mapvisited[objvars->pos.y][objvars->pos.x] |= idir;
        vg4->audio->play(objvars->sound.wall_portal, VG_FALSE, VG_FALSE);
      }
      if (objvars->pos.dir == WAND_NORD) {
        objvars->pos.y++;
      } else if (objvars->pos.dir == WAND_OST) {
        objvars->pos.x--;
      } else if (objvars->pos.dir == WAND_SUED) {
        objvars->pos.y--;
      } else if (objvars->pos.dir == WAND_WEST) {
        objvars->pos.x++;
      }
      objvars->go = -100;
      sgame->ply.x = objvars->pos.x;
      sgame->ply.y = objvars->pos.y;
      sgame->ply.go = objvars->go;
      vg4->audio->play(objvars->sound.footstep[objvars->sound.step], VG_FALSE, VG_FALSE);
      objvars->sound.step = !objvars->sound.step;

    } else if (sgame->maze.exit_x == objvars->pos.x
               && sgame->maze.exit_y == objvars->pos.y
               && sgame->maze.exit_dir == idir) {
      retw = VG_FALSE;
      goto endrun;
    }
  }

  if (objvars->pos.x < 0 || objvars->pos.x >= sgame->maze.wsize || objvars->pos.y < 0 || objvars->pos.y >= sgame->maze.hsize) {
    retw = VG_FALSE;
    goto endrun;
  }

  /* turn right? */
  if (vg4->input->key_newpressed(sgame->kref.k_turn_right)) {
    if (objvars->pos.dir == WAND_NORD) {
      objvars->pos.dir = WAND_OST;
    } else if (objvars->pos.dir == WAND_OST) {
      objvars->pos.dir = WAND_SUED;
    } else if (objvars->pos.dir == WAND_SUED) {
      objvars->pos.dir = WAND_WEST;
    } else if (objvars->pos.dir == WAND_WEST) {
      objvars->pos.dir = WAND_NORD;
    }
    objvars->turn = 100;
    sgame->ply.dir = objvars->pos.dir;
    sgame->ply.turn = objvars->turn;
  }

  /* turn left? */
  if (vg4->input->key_newpressed(sgame->kref.k_turn_left)) {
    if (objvars->pos.dir == WAND_NORD) {
      objvars->pos.dir = WAND_WEST;
    } else if (objvars->pos.dir == WAND_OST) {
      objvars->pos.dir = WAND_NORD;
    } else if (objvars->pos.dir == WAND_SUED) {
      objvars->pos.dir = WAND_OST;
    } else if (objvars->pos.dir == WAND_WEST) {
      objvars->pos.dir = WAND_SUED;
    }
    objvars->turn = -100;
    sgame->ply.dir = objvars->pos.dir;
    sgame->ply.turn = objvars->turn;
  }

endrun:
  return retw;
}


/* get/set data from/for object-instance, called from vg4->object->call_data() */
static int
f_data(void *vgame, struct VG_Object *objp, void *vptr)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_player *objvars = (struct sobj_player *)objp->opriv;
  struct obj_pos *opos = (struct obj_pos *)vptr;
  int flag;

  if (sgame == NULL || opos == NULL) { return 0; }

  flag = opos->dir;

  memset(opos, 0, sizeof(*opos));
  opos->x = objvars->pos.x;
  opos->y = objvars->pos.y;
  opos->dir = objvars->pos.dir;

  if (flag == 1) {
    opos->img = objvars->img;
  } else {
    opos->img = NULL;
  }

  VG_IMAGECOPY_ATTR_DEFAULT(&opos->attr);
  opos->u_pg = 1;
  opos->u.ply.lamp_ontime = (objvars->lamp_ontime >= 0 ? objvars->lamp_ontime : -objvars->lamp_ontime);
  opos->u.ply.garlic = (objvars->garlic_smell > 0 ? VG_TRUE : VG_FALSE);

  return 1;
}
