/* Copyright 2025 Kurt Nienhaus
 *
 * This is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 * This is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License
 * along with this software.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "main.h"

/* settings.c: load and save system-settings */
extern void settings_load(void);
extern void settings_save(void);

static VG_BOOL cvexec_main_menu(VG_BOOL *);
static VG_BOOL show_help(void);
static VG_BOOL dgexec_sysmenu(void);
static void astack_free(unsigned int, void *);


/* load and execute canvas main-menu; return: VG_TRUE=start-game, VG_FALSE=exit-request */
static VG_BOOL
cvexec_main_menu(VG_BOOL *ishard)
{
  const char *selname;
  struct VG_Canvas *cvas;
  int retw = VG_TRUE;

  if (ishard == NULL) { return VG_FALSE; }
  *ishard = VG_FALSE;

cvmm_start:
  vg4->window->clear();

  cvas = vg4->canvas->load("files/canvas/main-menu/top.cvas", NULL);
  if (cvas == NULL) { return VG_FALSE; }

cvmm_redo:
  if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; goto cvmm_end; }
  if (selname == NULL || *selname == '\0') { goto cvmm_redo; }

  if (strcmp(selname, "bt_help") == 0) {
    if (!show_help()) { retw = VG_FALSE; goto cvmm_end; }

  } else if (strcmp(selname, "bt_sysmenu") == 0) {
    if (!dgexec_sysmenu()) { retw = VG_FALSE; goto cvmm_end; }

  } else if (strcmp(selname, "bt_start") == 0) {
    struct VG_Canvas *cvas_sub;
    cvas_sub = vg4->canvas->load("files/canvas/difficulty/top.cvas", NULL);
    if (cvas_sub != NULL) {
      struct VG_Position possub;
      struct VG_ImagecopyAttrPixel iattr_pixel;
      VG_IMAGECOPY_ATTRPIXEL_DEFAULT(&iattr_pixel);
      iattr_pixel.brightness = 60;
      possub = vg4->canvas->subcanvas(cvas, &iattr_pixel);
cvmm_redo_sub:
      if (!vg4->canvas->exec(cvas_sub, &possub, &selname)) { vg4->canvas->destroy(cvas_sub); retw = VG_FALSE; goto cvmm_end; }
      if (selname == NULL || *selname == '\0') { goto cvmm_redo_sub; }
      if (strcmp(selname, "easy") == 0) {
        *ishard = VG_FALSE;
      } else {
        *ishard = VG_TRUE;
      }
      vg4->canvas->destroy(cvas_sub);
    }
    retw = VG_TRUE;
    goto cvmm_end;

  } else if (strcmp(selname, "bt_exit") == 0) {
    retw = VG_FALSE; 
    goto cvmm_end;
  }

  vg4->canvas->destroy(cvas);
  goto cvmm_start;

cvmm_end:
  vg4->canvas->destroy(cvas);
  return retw;
}


/* show help */
static VG_BOOL
show_help(void)
{
  struct VG_Canvas *cvas;
  const char *selname;
  struct VG_Hash *hvar;
  VG_BOOL retw, asusp;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "title", vg4->mlang->get("menu", "help-title"));
  vg4->hash->setstr(hvar, "text", vg4->mlang->get("menu", "help-text"));

  /* load canvas */
  cvas = vg4->canvas->load("files/canvas/help/top.cvas", hvar);
  if (cvas == NULL) { return VG_FALSE; }

  asusp = vg4->audio->suspend(VG_TRUE);
  retw = VG_TRUE;

  for (;;) {
    /* execute canvas */
    if (!vg4->canvas->exec(cvas, NULL, &selname)) { retw = VG_FALSE; break; }
    if (selname == NULL) { break; }
    if (*selname == '\0' || strcmp(selname, "close") == 0) { break; }
  }

  vg4->canvas->destroy(cvas);
  vg4->hash->destroy(hvar);
  vg4->audio->suspend(asusp);

  return retw;
}


/* execute system-menu dialog */
static VG_BOOL
dgexec_sysmenu(void)
{
  struct VG_Hash *hvar;
  VG_BOOL retw;

  hvar = vg4->hash->create();

  vg4->hash->setstr(hvar, "top:title", vg4->mlang->get("menu", "System-menu"));
  vg4->hash->setstr(hvar, "exit:top:title", vg4->mlang->get("menu", "Quit game?"));
  vg4->hash->setstr(hvar, "volume:top:title", vg4->mlang->get("menu", "Set audio volumes"));
  vg4->hash->setstr(hvar, "keydef:top:title", vg4->mlang->get("menu", "Key Redefinition"));
  vg4->hash->setstr(hvar, "keydef:press:title", vg4->mlang->get("menu", "Press key"));
  vg4->hash->setstr(hvar, "windowsize:top:title", vg4->mlang->get("menu", "Resize window"));
  vg4->hash->setstr(hvar, "winbright:top:title", vg4->mlang->get("menu", "Brightness"));

  vg4->audio->suspend(VG_TRUE);
  retw = vg4->dialog->sysmenu(NULL, NULL, hvar, NULL, NULL, NULL, NULL);
  vg4->audio->suspend(VG_FALSE);
  vg4->hash->destroy(hvar);

  return retw;
}


/* free function for actionstack element-data */
static void
astack_free(unsigned int name_id, void *vdata)
{
  struct VG_Hash *hparm = (struct VG_Hash *)vdata;

  (void)name_id;
  if (hparm != NULL) { vg4->hash->destroy(hparm); }
}


int main(int argc, char **argv) {
  struct s_game sgame;
  VG_BOOL filmskip, ishard;
  enum { GAMEPLAY_NOOP = 0, GAMEPLAY_DONE, GAMEPLAY_MENU } gplay;

  (void)argc; (void)argv;

  /* +++ initializations +++ */

  memset(&sgame, 0, sizeof(sgame));

  if (!VG_init("Black Smurfs")) { exit(1); }
  if (!vg4->window->open(VG_WINDOW_SIZE_HIGH, VG_WINDOW_SCALE_BEST)) { VG_dest(); exit(1); }
  vg4->window->getsize(&sgame.winw, &sgame.winh);
  if (!vg4->audio->open(VG_AUDIO_FREQ_MEDIUM, VG_TRUE)) { VG_dest(); exit(1); }
  vg4->input->mouse_grabbing(VG_FALSE);

  vg4->mlang->fb_locale("en");
  vg4->mlang->add("files/mlang");

  /* setting keys */

  /* quit with ALT+Q, not changeable, local key */
  if ((sgame.kref.k_quit_lalt = vg4->input->key_insert("Quit-LALT", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_lalt, VG_INPUT_KBDCODE_LALT);
  if ((sgame.kref.k_quit_q = vg4->input->key_insert("Quit-Q", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_quit_q, VG_INPUT_KBDCODE_Q);
  /* system-menu with ESC, not changeable, local key */
  if ((sgame.kref.k_sysmenu = vg4->input->key_insert("System-menu", VG_FALSE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_sysmenu, VG_INPUT_KBDCODE_ESCAPE);
  /* help with H, changeable, local key */
  if ((sgame.kref.k_help = vg4->input->key_insert(vg4->mlang->get("menu", "Help"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_help, VG_INPUT_KBDCODE_H);
  /* pause with P, changeable, local key */
  if ((sgame.kref.k_pause = vg4->input->key_insert(vg4->mlang->get("menu", "Pause"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_pause, VG_INPUT_KBDCODE_P);

  /* go-left with cursor-key LEFT, changeable, local key */
  if ((sgame.kref.k_go_left = vg4->input->key_insert(vg4->mlang->get("menu", "Go left"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_go_left, VG_INPUT_KBDCODE_LCURS);
  /* go-right with cursor-key RIGHT, changeable, local key */
  if ((sgame.kref.k_go_right = vg4->input->key_insert(vg4->mlang->get("menu", "Go right"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_go_right, VG_INPUT_KBDCODE_RCURS);
  /* stop-dice with Spacebar, changeable, local key */
  if ((sgame.kref.k_stop = vg4->input->key_insert(vg4->mlang->get("menu", "Stop dice"), VG_TRUE, VG_FALSE)) == 0) { VG_dest(); exit(1); }
  vg4->input->key_setkbd(sgame.kref.k_stop, VG_INPUT_KBDCODE_SPACE);

  /* create collision-tag with size of window */
  sgame.coll_tag = vg4->collision->create(NULL, 0, 0);

  /* set actionstack's free function */
  vg4->actionstack->freefunction(astack_free);

  /* load settings */
  settings_load();

  /* play intro */
  if (!vg4->film->play("files/film/intro", NULL, &filmskip, NULL)) { VG_dest(); exit(0); }

main_menu:
  if (!cvexec_main_menu(&ishard)) { goto endgame; }

  /* +++ create object-instances +++ */

  sgame.smurf_total = sgame.smurf_pending = sgame.smurf_saved = 0;
  sgame.game_won = VG_FALSE;

  /* create playfield */
  if (objnew_PLAYFIELD(&sgame, ishard) == 0) { goto endgame; }

  /* create the management of smurfs */
  if (!objmgmt_SMURF(&sgame)) { goto endgame; }

  /* create papasmurf and fly */
  if (objnew_SMURF(&sgame, SPECIES_PAPASMURF) == 0) { goto endgame; }
  if (objnew_SMURF(&sgame, SPECIES_FLY) == 0) { goto endgame; }

  /* create dirsel */
  if (objnew_DIRSEL(&sgame) == 0) { goto endgame; }

  /* create mousesel */
  if (objnew_MOUSESEL(&sgame) == 0) { goto endgame; }

  /* create dice */
  if (objnew_DICE(&sgame) == 0) { goto endgame; }

  /* create animation */
  if (objnew_ANIMATION(&sgame) == 0) { goto endgame; }

  /* create music */
  if (objnew_MUSIC(&sgame) == 0) { goto endgame; }

  /* push starting animation into actionstack */
  { struct VG_Hash *hparm = vg4->hash->create();
    vg4->hash->setint(hparm, "num", ANIMATION_STARTING);
    vg4->actionstack->push(UART_ANIMATION, hparm);
  }

  /* +++ game loop +++ */

  gplay = GAMEPLAY_NOOP;
  for (;;) {
    if (!vg4->input->update(VG_TRUE)) { goto endgame; }

    /* quit to main-menu? */
    if (vg4->input->key_newpressed(sgame.kref.k_quit_q) && vg4->input->key_pressed(sgame.kref.k_quit_lalt)) {
      gplay = GAMEPLAY_MENU;
      break;
    }

    /* help? */
    if (vg4->input->key_newpressed(sgame.kref.k_help)) {
      if (!show_help()) { goto endgame; }
    }

    /* pause? */
    if (vg4->input->key_newpressed(sgame.kref.k_pause)) {
      if (!vg4->misc->pause()) { goto endgame; }
    }

    /* system-menu? */
    if (vg4->input->key_newpressed(sgame.kref.k_sysmenu)) {
      if (!dgexec_sysmenu()) { goto endgame; }
    }

    /* call object-management-function f_run() for all object-managements */
    vg4->object->call_mgmt_run(&sgame);

    /* call f_run() of all object-instances */
    vg4->object->call_run(&sgame);

    /* call f_draw() of all object-instances */
    vg4->window->clear();
    vg4->object->call_draw(&sgame);

    /* flush contents to window and wait */
    vg4->window->flush();
    vg4->misc->wait_time(70);

    /* game won? */
    if (sgame.game_won) { gplay = GAMEPLAY_DONE; break; }
  }

  /* reset window parameters */
  vg4->window->setattr(NULL);

  /* destroy object-instances and -management */
  vg4->object->destroy_objid(&sgame, OBJID_MUSIC, 0);
  vg4->object->destroy_objid(&sgame, OBJID_ANIMATION, 0);
  vg4->object->destroy_objid(&sgame, OBJID_DICE, 0);
  vg4->object->destroy_objid(&sgame, OBJID_MOUSESEL, 0);
  vg4->object->destroy_objid(&sgame, OBJID_DIRSEL, 0);
  vg4->object->mgmt_destroy(OBJID_SMURF);
  vg4->object->destroy_objid(&sgame, OBJID_SMURF, 0);
  vg4->actionstack->clear();

  if (gplay == GAMEPLAY_DONE) {  /* game won */
    struct VG_Hash *hparm = vg4->hash->create();
    char buf[64];
    struct VG_Image *wimg;
    unsigned int name_id;

    if (objnew_ANIMATION(&sgame) == 0) { goto endgame; }

    vg4->hash->setint(hparm, "num", ANIMATION_ENDING);
    snprintf(buf, sizeof(buf), "%d / %d\n\n%s", sgame.smurf_saved, sgame.smurf_total, vg4->mlang->get("game", "smurf-saved"));
    vg4->hash->setstr(hparm, "text", buf);
    vg4->actionstack->push(UART_ANIMATION, hparm);

    wimg = vg4->window->clone(NULL, NULL);
    for (;;) {
      if (!vg4->input->update(VG_TRUE)) { goto endgame; }
      if (vg4->input->key_newpressed(sgame.kref.k_stop)) { break; }
      vg4->actionstack->get(&name_id);
      if (name_id == 0) { break; }  /* animation done */
      vg4->object->call_run(&sgame);
      vg4->window->clear();
      vg4->window->copy(wimg, NULL, NULL);
      vg4->object->call_draw(&sgame);
      vg4->window->flush();
      vg4->misc->wait_time(70);
    }
    vg4->image->destroy(wimg);
    vg4->actionstack->clear();

    vg4->object->destroy_objid(&sgame, OBJID_ANIMATION, 0);
  }

  vg4->object->destroy_objid(&sgame, OBJID_PLAYFIELD, 0);

  goto main_menu;

endgame:
  /* save settings */
  settings_save();

  /* destroy and exit */
  VG_dest();
  exit(0);
}
