/* object OBJID_SMURF */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "obj-smurf.h"

unsigned int objnew_SMURF(void *, int);

static void f_free(void *, void *);
static VG_BOOL f_run(void *, struct VG_Object *);
static void f_draw(void *, struct VG_Object *);
static int f_data(void *, struct VG_Object *, void *);

#define MAX_FUEL 5
#define MAX_VANISH 50


/* export-function to create a new object-instance of OBJID_SMURF
 * @param vgame       private structure of the game
 * @param species     smurf-species (SPECIES_*)
 * @return  instance-ID or 0 = error
 */
unsigned int
objnew_SMURF(void *vgame, int species)
{
  const int coll_percent = 90;
  struct s_game *sgame = (struct s_game *)vgame;
  struct VG_Object *objp;
  struct sobj_smurf *objvars;
  int step;

  if (sgame == NULL) { return 0; }

  /* create private struct */

  objvars = calloc(1, sizeof(*objvars));
  if (objvars == NULL) { return 0; }

  /* load images and audio */
  if (species == SPECIES_FLY) {
    objvars->imgp_normal = vg4->image->load("files/images/smurfs/fly.bmp");
    if (objvars->imgp_normal == NULL) { f_free(vgame, objvars); return 0; }
    objvars->papa.imgp_empty = objvars->imgp_black = objvars->imgp_mouseover = NULL;
    objvars->snd_flymoving = vg4->audio->load("files/audio/fly.wav", 100, VG_AUDIO_VOLUME_SOUND);
  } else if (species == SPECIES_PAPASMURF || species == SPECIES_PAPASMURF_BLACK) {
    objvars->imgp_normal = vg4->image->load("files/images/smurfs/papasmurf-blue.bmp");
    if (objvars->imgp_normal == NULL) { f_free(vgame, objvars); return 0; }
    objvars->papa.imgp_empty = vg4->image->load("files/images/smurfs/papasmurf-blue-empty.bmp");
    if (objvars->papa.imgp_empty == NULL) { f_free(vgame, objvars); return 0; }
    objvars->imgp_mouseover = vg4->image->load("files/images/smurfs/papasmurf-blue-mouseover.bmp");
    if (objvars->imgp_mouseover == NULL) { f_free(vgame, objvars); return 0; }
    objvars->imgp_black = vg4->image->load("files/images/smurfs/papasmurf-black.bmp");
    if (objvars->imgp_black == NULL) { f_free(vgame, objvars); return 0; }
    objvars->papa.fuel = MAX_FUEL;
  } else {
    objvars->imgp_normal = vg4->image->load("files/images/smurfs/smurf-blue.bmp");
    if (objvars->imgp_normal == NULL) { f_free(vgame, objvars); return 0; }
    objvars->imgp_mouseover = vg4->image->load("files/images/smurfs/smurf-blue-mouseover.bmp");
    if (objvars->imgp_mouseover == NULL) { f_free(vgame, objvars); return 0; }
    objvars->imgp_black = vg4->image->load("files/images/smurfs/smurf-black.bmp");
    if (objvars->imgp_black == NULL) { f_free(vgame, objvars); return 0; }
    objvars->papa.imgp_empty = NULL;
  }

  /* select starting field */
  objvars->pfstep = -1;
  for (step = 0; step < PLAYFIELD_STEPS; step++) {
    if (sgame->pfmap[step].starting > 0) {
      if (species == SPECIES_FLY) {
        if (sgame->pfmap[step].starting == 3) { objvars->pfstep = step; }
      } else if (species == SPECIES_PAPASMURF || species == SPECIES_PAPASMURF_BLACK) {
        if (sgame->pfmap[step].starting == 2) { objvars->pfstep = step; }
      } else {
        if (sgame->pfmap[step].starting == 1) { objvars->pfstep = step; }
      }
    }
  }

  vg4->image->getsize(objvars->imgp_normal, NULL, &objvars->rect.w, &objvars->rect.h);
  objvars->rect.x = sgame->pfmap[objvars->pfstep].xpos - objvars->rect.w / 2;
  objvars->rect.y = sgame->pfmap[objvars->pfstep].ypos - objvars->rect.h / 2;

  /* create object-instance */
  objp = vg4->object->create(OBJID_SMURF, species, 0, (species == SPECIES_FLY ? 3 : 2), objvars);
  objp->f_free = f_free;
  objp->f_run = f_run;
  objp->f_draw = f_draw;
  objp->f_data = f_data;

  /* insert object-instance into collision-tag if smurf */
  if (objp->subid == SPECIES_SMURF || objp->subid == SPECIES_PAPASMURF) {
    vg4->collision->insert(sgame->coll_tag, objp->instanceid, &objvars->rect, coll_percent);
  }

  /* is field is occupated, move it one step */
  { int numberof, ipos;
    unsigned int *idlist;
    struct VG_Object *objp2 = NULL;
    struct sobj_smurf *objvars2;

    numberof = vg4->object->list(OBJID_SMURF, 0, &idlist);
    for (ipos = 0; ipos < numberof; ipos++) {
      if (idlist[ipos] == objp->instanceid) { continue; }
      objp2 = vg4->object->instance_getobj(idlist[ipos]);
      objvars2 = (struct sobj_smurf *)objp2->opriv;
      if (objvars->pfstep == objvars2->pfstep) { break; }
    }
    if (idlist != NULL) { free(idlist); }

    if (ipos < numberof) {  /* move objp2 one step */
      struct VG_Hash *hparm = vg4->hash->create();
      vg4->hash->setint(hparm, "instanceid", objp2->instanceid);
      vg4->hash->setint(hparm, "steps", 1);
      vg4->hash->setint(hparm, "direction", 0);
      vg4->actionstack->push(UART_MOVSMURF, hparm);
    }
  }

  return objp->instanceid;
}


/* free private struct of object-instance, called from vg4->object->destroy() */
static void
f_free(void *vgame, void *opriv)
{
  struct sobj_smurf *objvars = (struct sobj_smurf *)opriv;

  (void)vgame;

  /* free private struct */
  if (objvars->imgp_normal != NULL) { vg4->image->destroy(objvars->imgp_normal); }
  if (objvars->papa.imgp_empty != NULL) { vg4->image->destroy(objvars->papa.imgp_empty); }
  if (objvars->imgp_mouseover != NULL) { vg4->image->destroy(objvars->imgp_mouseover); }
  if (objvars->imgp_black != NULL) { vg4->image->destroy(objvars->imgp_black); }
  if (objvars->snd_flymoving > 0) { vg4->audio->unload(objvars->snd_flymoving); }
  free(objvars);
}


/* move object-instance, called from vg4->object->call_run() */
static VG_BOOL
f_run(void *vgame, struct VG_Object *objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_smurf *objvars = (struct sobj_smurf *)objp->opriv;
  struct VG_Hash *hparm;
  unsigned int name_id;
  struct VG_Rect rct1, rct2;
  int pfstep1, pfstep2;

  if (sgame == NULL) { return VG_TRUE; }

  hparm = vg4->actionstack->get(&name_id);
  if (name_id != UART_MOVSMURF) { return VG_TRUE; }
  if ((unsigned int)vg4->hash->getint(hparm, "instanceid") != objp->instanceid) { return VG_TRUE; }

  /* move smurf-instance */
  while (objvars->pos.movmax == 0 || objvars->pos.movmom == objvars->pos.movmax) {
    if (objvars->pos.rectmov != NULL) { free(objvars->pos.rectmov); objvars->pos.rectmov = NULL; }

    /* target reached? */
    if (vg4->hash->getint(hparm, "steps") == 0) {
      vg4->actionstack->pop();
      name_id = 0;
      if (sgame->pfmap[objvars->pfstep].village) {  /* village reached */
        vg4->object->destroy(vgame, objp->instanceid);
        sgame->smurf_saved++;
        return VG_TRUE;
      }

      /* fill bellows if papasmurf is in sarsaparille */
      if ((objp->subid == SPECIES_PAPASMURF || objp->subid == SPECIES_PAPASMURF_BLACK)
          && sgame->pfmap[objvars->pfstep].sarsaparille
         ) {
        objvars->papa.fuel = MAX_FUEL;
      }
      break;
    }

    /* fly: play audio */
    if (objp->subid == SPECIES_FLY) {
      if (objvars->snd_flymoving > 0 && !vg4->audio->is_playing(objvars->snd_flymoving, NULL)) {
        vg4->audio->play(objvars->snd_flymoving, VG_TRUE, VG_FALSE);
      }
    }

    /* select direction? */
    if (vg4->hash->getint(hparm, "direction") == 0 && sgame->pfmap[objvars->pfstep].next[1] >= 0) {
      check_direction(sgame, objvars->pfstep, vg4->hash->getint(hparm, "steps"), objp->subid, &pfstep1, &pfstep2);

      if (pfstep1 >= 0 && pfstep2 >= 0) {
        if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK || objp->subid == SPECIES_FLY) {
          int numberof, ipos;
          unsigned int *idlist;
          struct VG_Object *objp2;
          struct sobj_smurf *objvars2;
          int pf1, pf2;

          pf1 = pf2 = -1;
          numberof = vg4->object->list(OBJID_SMURF, 0, &idlist);
          for (ipos = 0; ipos < numberof; ipos++) {
            if (idlist[ipos] == objp->instanceid) { continue; }
            objp2 = vg4->object->instance_getobj(idlist[ipos]);
            if (objp2->subid != SPECIES_SMURF && objp2->subid != SPECIES_PAPASMURF) { continue; }
            objvars2 = (struct sobj_smurf *)objp2->opriv;
            if (objvars->pfstep == objvars2->pfstep) { break; }
            if (pfstep1 == objvars2->pfstep) { pf1 = pfstep1; }
            if (pfstep2 == objvars2->pfstep) { pf2 = pfstep2; }
          }
          if (idlist != NULL) { free(idlist); }

          if (pf1 >= 0 && pf2 >= 0) {
            vg4->hash->setint(hparm, "direction", ZUFALL(1, 2));
          } else if (pf1 >= 0) {
            vg4->hash->setint(hparm, "direction", 1);
          } else if (pf2 >= 0) {
            vg4->hash->setint(hparm, "direction", 2);
          } else {
            vg4->hash->setint(hparm, "direction", ZUFALL(1, 2));
          }

        } else {
          struct VG_Hash *hparm2 = vg4->hash->create();
          vg4->hash->setint(hparm2, "instanceid", objp->instanceid);
          vg4->actionstack->push(UART_DIRSEL, hparm2);
          return VG_TRUE;
        }

      } else if (pfstep1 >= 0) {  /* left */
        vg4->hash->setint(hparm, "direction", 1);
      } else if (pfstep2 >= 0) {  /* right */
        vg4->hash->setint(hparm, "direction", 2);
      } else {  /* impossible */
        vg4->hash->setint(hparm, "steps", 0);
        continue;
      }
    }

    /* prepare moving to the next step */
    vg4->hash->setint(hparm, "steps", vg4->hash->getint(hparm, "steps") - 1);
    rct1.x = sgame->pfmap[objvars->pfstep].xpos; rct1.w = 1;
    rct1.y = sgame->pfmap[objvars->pfstep].ypos; rct1.h = 1;
    if (vg4->hash->getint(hparm, "direction") == 2) {  /* to the right */
      objvars->pfstep = sgame->pfmap[objvars->pfstep].next[1];
    } else {  /* default or to the left */
      objvars->pfstep = sgame->pfmap[objvars->pfstep].next[0];
    }
    vg4->hash->setint(hparm, "direction", 0);
    rct2.x = sgame->pfmap[objvars->pfstep].xpos; rct2.w = 1;
    rct2.y = sgame->pfmap[objvars->pfstep].ypos; rct2.h = 1;
    objvars->pos.movmax = vg4->misc->line_positions(&rct1, &rct2, 5, &objvars->pos.rectmov);
    objvars->pos.movmom = 0;
  }

  if (name_id > 0) {  /* move */
    objvars->rect.x = objvars->pos.rectmov[objvars->pos.movmom].x - objvars->rect.w / 2;
    objvars->rect.y = objvars->pos.rectmov[objvars->pos.movmom].y - objvars->rect.h / 2;
    objvars->pos.movmom++;

  } else {  /* check if the target position is occupated */
    int numberof, ipos;
    unsigned int *idlist;
    struct VG_Object *objp2, *objp_smurf, *objp_fly;
    struct sobj_smurf *objvars2;
    VG_BOOL is_sarsaparille;

    if (objp->subid == SPECIES_FLY && objvars->pos.movmax > 0) { vg4->audio->stop(objvars->snd_flymoving, VG_FALSE); }
    objp_smurf = objp_fly = NULL;

    /* occupated? */
    numberof = vg4->object->list(OBJID_SMURF, 0, &idlist);
    for (ipos = 0; ipos < numberof; ipos++) {
      if (idlist[ipos] == objp->instanceid) { continue; }
      objp2 = vg4->object->instance_getobj(idlist[ipos]);
      objvars2 = (struct sobj_smurf *)objp2->opriv;
      if (objvars->pfstep == objvars2->pfstep) {  /* occupated */
        if (objp2->subid == SPECIES_FLY) { objp_fly = objp2; } else { objp_smurf = objp2; }
      }
    }
    if (idlist != NULL) { free(idlist); }

    /* smurf becomes blue if in sarsaparille */
    is_sarsaparille = VG_FALSE;
    if (sgame->pfmap[objvars->pfstep].sarsaparille) {
      if (objp->subid == SPECIES_SMURF_BLACK) {
        objp->subid = SPECIES_SMURF;
        is_sarsaparille = VG_TRUE;
      } else if (objp->subid == SPECIES_PAPASMURF_BLACK) {
        objp->subid = SPECIES_PAPASMURF;
        is_sarsaparille = VG_TRUE;
      }
    }

    objp2 = objp_smurf;
    if (objp2 == NULL) { objp2 = objp_fly; objp_fly = NULL; }
    if (objp2 != NULL) {  /* occupated by objp2 */
      int animation;

      objvars2 = (struct sobj_smurf *)objp2->opriv;

      /* move objp2 one step */
      if (objp->subid != SPECIES_FLY && objp2->subid != SPECIES_FLY) {
        struct VG_Hash *hparm2 = vg4->hash->create();
        vg4->hash->setint(hparm2, "instanceid", objp2->instanceid);
        vg4->hash->setint(hparm2, "steps", 1);
        vg4->hash->setint(hparm2, "direction", 0);
        vg4->actionstack->push(UART_MOVSMURF, hparm2);
        for (;;) {  /* don't stop on invalid target step */
          check_direction(sgame, objvars2->pfstep, vg4->hash->getint(hparm2, "steps"), objp2->subid, &pfstep1, &pfstep2);
          if (pfstep1 >= 0 || pfstep2 >= 0) { break; }
          vg4->hash->setint(hparm2, "steps", vg4->hash->getint(hparm2, "steps") + 1);
        }
      }

      /* animation "change to blue", if in sarsaparille */
      if (is_sarsaparille) {
        struct VG_Hash *hparm2 = vg4->hash->create();
        if (objp->subid == SPECIES_PAPASMURF) {
          animation = ANIMATION_SMURF_BLUE;
        } else {
          animation = ANIMATION_SMURF_BLUE;
        }
        vg4->hash->setint(hparm2, "num", animation);
        vg4->actionstack->push(UART_ANIMATION, hparm2);
      }

      /* change one of the smurfs to black or blue? */
      animation = 0;
      if (objp->subid == SPECIES_PAPASMURF && objp_fly != NULL) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_PAPASMURF_BLACK;
      } else if (objp->subid == SPECIES_PAPASMURF && objp2->subid == SPECIES_SMURF_BLACK) {
        if (objvars->papa.fuel > 0) {
          animation = ANIMATION_SMURF_BLUE;
          objp2->subid = SPECIES_SMURF;
          objvars->papa.fuel--;
        } else {
          animation = ANIMATION_SMURF_BLACK;
          objp->subid = SPECIES_PAPASMURF_BLACK;
        }
      } else if (objp->subid == SPECIES_SMURF_BLACK && objp2->subid == SPECIES_PAPASMURF) {
        if (objvars2->papa.fuel > 0) {
          animation = ANIMATION_SMURF_BLUE;
          objp->subid = SPECIES_SMURF;
          objvars2->papa.fuel--;
        } else {
          animation = ANIMATION_SMURF_BLACK;
          objp2->subid = SPECIES_PAPASMURF_BLACK;
        }
      } else if (objp->subid == SPECIES_SMURF
                 && (objp2->subid == SPECIES_SMURF_BLACK || objp2->subid == SPECIES_PAPASMURF_BLACK)
                ) {
        animation = ANIMATION_SMURF_BLACK;
        objp->subid = SPECIES_SMURF_BLACK;
      } else if ((objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK)
                 && objp2->subid == SPECIES_SMURF
                ) {
        animation = ANIMATION_SMURF_BLACK;
        objp2->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_SMURF && objp2->subid == SPECIES_FLY) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_FLY && objp2->subid == SPECIES_SMURF) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp2->subid = SPECIES_SMURF_BLACK;
      } else if (objp->subid == SPECIES_PAPASMURF && objp2->subid == SPECIES_FLY) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp->subid = SPECIES_PAPASMURF_BLACK;
      } else if (objp->subid == SPECIES_FLY && objp2->subid == SPECIES_PAPASMURF) {
        animation = ANIMATION_SMURF_BLACKFLY;
        objp2->subid = SPECIES_PAPASMURF_BLACK;
      }

      if (animation > 0) {
        struct VG_Hash *hparm2 = vg4->hash->create();
        vg4->hash->setint(hparm2, "num", animation);
        vg4->actionstack->push(UART_ANIMATION, hparm2);
      }

    } else if (is_sarsaparille) {  /* animation "change to blue", if in sarsaparille */
      int animation;
      struct VG_Hash *hparm2 = vg4->hash->create();
      if (objp->subid == SPECIES_PAPASMURF) {
        animation = ANIMATION_SMURF_BLUE;
      } else {
        animation = ANIMATION_SMURF_BLUE;
      }
      vg4->hash->setint(hparm2, "num", animation);
      vg4->actionstack->push(UART_ANIMATION, hparm2);
    }   

    if (objp->subid == SPECIES_SMURF || objp->subid == SPECIES_PAPASMURF) {
      vg4->collision->setpos(sgame->coll_tag, objp->instanceid, &objvars->rect, NULL);
    }
  }

  return VG_TRUE;
}


/* draw object-instance, called from vg4->object->call_draw() */
static void
f_draw(void *vgame, struct VG_Object *objp)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_smurf *objvars = (struct sobj_smurf *)objp->opriv;
  struct VG_Hash *hparm;
  unsigned int name_id;
  struct VG_Position posi;
  struct VG_Image *imgp;
  struct VG_ImagecopyAttr iattr;
  VG_BOOL dirsel, mouseover;

  if (sgame == NULL) { return; }

  dirsel = VG_FALSE;
  hparm = vg4->actionstack->get(&name_id);
  if (name_id == UART_DIRSEL && (unsigned int)vg4->hash->getint(hparm, "instanceid") == objp->instanceid) { dirsel = VG_TRUE; }

  mouseover = VG_FALSE;
  if (objvars->mouseover) { mouseover = VG_TRUE; }

  /* draw smurf-instance */
  if (objp->subid == SPECIES_FLY) {
    imgp = objvars->imgp_normal;
  } else if (objp->subid == SPECIES_SMURF_BLACK || objp->subid == SPECIES_PAPASMURF_BLACK) {
    imgp = objvars->imgp_black;
  } else if (dirsel || mouseover) {
    imgp = objvars->imgp_mouseover;
    objvars->mouseover = VG_FALSE;
  } else if (objp->subid == SPECIES_PAPASMURF && objvars->papa.fuel == 0) {
    imgp = objvars->papa.imgp_empty;
  } else {
    imgp = objvars->imgp_normal;
  }

  VG_IMAGECOPY_ATTR_DEFAULT(&iattr);
  if (sgame->pfmap[objvars->pfstep].mirrordraw) {
    iattr.image.flip = VG_AXE_VERTICAL;
  }

  if (objvars->vanish > 0) {
    if (--objvars->vanish == 0) { vg4->object->destroy(vgame, objp->instanceid); return; }
    iattr.pixel.opaqueness = 100 * objvars->vanish / MAX_VANISH;
  }

  vg4->window->copy(imgp, vg4->misc->rect2position(&posi, &objvars->rect), &iattr);

  if (objp->subid == SPECIES_PAPASMURF && mouseover) {
    char buf[32];
    snprintf(buf, sizeof(buf), "%d%%", objvars->papa.fuel * 100 / MAX_FUEL);
    imgp = vg4->font->totext(buf, "[fgcolor=0xff0000 bgcolor=0x0000ff]", NULL, NULL, NULL);
    if (imgp != NULL) { 
      posi.x = objvars->rect.x + objvars->rect.w / 2;
      posi.y = objvars->rect.y + objvars->rect.h;
      posi.pos = VG_POS_CENTERED;
      vg4->window->copy(imgp, &posi, NULL);
      vg4->image->destroy(imgp);
    }
  }
}


/* get/set data from/for object-instance, called from vg4->object->call_data() */
static int
f_data(void *vgame, struct VG_Object *objp, void *vptr)
{
  struct s_game *sgame = (struct s_game *)vgame;
  struct sobj_smurf *objvars = (struct sobj_smurf *)objp->opriv;
  int *pfstep = (int *)vptr;

  if (pfstep != NULL) { *pfstep = 0; }
  if (sgame == NULL) { return 0; }

  if (pfstep == NULL) {
    objvars->vanish = MAX_VANISH;
  } else {
    *pfstep = objvars->pfstep;
  }

  return 1;
}
