#!/bin/sh
#
# config.sh: configure specific values for your system and create config
#


# set use_ip6_only=1 if you want to use only ipv6
# set use_ip6_only=0 if you want to use ipv4/ipv6 (ipv6 only if ipv4 fails)
use_ip6_only=0


# read in parameters
read CC
read CPP
read libpath
read incpath
read binpath

# check echo
ECHO="echo"; ECHOSTRIP="\c"
test "`$ECHO \"$ECHOSTRIP\"`" != "" && ECHO="echo -e"
test "`$ECHO \"$ECHOSTRIP\"`" != "" && { ECHO="echo -n"; ECHOSTRIP=""; }


# ***** Check for system parameters *****

echo
echo "### Testing your system ... ###"
rm -f config config.log vgag-config.sh _vgc
echo "####################" >> config.log
echo "### System Tests ###" >> config.log
echo "####################" >> config.log
cat vgag-config.tmpl | sed -e "s%##LIBPATH##%$libpath%" > vgag-config.sh
cat vgag-config.sh | sed -e "s%##BINPATH##%$binpath%" > _vgc; mv -f _vgc vgag-config.sh


# +++ check for kernel name +++

rm -f _uname.h
kernname=`uname`
test "$kernname" = "SunOS" && kernname="SOLARIS"
echo "#ifndef $kernname" > _uname.h
echo "  #define $kernname" >> _uname.h
echo "#endif" >> _uname.h
(echo; echo "*** Kernel name: $kernname ***") >> config.log
(echo; echo "####################") >> config.log


# +++ check for existence of IPv6 +++

rm -f netw.h
$ECHO "have IPv6? $ECHOSTRIP"
(echo; echo "*** Have IPv6? ***") >> config.log
cat > _xt.c <<_EOF_001
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>
int main() {
#ifndef AF_INET6
  #error "No IPv6 found"
#endif
  return(0);
}
_EOF_001
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
$CC -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  echo "no"
  echo "==> no" >> config.log
  use_ip6_only=2
else
  echo "yes"
  echo "==> yes" >> config.log
fi
rm -f _xt.c _xt.o

if test $use_ip6_only -eq 0
then
  echo "set: IPv4 (IPv6 only if IPv4 fails)"
  (echo; echo "==> set: IPv4 (IPv6 only if IPv4 fails) <==") >> config.log
elif test $use_ip6_only -eq 2
then
  echo "set: IPv4"
  (echo; echo "==> set: IPv4 <==") >> config.log
else
  echo "set: IPv6"
  (echo; echo "==> set: IPv6 <==") >> config.log
fi
if test $use_ip6_only -eq 1
then
  use_ip6_only="#define USE_AF_VER6"
else
  use_ip6_only="#undef USE_AF_VER6"
fi
echo "$use_ip6_only" >> netw.h
(echo; echo "####################") >> config.log


# +++ check for existence of sa_len in struct sockaddr +++

$ECHO "have sa_len in struct sockaddr? $ECHOSTRIP"
(echo; echo "*** Have sa_len in struct sockaddr? ***") >> config.log
cat > _xt.c <<_EOF_002
#include <sys/types.h>
#include <sys/socket.h>
int main() {
  unsigned int i=sizeof(((struct sockaddr *)0)->sa_len);
  return(0);
}
_EOF_002
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
$CC -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  echo "no"
  (echo; echo "==> no <==") >> config.log
  erg1="#undef HAVE_SOCKADDR_SA_LEN"
else
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  erg1="#define HAVE_SOCKADDR_SA_LEN"
fi
rm -f _xt.c _xt.o
echo "$erg1" >> netw.h
(echo; echo "####################") >> config.log


# +++ check for existence of function getaddrinfo +++

$ECHO "have getaddrinfo? $ECHOSTRIP"
(echo; echo "*** Have getaddrinfo? ***") >> config.log
cat > _xt.c <<_EOF_003
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
_EOF_003
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "Testing for existence of addrinfo while compiling: $CPP _xt.c") >> config.log
if $CPP _xt.c | grep addrinfo > /dev/null
then
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  erg1="#define HAVE_ADDRINFO_STRUCT"
else
  echo "no"
  (echo; echo "==> no <==") >> config.log
  erg1="#undef HAVE_ADDRINFO_STRUCT"
fi
rm -f _xt.c
echo "$erg1" >> netw.h
(echo; echo "####################") >> config.log


# +++ check for existence of socklen_t +++

$ECHO "have socklen_t? $ECHOSTRIP"
(echo; echo "*** Have socklen_t? ***") >> config.log
cat > _xt.c <<_EOF_004
#include <sys/types.h>
#include <sys/socket.h>
_EOF_004
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "Testing for existence of socklen_t while compiling: $CPP _xt.c") >> config.log
if $CPP _xt.c | grep socklen_t > /dev/null
then
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  erg1="#define HAVE_SOCKLEN_T"
else
  echo "no"
  (echo; echo "==> no <==") >> config.log
  erg1="#undef HAVE_SOCKLEN_T"
fi
rm -f _xt.c
echo "$erg1" >> netw.h
(echo; echo "####################") >> config.log


# +++ find libraries for network +++

rm -f _netw
$ECHO "looking for network libraries: $ECHOSTRIP"
(echo; echo "*** Looking for network libraries ***") >> config.log
cat > _xt.c <<_EOF_005
int main() {
  return(0);
}
_EOF_005
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
erg1="-lsocket -lnsl -lresolv"
(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt $erg1") >> config.log
$CC _xt.c -o _xt $erg1 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  echo "==> wrong" >> config.log
  erg1="-lsocket -lnsl"
  (echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt $erg1") >> config.log
  $CC _xt.c -o _xt $erg1 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -ne 0
  then
    echo "==> wrong" >> config.log
    erg1="-lsocket"
    (echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt $erg1") >> config.log
    $CC _xt.c -o _xt $erg1 2>> config.log
    x=$?
    echo "<-- End compiling error messages" >> config.log
    if [ "$x" -ne 0 ]
    then
      echo "==> wrong" >> config.log
      erg1=""
    fi
  else
    echo "==> ok" >> config.log
  fi
else
  echo "==> ok" >> config.log
fi
echo "\"$erg1\" - done"
rm -f _xt.c _xt.o _xt
echo "$erg1" > _netw
cat vgag-config.sh | sed -e "s/##LIBS##/$erg1/" > _vgc; mv -f _vgc vgag-config.sh
(echo; echo "==> using network libraries: \"$erg1\" <==") >> config.log
(echo; echo "####################") >> config.log


# +++ get time select() needs +++

rm -f _t_of_sel.h
$ECHO "get time select needs: $ECHOSTRIP"
(echo; echo "*** Get time select needs ***") >> config.log
cat > _xt.c <<_EOF_006
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
int main() {
  struct timeval tv;
  struct timezone tz;
  int ms,ws,i1;
  if (gettimeofday(&tv,&tz)!=0) {return(1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  for (i1=0;i1<100;i1++) {
    tv.tv_usec=10000;
    tv.tv_sec=0;
    select(0,NULL,NULL,NULL,&tv);
  }
  if (gettimeofday(&tv,&tz)!=0) {return(1);}
  ws=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ws<ms) {ws+=60000;}
  ws-=ms;
  ws-=1000;
  ws/=100;
  printf("%d\n",ws);
  return(0);
}
_EOF_006
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error get time for select (see config.log)"
  echo "==> Compiling error: aborting!" >> config.log
  exit 1
fi
(echo; echo "--> Begin executing ./_xt") >> config.log
erg1=`./_xt 2>> config.log`
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error get time for select (see config.log)"
  echo "==> Executing error: aborting!" >> config.log
  exit 1
fi
echo "#define TIME_OF_SELECT $erg1" > _t_of_sel.h
echo "\"$erg1 milliseconds\" - done"
(echo; echo "==> $erg1 milliseconds <==") >> config.log
rm -f _xt.c _xt.o _xt
(echo; echo "####################") >> config.log


# +++ find endian.h +++

rm -f _endian.h
$ECHO "looking for endian.h: $ECHOSTRIP"
(echo; echo "*** Looking for endian.h ***") >> config.log
erg1=""
for e1 in "endian.h" "sys/endian.h" "machine/endian.h"
do
  cat > _xt.c <<_EOF_007
#include <sys/types.h>
#include <$e1>
int main() {return(0);}
_EOF_007
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
  $CC -c _xt.c 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -eq 0
  then
    erg1="#include <$e1>"
    $ECHO "\"$e1\" $ECHOSTRIP"
    (echo; echo "==> $e1 <==") >> config.log
    echo "#include <sys/types.h>" > _endian.h
    echo "$erg1" >> _endian.h
    break
  fi
done
rm -f _xt.c _xt.o
if test "$erg1" = ""
then
  echo "#define LITTLE_ENDIAN 1234" > _endian.h
  echo "#define BIG_ENDIAN 4321" >> _endian.h
  cat > _xt.c <<_EOF_007a
#include <stdio.h>
int main() {
  union {
    short a;
    char b[2];
  } ue;
  ue.a=1;
  if (ue.b[0]==1) {return(0);}
  return(1);
}
_EOF_007a
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
  $CC _xt.c -o _xt 2>> config.log
  x=$?
  if test $x -ne 0
  then
    rm -f _xt.c _xt.o _xt
    echo "- compiling error (see config.log)"
    echo "==> Compiling error: aborting!" >> config.log
    exit 1
  fi
  (echo; echo "--> Begin executing ./_xt") >> config.log
  ./_xt 2>> config.log
  x=$?
  echo "<-- End executing with return status $x" >> config.log
  if test $x -eq 0
  then
    echo "#define BYTE_ORDER LITTLE_ENDIAN" >> _endian.h
    $ECHO "\"LITTLE_ENDIAN\" $ECHOSTRIP"
    (echo; echo "==> LITTLE_ENDIAN <==") >> config.log
  else
    echo "#define BYTE_ORDER BIG_ENDIAN" >> _endian.h
    $ECHO "\"BIG_ENDIAN\" $ECHOSTRIP"
    (echo; echo "==> BIG_ENDIAN <==") >> config.log
  fi
  rm -f _xt.c _xt.o _xt
fi
echo "- done"
(echo; echo "####################") >> config.log


# +++ check for pipe bufferlength +++

rm -f _pipbfl.h
$ECHO "testing pipe bufferlength: $ECHOSTRIP"
(echo; echo "*** Testing pipe bufferlength ***") >> config.log
cat > _xt.c <<_EOF_008
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>
size_t blen;
FILE * pp;
void do_pipe(int signum) {
  signal(SIGPIPE,SIG_IGN);
  printf("%d\n",blen); fflush(stdout);
  pclose(pp);
  exit(0);
}
int main() {
  char buf[128];
  size_t i1;
  memset(buf,'0',sizeof(buf));
  signal(SIGPIPE,do_pipe);
  if ((pp=popen("./_xts","w"))==NULL) {return(1);}
  blen=0;
  while (1) {
    i1=fwrite(buf,1,sizeof(buf),pp); fflush(pp);
    blen+=i1;
  }
  /* notreached */
  pclose(pp);
  return(0);
}
_EOF_008
cat > _xts.c <<_EOF_008a
#include <unistd.h>
int main() {
  sleep(2);
  return(0);
}
_EOF_008a
(echo; echo "--> Begin 1st test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End 1st test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error get bufferlength for pipes (see config.log)"
  echo "==> Compiling error: aborting!" >> config.log
  exit 1
fi
(echo; echo "--> Begin 2nd test file _xts.c:") >> config.log
cat _xts.c >> config.log
echo "<-- End 2nd test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xts.c -o _xts") >> config.log
$CC _xts.c -o _xts 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt _xts.c _xts.o _xts
  echo "- error get bufferlength for pipes (see config.log)"
  echo "==> Compiling error: aborting!" >> config.log
  exit 1
fi
(echo; echo "--> Begin executing ./_xt") >> config.log
erg1=`./_xt 2>> config.log`
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt _xts.c _xts.o _xts
  echo "- error get bufferlength for pipes (see config.log)"
  echo "==> Executing error: aborting!" >> config.log
  exit 1
fi
echo "#define PIPE_BFLEN $erg1" > _pipbfl.h
echo "\"$erg1 bytes\" - done"
(echo; echo "==> $erg1 bytes <==") >> config.log
rm -f _xt.c _xt.o _xt _xts.c _xts.o _xts
(echo; echo "####################") >> config.log


# +++ INCPATH recommended? +++

$ECHO "testing whether \"-I $incpath\" is recommended: $ECHOSTRIP"
(echo; echo "*** Testing whether \"-I $incpath\" is recommended ***") >> config.log
cat > _xt.c <<_EOF_009
#include <stdio.h>
int main() {return(0);}
_EOF_009
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -c -I $incpath _xt.c") >> config.log
x=`$CC -c -I $incpath _xt.c 2>&1`
echo "$x" >> config.log
echo "<-- End compiling error messages" >> config.log
if test "$x" != ""
then
  incpath=""
  echo "no"
  (echo; echo "==> no <==") >> config.log
else
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
fi
rm -f _xt.c _xt.o
cat vgag-config.sh | sed -e "s%##INCPATH##%$incpath%" > _vgc; mv -f _vgc vgag-config.sh
(echo; echo "####################") >> config.log


# +++ System tests done +++

echo "... system tests done"



# ***** check and write out installed graphic libraries *****

echo
echo "### Testing console libraries ... ###"
(echo; echo) >> config.log
echo "####################" >> config.log
echo "### Graphic Libs ###" >> config.log
echo "####################" >> config.log


# +++ svgalib +++

rm -f __lib_vga
$ECHO " - have installed svgalib (header files: devel)? $ECHOSTRIP"
(echo; echo "*** Have installed svgalib (header files: devel)? ***") >> config.log
cat > _xt.c <<_EOF_010
#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include <vgakeyboard.h>
int main() {
  vga_init();
  vga_setmode(TEXT);
  return(0);
}
_EOF_010
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
$CC -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  echo "no"
  (echo; echo "==> no <==") >> config.log
else
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  echo > __lib_vga
fi
rm -f _xt.c _xt.o
(echo; echo "####################") >> config.log


# +++ libvgl +++

nomachcons=0
rm -f __lib_vgl
$ECHO " - have installed libvgl (header files: devel)? $ECHOSTRIP"
(echo; echo "*** Have installed libvgl (header files: devel)? ***") >> config.log
cat > _xt.c <<_EOF_011a
#include <machine/console.h>
#include <vgl.h>
int main() {
  VGLInit(SW_VGA_CG320);
  VGLEnd();
  return(0);
}
_EOF_011a
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
$CC -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  $ECHO "no??? $ECHOSTRIP"
  (echo; echo "==> no??? <==") >> config.log
else
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  echo > __lib_vgl
fi
rm -f _xt.c _xt.o
if test $x -ne 0
then
  cat > _xt.c <<_EOF_011b
#include <sys/fbio.h>
#include <sys/consio.h>
#include <sys/kbio.h>
#include <vgl.h>
int main() {
  VGLInit(SW_VGA_CG320);
  VGLEnd();
  return(0);
}
_EOF_011b
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
  $CC -c _xt.c 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -ne 0
  then
    echo "no"
    (echo; echo "==> no <==") >> config.log
  else
    echo "yes"
    (echo; echo "==> yes <==") >> config.log
    echo > __lib_vgl
    nomachcons=1
  fi
  rm -f _xt.c _xt.o
fi
rm -f vgagames_vgl.h
if test "$nomachcons" -eq 1
then
  cat vgagames_vgl.h_ | sed -e 's/###VGL_NOMACHCONS###/#define VGL_NOMACHCONS/' > vgagames_vgl.h
else
  cat vgagames_vgl.h_ | sed -e 's/###VGL_NOMACHCONS###/#undef VGL_NOMACHCONS/' > vgagames_vgl.h
fi
(echo; echo "####################") >> config.log


# +++ X window +++

rm -f __lib_x __lib_x_exts
echo
echo "Testing X window ..."
(echo; echo "### Testing X window ###") >> config.log

$ECHO " - have installed X window (header files: devel)? $ECHOSTRIP"
(echo; echo "*** Have installed X window (header files: devel)? ***") >> config.log
cat > _xt.c <<_EOF_012
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/cursorfont.h>
int main() {
  return(0);
}
_EOF_012
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC -I/usr/X11R6/include -c _xt.c") >> config.log
$CC -I/usr/X11R6/include -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
rm -f _xt.c _xt.o
if test $x -ne 0
then
  echo "no"
  (echo; echo "==> no <==") >> config.log
  echo "   --> to use VgaGames with X window the development package is needed,"
  echo "       (for Linux often named xdevel-* or XFree86-devel* or *-dev or similar)."
  $ECHO "Press [Return] to continue $ECHOSTRIP"
  read x < /dev/tty
else
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  echo > __lib_x
  $ECHO " - have installed X window extensions? $ECHOSTRIP"
  (echo; echo "*** Have installed X window extensions? ***") >> config.log
  cat > _xt.c <<_EOF_012a
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/extensions/xf86vmode.h>
int main() {
  Display * d=NULL;
  XF86VidModeModeInfo ** mi=NULL;
  XF86VidModeSwitchToMode(d,DefaultScreen(d),mi[0]);
  return(0);
}
_EOF_012a
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC -I/usr/X11R6/include _xt.c -L/usr/lib/X11 -L/usr/X11/lib -L/usr/X11R6/lib -lXxf86vm -lXext -lX11 -o _xt") >> config.log
  $CC -I/usr/X11R6/include _xt.c -L/usr/lib/X11 -L/usr/X11/lib -L/usr/X11R6/lib -lXxf86vm -lXext -lX11 -o _xt 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -ne 0
  then
    echo "no"
    (echo; echo "==> no <==") >> config.log
    cat vgag-config.sh | sed -e "s%##X_EXTS##%%" > _vgc; mv -f _vgc vgag-config.sh
  else
    echo "yes"
    (echo; echo "==> yes <==") >> config.log
    echo > __lib_x_exts
    cat vgag-config.sh | sed -e "s%##X_EXTS##%1%" > _vgc; mv -f _vgc vgag-config.sh
  fi
  rm -f _xt.c _xt.o _xt
fi
(echo; echo "####################") >> config.log


# +++ Graphic tests done +++

echo "... graphic tests done"



# ***** test sound libraries *****

echo
echo "### Testing sound libraries ... ###"
(echo; echo) >> config.log
echo "####################" >> config.log
echo "###  Sound Libs  ###" >> config.log
echo "####################" >> config.log


rm -f _c_wave.h _c_wave.lib
echo > _c_wave.h
echo > _c_wave.lib


# +++ OSS +++

$ECHO "have OSS?       $ECHOSTRIP"
(echo; echo "*** Have OSS? ***") >> config.log
cat > _xt.c <<_EOF_013
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/soundcard.h>
int main() {
  int fd;
  fd=open("/dev/dsp",O_WRONLY);
  if (fd<0) {
    if ((errno==ENOENT) || (errno==ENODEV)) {return(1);}
  } else {close(fd);}
  return(0);
}
_EOF_013
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -eq 0
then
  (echo; echo "--> Begin executing ./_xt") >> config.log
  ./_xt 2>> config.log
  x=$?
  echo "<-- End executing with return status $x" >> config.log
  if test $x -eq 0
  then
    echo "- seems to have OSS"
    (echo; echo "==> yes <==") >> config.log
    echo "#define C_WAVE__OSS" >> _c_wave.h
  else
    echo "- i don't think so"
    (echo; echo "==> no <==") >> config.log
    echo "#undef C_WAVE__OSS" >> _c_wave.h
  fi
else
  echo "- i don't think so"
  (echo; echo "==> no <==") >> config.log
  echo "#undef C_WAVE__OSS" >> _c_wave.h
fi
(echo; echo "####################") >> config.log


# +++ ALSA 0.9 +++

$ECHO "have ALSA 0.9?  $ECHOSTRIP"
(echo; echo "*** Have ALSA 0.9? ***") >> config.log
cat > _xt.c <<_EOF_014
#define ALSA_PCM_NEW_HW_PARAMS_API
#define ALSA_PCM_NEW_SW_PARAMS_API
#include <alsa/asoundlib.h>
int main() {
  snd_pcm_stream_t stream=SND_PCM_STREAM_PLAYBACK;
  snd_pcm_t * pcm;
  int err=snd_pcm_open(&pcm,"default",stream,SND_PCM_NONBLOCK);
  if (err<0) {
    if ((err!=-EACCES) && (err!=-EBUSY)) {return(1);}
  } else {
    snd_pcm_hw_params_t * hw;
    int hw_dir=0;
    unsigned int rate;
    if (0) snd_pcm_hw_params_set_rate_near(pcm,hw,&rate,&hw_dir);
    snd_pcm_close(pcm);
  }
  return(0);
}
_EOF_014
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xt.c -lasound -o _xt") >> config.log
erg1=`$CC _xt.c -lasound -o _xt 2>&1`
x=$?
echo "$erg1" >> config.log
echo "<-- End compiling error messages" >> config.log
if test $x -eq 0
then
  (echo; echo "--> Begin executing ./_xt") >> config.log
  ./_xt 2>> config.log
  x=$?
  echo "<-- End executing with return status $x" >> config.log
  if test $x -eq 0
  then
    if echo "$erg1" | grep set_rate_near > /dev/null
    then
      echo "- incomplete (too old ALSA-style found)"
      (echo; echo "==> incomplete (too old ALSA-style found) <==") >> config.log
      echo "#undef C_WAVE__ALSA09" >> _c_wave.h
    else
      echo "- seems to have ALSA 0.9"
      (echo; echo "==> yes <==") >> config.log
      echo "#define C_WAVE__ALSA09" >> _c_wave.h
      echo "-lasound" >> _c_wave.lib
    fi
  fi
else
  echo "- i don't think so"
  (echo; echo "==> no <==") >> config.log
  echo "#undef C_WAVE__ALSA09" >> _c_wave.h
fi
rm -f _xt _xt.c _xt.o
(echo; echo "####################") >> config.log


# +++ Sun audio +++

$ECHO "have SUN-AUDIO? $ECHOSTRIP"
(echo; echo "*** Have SUN-AUDIO? ***") >> config.log
cat > _xt.c <<_EOF_015
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>
#ifndef AUDIO_ENCODING_ULINEAR_LE
  #define AUDIO_ENCODING_ULINEAR_LE AUDIO_ENCODING_LINEAR8
#endif
#ifndef AUDIO_ENCODING_SLINEAR_LE
  #define AUDIO_ENCODING_SLINEAR_LE AUDIO_ENCODING_LINEAR
#endif
int main() {
  int fd;
  audio_info_t auinfo;
  if ((fd=open("/dev/audio",O_WRONLY))<0) {
    if (errno==EBUSY) {return(0);}
    return(1);
  }
  AUDIO_INITINFO(&auinfo);
  auinfo.play.encoding=AUDIO_ENCODING_ULINEAR_LE;
  auinfo.play.precision=8;
  if (ioctl(fd,AUDIO_SETINFO,&auinfo)<0) {
    AUDIO_INITINFO(&auinfo);
    auinfo.play.encoding=AUDIO_ENCODING_SLINEAR_LE;
    auinfo.play.precision=16;
    if (ioctl(fd,AUDIO_SETINFO,&auinfo)<0) {close(fd); return(1);}
  }
  close(fd);
  return(0);
}
_EOF_015
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
erg1=0
if test $x -eq 0
then
  (echo; echo "--> Begin executing ./_xt") >> config.log
  ./_xt 2>> config.log
  x=$?
  echo "<-- End executing with return status $x" >> config.log
  if test $x -eq 0
  then
    echo "- seems to have SUN-AUDIO"
    (echo; echo "==> yes <==") >> config.log
    echo "#define C_WAVE__SAUDIO" >> _c_wave.h
    erg1=1
  else
    echo "- incomplete (device or linear mode failure?)"
    (echo; echo "==> no (incomplete) <==") >> config.log
    echo "#undef C_WAVE__SAUDIO" >> _c_wave.h
  fi
else
  echo "- i don't think so"
  (echo; echo "==> no <==") >> config.log
  echo "#undef C_WAVE__SAUDIO" >> _c_wave.h
fi
rm -f _xt _xt.c _xt.o

if test $erg1 -eq 1
then
  $ECHO "have SUN-AUDIO-BLOCKSIZE? $ECHOSTRIP"
  (echo; echo "*** Have SUN-AUDIO-BLOCKSIZE? ***") >> config.log
  cat > _xt.c <<_EOF_015a
#include <sys/types.h>
#include <sys/audioio.h>
int main() {
  audio_info_t auinfo;
  auinfo.hiwat=2;
  auinfo.blocksize=(1<<9);
  return(0);
}
_EOF_015a
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
  $CC -c _xt.c 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -eq 0
  then
    echo "- seems to have SUN-AUDIO-BLOCKSIZE"
    (echo; echo "==> yes <==") >> config.log
    echo "#define C_WAVE__SAUDIOBLKSZ" >> _c_wave.h
  else
    echo "- i don't think so"
    (echo; echo "==> no <==") >> config.log
    echo "#undef C_WAVE__SAUDIOBLKSZ" >> _c_wave.h
  fi
  rm -f _xt.c _xt.o
fi

if test $erg1 -eq 1
then
  $ECHO "have SUN-AUDIO-MIXER? $ECHOSTRIP"
  (echo; echo "*** Have SUN-AUDIO-MIXER? ***") >> config.log
  cat > _xt.c <<_EOF_015b
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/audioio.h>
int main() {
  mixer_ctrl_t mixctrl;
  mixer_devinfo_t mixdev;
  mixdev.index=1;
  mixctrl.dev=1;
  mixdev.type=AUDIO_MIXER_CLASS;
  mixdev.type=AUDIO_MIXER_VALUE;
  ioctl(1,AUDIO_MIXER_READ,&mixctrl);
  ioctl(1,AUDIO_MIXER_WRITE,&mixctrl);
  return(0);
}
_EOF_015b
  (echo; echo "--> Begin test file _xt.c:") >> config.log
  cat _xt.c >> config.log
  echo "<-- End test file" >> config.log
  (echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
  $CC -c _xt.c 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -eq 0
  then
    echo "- seems to have SUN-AUDIO-MIXER"
    (echo; echo "==> yes <==") >> config.log
    echo "#define C_WAVE__SAUDIOMIXER" >> _c_wave.h
  else
    echo "- i don't think so"
    (echo; echo "==> no <==") >> config.log
    echo "#undef C_WAVE__SAUDIOMIXER" >> _c_wave.h
  fi
  rm -f _xt.c _xt.o
fi
(echo; echo "####################") >> config.log


# +++ sound libraries done +++

echo "... sound tests done"
echo "*** Please see file \"wave.conf\" for configuration settings."
echo "*** Find it in /usr/share/vgagames"
sleep 3
echo



# ***** All done *****

echo > config
echo
echo "... all tests done (see config.log)"
echo
exit 0
