/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* Example program for "Graphic box functions - manipulating"
**   rotate a propeller more and more quickly and then zoom it smaller
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <vgagames.h>

int get_opt(int,char *[]);
int get_space(int);
void end_err(const char *,const char *);


int main(int argc,char * argv[]) {
  /* variables */
  grafik * grf1,* grf2;
  int i1,i2,i3,w,h,x,y,wx,wy;
  float f1;
  char buf[64];

  /* open window */
  if (open_window("Example",get_opt(argc,argv))==-1) {
    fprintf(stderr,"%s: %s\n",argv[0],errmsg);
    exit(1);
  }

  /* load propeller graphic box */
  if ((grf1=load_grafik("./propell.bmp"))==NULL) {
    end_err(argv[0],"load \"propell.bmp\"");
  }
  w=GRAFIK_WIDTH(grf1);  /* get width of graphic */
  h=GRAFIK_HEIGHT(grf1);  /* get height of graphic */

  /* put graphic out to window */
  /* get coordinates to put graphic to middle of window */
  wx=(SC_WIDTH-w)/2;
  wy=(SC_HEIGHT-h)/2;
  copy_grafik(NULL,wx,wy,grf1,0,0,0,0,RGB_FULL);
  flush_window();
  get_space(0);


  /* +++ now we want to rotate it: create second graphic +++ */
  /* create grf2 with width and height of largest size of "propell.bmp" */
  if ((grf2=create_grafik(w>h?w+1:h+1,h>w?h+1:w+1))==NULL) {
    end_err(argv[0],"create grf2");
  }

  /* now let the show begin */
  /* i1 is for speed, i2 for rotating degrees, i3 for acceleration */
  i3=1;
  for (i1=i2=0;i1>=0;) {
    i2+=(i1/2);
    x=y=w=h=0;  /* rotate whole graphics; else try x=65;y=6;w=30;h=9; */
    /* get grf1 rotated into grf2, don't change grf1 */
    if (rotate_grafik(grf2,grf1,i2%360,&x,&y,&w,&h)==NULL) {
      end_err(argv[0],"rotate graphic");
    }
    CLEAR_BOX(NULL,RGB_BLACK);
    /* put grf2 to window (backbuffer),
    ** use x,y,w,h to put it again at the middle of the window
    */
    copy_grafik(NULL,wx+x,wy+y,grf2,0,0,w,h,RGB_FULL);
    /* draw procent info */
    snprintf(buf,sizeof(buf),"%d%%",(i1*100)/200);
    draw_text(NULL,RGB_WHITE,SC_WIDTH/2-12,180,buf,strlen(buf),RGB_TRANS);
    flush_window();
    wait_time(30);
    get_keys();  /* only for correcting screen in X window */
    if (i1>=200) {i3=-1;}
    i1+=i3;
  }
  /* wait for space key */
  get_space(0);


  /* +++ now we want to zoom it smaller +++ */
  /* rotate "propell.bmp" at 90 deg and get it into a new allocated grf1 */
  x=y=w=h=0;
  /* get grf1 rotated into grf2 */
  if (rotate_grafik(grf2,grf1,90,&x,&y,&w,&h)==NULL) {
    end_err(argv[0],"rotate graphic");
  }
  free_grafik(grf1);
  /* create grf1 with width and height of image in grf2 */
  if ((grf1=create_grafik(w,h))==NULL) {
    end_err(argv[0],"create grf1");
  }
  i1=w; i2=h;  /* save width and height of image in grf2 */
  /* copy image of width i1 and height i2 in grf2 into grf1 */
  copy_grafik(grf1,0,0,grf2,0,0,i1,i2,RGB_FULL);
  /* middle of screen */
  wx=(SC_WIDTH-i1)/2;
  wy=(SC_HEIGHT-i2)/2;

  /* now let the show begin */
  for (f1=1.;f1>.01;f1-=.01) {
    /* we use 0 for x, y, w and h because we allocated grf1 with the width
       and height of the image.
       If grf1 was greater, we still wanted to zoom only the image and
       therefore had to set the width and height of the image in
       zoom_grafik(), which values we had saved before in i1,i2 above:
        --> x=y=0; w=i1; h=i2;
            zoom_grafik(grf2,grf1,f1,f1,&x,&y,&w,&h);
    */
    x=y=w=h=0;
    /* get grf1 zoomed into grf2 */
    if (zoom_grafik(grf2,grf1,f1,f1,&x,&y,&w,&h)==NULL) {
      end_err(argv[0],"zoom graphic");
    }
    CLEAR_BOX(NULL,RGB_BLACK);
    /* put grf2 to window (backbuffer),
    ** use x,y,w,h to put it again at the middle of the window
    */
    copy_grafik(NULL,wx+x,wy+y,grf2,0,0,w,h,RGB_FULL);
    flush_window();
    wait_time(30);
    get_keys();  /* only for correcting screen in X window */
  }
  /* wait for space key */
  get_space(0);


  /* free graphic boxes */
  free_grafik(grf1);
  free_grafik(grf2);

  /* do ending */
  close_window();
  exit(0);
}


int get_opt(int argc,char * argv[]) {
/* examinate how to open window
** parameters: "-f"=full window, "-1" to "-3"=scaling factor 1 to 3
*/
  int winflag=0;
#ifdef WINDOW_RESIZABLE  /* we can change window size */
  int i1;
  for (i1=1;i1<argc;i1++) {  /* find scaling factor */
    if (argv[i1][0]=='-') {
      if (strchr(argv[i1]+1,'1')!=NULL) {winflag=VGAWINDOW_1; break;}
      else if (strchr(argv[i1]+1,'2')!=NULL) {winflag=VGAWINDOW_2; break;}
      else if (strchr(argv[i1]+1,'3')!=NULL) {winflag=VGAWINDOW_3; break;}
    }
  }
  for (i1=1;i1<argc;i1++) {  /* full window? */
    if (argv[i1][0]=='-') {
      if (strchr(argv[i1]+1,'f')!=NULL) {winflag+=VGAWINDOW_FULL; break;}
    }
  }
#endif  /* WINDOW_RESIZABLE */
  return(winflag);
}


int get_space(int flag) {
/* check for pressing space key */
  static int first=1;
  if (first==1) {
    CLEAR_KEYS;
    ADD_KEYS(KEY_SPACE,SHORT_KEY);
    first=0;
  }
  if (flag==0) {
    draw_text(NULL,RGB_WHITE,0,180,"Press Space",11,RGB_TRANS);
    flush_window();
    clearstate();
  }
  do {
    get_keys();
    wait_time(flag==0?100:flag);
    if (flag>0) {break;}
  } while (IS_KEYS(KEY_SPACE)==0);
  return(IS_KEYS(KEY_SPACE));
}


void end_err(const char * prg,const char * msg) {
/* exit with an error messages */
  close_window();
  fprintf(stderr,"%s: %s: %s\n",prg,msg,errmsg);
  exit(1);
}
