/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* Example program for "network functions"
**   move 4 sunnyboys belonging up to 4 players (else virtual players)
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <vgagames.h>

int get_opt(int,char *[]);
void end_err(const char *,const char *);


int main(int argc,char * argv[]) {
/* we want to move 4 sunnyboys. Up to 4 players can move them. If there
** are less players the master computer moves the missing players as
** virtual players.
** From time to time the colormap changes into color or grey.
*/
  /* variables */
  grafik * sunnyboy[4];
  int col_ind[4];
  int i1,i2,i3,x,y,quit;
  char buf[256];
  int no_of_ply,ply_pos;  /* number of players, player position */
  int virt_ply=0;  /* number of virtual players */
  const unsigned short port=1235;  /* server port */
  void * pdata;   /* network data */
  char * dfield;  /* data field in network data
                  ** byte 0 = reserved
                  ** byte 1+2 = x position of sunnyboy (high byte + low byte)
                  ** byte 3 = y position of sunnyboy
                  */
  const int dfieldlen=4;  /* length of each data field in network data */
  int master;     /* whether you are master player */

  /* open window */
  if (open_window("Example",get_opt(argc,argv))==-1) {
    fprintf(stderr,"%s: %s\n",argv[0],errmsg);
    exit(1);
  }

  /* input number of players if you are master */
  draw_text(NULL,RGB_WHITE,0,0,"If you are master",17,RGB_FULL);
  draw_text(NULL,RGB_WHITE,0,10,"enter number of players",23,RGB_FULL);
  draw_text(NULL,RGB_WHITE,0,20,"otherwise enter return:",23,RGB_FULL);
  flush_window();
  CLEAR_KEYS;
  ADD_KEYS(KEY_1,SHORT_KEY);
  ADD_KEYS(KEY_2,SHORT_KEY);
  ADD_KEYS(KEY_3,SHORT_KEY);
  ADD_KEYS(KEY_4,SHORT_KEY);
  ADD_KEYS(KEY_ENTER,SHORT_KEY);
  no_of_ply=0;
  clearstate();
  while (1) {
    get_keys();  /* update input events */
    /* test key events */
    if (IS_KEYS(KEY_1)==1) {no_of_ply=1; break;}
    else if (IS_KEYS(KEY_2)==1) {no_of_ply=2; break;}
    else if (IS_KEYS(KEY_3)==1) {no_of_ply=3; break;}
    else if (IS_KEYS(KEY_4)==1) {no_of_ply=4; break;}
    else if (IS_KEYS(KEY_ENTER)==1) {no_of_ply=0; break;}
    wait_time(40);  /* reduce cpu usage */
  }
  if (no_of_ply>0) {  /* master: number of players */
    master=1;
    virt_ply=4-no_of_ply;  /* rest to 4 players are virtual players */
    snprintf(buf,sizeof(buf),"%d",no_of_ply);
    no_of_ply=4;  /* always 4 players to move */
    srand(time(NULL)%1000);  /* initialize random number generator */
  } else {  /* no master: got only [return] */
    master=0;
    strcpy(buf,"[return]");
  }
  draw_text(NULL,RGB_WHITE,25*8,20,buf,strlen(buf),RGB_FULL);
  draw_text(NULL,RGB_WHITE,0,50,"Use cursor keys",15,RGB_FULL);
  flush_window();
  sleep(1);
  CLEAR_BOX(NULL,RGB_BLACK);

  /* create sunnyboys */
  col_ind[0]=color_index(CL_YELLOW,100);
  col_ind[1]=color_index(CL_RED,100);
  col_ind[2]=color_index(CL_GREEN,100);
  col_ind[3]=color_index(CL_BLUE,100);
  for (i3=0;i3<4;i3++) {
    if ((sunnyboy[i3]=create_grafik(20,20))==NULL) {
      end_err(argv[0],"create graphic");
    }
    /* i1,i2 is middle of graphic box */
    i1=GRAFIK_WIDTH(sunnyboy[i3])/2;
    i2=GRAFIK_HEIGHT(sunnyboy[i3])/2;
    /* draw face with radius 9 and not filled */
    draw_circle(sunnyboy[i3],i1,i2,9,col_ind[i3],0);
    /* draw eyes */
    draw_fillbox(sunnyboy[i3],5,5,3,3,col_ind[i3]);
    draw_fillbox(sunnyboy[i3],13,5,3,3,col_ind[i3]);
    /* draw nose */
    draw_line(sunnyboy[i3],10,8,10,12,col_ind[i3]);
    /* draw mouth */
    draw_line(sunnyboy[i3],7,15,13,15,col_ind[i3]);
    draw_line(sunnyboy[i3],5,11,7,15,col_ind[i3]);
    draw_line(sunnyboy[i3],13,15,15,11,col_ind[i3]);
  }

  /* set the key field */
  CLEAR_KEYS;  /* clear key field */
  /* add cursor keys for moving and space key for ending */
  ADD_KEYS(KEY_RCURS,LONG_KEY);
  ADD_KEYS(KEY_LCURS,LONG_KEY);
  ADD_KEYS(KEY_UCURS,LONG_KEY);
  ADD_KEYS(KEY_DCURS,LONG_KEY);
  ADD_KEYS(KEY_SPACE,SHORT_KEY);
  clearstate();

  /* start network server (only master)
  ** we want to have always 4 sunnyboys at screen. If there are less players
  ** then the rest is simulated via virtual players.
  */
  if (master) {
    if (start_nettcp(no_of_ply,virt_ply,port)==-1) {
      end_err(argv[0],"start network server");
    }
  }

  /* connect to network server via broadcast/multicast */
  draw_text(NULL,RGB_WHITE,0,30,"Connecting to server ...",24,RGB_FULL);
  flush_window();
  /* get network memory pointer, your player position and number of players */
  pdata=connect_nettcp(NULL,port,dfieldlen,&ply_pos,&no_of_ply);
  if (pdata==NULL) {end_err(argv[0],"connect to network server");}
  
  /* set your own data field */
  /* you got in pdata the network memory, that are the data fields of all
     players, and you got your own player position in ply_pos
  */
  dfield=pdata;  /* set dfield at begin of network memory */
  dfield+=ply_pos*dfieldlen;  /* set dfield to your own position */
  /* now initialize your data field */
  dfield[0]=1;  /* dfield[0] is reserved, must be initialized to 1 */
  dfield[1]=dfield[2]=0;  /* x position of your sunnyboy (high, low) */
  dfield[3]=0;  /* y position of your sunnyboy */
  /* if you are master initialize also all virtual players data fields.
  ** As master your ply_pos is always 1. The virtual players come next.
  */
  if (master) {
    for (i1=0;i1<virt_ply;i1++) {  /* set virtual players */
      dfield+=dfieldlen;  /* set dfield to next position */
      dfield[0]=1;  /* dfield[0] is reserved, must be initialized to 1 */
      dfield[1]=dfield[2]=0;  /* x position of sunnyboy (high, low) */
      dfield[3]=0;  /* y position of sunnyboy */
    }
    /* as master also initialize global data field
    ** global data field (begins at pdata, length=dfieldlen):
    **   byte 0: we let it reserved and don't use it
    **   byte 1: unsigned char: after how many loops to change colormap
    **   byte 2: 0=color or 1=grey actually shown
    **   byte 3: unused
    */
    dfield=pdata;  /* set dfield at begin of network memory */
    dfield[1]=0;
    dfield[2]=0;
  }

  /* now let the show begin */
  quit=0;
  while (1) {
    /* network update */
    if (talk_nettcp()==-1) {end_err(argv[0],"connection error");}

    CLEAR_BOX(NULL,RGB_BLACK);  /* clear window */

    quit=1;  /* perhaps you can quit, if there are no other real players */
    dfield=pdata;  /* set dfield at begin of network memory */

    /* all players can read global data field */
    if (dfield[1]==0) {  /* change colormap */
      if (dfield[2]==0) {load_colormap(RGB256);}
      else {load_colormap(GREY256);}
      if (master) {  /* only master can set global data field */
        /* set a time how long to show in color and how long in grey */
        i1=100+(int)(100.*rand()/(RAND_MAX+1.));
        if (dfield[2]==0) {dfield[2]=1;} else {dfield[2]=0;}
        if (dfield[2]==0) {i1/=3;}  /* show grey shorter than color */
        dfield[1]=(unsigned char)i1;
      }
    } else if (master) {dfield[1]--;}  /* master must decrement time value */

    /* all players read data fields of each player */
    for (i1=1;i1<=no_of_ply;i1++) {  /* for each player (from 1) */
      /* set data field pointer to selected player */
      dfield+=dfieldlen;

      if (dfield[0]==0) {  /* player disconnected (or gone) */
        continue;
      }
      x=((unsigned char)dfield[1]<<8)+(unsigned char)dfield[2];
      y=(unsigned char)dfield[3];

      if (i1==ply_pos) {  /* own sunnyboy */
        quit=0;  /* there are still real players (you!), you can't quit */
        /* test keyboard input */
        get_keys();  /* update input events */
        if (IS_KEYS(KEY_LCURS)==1) {  /* left cursor key */
          if (--x<0) {x=0;}
        }
        if (IS_KEYS(KEY_RCURS)==1) {  /* right cursor key */
          if (++x>SC_WIDTH-GRAFIK_WIDTH(sunnyboy[i1-1])) {x=SC_WIDTH-GRAFIK_WIDTH(sunnyboy[i1-1]);}
        }
        if (IS_KEYS(KEY_UCURS)==1) {  /* cursor key up */
          if (--y<0) {y=0;}
        }
        if (IS_KEYS(KEY_DCURS)==1) {  /* cursor key down */
          if (++y>SC_HEIGHT-GRAFIK_HEIGHT(sunnyboy[i1-1])) {y=SC_HEIGHT-GRAFIK_HEIGHT(sunnyboy[i1-1]);}
        }
        if (IS_KEYS(KEY_SPACE)==1) {  /* space key: want to exit */
          if (master) {
            /* if you are master then don't quit, because you have still to
            ** move virtual players
            */
            dfield[0]=0;  /* set reserved byte to 0: player is gone */
          } else {
            /* you may quit */
            quit=1;
            break;
          }
        }
        /* update own data field */
        dfield[1]=(x&0xff00)>>8;  /* high byte */
        dfield[2]=(x&0xff);  /* low byte */
        dfield[3]=y;
      } else if (master) {  /* if not your sunnyboy but you are master */
        if (i1<=1+virt_ply) {
          /* move a virtual player. Because as master your ply_pos is
          ** always 1 and the next are the virtual players, you get with
          ** (i1<=1+virt_ply) all virtual players
          */
          static int x_mv[4]={},y_mv[4]={};
          if ((x_mv[i1-1]==0) && (y_mv[i1-1]==0)) {
            x_mv[i1-1]=-160+(int)(320.*rand()/(RAND_MAX+1.));   /* x movement */
            y_mv[i1-1]=-100+(int)(200.*rand()/(RAND_MAX+1.));   /* y movement */
          }
          if (x_mv[i1-1]<0) {  /* move left */
            if (--x<0) {x=0;}
            x_mv[i1-1]++;
          } else if (x_mv[i1-1]>0) {  /* move right */
            if (++x>SC_WIDTH-GRAFIK_WIDTH(sunnyboy[i1-1])) {x=SC_WIDTH-GRAFIK_WIDTH(sunnyboy[i1-1]);}
            x_mv[i1-1]--;
          }
          if (y_mv[i1-1]<0) {  /* move up */
            if (--y<0) {y=0;}
            y_mv[i1-1]++;
          } else if (y_mv[i1-1]>0) {  /* move down */
            if (++y>SC_HEIGHT-GRAFIK_HEIGHT(sunnyboy[i1-1])) {y=SC_HEIGHT-GRAFIK_HEIGHT(sunnyboy[i1-1]);}
            y_mv[i1-1]--;
          }
          /* update virtual players data field */
          dfield[1]=(x&0xff00)>>8;  /* high byte */
          dfield[2]=(x&0xff);  /* low byte */
          dfield[3]=y;
        } else {
          /* you are master and i1 represents another real player */
          quit=0;  /* there are still real players, you can't quit */
        }
      }

      /* draw players sunnyboy */
      copy_grafik(NULL,x,y,sunnyboy[i1-1],0,0,0,0,RGB_TRANS);
    }
    if (quit==1) {break;}

    flush_window();
    wait_time(20);  /* wait up to 20 milliseconds */
  }

  /* disconnect from network server */
  close_nettcp();

  /* free sunnyboys */
  for (i1=0;i1<4;i1++) {
    free_grafik(sunnyboy[i1]);
  }

  /* do ending */
  close_window();
  exit(0);
}


int get_opt(int argc,char * argv[]) {
/* examinate how to open window
** parameters: "-f"=full window, "-1" to "-3"=scaling factor 1 to 3
*/
  int winflag=0;
#ifdef WINDOW_RESIZABLE  /* we can change window size */
  int i1;
  for (i1=1;i1<argc;i1++) {  /* find scaling factor */
    if (argv[i1][0]=='-') {
      if (strchr(argv[i1]+1,'1')!=NULL) {winflag=VGAWINDOW_1; break;}
      else if (strchr(argv[i1]+1,'2')!=NULL) {winflag=VGAWINDOW_2; break;}
      else if (strchr(argv[i1]+1,'3')!=NULL) {winflag=VGAWINDOW_3; break;}
    }
  }
  for (i1=1;i1<argc;i1++) {  /* full window? */
    if (argv[i1][0]=='-') {
      if (strchr(argv[i1]+1,'f')!=NULL) {winflag+=VGAWINDOW_FULL; break;}
    }
  }
#endif  /* WINDOW_RESIZABLE */
  return(winflag);
}


void end_err(const char * prg,const char * msg) {
/* exit with an error messages */
  close_nettcp();
  close_window();
  fprintf(stderr,"%s: %s: %s\n",prg,msg,errmsg);
  exit(1);
}
