/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* nettcp: network support
   define HAVE_ADDRINFO_STRUCT, if you have getaddrinfo()
     or undef HAVE_ADDRINFO_STRUCT, if you have not getaddrinfo().
   define HAVE_SOCKADDR_SA_LEN, if you have sin_len
     or undef HAVE_SOCKADDR_SA_LEN, if you have not sin_len.
   compile if needed with -lsocket -lnsl [-lresolv]
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <limits.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>
#include "netw.h"


#ifndef HAVE_SOCKLEN_T
  typedef int socklen_t;
#endif
#ifndef HAVE_ADDRINFO_STRUCT
  #include "nettcp_gai.h"
#else
  #include <netinet/in.h>
  #include <arpa/nameser.h>
  #include <resolv.h>
#endif
#include "pfad.h"

#ifdef AF_INET6
  #ifndef IPV6_ADD_MEMBERSHIP
    #define IPV6_ADD_MEMBERSHIP IPV6_JOIN_GROUP
  #endif
#else
  #ifdef USE_AF_VER6
    #undef USE_AF_VER6
  #endif
#endif

#define VGAG_NETTCP_SERVER BIN_PATH "/vgag-nettcp"

extern char errmsg[2048];
char * netmem;
static int cfd=-1,plynr=-1,plypos=-1,plyvirt=-1,inetv=0,master=0;
static size_t plysz=0;
static volatile sig_atomic_t sig_alrm=0;

int start_nettcp(int,int,unsigned short);
void * connect_nettcp(const char *,unsigned short,size_t,int *,int *);
int talk_nettcp(void);
void close_nettcp(void);

static ssize_t writen(int,const void *,size_t);
static ssize_t readn(int,void *,size_t);
#ifndef USE_AF_VER6
  static int get_hostip_bc(char *,const char *,const char *);
#endif
#ifdef AF_INET6
  static int get_hostip_mc(char *,const char *,const char *,int);
#endif
static void do_alrm(int);


int start_nettcp(int anz,int vanz,unsigned short port) {
/* Only called once, from the master player, to start server
** for all players to connect to.
** 1.arg: number of players to wait for connection (from 1 to 8)
** 2.arg: how many virtual players (=guided by computer) are included in 1.arg
** 3.arg: portnumber listening to
** return: 0=ok or -1=error
*/
  char serv[8],anzp[2],vanzp[2];
  pid_t cpid;
  if ((anz<1) || (anz>8) || (vanz<0) || (vanz>=anz)) {strcpy(errmsg,"Wrong number of players"); return(-1);}
  anzp[0]=anz+'0'; anzp[1]='\0';
  vanzp[0]=vanz+'0'; vanzp[1]='\0';
  snprintf(serv,sizeof(serv),"%u",port);
  if ((cpid=fork())==-1) {snprintf(errmsg,sizeof(errmsg),"Fork: %s",strerror(errno)); return(-1);}
  if (cpid==0) {  /* Child */
    execlp(VGAG_NETTCP_SERVER,VGAG_NETTCP_SERVER,anzp,vanzp,serv,(void *)0);
    fprintf(stderr,"Cannot execute %s\n",VGAG_NETTCP_SERVER);
    exit(1);
  }
  master=1;
  plyvirt=vanz;
  sleep(3);
  return(0);
} /* Ende start_nettcp */


void * connect_nettcp(const char * host,unsigned short port,size_t stsz,int * plp,int * pln) {
/* connect to server, waiting until all players are connected
** 1.arg: NULL=get host via broadcast/multicast
**        or hostname or ip of server
** 2.arg: portnumber of server
** 3.arg: size of data field (number of characters) (1 to 128)
** 4.arg: address for player position
** 5.arg: address for number of players
** return: address of network memory or NULL=error
*/
  int n1;
  char serv[8],buf[64],hostp[128+64];
  struct addrinfo adi0,* adi1,* adi2;
#ifdef AF_INET6
  struct if_nameindex * ifa,* ifi;
#endif
  if ((stsz==0) || (stsz>128)) {strcpy(errmsg,"Wrong data size (must be 1 to 128)"); return(NULL);}
  snprintf(serv,sizeof(serv),"%u",port);
#ifdef AF_INET6
  ifa=if_nameindex();
  if (ifa==NULL) {
    printf("Error getting interfaces for multicast!\n"); fflush(stdout);
  }
  ifi=ifa;
_cast_again:
#endif
#if 1
  if (master) {strcpy(hostp,"localhost");}
  else if (host==NULL) {
#else
  if (host==NULL) {
#endif
#ifndef USE_AF_VER6
  #ifdef AF_INET6
    if (ifi==ifa) {  /* first time */
  #endif
      n1=get_hostip_bc(hostp,serv,"255.255.255.255");
      if (n1<0) {
        n1=get_hostip_bc(hostp,serv,"127.255.255.255");
      }
  #ifdef AF_INET6
    } else {n1=-1;}
  #endif
#else
    n1=-1;
#endif
    if (n1<0) {
#ifdef AF_INET6
      if (ifa!=NULL) {
        while (ifi->if_index>0) {
          n1=get_hostip_mc(hostp,serv,"ff02::1",ifi->if_index);
          ifi++;
          if (n1==0) {break;}
        }
      }
    }
    if (n1<0) {
      strcpy(errmsg,"Got no server ip via multicast");
#endif
      return(NULL);
    }
  } else {strncpy(hostp,host,128-1); hostp[128-1]='\0';}
  /* open connection */
#ifdef AF_INET6
_conn_again:
#endif
  memset(&adi0,0,sizeof(struct addrinfo));
#ifndef USE_AF_VER6
  adi0.ai_family=AF_UNSPEC;
#else
  adi0.ai_family=AF_INET6;
#endif
  adi0.ai_socktype=SOCK_STREAM;
  if ((n1=getaddrinfo(hostp,serv,&adi0,&adi1))!=0) {
    snprintf(errmsg,sizeof(errmsg),"Socket error (%s): %s",hostp,gai_strerror(n1));
    return(NULL);
  }
  adi2=adi1;
  do {
    cfd=socket(adi1->ai_family,adi1->ai_socktype,adi1->ai_protocol);
    if (cfd<0) {continue;}
    if (connect(cfd,adi1->ai_addr,adi1->ai_addrlen)==0) {break;}
    close(cfd); cfd=-1;
  } while ((adi1=adi1->ai_next)!=NULL);
  if (adi1==NULL) {
    freeaddrinfo(adi2);
#ifdef AF_INET6
    if ((ifa!=NULL) && (ifi!=ifa)) {  /* multicast */
      static char * pdev=NULL;
      if (pdev==NULL) {
        ifi--;
        if ((pdev=ifi->if_name)==NULL) {ifi++; goto _cast_again;}
        ifi++;
        snprintf(hostp+strlen(hostp),64,"%%%s",pdev);
        goto _conn_again;
      } else {pdev=NULL;}
      goto _cast_again;
    }
#endif
    snprintf(errmsg,sizeof(errmsg),"Cannot connect to %s (%s)",hostp,serv);
    return(NULL);
  }
  inetv=adi1->ai_addr->sa_family;
  freeaddrinfo(adi2);
#ifdef AF_INET6
  if (ifa!=NULL) {if_freenameindex(ifa);}
#endif
  printf("Connected to server %s at port %s\n",hostp,serv); fflush(stdout);
  /* set network memory */
  if ((netmem=calloc(1+8,stsz))==NULL) {
    strcpy(errmsg,"Cannot allocate network memory");
    close(cfd); cfd=-1;
    return(NULL);
  }
  netmem[0]=1;
  /* write data size + master identification */
  snprintf(buf,sizeof(buf),"%04u%d",stsz,master);
  if (writen(cfd,buf,5)==-1) {
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  plysz=stsz;
  /* read number of players and player position */
  if (read(cfd,buf,2)<2) {
    snprintf(errmsg,sizeof(errmsg),"Cannot read from %s (%s)",hostp,serv);
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  plynr=(int)buf[0]-'0';
  plypos=(int)buf[1]-'0';
  if ((plynr<1) || (plynr>8) || (plypos<1) || (plypos>8)) {
    snprintf(errmsg,sizeof(errmsg),"Wrong answer read from %s (%s)",hostp,serv);
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  if (pln!=NULL) {*pln=plynr;}
  if (plp!=NULL) {*plp=plypos;}
  netmem[plypos*plysz]=1;
  if (master) {
    for (n1=1;n1<=plyvirt;n1++) {netmem[(plypos+n1)*plysz]=1;}
  }
  return((void *)netmem);
} /* Ende connect_nettcp */


int talk_nettcp() {
/* sends own data and loads data of the other players
** return: 0=ok or -1=error
*/
  const int rnr=(1+plynr)*plysz;
  ssize_t rr;
  size_t slen;
  char * nmm;
  if (cfd<0) {strcpy(errmsg,"Not connected"); return(-1);}
  if (master) {nmm=netmem; slen=plysz*(1+1+plyvirt);}
  else {nmm=netmem+plypos*plysz; slen=plysz;}
  if (writen(cfd,nmm,slen)==-1) {
    strcpy(errmsg,"Write error to server");
    close(cfd); cfd=-1;
    return(-1);
  }
  if ((rr=readn(cfd,netmem,rnr))!=rnr) {
    strcpy(errmsg,"Read error from server");
    close(cfd); cfd=-1;
    return(-1);
  }
  return(0);
} /* Ende talk_nettcp */


void close_nettcp() {
/* close socket and free network memory */
  if (cfd>=0) {
    close(cfd);
    free(netmem);
  }
  cfd=-1;
} /* Ende close_nettcp */


static ssize_t writen(int fd,const void * vptr,size_t n) {
/* force number of bytes to write
** 1.arg: socket descriptor
** 2.arg: buffer to write
** 3.arg: number of characters to write
** return: 3.arg or -1=error
*/
  static int spip=0;
  size_t nleft;
  ssize_t nwritten;
  const char * ptr;
  if (spip==0) {
    struct sigaction sa1;
    spip=1;
    memset(&sa1,0,sizeof(sa1));
    sa1.sa_handler=SIG_IGN;
    sigaction(SIGPIPE,&sa1,NULL);
  }
  ptr=vptr;
  nleft=n;
  while (nleft>0) {
    if ((nwritten=write(fd,ptr,nleft))<0) {
      if (errno==EAGAIN) {continue;}
      strcpy(errmsg,"Error writing to socket");
      return(-1);
    }
    nleft-=nwritten;
    ptr+=nwritten;
  }
  return(n);
} /* Ende writen */


static ssize_t readn(int fd,void * vptr,size_t n) {
/* force number of bytes to read
** 1.arg: socket descriptor
** 2.arg: address of buffer
** 3.arg: number of characters to read
** return: 0 to 3.arg or -1=error or -2=no data
*/
  size_t nleft;
  ssize_t nread;
  char * ptr;
  ptr=(char *)vptr;
  nleft=n;
  while (nleft>0) {
    if ((nread=read(fd,ptr,nleft))<0) {
      if (errno==EAGAIN) {
        if (nleft!=n) {continue;}
        return(-2);
      }
      strcpy(errmsg,"Error reading from socket");
      return(-1);
    } else if (nread==0) {break;}
    nleft-=nread;
    ptr+=nread;
  }
  return(n-nleft);
} /* Ende readn */


#ifndef USE_AF_VER6
static int get_hostip_bc(char * host,const char * serv,const char * bcadd) {
/* get server ip via broadcast
** 1.arg: address for server ip (size >= 128)
** 2.arg: portnumber of broadcast server
** 3.arg: broadcast address
** return: 0=ok or -1=error or -2=no answer (alarm clock)
*/
  const int altm=7;  /* alarm time for recv */
  const int on=1;
  int sockfd,anz;
  char buf[2];
  socklen_t salen,slen;
  struct sockaddr * sa,* servaddr;
  struct sockaddr_in * sain;
  struct addrinfo hints,* res,* ressave;
  struct sigaction sac;
  *host='\0';
  memset(&sac,0,sizeof(sac));
  sac.sa_handler=do_alrm;
  sigaction(SIGALRM,&sac,NULL);
  memset(&sac,0,sizeof(sac));
  sac.sa_handler=SIG_DFL;
  /* open socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_family=AF_INET;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(bcadd,serv,&hints,&res))!=0) {
    snprintf(errmsg,sizeof(errmsg),"Cannot get server ip: %s",gai_strerror(anz));
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  ressave=res;
  do {
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd>=0) {break;}
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot get server ip for port %s",serv);
    freeaddrinfo(ressave);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  if ((sa=malloc(res->ai_addrlen))==NULL) {
    strcpy(errmsg,"Get server ip: cannot allocate memory");
    freeaddrinfo(ressave);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  memmove(sa,res->ai_addr,res->ai_addrlen);
  salen=res->ai_addrlen;
  freeaddrinfo(ressave);
  /* prepare for getting server ip */
  if ((servaddr=malloc(salen))==NULL) {
    strcpy(errmsg,"Get server ip: cannot allocate memory\n");
    close(sockfd);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  if (setsockopt(sockfd,SOL_SOCKET,SO_BROADCAST,&on,sizeof(on))<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip via broadcast: setsockopt error: %s\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  /* send broadcast and read first answer */
  strcpy(buf,"\n");
  if (sendto(sockfd,buf,strlen(buf),0,sa,salen)<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip (sending): %s!\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  slen=salen;
  alarm(altm);
  anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,servaddr,&slen);
  alarm(0);
  sigaction(SIGALRM,&sac,NULL);
  if (anz<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip (receiving): %s!\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    if (sig_alrm==1) {return(-2);}
    return(-1);
  }
  sain=(struct sockaddr_in *)servaddr;
  if (inet_ntop(AF_INET,&sain->sin_addr,host,128)==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip: cannot get ip");
    close(sockfd);
    free(servaddr);
    *host='\0';
    return(-1);
  }
  close(sockfd);
  free(servaddr);
  return(0);
} /* Ende get_hostip_bc */
#endif


#ifdef AF_INET6
static int get_hostip_mc(char * host,const char * serv,const char * mcadd,int ifn) {
/* get server ip via multicast
** 1.arg: address for server ip (size >= 128)
** 2.arg: portnumber of multicast server
** 3.arg: multicast address
** 4.arg: interface index
** return: 0=ok or -1=error or -2=no answer (alarm clock) or -3=setsockopt error
*/
  const int altm=7;  /* alarm time for recv */
  int sockfd=-1,anz;
  char buf[2];
  socklen_t salen,slen;
  struct sockaddr * sa,* servaddr;
  struct sockaddr_in6 * sain;
  struct addrinfo hints,* res,* ressave;
  struct sigaction sac;
  *host='\0';
  memset(&sac,0,sizeof(sac));
  sac.sa_handler=do_alrm;
  sigaction(SIGALRM,&sac,NULL);
  memset(&sac,0,sizeof(sac));
  sac.sa_handler=SIG_DFL;
  /* open socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_family=AF_INET6;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(mcadd,serv,&hints,&res))!=0) {
    snprintf(errmsg,sizeof(errmsg),"Cannot get server ip: %s",gai_strerror(anz));
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  ressave=res;
  do {
    if (res->ai_family!=AF_INET6) {continue;}
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd>=0) {break;}
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot get server ip for port %s",serv);
    freeaddrinfo(ressave);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  if ((sa=malloc(res->ai_addrlen))==NULL) {
    strcpy(errmsg,"Get server ip: cannot allocate memory");
    freeaddrinfo(ressave);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  memmove(sa,res->ai_addr,res->ai_addrlen);
  salen=res->ai_addrlen;
  freeaddrinfo(ressave);
  /* prepare for getting server ip */
  if ((servaddr=malloc(salen))==NULL) {
    strcpy(errmsg,"Get server ip: cannot allocate memory\n");
    close(sockfd);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  if (setsockopt(sockfd,IPPROTO_IPV6,IPV6_MULTICAST_IF,&ifn,sizeof(ifn))<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip via multicast: setsockopt error: %s\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    sigaction(SIGALRM,&sac,NULL);
    return(-3);
  }
  /* send multicast and read first answer */
  strcpy(buf,"\n");
  if (sendto(sockfd,buf,strlen(buf),0,sa,salen)<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip (send): %s!\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    sigaction(SIGALRM,&sac,NULL);
    return(-1);
  }
  slen=salen;
  alarm(altm);
  anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,servaddr,&slen);
  alarm(0);
  sigaction(SIGALRM,&sac,NULL);
  if (anz<0) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip (recv): %s!\n",strerror(errno));
    close(sockfd);
    free(servaddr);
    if (sig_alrm==1) {return(-2);}
    return(-1);
  }
  sain=(struct sockaddr_in6 *)servaddr;
  if (inet_ntop(AF_INET6,&sain->sin6_addr,host,128)==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Get server ip: cannot get ip");
    close(sockfd);
    free(servaddr);
    *host='\0';
    return(-1);
  }
  close(sockfd);
  free(servaddr);
  return(0);
} /* Ende get_hostip_mc */
#endif


static void do_alrm(int signum) {
  sig_alrm=1;
} /* Ende do_alrm */
