/* *****************************************************************
   from: UNIX Network Programming Vol. 1, 2nd Ed.
         W. Richard Stevens

   needed only for older unix's which have no getaddrinfo()
   ***************************************************************** */

struct addrinfo {
  int ai_flags;               /* AI_PASSIVE, AI_CANONNAME */
  int ai_family;              /* PF_xxx */
  int ai_socktype;            /* SOCK_xxx */
  int ai_protocol;            /* IPPROTO_xxx for IPv4_ and IPv6_ */
  size_t ai_addrlen;          /* length of ai_addr */
  char * ai_canonname;        /* canonical name for host */
  struct sockaddr * ai_addr;  /* binary address */
  struct addrinfo * ai_next;  /* next structure in linked list */
};
#define AI_PASSIVE      1  /* socket is intended for bind()+listen() */
#define AI_CANONNAME    2  /* return canonical name */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/nameser.h>
#include <resolv.h>
#include <ctype.h>

#ifndef AI_CLONE
  #define AI_CLONE  4
#endif

#define EAI_ADDRFAMILY  1  /* address family for host not supported */
#define EAI_AGAIN       2  /* temporary failure in name resolution */
#define EAI_BADFLAGS    3  /* invalid value for ai_flags */
#define EAI_FAIL        4  /* non-recoverable failure in name resolution */
#define EAI_FAMILY      5  /* ai_family not supported */
#define EAI_MEMORY      6  /* memory allocation failure */
#define EAI_NODATA      7  /* no address associated with host */
#define EAI_NONAME      8  /* host nor service provided, or not known */
#define EAI_SERVICE     9  /* service not supported for ai_socktype */
#define EAI_SOCKTYPE   10  /* ai_socktype not supported */
#define EAI_SYSTEM     11  /* system error returned in errno */

struct search {
  const char * host;  /* hostname or address string */
  int family;  /* AF_xxx */
};

static int ga_aistruct(struct addrinfo ***,const struct addrinfo *,const void *,int);
static int ga_echeck(const char *,const char *,int,int,int,int);
static struct addrinfo * ga_clone(struct addrinfo *);
static int ga_nsearch(const char *,const struct addrinfo *,struct search *);
static int ga_port(struct addrinfo *,int,int);
static int ga_serv(struct addrinfo *,const struct addrinfo *,const char *);

void freeaddrinfo(struct addrinfo *);
char * gai_strerror(int);
int getaddrinfo(const char *,const char *,const struct addrinfo *,struct addrinfo **);

#ifndef INADDRSZ
  #define INADDRSZ    4
#endif
int inet_pton(int,const char *,void *);
static int _inet_pton4(const char *,void *);
const char * inet_ntop(int,const void *,char *,size_t);
static const char * _inet_ntop4(const void *,char *,size_t);



int getaddrinfo(const char * hostname,const char * servname,const struct addrinfo * hintsp,struct addrinfo ** result) {
  int rc,error;
  char * canon;
  struct addrinfo hints,* aihead,** aipnext;
  #define gaderror(e) {error=(e); goto bad;}
  int nsearch;
  char ** ap;
  struct hostent * hptr;
  struct search search[3],* sptr;

  aihead=NULL;
  aipnext=&aihead;
  canon=NULL;
  if (hintsp==NULL) {
    memset(&hints,0,sizeof(hints));
    hints.ai_family=AF_UNSPEC;
  } else {
    hints=*hintsp;
  }
  if ((rc=ga_echeck(hostname,servname,hints.ai_flags,hints.ai_family,hints.ai_socktype,hints.ai_protocol))!=0) {gaderror(rc);}

  if ((nsearch=ga_nsearch(hostname,&hints,&search[0]))==-1) {gaderror(-1);}
  for (sptr=&search[0];sptr<&search[nsearch];sptr++) {
    if (isdigit((int)sptr->host[0])) {
      struct in_addr inaddr;
      if (inet_pton(AF_INET,sptr->host,&inaddr)==1) {
        if ((hints.ai_family!=AF_UNSPEC) && (hints.ai_family!=AF_INET)) {gaderror(EAI_ADDRFAMILY);}
        if (sptr->family!=AF_INET) {continue;}
        rc=ga_aistruct(&aipnext,&hints,&inaddr,AF_INET);
        if (rc!=0) {gaderror(rc);}
        continue;
      }
    }

    if ((_res.options&RES_INIT)==0) {res_init();}
    if (nsearch==2) {
/*    hptr=gethostbyname2(sptr->host,sptr->family); */
      hptr=gethostbyname(sptr->host);
    } else {
      hptr=gethostbyname(sptr->host);
    }
    if (hptr==NULL) {
      if (nsearch==2) {continue;}
      switch (h_errno) {
        case HOST_NOT_FOUND:  gaderror(EAI_NONAME);
        case TRY_AGAIN:       gaderror(EAI_AGAIN);
        case NO_RECOVERY:     gaderror(EAI_FAIL);
        case NO_DATA:         gaderror(EAI_NODATA);
        default:              gaderror(EAI_NONAME);
      }
    }
    if ((hints.ai_family!=AF_UNSPEC) && (hints.ai_family!=hptr->h_addrtype)) {gaderror(EAI_ADDRFAMILY);}
    if ((hostname!=NULL) && (hostname[0]!='\0') && (hints.ai_flags&AI_CANONNAME) && (canon==NULL)) {
      if ((canon=strdup(hptr->h_name))==NULL) {gaderror(EAI_MEMORY);}
    }
    for (ap=hptr->h_addr_list;*ap!=NULL;ap++) {
      rc=ga_aistruct(&aipnext,&hints,*ap,hptr->h_addrtype);
      if (rc!=0) {gaderror(rc);}
    }
  }
  if (aihead==NULL) {gaderror(EAI_NONAME);}

  if ((hostname!=NULL) && (hostname[0]!='\0') && (hints.ai_flags&AI_CANONNAME)) {
    if (canon!=NULL) {aihead->ai_canonname=canon;}
    else {
      if ((aihead->ai_canonname=strdup(search[0].host))==NULL) {gaderror(EAI_MEMORY);}
    }
  }
  if ((servname!=NULL) && (servname[0]!='\0')) {
    if ((rc=ga_serv(aihead,&hints,servname))!=0) {gaderror(rc);}
  }
  *result=aihead;
  return(0);

bad:
  freeaddrinfo(aihead);
  return(error);
} /* Ende getaddrinfo */


void freeaddrinfo(struct addrinfo * aihead) {
  struct addrinfo * ai,* ainext;
  for (ai=aihead;ai!=NULL;ai=ainext) {
    if (ai->ai_addr!=NULL) {free(ai->ai_addr);}
    if (ai->ai_canonname!=NULL) {free(ai->ai_canonname);}
    ainext=ai->ai_next;
    free(ai);
  }
} /* Ende freeaddrinfo */


static int ga_aistruct(struct addrinfo *** paipnext,const struct addrinfo * hintsp,const void * addr,int family) {
  struct addrinfo * ai;
  if ((ai=calloc(1,sizeof(struct addrinfo)))==NULL) {return(EAI_MEMORY);}
  ai->ai_next=NULL;
  ai->ai_canonname=NULL;
  **paipnext=ai;
  *paipnext=&ai->ai_next;
  if ((ai->ai_socktype=hintsp->ai_socktype)==0) {ai->ai_flags|=AI_CLONE;}
  ai->ai_protocol=hintsp->ai_protocol;

  switch((ai->ai_family=family)) {
    case AF_INET: {
      struct sockaddr_in * sinptr;
      if ((sinptr=calloc(1,sizeof(struct sockaddr_in)))==NULL) {return(EAI_MEMORY);}
#ifdef HAVE_SOCKADDR_SA_LEN
      sinptr->sin_len=sizeof(struct sockaddr_in);
#endif
      sinptr->sin_family=AF_INET;
      memcpy(&sinptr->sin_addr,addr,sizeof(struct in_addr));
      ai->ai_addr=(struct sockaddr *)sinptr;
      ai->ai_addrlen=sizeof(struct sockaddr_in);
      break;
    }
  }
  return(0);
} /* Ende ga_aistruct */


static struct addrinfo * ga_clone(struct addrinfo * ai) {
  struct addrinfo * new;
  if ((new=calloc(1,sizeof(struct addrinfo)))==NULL) {return(NULL);}
  new->ai_next=ai->ai_next;
  ai->ai_next=new;
  new->ai_flags=0;
  new->ai_family=ai->ai_family;
  new->ai_socktype=ai->ai_socktype;
  new->ai_protocol=ai->ai_protocol;
  new->ai_canonname=NULL;
  new->ai_addrlen=ai->ai_addrlen;
  if ((new->ai_addr=malloc(ai->ai_addrlen))==NULL) {return(NULL);}
  memcpy(new->ai_addr,ai->ai_addr,ai->ai_addrlen);
  return(new);
} /* Ende ga_clone */


static int ga_echeck(const char * hostname,const char * servname,int flags,int family,int socktype,int protocol) {
  if (flags&~(AI_PASSIVE|AI_CANONNAME)) {return(EAI_BADFLAGS);}
  if ((hostname==NULL) || (hostname[0]=='\0')) {
    if ((servname==NULL) || (servname[0]=='\0')) {return(EAI_NONAME);}
  }
  switch(family) {
    case AF_UNSPEC:
      break;
    case AF_INET:
      if ((socktype!=0) && (socktype!=SOCK_STREAM) && (socktype!=SOCK_DGRAM) && (socktype!=SOCK_RAW)) {return(EAI_SOCKTYPE);}
      break;
    default:
      return(EAI_FAMILY);
  }
  return(0);
} /* Ende ga_echeck */


static int ga_nsearch(const char * hostname,const struct addrinfo * hintsp,struct search * search) {
  int nsearch=0;
  if (hostname==NULL) {
    if (hintsp->ai_flags&AI_PASSIVE) {  /* no host and AI_PASSIVE: wildcard bind */
      switch (hintsp->ai_family) {
        case AF_INET:
          search[nsearch].host="0.0.0.0";
          search[nsearch].family=AF_INET;
          nsearch++;
          break;
        case AF_UNSPEC:
          search[nsearch].host="0.0.0.0";
          search[nsearch].family=AF_INET;
          nsearch++;
          break;
      }

    } else {  /* no host and not AI_PASSIVE: connect to local host */
      switch (hintsp->ai_family) {
        case AF_INET:
          search[nsearch].host="localhost";
          search[nsearch].family=AF_INET;
          nsearch++;
          break;
        case AF_UNSPEC:
          search[nsearch].host="localhost";
          search[nsearch].family=AF_INET;
          nsearch++;
          break;
      }
    }

  } else {  /* host is specified */
    switch (hintsp->ai_family) {
      case AF_INET:
        search[nsearch].host=hostname;
        search[nsearch].family=AF_INET;
        nsearch++;
        break;
      case AF_UNSPEC:
        search[nsearch].host=hostname;
        search[nsearch].family=AF_INET;
        nsearch++;
        break;
    }
  }
  if ((nsearch<1) || (nsearch>2)) {return(-1);}
  return(nsearch);
} /* Ende ga_nsearch */


static int ga_port(struct addrinfo * aihead,int port,int socktype) {
  int nfound=0;
  struct addrinfo * ai;
  for (ai=aihead;ai!=NULL;ai=ai->ai_next) {
    if (ai->ai_flags&AI_CLONE) {
      if (ai->ai_socktype!=0) {
        if ((ai=ga_clone(ai))==NULL) {return(-1);}
      }
    } else if (ai->ai_socktype!=socktype) {continue;}
    ai->ai_socktype=socktype;
    switch (ai->ai_family) {
      case AF_INET:
        ((struct sockaddr_in *)ai->ai_addr)->sin_port=port;
        nfound++;
        break;
    }
  }
  return(nfound);
} /* Ende ga_port */


static int ga_serv(struct addrinfo * aihead,const struct addrinfo * hintsp,const char * serv) {
  int port,rc,nfound;
  struct servent * sptr;
  nfound=0;
  if (isdigit((int)serv[0])) {
    port=htons(atoi(serv));
    if (hintsp->ai_socktype) {
      if ((rc=ga_port(aihead,port,hintsp->ai_socktype))<0) {return(EAI_MEMORY);}
      nfound+=rc;
    } else {
      if ((rc=ga_port(aihead,port,SOCK_STREAM))<0) {return(EAI_MEMORY);}
      nfound+=rc;
      if ((rc=ga_port(aihead,port,SOCK_DGRAM))<0) {return(EAI_MEMORY);}
      nfound+=rc;
    }
  } else {
    if ((hintsp->ai_socktype==0) || (hintsp->ai_socktype==SOCK_STREAM)) {
      if ((sptr=getservbyname((char *)serv,"tcp"))!=NULL) {
        if ((rc=ga_port(aihead,sptr->s_port,SOCK_STREAM))<0) {return(EAI_MEMORY);}
        nfound+=rc;
      }
    }
    if ((hintsp->ai_socktype==0) || (hintsp->ai_socktype==SOCK_DGRAM)) {
      if ((sptr=getservbyname((char *)serv,"udp"))!=NULL) {
        if ((rc=ga_port(aihead,sptr->s_port,SOCK_DGRAM))<0) {return(EAI_MEMORY);}
        nfound+=rc;
      }
    }
  }
  if (nfound==0) {
    if (hintsp->ai_socktype==0) {return(EAI_NONAME);} else {return(EAI_SERVICE);}
  }
  return(0);
} /* Ende ga_serv */


char * gai_strerror(int err) {
  switch(err) {
    case EAI_ADDRFAMILY: return("address family for host not supported");
    case EAI_AGAIN: return("temporary failure in name resolution");
    case EAI_BADFLAGS: return("invalid flags value");
    case EAI_FAIL: return("non-recoverable failure in name resolution");
    case EAI_FAMILY: return("address family not supported");
    case EAI_MEMORY: return("memory allocation failure");
    case EAI_NODATA: return("no address associated with host");
    case EAI_NONAME: return("host nor service provided, or not known");
    case EAI_SERVICE: return("service not supported for socket type");
    case EAI_SOCKTYPE: return("socket type not supported");
    case EAI_SYSTEM: return("system error");
    default: return("unknown getaddrinfo() error");
  }
} /* Ende gai_strerror */


/* from: Paul Vixie, 1996 */
int inet_pton(int af,const char * src,void * dst) {
  if (af==AF_INET) {
    return(_inet_pton4(src,dst));
  }
  errno=EAFNOSUPPORT;
  return(-1);
} /* Ende inet_pton */


static int _inet_pton4(const char * src,void * dst) {
  unsigned int punktanz=0,imrk;
  unsigned char mrk[INADDRSZ],* pmrk;
  char c1;
  memset(mrk,0,INADDRSZ);
  pmrk=mrk;
  while ((c1=*src++)!='\0') {
    switch(c1) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9': imrk=*pmrk*10+(c1-'0');
                if (imrk>255) {return(0);}
                *pmrk=imrk;
                break;
      case '.': if (++punktanz>=INADDRSZ) {return(0);}
                pmrk++;
                break;
      default: return(0);
    }
  }
  if (dst!=NULL) {memmove(dst,mrk,INADDRSZ);}
  return(1);
} /* Ende _inet_pton4 */


/* from: Paul Vixie, 1996 */
const char * inet_ntop(int af,const void * src,char * dst,size_t size) {
  if (af==AF_INET) {
    return(_inet_ntop4(src,dst,size));
  }
  errno=EAFNOSUPPORT;
  return(NULL);
} /* Ende inet_ntop */


static const char * _inet_ntop4(const void * src,char * dst,size_t size) {
  static const char fmt[]="%u.%u.%u.%u";
  char tmp[16];
  unsigned char * csrc;
  csrc=(unsigned char *)src;
  sprintf(tmp,fmt,csrc[0],csrc[1],csrc[2],csrc[3]);
  if (strlen(tmp)>size) {errno=ENOSPC; return(NULL);}
  if (dst!=NULL) {strcpy(dst,tmp);}
  return(dst);
} /* Ende _inet_ntop4 */
