/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* sound: handle sound events. Sets SIGPIPE to SIG_IGN! */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <fcntl.h>
#include <unistd.h>
#include <signal.h>
#include <errno.h>
#include "pfad.h"

#define VGAG_WAVE_SERVER BIN_PATH "/vgag-wave"

extern char errmsg[2048];
int pdw[2],pdr[2];
pid_t cpidw=0;

int init_sound(long,int);
int load_wave(const char *,int);
void play_wave(int,int,short,short);
void stop_wave(int,short);
void pause_wave(int);
void continue_wave(int);
void volume_wave(int,short);
void accel_wave(short,short);
void end_sound(void);
int set_vol(int,int);



int init_sound(long smpfq,int moso) {
/* start wave server.
** 1.arg: samplefrequence to set
** 2.arg: 0=set mono or 1=set stereo
** return: 0=ok or -1=error
*/
  struct sigaction sa;
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=SIG_IGN;
  if (sigaction(SIGPIPE,&sa,NULL)) {strcpy(errmsg,"Error sigaction"); return(-1);}
  if (pipe(pdw)==-1) {snprintf(errmsg,sizeof(errmsg),"Pipe: %s",strerror(errno)); return(-1);}
  if (pipe(pdr)==-1) {snprintf(errmsg,sizeof(errmsg),"Pipe: %s",strerror(errno)); return(-1);}
  if ((cpidw=fork())==-1) {snprintf(errmsg,sizeof(errmsg),"Fork: %s",strerror(errno)); return(-1);}
  if (cpidw==0) {  /* Child */
    char istr[64];
    close(pdw[1]);
    close(pdr[0]);
    if (pdw[0]!=STDIN_FILENO) {
      if (dup2(pdw[0],STDIN_FILENO)==-1) {snprintf(errmsg,sizeof(errmsg),"Dup2: %s",strerror(errno)); return(-1);}
      close(pdw[0]);
    }
    if (pdr[1]!=STDERR_FILENO) {
      if (dup2(pdr[1],STDERR_FILENO)==-1) {snprintf(errmsg,sizeof(errmsg),"Dup2: %s",strerror(errno)); return(-1);}
      close(pdr[1]);
    }
    snprintf(istr,sizeof(istr),"-s%ld%c",smpfq,moso==1?'S':'M');
    execlp(VGAG_WAVE_SERVER,VGAG_WAVE_SERVER,"-q",istr,(void *)0);
    fprintf(stderr,"Cannot execute %s\n",VGAG_WAVE_SERVER);
    exit(1);
  }
  close(pdw[0]);
  close(pdr[1]);
  return(0);
} /* Ende init_sound */


int load_wave(const char * wvfile,int emul) {
/* connect a wave file to a number
** 1.arg: wave filename
** 2.arg: emulation type: TYP_IS_WAVE
**                        TYP_IS_MIDI
**                        TYP_IS_MP3
**                        TYP_IS_EXTENSION
** return value: wave number or 0=error (server exit)
*/
  static int wvnr=0;
  char buf[64];
  if ((cpidw==0) || (wvfile==NULL)) {return(0);}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return(0);}
  wvnr++;
  snprintf(buf,sizeof(buf),"L %d %d ",wvnr,emul);
  write(pdw[1],buf,strlen(buf));
  write(pdw[1],wvfile,strlen(wvfile));
  if (write(pdw[1],"\n",1)!=1) {close(pdw[1]); close(pdr[0]); cpidw=0; return(0);}
  return(wvnr);
} /* Ende load_wave */


void play_wave(int wvnr,int chnr,short loop,short voltime) {
/* plays wave file assigned to number
** 1.arg: wave number
** 2.arg: play one of these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 3.arg: 0=play once or >0=play looping with pause of 3.arg seconds
** 4.arg: 0=normal or >0=time in seconds to increase volume to 100%
*/
  int buflen;
  char buf[64];
  if ((cpidw==0) || (wvnr<=0) || (chnr<=0)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"S %d %d %d/%d\n",wvnr,chnr,loop,voltime);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende play_wave */


void stop_wave(int chnr,short voltime) {
/* stops playing wave file at channel numbers
** 1.arg: stop these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 2.arg: 0=at once or >0=time in seconds to decrease volume to 0%
*/
  int buflen;
  char buf[64];
  if ((cpidw==0) || (chnr<=0)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"E 0 %d %d\n",chnr,voltime);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende stop_wave */


void pause_wave(int chnr) {
/* pauses playing wave file at channel numbers
** 1.arg: pause these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
*/
  int buflen;
  char buf[64];
  if ((cpidw==0) || (chnr<=0)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"P 0 %d \n",chnr);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende pause_wave */


void continue_wave(int chnr) {
/* continues playing wave file at channel numbers
** 1.arg: continue these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
*/
  int buflen;
  char buf[64];
  if ((cpidw==0) || (chnr<=0)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"C 0 %d \n",chnr);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende continue_wave */


void volume_wave(int chnr,short perc) {
/* changes volume percent at channel numbers
** 1.arg: change these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 2.arg: 0 to 30 (where 0=0%, 1=10%, 2=20%, ..., 10=100%, ..., 30=300%)
*/
  int buflen;
  char buf[64];
  if ((cpidw==0) || (chnr<=0)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"V 0 %d %d\n",chnr,perc-10);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende volume_wave */


void accel_wave(short smpfq,short moso) {
/* accelerate playing wave/midi at cost of quality
** 1.arg: 0=don't change samplefrequence factor
**        1-4=divide samplefrequence with this number
** 2.arg: 0=don't change mono/stereo settings
**        1=set all to mono
**        2=set mono files to mono and stereo files to stereo
*/
  int buflen;
  char buf[64];
  if (cpidw==0) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return;}
  buflen=snprintf(buf,sizeof(buf),"A 0 %d %d\n",smpfq,moso);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
} /* Ende accel_wave */


void end_sound() {
/* stops playing and exits server */
  if (cpidw!=0) {
    waitpid(cpidw,NULL,WNOHANG);
    close(pdw[1]);
    close(pdr[0]);
    cpidw=0;
  }
} /* Ende end_sound */


int set_vol(int volm,int flag) {
/* set volume to 1.arg
** 1.arg: 0-100 or <0 to get only actual volume
** 2.arg: 0=main volume or 1=pcm volume (wave)
** return: <volume before (>=0)>=ok or -1=error
*/
  int buflen;
  char buf[64],* p1;
  if (cpidw==0) {return(-1);}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {close(pdw[1]); close(pdr[0]); cpidw=0; return(-1);}
  buflen=snprintf(buf,sizeof(buf),"M 0 %d %d\n",volm,flag);
  if (write(pdw[1],buf,buflen)!=buflen) {close(pdw[1]); close(pdr[0]); cpidw=0;}
  p1=buf;
  while ((buflen=read(pdr[0],p1,1))>0) {
    if (*p1=='\n') {break;}
    p1++; if (p1-buf>=(int)sizeof(buf)) {p1--;}
  }
  if (buflen==-1) {
    close(pdw[1]);
    close(pdr[0]);
    cpidw=0;
  } else {
    *p1='\0';
    buflen=atoi(buf);
  }
  return(buflen);
} /* Ende set_vol */
