/*
  For vgagames modified from:
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2001 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#define HAVE_SIGNAL

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h> /* for open */
#include <ctype.h>

#include <signal.h>

#if defined(__FreeBSD__)
#include <floatingpoint.h> /* For FP exceptions */
#endif

#include "tmidi.h"
#include "common.h"
#include "instrum.h"
#include "playmidi.h"
#include "readmidi.h"
#include "output.h"
#include "controls.h"
#include "tables.h"
#include "miditrace.h"
#include "reverb.h"
#include "recache.h"
#include "strtab.h"
#define DEFINE_GLOBALS
#include "aq.h"
#include "mix.h"
#include "unimod.h"

#define OPTCOMMANDS "4A:aB:b:C:c:D:d:E:eFfg:hI:i:jk:L:M:m:n:O:o:P:p:Q:q:R:rS:s:t:T:UW:w:x:Z:"	/* Only GHJKlNuVvXz are remain... */
#define INTERACTIVE_INTERFACE_IDS "kmqagrwA"
char *timidity_version = "2.10.4";

/* main interfaces (To be used another main) */
#if defined(main) || defined(ANOTHER_MAIN)
#define MAIN_INTERFACE
#else
#define MAIN_INTERFACE static
#endif /* main */

MAIN_INTERFACE void timidity_start_initialize(void);
MAIN_INTERFACE int timidity_pre_load_configuration(void);
MAIN_INTERFACE int timidity_post_load_configuration(void);
MAIN_INTERFACE void timidity_init_player(void);
MAIN_INTERFACE int timidity_play_main(int nfiles, char **files);
MAIN_INTERFACE int got_a_configuration;
char *opt_aq_max_buff = NULL,
     *opt_aq_fill_buff = NULL;
void timidity_init_aq_buff(void);
int opt_control_ratio = 0; /* Save -C option */

extern struct URL_module URL_module_file;

MAIN_INTERFACE struct URL_module *url_module_list[] =
{
    &URL_module_file,
    NULL
};

#ifndef MAXPATHLEN
#define MAXPATHLEN 1024
#endif /* MAXPATHLEN */

int free_instruments_afterwards=0;
char def_instr_name[256]="";
VOLATILE int intr = 0;


int effect_lr_mode = -1;
/* 0: left delay
 * 1: right delay
 * 2: rotate
 * -1: not use
 */
int effect_lr_delay_msec = 25;

extern char* pcm_alternate_file;
/* NULL, "none": disabled (default)
 * "auto":       automatically selected
 * filename:     use the one.
 */

#ifndef atof
extern double atof(const char *);
#endif

#define dynamic_interface_module(dmy) NULL
#define dynamic_interface_info(dmy) NULL


static void help(void)
{
  PlayMode **pmp=play_mode_list;
  int i, j;
  static char *help_args[3];
  FILE *fp;
  static char *help_list[] = {
" MIDI to WAVE converter from:",
" TiMidity++ version %s (C) 1999-2001 Masanao Izumo <mo@goice.co.jp>",
" The original version (C) 1995 Tuukka Toivonen <tt@cgs.fi>",
" TiMidity is free software and comes with ABSOLUTELY NO WARRANTY.",
"",
"",
"Usage:",
"  %s [options] filename [...]",
"",
"  Use \"-\" as filename to read a MIDI file from stdin",
"",
"Options:",
"  -A n    Amplify volume by n percent (may cause clipping)",
"  -a      Enable the antialiasing filter",
"  -B n,m  Set number of buffer fragments(n), and buffer size(2^m)",
"  -C n    Set ratio of sampling and control frequencies",
"  -D n    Play drums on channel n",
"  -F      Disable/Enable fast panning (toggle on/off, default is on)",
"  -f      "
#ifdef FAST_DECAY
           "Disable"
#else
	   "Enable"
#endif
" fast decay mode (toggle)",
"  -h      Display this help message",
"  -I n    Use program n as the default",
"  -j      Realtime load instrument (toggle on/off)",
"  -k msec Specify audio queue time limit to reduce voice",
"  -m msec Minimum time for a full volume sustained note to decay, 0 disables",
"  -O mode Select output mode and format (see below for list)",
"  -o file Output to another file (or device/server)  (Use \"-\" for stdout)",
"  -P file Use patch file for all programs",
"  -p n(a) Allow n-voice polyphony.  Optional auto polyphony reduction toggle."
,
"  -p a    Toggle automatic polyphony reduction.  Enabled by default.",
"  -Q n    Ignore channel n",
"  -q m/n  Specify audio buffer in seconds",
"            m:Maxmum buffer, n:Filled to start   (default is 5.0/100%%)",
"            (size of 100%% equals device buffer size)",
"  -R n    Pseudo Reveb (set every instrument's release to n ms",
"            if n=0, n is set to 800(default)",
"  -s f    Set sampling frequency to f (Hz or kHz)",
"  -T n    Adjust tempo to n%%; 120=play MOD files with an NTSC Amiga's timing",
NULL
};

  fp = stdout;
  j = 0; /* index of help_args */
  help_args[0] = timidity_version;
  help_args[1] = program_name;
  help_args[2] = NULL;

  for(i = 0; help_list[i]; i++)
  {
      char *h, *p;

      h = help_list[i];
      if((p = strchr(h, '%')) != NULL)
      {
	  if(*(p + 1) != '%')
	      fprintf(fp, h, help_args[j++]);
	  else
	      fprintf(fp, h);
      }
      else
	  fputs(h, fp);
      fputs(NLS, fp);
  }

  fputs(NLS, fp);
  fputs("Available output modes (-O option):" NLS, fp);
  while(*pmp)
  {
      fprintf(fp, "  -O%c     %s" NLS, (*pmp)->id_character, (*pmp)->id_name);
      pmp++;
  }
  fputs(NLS, fp);
  fputs("Output format options (append to -O? option):" NLS
	"   `M'    monophonic" NLS
	"   `S'    stereo" NLS
	"   `x'    byte-swapped output" NLS
	, fp);
  fputs(NLS, fp);
}

static void interesting_message(void)
{
  printf(
NLS
" MIDI to WAVE converter from TiMidity++ version %s" NLS
" Copyright (C) 1999-2001 Masanao Izumo <mo@goice.co.jp>" NLS
" Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>" NLS
NLS
" This program is free software; you can redistribute it and/or modify" NLS
" it under the terms of the GNU General Public License as published by" NLS
" the Free Software Foundation; either version 2 of the License, or" NLS
" (at your option) any later version." NLS
NLS
" This program is distributed in the hope that it will be useful," NLS
" but WITHOUT ANY WARRANTY; without even the implied warranty of" NLS
" MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the" NLS
" GNU General Public License for more details." NLS
NLS
" You should have received a copy of the GNU General Public License" NLS
" along with this program; if not, write to the Free Software" NLS
" Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA." NLS
NLS,
timidity_version
);
}

static int set_channel_flag(ChannelBitMask *flags, int32 i, char *name)
{
    if(i == 0)
	memset(flags, 0, sizeof(ChannelBitMask));
    else if((i < 1 || i > MAX_CHANNELS) && (i < -MAX_CHANNELS || i > -1))
    {
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		"%s must be between 1 and %d, or between -1 and -%d, or 0"
		NLS, name, MAX_CHANNELS, MAX_CHANNELS);
	return -1;
    }
    else
    {
	if(i > 0)
	    SET_CHANNELMASK(*flags, i - 1);
	else
	    UNSET_CHANNELMASK(*flags, -i - 1);
    }
    return 0;
}

static int set_value(int32 *param, int32 i, int32 low, int32 high, char *name)
{
  if (i<low || i > high)
    {
      	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "%s must be between %ld and %ld",
		  name, low, high);
	return -1;
    }
  else *param=i;
  return 0;
}

MAIN_INTERFACE int set_default_prog(char *opt)
{
    int prog, ch;
    char *p;

    prog = atoi(opt);
    if(prog < 0 || prog > 127)
    {
      	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "Default program must be between 0 and 127");
	return -1;
    }

    p = strchr(opt, '/');
    if(p == NULL)
	for(ch = 0; ch < MAX_CHANNELS; ch++)
	    default_program[ch] = prog;
    else
    {
	ch = atoi(p + 1) - 1;
	if(ch < 0 || ch >= MAX_CHANNELS)
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "Default program channel must be between 1 and %d",
		      MAX_CHANNELS);
	    return -1;
	}
	default_program[ch] = prog;
    }
    return 0;
}

int set_play_mode(char *cp)
{
    PlayMode *pmp, **pmpp=play_mode_list;

    while((pmp=*pmpp++))
    {
        pmp->encoding &= ~PE_16BIT;
        pmp->encoding |= PE_MONO;
	if(pmp->id_character == *cp)
	{
	    play_mode=pmp;
	    while(*(++cp))
		switch(*cp)
		{
		  case 'M': pmp->encoding |= PE_MONO; break;
		  case 'S': pmp->encoding &= ~PE_MONO; break; /* stereo */

		  case 'x':
		    pmp->encoding ^= PE_BYTESWAP; /* toggle */
		    break; 

		  default:
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "Unknown format modifier `%c'", *cp);
		    return 1;
		}
	    return 0;
	}
    }

    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
	      "Playmode `%c' is not compiled in.", *cp);
    return 1;
}


static void copybank(ToneBank *to, ToneBank *from)
{
    int i;

    if(from == NULL)
	return;
    for(i = 0; i < 128; i++)
    {
	ToneBankElement *toelm, *fromelm;

	toelm   = &to->tone[i];
	fromelm = &from->tone[i];

	if(fromelm->name == NULL)
	    continue;

	if(toelm->name)
	    free(toelm->name);
	if(toelm->comment)
	    free(toelm->comment);
	memcpy(toelm, fromelm, sizeof(ToneBankElement));
	if(toelm->name)
	    toelm->name = safe_strdup(toelm->name);
	if(toelm->comment)
	    toelm->comment = safe_strdup(toelm->comment);
	toelm->instrument = NULL;
    }
}

static int set_gus_patchconf_opts(char *name, int line, char *opts,
				  ToneBankElement *tone)
{
    char *cp;
    int k;

    if(!(cp = strchr(opts, '=')))
    {
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "%s: line %d: bad patch option %s",
		  name, line, opts);
	return 1;
    }

    *cp++ = 0;
    if(!strcmp(opts, "amp"))
    {
	k = atoi(cp);
	if((k < 0 || k > MAX_AMPLIFICATION) ||
	   (*cp < '0' || *cp > '9'))
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: amplification must be between "
		      "0 and %d", name, line, MAX_AMPLIFICATION);
	    return 1;
	}
	tone->amp = k;
    }
    else if(!strcmp(opts, "note"))
    {
	k = atoi(cp);
	if((k < 0 || k > 127) || (*cp < '0' || *cp > '9'))
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: note must be between 0 and 127",
		      name, line);
	    return 1;
	}
	tone->note = k;
    }
    else if(!strcmp(opts, "pan"))
    {
	if(!strcmp(cp, "center"))
	    k = 64;
	else if(!strcmp(cp, "left"))
	    k = 0;
	else if(!strcmp(cp, "right"))
	    k = 127;
	else
	    k = ((atoi(cp) + 100) * 100) / 157;
	if((k < 0 || k > 127) ||
	   (k == 0 && *cp != '-' && (*cp < '0' || *cp > '9')))
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: panning must be left, right, "
		      "center, or between -100 and 100",
		      name, line);
	    return 1;
	}
	tone->pan = k;
    }
    else if(!strcmp(opts, "keep"))
    {
	if(!strcmp(cp, "env"))
	    tone->strip_envelope = 0;
	else if(!strcmp(cp, "loop"))
	    tone->strip_loop = 0;
	else
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: keep must be env or loop",
		      name, line);
	    return 1;
	}
    }
    else if(!strcmp(opts, "strip"))
    {
	if(!strcmp(cp, "env"))
	    tone->strip_envelope = 1;
	else if(!strcmp(cp, "loop"))
	    tone->strip_loop = 1;
	else if(!strcmp(cp, "tail"))
	    tone->strip_tail = 1;
	else
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: strip must be "
		      "env, loop, or tail", name, line);
	    return 1;
	}
    }
    else if(!strcmp(opts, "comm"))
    {
	char *p;
	if(tone->comment)
	    free(tone->comment);
	p = tone->comment = safe_strdup(cp);
	while(*p)
	{
	    if(*p == ',') *p = ' ';
	    p++;
	}
    }
    else
    {
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "%s: line %d: bad patch option %s",
		  name, line, opts);
	return 1;
    }
    return 0;
}


static int set_gus_patchconf(char *name, int line,
			     ToneBankElement *tone, char *pat, char **opts)
{
    int j;

    if(tone->name)
    {
	free(tone->name);
	tone->name = NULL;
    }
    tone->note = tone->pan =
	tone->strip_loop = tone->strip_envelope =
	    tone->strip_tail = -1;
    tone->amp = -1;

    if(strcmp(pat, "%font") == 0) /* Font extention */
    {
	/* %font filename bank prog [note-to-use]
	 * %font filename 128 bank key
	 */

	if(opts[0] == NULL || opts[1] == NULL || opts[2] == NULL ||
	   (atoi(opts[1]) == 128 && opts[3] == NULL))
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "%s: line %d: Syntax error", name, line);
	    return 1;
	}
	tone->name = safe_strdup(opts[0]);
	tone->instype = 1;
	if(atoi(opts[1]) == 128) /* drum */
	{
	    tone->font_bank = 128;
	    tone->font_preset = atoi(opts[2]);
	    tone->note = atoi(opts[3]);
	    opts += 4;
	}
	else
	{
	    tone->font_bank = atoi(opts[1]);
	    tone->font_preset = atoi(opts[2]);

	    if(opts[3] && ('0' <= opts[3][0] && opts[3][0] <= '9'))
	    {
		tone->note = atoi(opts[3]);
		opts += 4;
	    }
	    else
	    {
		tone->note = -1;
		opts += 3;
	    }
	}
    }
    else
    {
	tone->instype = 0;
	tone->name = safe_strdup(pat);
    }

    for(j = 0; opts[j] != NULL; j++)
    {
	int err;
	if((err = set_gus_patchconf_opts(name, line, opts[j], tone)) != 0)
	    return err;
    }
    if(tone->comment == NULL)
	tone->comment = safe_strdup(tone->name);
    return 0;
}

static int mapname2id(char *name, int *isdrum)
{
    if(strcmp(name, "sc55") == 0)
    {
	*isdrum = 0;
	return SC_55_TONE_MAP;
    }

    if(strcmp(name, "sc55drum") == 0)
    {
	*isdrum = 1;
	return SC_55_DRUM_MAP;
    }

    if(strcmp(name, "sc88") == 0)
    {
	*isdrum = 0;
	return SC_88_TONE_MAP;
    }

    if(strcmp(name, "sc88drum") == 0)
    {
	*isdrum = 1;
	return SC_88_DRUM_MAP;
    }

    if(strcmp(name, "sc88pro") == 0)
    {
	*isdrum = 0;
	return SC_88PRO_TONE_MAP;
    }

    if(strcmp(name, "sc88prodrum") == 0)
    {
	*isdrum = 1;
	return SC_88PRO_DRUM_MAP;
    }

    if(strcmp(name, "xg") == 0)
    {
	*isdrum = 0;
	return XG_SFX64_MAP;
    }

    if(strcmp(name, "xgsfx64") == 0)
    {
	*isdrum = 0;
	return XG_SFX64_MAP;
    }

    if(strcmp(name, "xgsfx126") == 0)
    {
	*isdrum = 1;
	return XG_SFX126_MAP;
    }

    if(strcmp(name, "xgdrum") == 0)
    {
	*isdrum = 1;
	return XG_DRUM_MAP;
    }
    return -1;
}

#define MAXWORDS 130
#define CHECKERRLIMIT \
  if(++errcnt >= 10) { \
    ctl->cmsg(CMSG_ERROR, VERB_NORMAL, \
      "Too many errors... Give up read %s", name); \
    close_file(tf); return 1; }

MAIN_INTERFACE int set_tim_opt(int c, char *optarg);
MAIN_INTERFACE int read_config_file(char *name, int self)
{
    struct timidity_file *tf;
    char tmp[1024], *w[MAXWORDS + 1], *cp;
    ToneBank *bank = NULL;
    int i, j, k, line = 0, words, errcnt = 0;
    static int rcf_count = 0;
    int dr = 0, bankno = 0;
    int extension_flag;

    if(rcf_count > 50)
    {
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "Probable source loop in configuration files");
	return 2;
    }

    if(self)
    {
	tf = open_with_mem(name, (int32)strlen(name), OF_VERBOSE);
	name = "(configuration)";
    }
    else
	tf = open_file(name, 1, OF_VERBOSE);
    if(tf == NULL)
	return 1;

    errno = 0;
    while(tf_gets(tmp, sizeof(tmp), tf))
    {
	line++;
	if(strncmp(tmp, "#extension", 10) == 0) {
	    extension_flag = 1;
	    i = 10;
	}
	else
	{
	    extension_flag = 0;
	    i = 0;
	}

	/* Chop the comment.
	 * Comment delimiter =~ /^#|\s#|#[ \t]/
	 */
	if((cp = strchr(tmp + i, '#')) != NULL) {
	    if(cp == tmp + i ||		/* beginning of line */
	       isspace((int)cp[-1]) ||	/* "\s#" */
	       cp[1] == ' ' ||		/* "#[ \t]" */
	       cp[1] == '\t')
		*cp = '\0';
	}

	if((w[0] = strtok(tmp + i, " \t\r\n\240")) == NULL)
	    continue;

	words = 0;
	while(w[words] && words < MAXWORDS)
	    w[++words] = strtok(NULL," \t\r\n\240");
	w[words] = NULL;

	/*
	 * #extension [something...]
	 */

	/* #extension comm program comment */
	if(strcmp(w[0], "comm") == 0)
	{
	    char *p;

	    if(words != 3)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum "
			  "set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension comm must be "
			  "between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(bank->tone[i].comment)
		free(bank->tone[i].comment);
	    p = bank->tone[i].comment = safe_strdup(w[2]);
	    while(*p)
	    {
		if(*p == ',') *p = ' ';
		p++;
	    }
	}
	/* #extension timeout program sec */
	else if(strcmp(w[0], "timeout") == 0)
	{
	    if(words != 3)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension timeout "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    bank->tone[i].loop_timeout = atoi(w[2]);
	}
	/* #extension copydrumset drumset */
	else if(strcmp(w[0], "copydrumset") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No copydrumset number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension copydrumset "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    copybank(bank, drumset[i]);
	}
	/* #extension copybank bank */
	else if(strcmp(w[0], "copybank") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No copybank number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension copybank "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    copybank(bank, tonebank[i]);
	}
	/* #extension HTTPproxy hostname:port */
	else if(strcmp(w[0], "HTTPproxy") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No proxy name given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    /* If network is not supported, this extension is ignored. */
	}
	/* #extension FTPproxy hostname:port */
	else if(strcmp(w[0], "FTPproxy") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No proxy name given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    /* If network is not supported, this extension is ignored. */
	}
	/* #extension mailaddr somebody@someware.domain.com */
	else if(strcmp(w[0], "mailaddr") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No mail address given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(strchr(w[1], '@') == NULL) {
		ctl->cmsg(CMSG_WARNING, VERB_NOISY,
			  "%s: line %d: Warning: Mail address %s is not valid",
			  name, line);
	    }

	    /* If network is not supported, this extension is ignored. */
	}
	/* #extension opt [-]{option}[optarg] */
	else if(strcmp(w[0], "opt") == 0)
	{
	    int c, err;
	    char *arg, *cmd, *p;

	    if(words != 2 && words != 3)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    cmd = w[1];
	    if(*cmd == '-')
		cmd++;
	    c = *cmd;
	    p = strchr(OPTCOMMANDS, c);
	    if(p == NULL)
		err = 1;
	    else
	    {
		if(*(p + 1) == ':')
		{
		    if(words == 2)
			arg = cmd + 1;
		    else
			arg = w[2];
		}
		else
		    arg = "";
		err = set_tim_opt(c, arg);
	    }
	    if(err)
	    {
		/* error */
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Invalid command line option",
			  name, line);
		errcnt += err - 1;
		CHECKERRLIMIT;
		continue;
	    }
	}
	/* #extension undef program */
	else if(strcmp(w[0], "undef") == 0)
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No undef number given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: extension undef "
			  "must be between 0 and 127", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or "
			  "drum set before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(bank->tone[i].name)
	    {
		free(bank->tone[i].name);
		    bank->tone[i].name = NULL;
	    }
	    if(bank->tone[i].comment)
	    {
		free(bank->tone[i].comment);
		bank->tone[i].comment = NULL;
	    }
	}
	/* #extension altassign numbers... */
	else if(strcmp(w[0], "altassign") == 0)
	{
	    ToneBank *bk;

	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before altassign", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No alternate assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    if(!dr) {
		ctl->cmsg(CMSG_WARNING, VERB_NORMAL,
			  "%s: line %d: Warning: Not a drumset altassign"
			  " (ignored)",
			  name, line);
		continue;
	    }

	    bk = drumset[bankno];
	    bk->alt = add_altassign_string(bk->alt, w + 1, words - 1);
	}
	else if(!strcmp(w[0], "soundfont"))
	{
	    int order, cutoff, isremove, reso, amp;
	    char *sf_file;

	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No soundfont file given",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    sf_file = w[1];
	    order = cutoff = reso = amp = -1;
	    isremove = 0;
	    for(j = 2; j < words; j++)
	    {
		if(strcmp(w[j], "remove") == 0)
		{
		    isremove = 1;
		    break;
		}
		if(!(cp = strchr(w[j], '=')))
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: bad patch option %s",
			      name, line, w[j]);
		    CHECKERRLIMIT;
		    break;
		}
		*cp++=0;
		k = atoi(cp);
		if(!strcmp(w[j], "order"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: order must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    order = k;
		}
		else if(!strcmp(w[j], "cutoff"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: cutoff must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    cutoff = k;
		}
		else if(!strcmp(w[j], "reso"))
		{
		    if(k < 0 || (*cp < '0' || *cp > '9'))
		    {
			ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
				  "%s: line %d: reso must be a digit",
				  name, line);
			CHECKERRLIMIT;
			break;
		    }
		    reso = k;
		}
		else if(!strcmp(w[j], "amp"))
		{
		    amp = k;
		}
	    }
	    if(isremove)
		remove_soundfont(sf_file);
	    else
		add_soundfont(sf_file, order, cutoff, reso, amp);
	}
	else if(!strcmp(w[0], "font"))
	{
	    int bank, preset, keynote;
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: no font command", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!strcmp(w[1], "exclude"))
	    {
		if(words < 3)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No bank/preset/key is given",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}
		bank = atoi(w[2]);
		if(words >= 4)
		    preset = atoi(w[3]) - progbase;
		else
		    preset = -1;
		if(words >= 5)
		    keynote = atoi(w[4]);
		else
		    keynote = -1;
		if(exclude_soundfont(bank, preset, keynote))
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No soundfont is given",
			      name, line);
		    CHECKERRLIMIT;
		}
	    }
	    else if(!strcmp(w[1], "order"))
	    {
		int order;
		if(words < 4)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No order/bank is given",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}
		order = atoi(w[2]);
		bank = atoi(w[3]);
		if(words >= 5)
		    preset = atoi(w[4]) - progbase;
		else
		    preset = -1;
		if(words >= 6)
		    keynote = atoi(w[5]);
		else
		    keynote = -1;
		if(order_soundfont(bank, preset, keynote, order))
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: No soundfont is given",
			      name, line);
		    CHECKERRLIMIT;
		}
	    }
	}
	else if(!strcmp(w[0], "progbase"))
	{
	    if(words < 2 || *w[1] < '0' || *w[1] > '9')
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    progbase = atoi(w[1]);
	}
	else if(!strcmp(w[0], "map")) /* map <name> set1 elem1 set2 elem2 */
	{
	    int arg[5], isdrum;

	    if(words != 6)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    if((arg[0] = mapname2id(w[1], &isdrum)) == -1)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Invalid map name: %s", w[1]);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 2; i < 6; i++)
		arg[i - 1] = atoi(w[i]);
	    if(isdrum)
	    {
		arg[1] -= progbase;
		arg[3] -= progbase;
	    }
	    else
	    {
		arg[2] -= progbase;
		arg[4] -= progbase;
	    }

	    for(i = 1; i < 5; i++)
		if(arg[i] < 0 || arg[i] > 127)
		    break;
	    if(i != 5)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Invalid parameter", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    set_instrument_map(arg[0], arg[1], arg[2], arg[3], arg[4]);
	}

	/*
	 * Standard configurations
	 */
	else if(!strcmp(w[0], "dir"))
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No directory given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 1; i < words; i++)
		add_to_pathlist(w[i]);
	}
	else if(!strcmp(w[0], "source"))
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No file name given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    for(i = 1; i < words; i++)
	    {
		int status;
		rcf_count++;
		status = read_config_file(w[i], 0);
		rcf_count--;
		if(status == 2)
		{
		    close_file(tf);
		    return 2;
		}
		else if(status != 0)
		{

		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else if(!strcmp(w[0], "default"))
	{
	    if(words != 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify exactly one patch name",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    strncpy(def_instr_name, w[1], 255);
	    def_instr_name[255] = '\0';
	    default_instrument_name = def_instr_name;
	}
	else if(!strcmp(w[0], "drumset"))
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No drum set number given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]) - progbase;
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Drum set must be between %d and %d",
			  name, line,
			  progbase, progbase + 127);
		CHECKERRLIMIT;
		continue;
	    }

	    alloc_instrument_bank(1, i);

	    if(words == 2)
	    {
		bank = drumset[i];
		bankno = i;
		dr = 1;
	    }
	    else
	    {
		ToneBank *localbank;

		localbank = drumset[i];

		if(words < 4 || *w[2] < '0' || *w[2] > '9')
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: syntax error", name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		i = atoi(w[2]);
		if(i < 0 || i > 127)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Drum number must be between "
			      "0 and 127",
			      name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		if(set_gus_patchconf(name, line, &localbank->tone[i],
				     w[3], w + 4))
		{
		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else if(!strcmp(w[0], "bank"))
	{
	    if(words < 2)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: No bank number given", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[1]);
	    if(i < 0 || i > 127)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Tone bank must be between 0 and 127",
			  name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    alloc_instrument_bank(0, i);

	    if(words == 2)
	    {
		bank = tonebank[i];
		bankno = i;
		dr = 0;
	    }
	    else
	    {
		ToneBank *localbank;

		localbank = tonebank[i];

		if(words < 4 || *w[2] < '0' || *w[2] > '9')
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: syntax error", name, line);
		    CHECKERRLIMIT;
		    continue;
		}

		i = atoi(w[2]) - progbase;
		if(i < 0 || i > 127)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Program must be between "
			      "%d and %d",
			      name, line,
			      progbase, 127 + progbase);
		    CHECKERRLIMIT;
		    continue;
		}

		if(set_gus_patchconf(name, line, &localbank->tone[i],
				     w[3], w + 4))
		{
		    CHECKERRLIMIT;
		    continue;
		}
	    }
	}
	else
	{
	    if(words < 2 || *w[0] < '0' || *w[0] > '9')
	    {
		if(extension_flag)
		    continue;
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: syntax error", name, line);
		CHECKERRLIMIT;
		continue;
	    }
	    i = atoi(w[0]);
	    if(!dr)
		i -= progbase;
	    if(i < 0 || i > 127)
	    {
		if(dr)
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Drum number must be between "
			      "0 and 127",
			      name, line);
		else
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "%s: line %d: Program must be between "
			      "%d and %d",
			      name, line,
			      progbase, 127 + progbase);
		CHECKERRLIMIT;
		continue;
	    }
	    if(!bank)
	    {
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "%s: line %d: Must specify tone bank or drum set "
			  "before assignment", name, line);
		CHECKERRLIMIT;
		continue;
	    }

	    if(set_gus_patchconf(name, line, &bank->tone[i], w[1], w + 2))
	    {
		CHECKERRLIMIT;
		continue;
	    }
	}
    }
    if(errno)
    {
	ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		  "Can't read %s: %s", name, strerror(errno));
	errcnt++;
    }
    close_file(tf);
    return errcnt != 0;
}


static int parse_effect_option(char *effect_opts)
{
    int i;

    if(strncmp(effect_opts, "delay=", 6) == 0)
    {
	switch(effect_opts[6])
	{
	  case 'l':
	    effect_lr_mode = 0;
	    break;
	  case 'r':
	    effect_lr_mode = 1;
	    break;
	  case 'b':
	    effect_lr_mode = 2;
	    break;
	  case '0':
	    effect_lr_mode = -1;
	    return 0;
	}
	if(effect_opts[7] == ',')
	{
	    effect_lr_delay_msec = atoi(effect_opts + 8);
	    if(effect_lr_delay_msec < 0)
	    {
		effect_lr_delay_msec = 0;
		effect_lr_mode = -1;
		ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			  "Invalid -EFdelay parameter.");
		return 1;
	    }
	}
	return 0;
    }

    if(strncmp(effect_opts, "reverb=", 7) == 0)
    {
	effect_opts += 7;
	switch(*effect_opts)
	{
	  case '0':
	    opt_reverb_control = 0;
	    break;
	  case '1':
	    if(*(effect_opts + 1) == ',')
		opt_reverb_control = -(atoi(effect_opts + 2) & 0x7f);
	    else
		opt_reverb_control = 1;
	    break;
	  case '2':
	    opt_reverb_control = 2;
	    break;
	  default:
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "Invalid -EFreverb parameter.");
	    return 1;
	}
	return 0;
    }

    if(strncmp(effect_opts, "chorus=", 7) == 0)
    {
	effect_opts += 7;
	switch(*effect_opts)
	{
	  case '0':
	    opt_chorus_control = 0;
	    opt_surround_chorus = 0;
	    break;

	  case '1':
	  case '2':
	    opt_surround_chorus = (*effect_opts == '2');
	    if(*(effect_opts + 1) == ',')
		opt_chorus_control = -(atoi(effect_opts + 2) & 0x7f);
	    else
		opt_chorus_control = 1;
	    break;
	  default:
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "Invalid -EFchorus parameter.");
	    return 1;
	}
	return 0;
    }

    if(strncmp(effect_opts, "ns=", 3) == 0)
    {
	/* Noise Shaping filter from
	 * Kunihiko IMAI <imai@leo.ec.t.kanazawa-u.ac.jp>
	 */

	i = atoi(effect_opts + 3);
	if(i < 0 || i > 4)
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "-EFns argument range is 0 to 4" );
	    return 1;
	}
	noise_sharp_type = i;
	return 0;
    }

    return 1;
}

int set_extension_modes(char *flag)
{
    int err;

    err = 0;
    while(*flag)
    {
	switch(*flag)
	{
	  case 'w':
	    opt_modulation_wheel = 1;
	    break;
	  case 'W':
	    opt_modulation_wheel = 0;
	    break;
	  case 'p':
	    opt_portamento = 1;
	    break;
	  case 'P':
	    opt_portamento = 0;
	    break;
	  case 'v':
	    opt_nrpn_vibrato = 1;
	    break;
	  case 'V':
	    opt_nrpn_vibrato = 0;
	    break;

	  case 'R':
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "-ER option is obsoleted.  Please use -EFreverb=0");
	    err++;
	    break;

	  case 'r':
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "-Er option is obsoleted.  Please use -EFreverb=2");
	    err++;
	    break;

	  case 'C':
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "-EC option is obsoleted.  Please use -EFchorus=0");
	    err++;
	    break;

	  case 'c':
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "-Ec option is obsoleted.  "
		      "Please use -EFchorus (toggle on/off)");
	    err++;
	    break;

	  case 's':
	    opt_channel_pressure = 1;
	    break;
	  case 'S':
	    opt_channel_pressure = 0;
	    break;
	  case 't':
	    opt_trace_text_meta_event = 1;
	    break;
	  case 'T':
	    opt_trace_text_meta_event = 0;
	    break;
	  case 'o':
	    opt_overlap_voice_allow = 1;
	    break;
	  case 'O':
	    opt_overlap_voice_allow = 0;
	    break;
	  case 'm':
	    {
		int val;
		val = str2mID(flag + 1);
		if(val == 0)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "-Em: Illegal value");
		    err++;
		}
		else
		    opt_default_mid = val;
		flag += 2;
	    }
	    break;
	  case 'M':
	    {
		int val;
		val = str2mID(flag + 1);
		if(val == 0)
		{
		    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
			      "-EM: Illegal value");
		    err++;
		}
		else
		    opt_system_mid = val;
		flag += 2;
	    }
	    break;
	  case 'b':
	    if(flag[1] < '0' || flag[1] > '9')
		default_tonebank = 0;
	    else
	    {
		flag++;
		default_tonebank = 0;
		while('0' <= *flag && *flag <= '9')
		    default_tonebank = default_tonebank * 10 + *flag++ - '0';
		default_tonebank &= 0x7f;
		flag--; /* to be inc. */
	    }
	    break;

	  case 'B':
	    if(flag[1] < '0' || flag[1] > '9')
		special_tonebank = -1;
	    else
	    {
		flag++;
		special_tonebank = 0;
		while('0' <= *flag && *flag <= '9')
		    special_tonebank = special_tonebank * 10 + *flag++ - '0';
		special_tonebank &= 0x7f;
		flag--; /* to be inc. */
	    }
	    break;

	  case 'F':
	    if(parse_effect_option(flag + 1))
	    {
		ctl->cmsg(CMSG_ERROR,
			  VERB_NORMAL, "-E%s: unsupported effect", flag);
		return 1+err;
	    }
	    return err;

	  default:
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL, "-E: Illegal mode `%c'", *flag);
	    err++;
	    break;
	}
	flag++;
    }
    return err;
}

static int   try_config_again = 0;
int32 opt_output_rate = 0;
static char *opt_output_name = NULL;
static StringTable opt_config_string;
int opt_buffer_fragments = -1;

static int parse_opt_B(char *opt)
{
  char *p;
  int32 val;

  /* num */
  if(*opt != ',') {
    if(set_value(&val, atoi(opt), 0, 1000, "Buffer fragments (num)"))
      return -1;
    if(val >= 0)
      opt_buffer_fragments = val;
  }

  /* bits */
  if((p = strchr(opt, ',')) != NULL) {
    if(set_value(&val, atoi(p + 1), 0, AUDIO_BUFFER_BITS, "Buffer fragments (bit)"))
      return -1;
    if(val >= 0)
      audio_buffer_bits = val;
  }
  return 0;
}

MAIN_INTERFACE int set_tim_opt(int c, char *optarg)
{
    int32 tmpi32;

    switch(c)
    {
      case 'A':
	if(set_value(&amplification, atoi(optarg), 0, MAX_AMPLIFICATION,
		     "Amplification"))
	    return 1;
	break;

      case 'a':
	antialiasing_allowed = 1;
	break;

      case 'B':
	if(parse_opt_B(optarg))
	  return 1;
	break;

      case 'C':
	if(set_value(&control_ratio, atoi(optarg), 1, MAX_CONTROL_RATIO,
		     "Control ratio"))
	    return 1;
	opt_control_ratio = control_ratio;
	break;

      case 'D':
	tmpi32 = atoi(optarg);
	if(set_channel_flag(&default_drumchannels, tmpi32, "Drum channel"))
	    return 1;
	if(tmpi32 < 0)
	    tmpi32 = -tmpi32;
	set_channel_flag(&default_drumchannel_mask, tmpi32, "Drum channel");
	break;

      case 'F':
	adjust_panning_immediately = !adjust_panning_immediately;
	break;

      case 'f':
	fast_decay = (fast_decay) ? 0 : 1;
	break;

      case 'h':
	help();
	exit(0);

      case 'I':
	if(set_default_prog(optarg))
	    return -1;
	break;

      case 'j':
	opt_realtime_playing = !opt_realtime_playing;
	break;

      case 'k':
	reduce_voice_threshold = atoi(optarg);
	break;

      case 'm':
        min_sustain_time = atoi(optarg);
        if(min_sustain_time < 0) min_sustain_time = 0;
        break;

      case 'O': /* output mode */
	if(set_play_mode(optarg))
	    return 1;
	break;

      case 'o':
	if(opt_output_name != NULL)
	    free(opt_output_name);
	opt_output_name = safe_strdup(url_expand_home_dir(optarg));
	break;

      case 'P': /* set overriding instrument */
	strncpy(def_instr_name, optarg, 255);
	def_instr_name[255] = '\0';
	break;

      case 'p':
	if (strchr(optarg, 'a')) auto_reduce_polyphony =
	    !auto_reduce_polyphony;
	if (*optarg != 'a') {
	    if(set_value(&tmpi32, atoi(optarg), 1, MAX_VOICES, "Polyphony"))
		return 1;
	    voices = tmpi32;
	}
	break;

      case 'Q':
	if(set_channel_flag(&quietchannels, atoi(optarg), "Quiet channel"))
	    return 1;
	break;

      case 'q':
	if(strchr(optarg, '/') == NULL)
	{
	    if(opt_aq_max_buff)
		free(opt_aq_max_buff);
	    opt_aq_max_buff = safe_strdup(optarg);
	}
	else
	{
	    if(optarg[0] == '/')
	    {
		if(opt_aq_fill_buff)
		    free(opt_aq_fill_buff);
		opt_aq_fill_buff = safe_strdup(optarg + 1);
	    }
	    else
	    {
		char *max_buff, *fill_buff;

		max_buff = safe_strdup(optarg);
		fill_buff = strchr(max_buff, '/');
		*fill_buff++ = '\0';
		if(opt_aq_max_buff)
		    free(opt_aq_max_buff);
		if(opt_aq_fill_buff)
		    free(opt_aq_fill_buff);
		opt_aq_max_buff = max_buff;
		opt_aq_fill_buff = safe_strdup(fill_buff);
	    }
	}
	break;

      case 'R':
        tmpi32 = atoi(optarg);
	if(tmpi32 == -1) {
	    /* reset */
	    modify_release = 0;
	} else {
	    if(set_value(&modify_release, tmpi32, 0, MAX_MREL, "Modify Release"))
		return 1;
	    if (modify_release==0) modify_release=DEFAULT_MREL;
	}
        break;

      case 's': /* sampling rate */
	tmpi32 = atoi(optarg);
	if(tmpi32 < 100)
	    tmpi32 = (int32)(atof(optarg) * 1000.0 + 0.5);
	if(set_value(&opt_output_rate, tmpi32, MIN_OUTPUT_RATE,MAX_OUTPUT_RATE,
		     "Resampling frequency"))
	    return 1;
	break;

       case 'T':
         tmpi32 = atoi(optarg);
         if(set_value(&tmpi32, tmpi32, 10, 400, "Tempo adjust"))
 	    return 1;
 	tempo_adjust = 100.0 / tmpi32;
 	break;

      default:
	return 1;
    }
    return 0;
}

#ifdef HAVE_SIGNAL
static void sigterm_exit(int sig)
{
    if(sig == SIGINT && intr < 0)
    {
	intr++;
	signal(SIGINT, sigterm_exit); /* For SysV base */
    }
    else
	safe_exit(1);
}
#endif /* HAVE_SIGNAL */


MAIN_INTERFACE void timidity_start_initialize(void)
{
    int i;
    static int drums[] = DEFAULT_DRUMCHANNELS;
    static int is_first = 1;
#if defined(__FreeBSD__)
    fp_except_t fpexp;

    fpexp = fpgetmask();
    fpsetmask(fpexp & ~(FP_X_INV|FP_X_DZ));
#endif

    if(!output_text_code)
	output_text_code = safe_strdup(OUTPUT_TEXT_CODE);
    if(!opt_aq_max_buff)
	opt_aq_max_buff = safe_strdup("5.0");
    if(!opt_aq_fill_buff)
	opt_aq_fill_buff = safe_strdup("100%");

    /* Check the byte order */
    i = 1;
#ifdef LITTLE_ENDIAN
    if(*(char *)&i != 1)
#else
    if(*(char *)&i == 1)
#endif
    {
	fprintf(stderr, "Byte order is miss configured.\n");
	exit(1);
    }

    memset(&quietchannels, 0, sizeof(ChannelBitMask));
    memset(&default_drumchannels, 0, sizeof(ChannelBitMask));

    for(i = 0; drums[i] > 0; i++)
	SET_CHANNELMASK(default_drumchannels, drums[i] - 1);
#if MAX_CHANNELS > 16
    for(i = 16; i < MAX_CHANNELS; i++)
	if(IS_SET_CHANNELMASK(default_drumchannels, i & 0xF))
	    SET_CHANNELMASK(default_drumchannels, i);
#endif

    if(program_name == NULL)
	program_name = "TiMidity";
    uudecode_unquote_html = 1;
    for(i = 0; i < MAX_CHANNELS; i++)
    {
	default_program[i] = DEFAULT_PROGRAM;
	memset(channel[i].drums, 0, sizeof(channel[i].drums));
    }

    if(play_mode == NULL)
    {
	char *output_id;
	int i;

	play_mode = play_mode_list[0];
	output_id = getenv("TIMIDITY_OUTPUT_ID");
#ifdef TIMIDITY_OUTPUT_ID
	if(output_id == NULL)
	    output_id = TIMIDITY_OUTPUT_ID;
#endif /* TIMIDITY_OUTPUT_ID */
	if(output_id != NULL)
	{
	    for(i = 0; play_mode_list[i]; i++)
		if(play_mode_list[i]->id_character == *output_id)
		{
		    play_mode = play_mode_list[i];
		    break;
		}
	}
    }

    if(is_first) /* initialize once time */
    {
	got_a_configuration = 0;

	for(i = 0; url_module_list[i]; i++)
	    url_add_module(url_module_list[i]);
	init_string_table(&opt_config_string);
	init_tables();
	for(i = 0; i < NSPECIAL_PATCH; i++)
	    special_patch[i] = NULL;
	init_midi_trace();
	int_rand(-1);	/* initialize random seed */
	int_rand(42);	/* the 1st number generated is not very random */
	ML_RegisterAllLoaders ();
    }

    is_first = 0;
}

MAIN_INTERFACE int timidity_pre_load_configuration(void)
{
    if(!read_config_file(CONFIG_FILE, 0)) got_a_configuration = 1;
    return 0;
}

MAIN_INTERFACE int timidity_post_load_configuration(void)
{
    int cmderr;

    cmderr = 0;
    if(!got_a_configuration)
    {
	if(try_config_again && !read_config_file(CONFIG_FILE, 0))
	    got_a_configuration = 1;
    }

    if(opt_config_string.nstring > 0)
    {
	char **config_string_list;
	int i;

	config_string_list = make_string_array(&opt_config_string);
	if(config_string_list != NULL)
	{
	    for(i = 0; config_string_list[i]; i++)
	    {
		if(!read_config_file(config_string_list[i], 1))
		    got_a_configuration = 1;
		else
		    cmderr++;
	    }
	    free(config_string_list[0]);
	    free(config_string_list);
	}
    }

    if(!got_a_configuration)
	cmderr++;
    return cmderr;
}

MAIN_INTERFACE void timidity_init_player(void)
{
    /* Set play mode parameters */
    if(opt_output_rate != 0)
	play_mode->rate = opt_output_rate;
    else if(play_mode->rate == 0)
	play_mode->rate = DEFAULT_RATE;

    /* save defaults */
    memcpy(&drumchannels, &default_drumchannels, sizeof(ChannelBitMask));
    memcpy(&drumchannel_mask, &default_drumchannel_mask,
	   sizeof(ChannelBitMask));

    if(opt_buffer_fragments != -1)
    {
	if(play_mode->flag & PF_BUFF_FRAGM_OPT)
	    play_mode->extra_param[0] = opt_buffer_fragments;
	else
	    ctl->cmsg(CMSG_WARNING, VERB_NORMAL,
		      "%s: -B option is ignored", play_mode->id_name);
    }
}

void timidity_init_aq_buff(void)
{
    double time1, /* max buffer */
	   time2, /* init filled */
	   base;  /* buffer of device driver */

    if(!IS_STREAM_TRACE)
	return; /* Ignore */

    time1 = atof(opt_aq_max_buff);
    time2 = atof(opt_aq_fill_buff);
    base  = (double)aq_get_dev_queuesize() / play_mode->rate;
    if(strchr(opt_aq_max_buff, '%'))
    {
	time1 = base * (time1 - 100) / 100.0;
	if(time1 < 0)
	    time1 = 0;
    }
    if(strchr(opt_aq_fill_buff, '%'))
	time2 = base * time2 / 100.0;
    aq_set_soft_queue(time1, time2);
}

MAIN_INTERFACE int timidity_play_main(int nfiles, char **files)
{
    int need_stdin = 0, need_stdout = 0;
    int i;
    int output_fail = 0;

    if(nfiles == 0 && !strchr(INTERACTIVE_INTERFACE_IDS, ctl->id_character))
	return 0;

    if(opt_output_name)
    {
	play_mode->name = opt_output_name;
	if(!strcmp(opt_output_name, "-"))
	    need_stdout = 1;
    }

    for(i = 0; i < nfiles; i++)
	if (!strcmp(files[i], "-"))
	    need_stdin = 1;

    if(ctl->open(need_stdin, need_stdout))
    {
	fprintf(stderr, "Couldn't open %s (`%c')" NLS,
		ctl->id_name, ctl->id_character);
	play_mode->close_output();
	return 3;
    }

#ifdef HAVE_SIGNAL
	signal(SIGINT, sigterm_exit);
	signal(SIGTERM, sigterm_exit);
#ifdef SIGPIPE
	signal(SIGPIPE, sigterm_exit);    /* Handle broken pipe */
#endif /* SIGPIPE */
#endif /* HAVE_SIGNAL */

	/* Open output device */
	ctl->cmsg(CMSG_INFO, VERB_DEBUG_SILLY,
		  "Open output: %c, %s",
		  play_mode->id_character,
		  play_mode->id_name);

	if (play_mode->flag & PF_PCM_STREAM) {
	    play_mode->extra_param[1] = aq_calc_fragsize();
	    ctl->cmsg(CMSG_INFO, VERB_DEBUG_SILLY,
		      "requesting fragment size: %d",
		      play_mode->extra_param[1]);
	}
	if(play_mode->open_output() < 0)
	{
	    ctl->cmsg(CMSG_FATAL, VERB_NORMAL,
		      "Couldn't open %s (`%c')",
		      play_mode->id_name, play_mode->id_character);
	    output_fail = 1;
	    ctl->close();
	    return 2;
	}

	if(!control_ratio)
	{
	    control_ratio = play_mode->rate / CONTROLS_PER_SECOND;
	    if(control_ratio < 1)
		control_ratio = 1;
	    else if (control_ratio > MAX_CONTROL_RATIO)
		control_ratio = MAX_CONTROL_RATIO;
	}

	init_load_soundfont();
	if(!output_fail)
	{
	    aq_setup();
	    timidity_init_aq_buff();
	}
	if(allocate_cache_size > 0)
	    resamp_cache_reset();

	if(*def_instr_name)
	    set_default_instrument(def_instr_name);

	if(ctl->flags & CTLF_LIST_RANDOM)
	    randomize_string_list(files, nfiles);
	else if(ctl->flags & CTLF_LIST_SORT)
	    sort_pathname(files, nfiles);

	/* Return only when quitting */
	ctl->cmsg(CMSG_INFO, VERB_DEBUG_SILLY,
		  "pass_playing_list() nfiles=%d", nfiles);

	ctl->pass_playing_list(nfiles, files);

	if(intr)
	    aq_flush(1);

#ifdef XP_UNIX
	return 0;
#endif /* XP_UNIX */

	play_mode->close_output();
	ctl->close();

    return 0;
}


int main(int argc, char **argv)
{
    int c, err;
    int nfiles;
    char **files;

    if((program_name=strrchr(argv[0],PATH_SEP))) program_name++;
    else program_name=argv[0];

    if(argc == 1 && !strchr(INTERACTIVE_INTERFACE_IDS, ctl->id_character))
    {
	interesting_message();
	return 0;
    }

    timidity_start_initialize();

    if((err = timidity_pre_load_configuration()) != 0)
	return err;

    while((c = getopt(argc, argv, OPTCOMMANDS)) > 0)
	if((err = set_tim_opt(c, optarg)) != 0)
	    break;

    err += timidity_post_load_configuration();

    /* If there were problems, give up now */
    if(err || (optind >= argc &&
	       !strchr(INTERACTIVE_INTERFACE_IDS, ctl->id_character)))
    {
	if(!got_a_configuration)
	{
	    ctl->cmsg(CMSG_FATAL, VERB_NORMAL,
		      "%s: Can't read any configuration file.\nPlease check "
		      CONFIG_FILE, program_name);
	}
	else
	{
	    ctl->cmsg(CMSG_ERROR, VERB_NORMAL,
		      "Try %s -h for help", program_name);
	}
	return 1; /* problems with command line */
    }

    timidity_init_player();

    nfiles = argc - optind;
    files  = argv + optind;
    if(nfiles > 0 && ctl->id_character != 'r' && ctl->id_character != 'A') {
	files = safe_malloc(sizeof(char *)*2);
        files[0]=safe_strdup(argv[optind]);
    }
    if(dumb_error_count)
	sleep(1);

    return timidity_play_main(nfiles, files);
}
