/*
  For vgagames modified from:
    TiMidity++ -- MIDI to WAVE converter and player
    Copyright (C) 1999-2001 Masanao Izumo <mo@goice.co.jp>
    Copyright (C) 1995 Tuukka Toivonen <tt@cgs.fi>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#define HAVE_MMAP

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>

#include "tmidi.h"

#ifdef HAVE_MMAP
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/mman.h>
#ifndef MAP_FAILED
#define MAP_FAILED ((caddr_t)-1)
#endif /* MAP_FAILED */

#else
/* mmap is not supported */
#define try_mmap(dmy1, dmy2) NULL
#define munmap(addr, size) /* Do nothing */
#endif /* HAVE_MMAP */

#include "url.h"

#if !defined(O_BINARY)
#define O_BINARY 0
#endif

typedef struct _URL_file
{
    char common[sizeof(struct _URL)];

    char *mapptr;		/* Non NULL if mmap is success */
    long mapsize;
    long pos;

    FILE *fp;			/* Non NULL if mmap is failure */
} URL_file;

static int name_file_check(char *url_string);
static long url_file_read(URL url, void *buff, long n);
static char *url_file_gets(URL url, char *buff, int n);
static int url_file_fgetc(URL url);
static long url_file_seek(URL url, long offset, int whence);
static long url_file_tell(URL url);
static void url_file_close(URL url);

struct URL_module URL_module_file =
{
    URL_file_t,			/* type */
    name_file_check,		/* URL checker */
    NULL,			/* initializer */
    url_file_open,		/* open */
    NULL			/* must be NULL */
};

static int name_file_check(char *s)
{
    int i;

    if(IS_PATH_SEP(s[0]))
	return 1;

    if(strncasecmp(s, "file:", 5) == 0)
	return 1;

    for(i = 0; s[i] && s[i] != ':' && s[i] != '/'; i++)
	;
    if(s[i] == ':' && s[i + 1] == '/')
	return 0;

    return 1;
}

#ifdef HAVE_MMAP
static char *try_mmap(char *path, long *size)
{
    int fd;
    char *p;
    struct stat st;

    errno = 0;
    fd = open(path, O_RDONLY | O_BINARY);
    if(fd < 0)
	return NULL;

    if(fstat(fd, &st) < 0)
    {
	int save_errno = errno;
	close(fd);
	errno = save_errno;
	return NULL;
    }

    p = (char *)mmap(0, st.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
    if(p == (char *)MAP_FAILED)
    {
	int save_errno = errno;
	close(fd);
	errno = save_errno;
	return NULL;
    }
    close(fd);
    *size = (long)st.st_size;
    return p;
}
#endif /* HAVE_MMAP */

URL url_file_open(char *fname)
{
    URL_file *url;
    char *mapptr;		/* Non NULL if mmap is success */
    long mapsize;
    FILE *fp;			/* Non NULL if mmap is failure */

#ifdef DEBUG
    printf("url_file_open(%s)\n", fname);
#endif /* DEBUG */

    if(!strcmp(fname, "-"))
    {
	mapptr = NULL;
	mapsize = 0;
	fp = stdin;
	goto done;
    }

    if(strncasecmp(fname, "file:", 5) == 0)
	fname += 5;
    if(*fname == '\0')
    {
	url_errno = errno = ENOENT;
	return NULL;
    }
    fname = url_expand_home_dir(fname);

    fp = NULL;
    mapsize = 0;
    errno = 0;
    mapptr = try_mmap(fname, &mapsize);
    if(errno == ENOENT || errno == EACCES)
    {
	url_errno = errno;
	return NULL;
    }

#ifdef DEBUG
    if(mapptr != NULL)
	printf("mmap - success. size=%d\n", mapsize);
#ifdef HAVE_MMAP
    else
	printf("mmap - failure.\n");
#endif
#endif /* DEBUG */


    if(mapptr == NULL)
    {
	fp = fopen(fname, "rb");
	if(fp == NULL)
	{
	    url_errno = errno;
	    return NULL;
	}
    }

  done:
    url = (URL_file *)alloc_url(sizeof(URL_file));
    if(url == NULL)
    {
	url_errno = errno;
	if(mapptr)
	    munmap(mapptr, mapsize);
	if(fp && fp != stdin)
	    fclose(fp);
	errno = url_errno;
	return NULL;
    }

    /* common members */
    URLm(url, type)      = URL_file_t;
    URLm(url, url_read)  = url_file_read;
    URLm(url, url_gets)  = url_file_gets;
    URLm(url, url_fgetc) = url_file_fgetc;
    URLm(url, url_close) = url_file_close;
    if(fp == stdin)
    {
	URLm(url, url_seek) = NULL;
	URLm(url, url_tell) = NULL;
    }
    else
    {
	URLm(url, url_seek) = url_file_seek;
	URLm(url, url_tell) = url_file_tell;
    }

    /* private members */
    url->mapptr = mapptr;
    url->mapsize = mapsize;
    url->pos = 0;
    url->fp = fp;

    return (URL)url;
}

static long url_file_read(URL url, void *buff, long n)
{
    URL_file *urlp = (URL_file *)url;

    if(urlp->mapptr != NULL)
    {
	if(urlp->pos + n > urlp->mapsize)
	    n = urlp->mapsize - urlp->pos;
	memcpy(buff, urlp->mapptr + urlp->pos, n);
	urlp->pos += n;
    }
    else
    {
	if((n = (long)fread(buff, 1, n, urlp->fp)) == 0)
	{
	    if(ferror(urlp->fp))
	    {
		url_errno = errno;
		return -1;
	    }
	    return 0;
	}
    }
    return n;
}

char *url_file_gets(URL url, char *buff, int n)
{
    URL_file *urlp = (URL_file *)url;

    if(urlp->mapptr != NULL)
    {
	long s;
	char *nlp, *p;

	if(urlp->mapsize == urlp->pos)
	    return NULL;
	if(n <= 0)
	    return buff;
	if(n == 1)
	{
	    *buff = '\0';
	    return buff;
	}
	n--; /* for '\0' */
	s = urlp->mapsize - urlp->pos;
	if(s > n)
	    s = n;
	p = urlp->mapptr + urlp->pos;
	nlp = (char *)memchr(p, url_newline_code, s);
	if(nlp != NULL)
	    s = nlp - p + 1;
	memcpy(buff, p, s);
	buff[s] = '\0';
	urlp->pos += s;
	return buff;
    }

    return fgets(buff, n, urlp->fp);
}

int url_file_fgetc(URL url)
{
    URL_file *urlp = (URL_file *)url;

    if(urlp->mapptr != NULL)
    {
	if(urlp->mapsize == urlp->pos)
	    return EOF;
	return urlp->mapptr[urlp->pos++] & 0xff;
    }

#ifdef getc
    return getc(urlp->fp);
#else
    return fgetc(urlp->fp);
#endif /* getc */
}

static void url_file_close(URL url)
{
    URL_file *urlp = (URL_file *)url;

    if(urlp->mapptr != NULL)
    {
	munmap(urlp->mapptr, urlp->mapsize);
    }
    if(urlp->fp != NULL)
    {
	if(urlp->fp == stdin)
	    rewind(stdin);
	else
	    fclose(urlp->fp);
    }
    free(url);
}

static long url_file_seek(URL url, long offset, int whence)
{
    URL_file *urlp = (URL_file *)url;
    long ret;

    if(urlp->mapptr == NULL)
	return fseek(urlp->fp, offset, whence);
    ret = urlp->pos;
    switch(whence)
    {
      case SEEK_SET:
	urlp->pos = offset;
	break;
      case SEEK_CUR:
	urlp->pos += offset;
	break;
      case SEEK_END:
	  urlp->pos = urlp->mapsize + offset;
	break;
    }
    if(urlp->pos > urlp->mapsize)
	urlp->pos = urlp->mapsize;
    else if(urlp->pos < 0)
	urlp->pos = 0;

    return ret;
}

static long url_file_tell(URL url)
{
    URL_file *urlp = (URL_file *)url;

    return urlp->mapptr ? urlp->pos : ftell(urlp->fp);
}
