/* vgag-bitmap.h: convert graphic file to vgagames format */

#include "_endian.h"


static FILE * _bmp2vga(FILE * HW,const char * filename) {
/* convert windows bitmap to vgagames format
** 1.arg: filepointer of windows bitmap file located at position 2 (from 0),
**        is closed before returning
** 2.arg: filename
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  union {
    short sw;
    char cw[2];
#if BYTE_ORDER == BIG_ENDIAN
  } _c2s,_c2st;
#else
  } _c2s;
#endif
  union {
    int iw;
    char cw[4];
#if BYTE_ORDER == BIG_ENDIAN
  } _c2i,_c2it;
#else
  } _c2i;
#endif
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  FILE * HX;
  char buf[8];
  int i1,i2,i3,i4,i5,ifl,bwid,bhei,anzfarb,bytedata;
  long offdat,hlen,lw;
  short bpp;
  unsigned char c[4];

  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  /* reading windows bitmap */
  fseek(HW,0x0a,SEEK_SET);  /* 0x0a: offset data */
  fread(&_c2i.iw,4,1,HW);
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  offdat=(long)_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x0e: header size */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  hlen=(long)_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x12: width of bitmap */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bwid=_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x16: height of bitmap */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bhei=_c2i.iw;
  fseek(HW,0x1C,SEEK_SET);  /* 0x1c: bits per pixel at data */
  fread(&_c2s.sw,2,1,HW);
#if BYTE_ORDER == BIG_ENDIAN
  _c2st.cw[0]=_c2s.cw[1]; _c2st.cw[1]=_c2s.cw[0];
  _c2s.sw=_c2st.sw;
#endif
  bpp=_c2s.sw;
  if (bpp>8) {
    snprintf(errmsg,sizeof(errmsg),"Too deep color depth (>256) in file \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  fread(&_c2i.iw,4,1,HW);  /* 0x1e: compression */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  lw=(long)_c2i.iw;
  if (lw!=0L) {
    snprintf(errmsg,sizeof(errmsg),"File \"%s\" uses compression: not supported",filename);
    fclose(HW);
    return(NULL);
  }

  anzfarb=(1<<(int)bpp);
  lw=14L+hlen+(long)(anzfarb*4);
  if (offdat!=lw) {
    snprintf(errmsg,sizeof(errmsg),"Corrupt colormap in file \"%s\": %ld!=%ld",filename,offdat,lw);
    fclose(HW);
    return(NULL);
  }

  fseek(HW,0x36,SEEK_SET);  /* 0x36: colormap */
  for (i1=0;i1<anzfarb;i1++) {
    fread(c,1,4,HW);
    fw[i1].r=(c[2]>>2);
    fw[i1].g=(c[1]>>2);
    fw[i1].b=(c[0]>>2);
  }
  lw=(long)(bwid*(int)bpp/32*32);
  if (bwid*(int)bpp%32>0) {lw+=32;}
  lw*=(long)bhei;
  bytedata=(int)(lw/8L);

  /* write vgagames graphic file */
  if ((HX=tmpfile())==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot write vgagames graphic file from \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  strcpy(buf,"VG");
  fwrite(buf,1,2,HX);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  fwrite(&buf,1,5,HX);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  fwrite(&buf,1,5,HX);  /* height */
  for (i1=0;i1<256;i1++) {  /* colormap */
    fwrite(&fw[i1].r,1,1,HX);  /* red byte */
    fwrite(&fw[i1].g,1,1,HX);  /* green byte */
    fwrite(&fw[i1].b,1,1,HX);  /* blue byte */
  }
  c[0]=0;
  for (i1=0;i1<bwid;i1++) {
    for (i2=0;i2<bhei;i2++) {fwrite(c,1,1,HX);}
  }

  fseek(HX,-bwid,SEEK_CUR);
  for (ifl=0,i4=bpp-1,i5=0,i1=0;i1<bytedata;i1+=4) {
    if (fread(c,1,4,HW)<4) {
      snprintf(errmsg,sizeof(errmsg),"Bitmap %s too short (at byte %d)",filename,i1+1);
      fclose(HW);
      fclose(HX);
      return(NULL);
    }
    for (i3=0;i3<4;i3++) {
      if (bpp==8) {
        if (i5++<bwid) {fwrite(&kv[c[i3]],1,1,HX);}
      } else {
        for (i2=7;i2>=0;i2--) {
          ifl|=(!!((unsigned int)c[i3]&(1<<i2))<<i4);
          if (--i4<0) {
            if (i5++<bwid) {fwrite(&kv[ifl],1,1,HX);}
            i4=bpp-1;
            ifl=0;
          }
        }
      }
    }
    if (i5>=bwid) {i5=0; i4=bpp-1; ifl=0; fseek(HX,-bwid*2,SEEK_CUR);}
  }
  fclose(HW);
  fflush(HX);
  fseek(HX,2L,SEEK_SET);
  return(HX);
} /* Ende _bmp2vga */


static FILE * _ppm2vga(FILE * HW,const char * filename,int raw) {
/* convert ppm to vgagames format
** 1.arg: filepointer of ppm file located at position 2 (from 0),
**        is closed before returning
** 2.arg: filename
** 3.arg: 0=true ppm or 1=raw ppm
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  FILE * HX;
  char buf[64];
  int ival,i1,i2,i3,i4,bwid=0,bhei=0,maxv=0;
  unsigned char c[3];

  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  for (i3=0;i3<3;i3++) {
    /* whitespace(s) */
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival>' ') {
        if ((unsigned char)ival=='#') {
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival=='\n') {break;}
          }
          if (ival==EOF) {break;}
        } else {break;}
      }
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }

    /* value */
    i2=0;
    buf[i2++]=(char)ival;
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival<=' ') {break;}
      if ((unsigned char)ival=='#') {break;}
      if (i2==(int)sizeof(buf)) {ival=EOF; break;}
      buf[i2++]=(char)ival;
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }
    buf[i2]='\0';
    if (i3==0) {bwid=atoi(buf);}
    else if (i3==1) {bhei=atoi(buf);}
    else if (i3==2) {maxv=atoi(buf);}
    if ((unsigned char)ival=='#') {
      while ((ival=fgetc(HW))!=EOF) {
        if ((unsigned char)ival=='\n') {break;}
      }
      if (ival==EOF) {
        snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
        fclose(HW);
        return(NULL);
      }
    }
  }
  /* single whitespace just read */

  if (maxv>255) {
    snprintf(errmsg,sizeof(errmsg),"file \"%s\" has too deep color depth (>255)",filename);
    fclose(HW);
    return(NULL);
  }

  if (raw==1) {
    /* whitespace(s) */
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival>' ') {break;}
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }
  }

  /* write vgagames graphic file */
  if ((HX=tmpfile())==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot write vgagames graphic file from \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  strcpy(buf,"VG");
  fwrite(buf,1,2,HX);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  fwrite(&buf,1,5,HX);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  fwrite(&buf,1,5,HX);  /* height */
  buf[0]=buf[1]=buf[2]=0;
  for (i1=0;i1<256;i1++) {
    fwrite(buf,1,3,HX);  /* room for colormap */
  }

  i4=0;
  for (i1=0;i1<bhei;i1++) {
    for (i2=0;i2<bwid;i2++) {
      if (raw==1) {
        int ii;
        for (ii=0;ii<3;ii++) {
          if (ival==EOF) {
            snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
            fclose(HW);
            return(NULL);
          }
          i3=0;
          buf[i3++]=(char)ival;
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival<=' ') {break;}
            if (i3==(int)sizeof(buf)) {ival=EOF; break;}
            buf[i3++]=(char)ival;
          }
          buf[i3]='\0';
          c[ii]=(unsigned char)atoi(buf);
          /* whitespace(s) */
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival>' ') {break;}
          }
        }
      } else {
        if (fread(c,1,3,HW)<3) {
          snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
          fclose(HW);
          return(NULL);
        }
      }
      for (i3=0;i3<i4;i3++) {
        if (((c[0]>>2)==fw[i3].r) \
        && ((c[1]>>2)==fw[i3].g) \
        && ((c[2]>>2)==fw[i3].b)) {
          fwrite(&kv[i3],1,1,HX);
          break;
        }
      }
      if (i3==i4) {
        if (++i4==256) {
          snprintf(errmsg,sizeof(errmsg),"Too many colors (>256) in file \"%s\"",filename);
          fclose(HW);
          fclose(HX);
          return(NULL);
        }
        fw[i3].r=(c[0]>>2);
        fw[i3].g=(c[1]>>2);
        fw[i3].b=(c[2]>>2);
        fwrite(&kv[i3],1,1,HX);
      }
    }
  }
  fseek(HX,12L,SEEK_SET);
  for (i1=0;i1<256;i1++) {  /* colormap */
    fwrite(&fw[i1].r,1,1,HX);  /* red byte */
    fwrite(&fw[i1].g,1,1,HX);  /* green byte */
    fwrite(&fw[i1].b,1,1,HX);  /* blue byte */
  }
  fclose(HW);
  fflush(HX);
  fseek(HX,2L,SEEK_SET);
  return(HX);
} /* Ende _ppm2vga */


static FILE * _vga_grafik(const char * filename) {
/* convert graphic file to vgagames format
** 1.arg: filename of graphic file
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  FILE * HW;
  char buf[8];

  /* read type of graphic file */
  if (filename==NULL) {strcpy(errmsg,"No graphic filename to open"); return(NULL);}
  if ((HW=fopen(filename,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot open graphic file \"%s\"",filename);
    return(NULL);
  }
  *buf='\0';
  fread(buf,1,2,HW);
  /* vgagames format */
  if (strncmp(buf,"VG",2)==0) {return(HW);}
  /* windows bitmap format */
  if (strncmp(buf,"BM",2)==0) {return(_bmp2vga(HW,filename));}
  /* ppm format */
  if (strncmp(buf,"P6",2)==0) {return(_ppm2vga(HW,filename,0));}
  if (strncmp(buf,"P3",2)==0) {return(_ppm2vga(HW,filename,1));}
  /* not supported */
  snprintf(errmsg,sizeof(errmsg),"File \"%s\" has no supported graphic format",filename);
  fclose(HW);
  return(NULL);
} /* Ende _vga_grafik */
