/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* vgag-nettcp: network server for nettcp
   define HAVE_ADDRINFO_STRUCT, if you have getaddrinfo()
     or undef HAVE_ADDRINFO_STRUCT, if you have not getaddrinfo().
   define HAVE_SOCKADDR_SA_LEN, if you have sin_len
     or undef HAVE_SOCKADDR_SA_LEN, if you have not sin_len.
   compile if needed with -lsocket -lnsl [-lresolv]
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stddef.h>
#include <limits.h>
#include <errno.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>
#include "netw.h"


#ifndef HAVE_SOCKLEN_T
  typedef int socklen_t;
#endif
#ifndef HAVE_ADDRINFO_STRUCT
  #include "nettcp_gai.h"
#else
  #include <netinet/in.h>
  #include <arpa/nameser.h>
  #include <resolv.h>
#endif

#ifdef AF_INET6
  #ifndef IPV6_ADD_MEMBERSHIP
    #define IPV6_ADD_MEMBERSHIP IPV6_JOIN_GROUP
  #endif
#else
  #ifdef USE_AF_VER6
    #undef USE_AF_VER6
  #endif
#endif

static volatile sig_atomic_t sig_alrm=0;
static int inetv=0;
static char * argv0=NULL;

void conn_serv(int [8],size_t,int,int,char *,size_t);

static ssize_t writen(int,const void *,size_t);
static ssize_t readn(int,void *,size_t);
#ifndef USE_AF_VER6
  static void serv_bcast(const char *,const char *,const int);
#endif
#ifdef AF_INET6
  static void serv_mcast(const char *,const char *,const int);
#endif
static void do_chld(int);
static void do_alrm(int);


int main(int argc,char ** argv) {
/* Wait for connections
** parameters: 
**  - number of players to wait for connection (from "1" to "8")
**  - number of virtual players included in number of players
**  - portnumber listening to
*/
  const int altm=90;  /* alarm time for accept */
  const int bcklg=SOMAXCONN;
  const int on=1;
  socklen_t addr_len,stlen;
  int anz,vanz,sfd,cfd[8],n1,n2,s1,master;
  char serv[8],buf[256],* shmem;
  size_t stsz,shsz;
  struct addrinfo adi0,* adi1,* adi2;
  struct sockaddr * ska;
  pid_t chpr;

  argv0=argv[0];
  if (argc!=4) {
    fprintf(stderr,"Server for multiplayer (1-8) games\n");
    fprintf(stderr,"Usage: %s <total number of players> <number of virtual players> <port>\n",argv[0]);
    exit(1);
  }
  anz=atoi(argv[1]);
  vanz=atoi(argv[2]);
  if ((anz<1) || (anz>8) || (vanz<0) || (vanz>=anz)) {fprintf(stderr,"%s: Wrong number of players\n",argv0); exit(1);}
  strncpy(serv,argv[3],sizeof(serv)-1);
  serv[sizeof(serv)-1]='\0';
  if ((atoi(serv)<1) || (atoi(serv)>65536)) {
    fprintf(stderr,"%s: Wrong port\n",argv0);
    exit(1);
  }

  /* open listening socket */
  memset(&adi0,0,sizeof(struct addrinfo));
  adi0.ai_flags=AI_PASSIVE;
  adi0.ai_socktype=SOCK_STREAM;
#ifndef USE_AF_VER6
  adi0.ai_family=AF_INET;
  strcpy(buf,"0.0.0.0");
  if ((n1=getaddrinfo(buf,serv,&adi0,&adi1))!=0) {
#ifdef AF_INET6
    adi0.ai_family=AF_INET6;
    strcpy(buf,"0::0");
    if ((n1=getaddrinfo(buf,serv,&adi0,&adi1))!=0) {
#endif
      fprintf(stderr,"%s: Socket error: %s\n",argv0,gai_strerror(n1));
      exit(1);
#ifdef AF_INET6
    }
#endif
#else  /* USE_AF_VER6 */
  adi0.ai_family=AF_INET6;
  strcpy(buf,"0::0");
  if ((n1=getaddrinfo(buf,serv,&adi0,&adi1))!=0) {
    fprintf(stderr,"%s: Socket error: %s\n",argv0,gai_strerror(n1));
    exit(1);
#endif  /* USE_AF_VER6 */
  }
  adi2=adi1;
  do {
    sfd=socket(adi1->ai_family,adi1->ai_socktype,adi1->ai_protocol);
    if (sfd<0) {continue;}
    if (setsockopt(sfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on))<0) {
      fprintf(stderr,"%s: Setsockopt error\n",argv0);
      freeaddrinfo(adi2);
      exit(1);
    }
    if (bind(sfd,adi1->ai_addr,adi1->ai_addrlen)==0) {break;}
    close(sfd);
  } while ((adi1=adi1->ai_next)!=NULL);
  if (adi1==NULL) {
    fprintf(stderr,"%s: Cannot get socket\n",argv0);
    freeaddrinfo(adi2);
    exit(1);
  }
  if (listen(sfd,bcklg)<0) {
    fprintf(stderr,"%s: Cannot listen at (%s)\n",argv0,serv);
    freeaddrinfo(adi2);
    exit(1);
  }
  addr_len=adi1->ai_addrlen;
  inetv=adi1->ai_addr->sa_family;
  freeaddrinfo(adi2);
  if ((ska=malloc(addr_len))==NULL) {
    fprintf(stderr,"%s: Cannot allocate memory\n",argv0);
    close(sfd);
    exit(1);
  }
  /* start broadcast/multicast server */
  if ((chpr=fork())==-1) {
    fprintf(stderr,"%s: Cannot fork\n",argv0);
    close(sfd);
    free(ska);
    exit(1);
  }
  if (chpr>0) {  /* parent */
    struct sigaction sa;
    memset(&sa,0,sizeof(sa));
    sa.sa_handler=do_chld;
    sigaction(SIGCHLD,&sa,NULL);
    close(sfd);
    free(ska);
#ifndef USE_AF_VER6
#ifdef AF_INET6
    if (inetv==AF_INET6) {
      serv_mcast("0::0",serv,altm);
    } else {
#endif
      serv_bcast("0.0.0.0",serv,altm);
#ifdef AF_INET6
    }
#endif
#else  /* USE_AF_VER6 */
    if (inetv==AF_INET6) {
      serv_mcast("0::0",serv,altm);
    }
#endif  /* USE_AF_VER6 */
    exit(1);
  }
  /* wait for connections */
  for (n1=0;n1<8;n1++) {cfd[n1]=-1;}
  stsz=0;
  n1=vanz+1;
  while (anz-vanz>0) {
    stlen=addr_len;
    alarm(altm);
    s1=accept(sfd,ska,&stlen);
    alarm(0);
    if (s1<0) {
      fprintf(stderr,"%s: Accept error\n",argv0);
      free(ska);
      for (n2=0;n2<8;n2++) {if (cfd[n2]>-1) {close(cfd[n2]);}}
      close(sfd);
      exit(1);
    }
    if (readn(s1,buf,5)!=5) {  /* read data size + master identification */
      free(ska);
      for (n2=0;n2<8;n2++) {if (cfd[n2]>-1) {close(cfd[n2]);}}
      close(sfd);
      exit(1);
    }
    if (buf[4]=='1') {master=1;} else {master=0;}
    buf[4]='\0';
    if (stsz==0) {stsz=atoi(buf);}
    if (atoi(buf)!=(int)stsz) {
      fprintf(stderr,"%s: Data size error\n",argv0);
      free(ska);
      for (n2=0;n2<8;n2++) {if (cfd[n2]>-1) {close(cfd[n2]);}}
      close(sfd);
      exit(1);
    }
    if (master) {cfd[0]=s1;} else {cfd[n1++]=s1;}
    anz--;
  }
  free(ska);
  close(sfd);
  /* allocate memory */
  shsz=(1+8)*stsz;
  if ((shmem=calloc(sizeof(char),shsz))==NULL) {
    fprintf(stderr,"%s: Cannot allocate memory\n",argv0);
    for (n2=0;n2<8;n2++) {if (cfd[n2]>-1) {close(cfd[n2]);}}
    exit(1);
  }
  /* reply to clients */
  anz=atoi(argv[1]);
  for (n1=1;n1<=anz;n1++) {
    if (cfd[n1-1]>-1) {
      snprintf(buf,sizeof(buf),"%d%d",anz,n1);
      writen(cfd[n1-1],buf,2);
    }
  }
  /* act as connected server */
  conn_serv(cfd,stsz,anz,vanz,shmem,shsz);
  free(shmem);
  exit(0);
} /* Ende main */


void conn_serv(int cfd[8],size_t stsz,int anz,int vanz,char * shmem,size_t shsz) {
/* acting as synchronisized connected server:
** args: socket descriptor field, data size, number of players (1-8),
**       virtual number of players, allocated memory, allocated memory size
** allocated memory:
**                   stsz * databytes for computerplayer
**                   stsz * databytes for player 1
**                   stsz * databytes for player 2
**                   ...
**                   stsz * databytes for player 8
*/
#undef __VGN_LAZY
  const int altm=5;  /* alarm time */
  const int rnr=(1+anz)*stsz;
  int i1,erg,manz;
  char * ptr;
  struct sigaction sa;
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=do_alrm;
  sigaction(SIGALRM,&sa,NULL);
  manz=anz-vanz;
  while (manz>0) {
    ptr=shmem;
    if (cfd[0]>-1) {  /* master */
      alarm(altm);
      erg=readn(cfd[0],ptr,stsz*(1+1+vanz));
      alarm(0);
      if (erg!=(int)(stsz*(1+1+vanz))) {
        close(cfd[0]);
        cfd[0]=-1;
        memset(ptr,0,stsz*(1+1+vanz));
        manz--;
      }
#ifdef __VGN_LAZY
      else {writen(cfd[0],shmem,rnr);}
#endif
    }
    ptr+=stsz*(1+1+vanz);
    for (i1=vanz+2;i1<=anz;i1++) {
      if (cfd[i1-1]>-1) {
        alarm(altm);
        erg=readn(cfd[i1-1],ptr,stsz);
        alarm(0);
        if (erg!=(int)stsz) {
          close(cfd[i1-1]);
          cfd[i1-1]=-1;
          memset(ptr,0,stsz);
          manz--;
        }
#ifdef __VGN_LAZY
        else {writen(cfd[i1-1],shmem,rnr);}
#endif
      }
      ptr+=stsz;
    }
#ifndef __VGN_LAZY
    if (cfd[0]>-1) {writen(cfd[0],shmem,rnr);}
    for (i1=vanz+2;i1<=anz;i1++) {
      if (cfd[i1-1]>-1) {writen(cfd[i1-1],shmem,rnr);}
    }
#endif
  }
} /* Ende conn_serv */


static ssize_t writen(int fd,const void * vptr,size_t n) {
/* force number of bytes to write
** 1.arg: socket descriptor
** 2.arg: buffer to write
** 3.arg: number of characters to write
** return: 3.arg or -1=error
*/
  static int spip=0;
  size_t nleft;
  ssize_t nwritten;
  const char * ptr;
  if (spip==0) {
    struct sigaction sa1;
    spip=1;
    memset(&sa1,0,sizeof(sa1));
    sa1.sa_handler=SIG_IGN;
    sigaction(SIGPIPE,&sa1,NULL);
  }
  ptr=vptr;
  nleft=n;
  while (nleft>0) {
    if ((nwritten=write(fd,ptr,nleft))<0) {
      if (errno==EAGAIN) {continue;}
      fprintf(stderr,"%s: Error writing to socket\n",argv0);
      return(-1);
    }
    nleft-=nwritten;
    ptr+=nwritten;
  }
  return(n);
} /* Ende writen */


static ssize_t readn(int fd,void * vptr,size_t n) {
/* force number of bytes to read
** 1.arg: socket descriptor
** 2.arg: address of buffer
** 3.arg: number of characters to read
** return: 0 to 3.arg or -1=error or -2=no data
*/
  size_t nleft;
  ssize_t nread;
  char * ptr;
  ptr=(char *)vptr;
  nleft=n;
  while (nleft>0) {
    if ((nread=read(fd,ptr,nleft))<0) {
      if (errno==EAGAIN) {
        if (nleft!=n) {continue;}
        return(-2);
      }
      return(-1);
    } else if (nread==0) {break;}
    nleft-=nread;
    ptr+=nread;
  }
  return(n-nleft);
} /* Ende readn */


#ifndef USE_AF_VER6
static void serv_bcast(const char * host,const char * serv,const int altm) {
  int sockfd,anz;
  char buf[2];
  socklen_t addrlen,alen;
  struct sockaddr * cliaddr;
  struct addrinfo hints,* res,* ressave;
  /* open broadcast server socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_flags=AI_PASSIVE;
  hints.ai_family=AF_INET;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(host,serv,&hints,&res))!=0) {
    fprintf(stderr,"%s: Broadcast server: %s\n",argv0,gai_strerror(anz));
    exit(1);
  }
  ressave=res;
  do {
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd<0) {continue;}
    if (bind(sockfd,res->ai_addr,res->ai_addrlen)==0) {break;}
    close(sockfd);
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    fprintf(stderr,"%s: Cannot get broadcast server socket\n",argv0);
    freeaddrinfo(ressave);
    exit(1);
  }
  addrlen=res->ai_addrlen;
  freeaddrinfo(ressave);
  /* prepare to receive data */
  if ((cliaddr=malloc(addrlen))==NULL) {
    fprintf(stderr,"%s: Cannot allocate memory\n",argv0);
    close(sockfd);
    exit(1);
  }
  /* wait for datagrams */
  while (1) {
    alen=addrlen;
    alarm(altm);
    anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,cliaddr,&alen);
    alarm(0);
    if (anz<0) {
      fprintf(stderr,"%s: Error receiving data: %s!\n",argv0,strerror(errno));
      close(sockfd);
      free(cliaddr);
      exit(1);
    }
    strcpy(buf,"\n");
    if (sendto(sockfd,buf,strlen(buf),0,cliaddr,alen)<0) {
      fprintf(stderr,"%s: Error sending to client: %s!\n",argv0,strerror(errno));
      close(sockfd);
      free(cliaddr);
      exit(1);
    }
  }
  close(sockfd);
  free(cliaddr);
  exit(0);
} /* Ende serv_bcast */
#endif


#ifdef AF_INET6
static void serv_mcast(const char * host,const char * serv,const int altm) {
  int sockfd=-1,anz;
  char buf[2];
  socklen_t addrlen,alen;
  struct sockaddr * cliaddr;
  struct addrinfo hints,* res,* ressave;
  struct ipv6_mreq mreq6;
  struct if_nameindex * ifa,* ifi;
  const char mcadd[]="ff02::1";
  inet_pton(AF_INET6,mcadd,&mreq6.ipv6mr_multiaddr);
  mreq6.ipv6mr_interface=0;
  /* open multicast server socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_flags=AI_PASSIVE;
  hints.ai_family=AF_INET6;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(host,serv,&hints,&res))!=0) {
    fprintf(stderr,"%s: Multicast server: %s\n",argv0,gai_strerror(anz));
    exit(1);
  }
  ressave=res;
  do {
    if (res->ai_family!=AF_INET6) {continue;}
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd<0) {continue;}
    if (bind(sockfd,res->ai_addr,res->ai_addrlen)==0) {break;}
    close(sockfd);
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    fprintf(stderr,"%s: Cannot get multicast server socket\n",argv0);
    freeaddrinfo(ressave);
    exit(1);
  }
  addrlen=res->ai_addrlen;
  freeaddrinfo(ressave);
  /* prepare to receive data */
  if ((cliaddr=malloc(addrlen))==NULL) {
    fprintf(stderr,"%s: Cannot allocate memory\n",argv0);
    close(sockfd);
    exit(1);
  }
  ifa=if_nameindex();
  printf("%s: Multicast server: add membership for \"",argv0);
  if (ifa!=NULL) {
    ifi=ifa;
    while (ifi->if_index>0) {
      mreq6.ipv6mr_interface=ifi->if_index;
      if (setsockopt(sockfd,IPPROTO_IPV6,IPV6_ADD_MEMBERSHIP,&mreq6,sizeof(mreq6))==0) {
        printf("%s ",ifi->if_name);
      }
      ifi++;
    }
    if_freenameindex(ifa);
  }
  printf("\"\n");
  /* wait for datagrams */
  while (1) {
    alen=addrlen;
    alarm(altm);
    anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,cliaddr,&alen);
    alarm(0);
    if (anz<0) {
      fprintf(stderr,"%s: Error receiving data: %s!\n",argv0,strerror(errno));
      close(sockfd);
      free(cliaddr);
      exit(1);
    }
    strcpy(buf,"\n");
    if (sendto(sockfd,buf,strlen(buf),0,cliaddr,alen)<0) {
      fprintf(stderr,"%s: Error sending to client: %s!\n",argv0,strerror(errno));
      close(sockfd);
      free(cliaddr);
      exit(1);
    }
  }
  close(sockfd);
  free(cliaddr);
  exit(0);
} /* Ende serv_mcast */
#endif


static void do_chld(int signum) {
  exit(1);
} /* Ende do_chld */


static void do_alrm(int signum) {
  sig_alrm=1;
} /* Ende do_alrm */
