/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* vgag-wave: multichannel wave server for pcm mode 8 bit
**   Use arguments: -q = quiet: no messages to stdout
**                  -s<samplefrequence><M|S> = set samplefrequence
**                                             and <M>ono or <S>tereo
**                                             (no space at all)
**   Reads config file: SHARE_PATH/wave.conf
*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <time.h>
#include "pfad.h"
#include "_pipbfl.h"
#include "_endian.h"

#include "vgagames_wave.h"
#include "soundserver.h"

volatile int sig_alarm=0;
char errmsg[2048];

#define VGAG_MIDI_TO_WAVE BIN_PATH "/vgag-midi"
#define VGAG_MP3_TO_WAVE BIN_PATH "/vgag-mp3"


/* WAV-File Header:
** "RIFF" (char[4])
**    filelength (ulong)
** "WAVE" (char[4])
** [ "????" (char[4])
**      blocklength (ulong)
**      data of length blocklength ]
** [ ... ]
** "fmt " (char[4])
**    blocklength: "16" (ulong)
**    PCM-code: "1" (ushort)
**    modus: "1"=mono or "2"=stereo (ushort)
**    samplefrequence (ulong)
**    bytes per second (ulong)
**    bytes per sample (ushort)
**    bits per sample (ushort)
** [ "????" (char[4])
**      blocklength (ulong)
**      data of length blocklength ]
** [ ... ]
** "data"
**    blocklength (ulong)
**    wavedata of length blocklength
** [ "????" (char[4])
**      blocklength (ulong)
**      data of length blocklength ]
** [ ... ]
*/

#define WAV_BLOCK 256*1024    /* load max. bytes per channel for wave */
#define MID_BLOCK PIPE_BFLEN  /* load max. bytes per channel for midi */

#define CN_N(A) (A-1)


int quiet;
int frgm,fct,mno;
int wdef_allo,wdef_used;
long wav_block,mid_block;
int no_sound;

typedef struct {
  short typ;                 /* TYP_IS_WAVE or TYP_IS_MIDI or TYP_IS_MP3 or TYP_IS_EXTENSION */
  char * fname;              /* file name */
  unsigned long bitsmp;      /* bits per sample, always 8 */
  unsigned long modus;       /* 0=mono, 1=stereo */
  unsigned long smpfq;       /* sample frequence */
  unsigned long bytsek;      /* bytes per second */
  size_t dbeg;               /* start of wave-data */
  size_t dlen;               /* number of bytes of wave-data */
  int w16to8;                /* whether convert 16LE bit to 8U bit */
} wavedef;
wavedef * wdef,mdef;

struct {
  int wvnr;  /* to which wdef[] it belongs */
  FILE * fp;  /* opened file */
  unsigned char data[WAV_BLOCK],* dptr;  /* WAV_BLOCK is largest */
  size_t dread,dnoch;  /* block read data, characters still to read */
  long loop,nlp;  /* looping: number of fragments to wait, looping counter */
  short volpc;  /* volume percent: -10 to 20
                  (-10=0%, -9=10%, -8=20%, ..., 0=100%, 1=110%, ..., 20=300%)
                */
  int vm,vv;  /* for in-/decreasing volume */
  short status;  /* 0=ended, 1=running, 2=stopped */
  int n_wvnr,n_chnn;  /* for waiting next file */
  short n_loopse,n_volse;  /* for waiting next file */
} chan_opt[CN_MAX];

short volpc[CN_MAX];

struct {  /* wave.conf */
  char lib[32];  /* not empty=only this sound library */
  char debugfile[512];  /* not empty=debug file */
  FILE * debugfd;  /* not NULL=debug descr. */
  unsigned long dfsfq;  /* sample frequence */
  unsigned long dfmod;  /* mono / stereo */
  char dev_pcm[128];  /* pcm device name */
  char dev_mixer[128];  /* mixer device name */
} wave_cf;

union {
  unsigned short sw;
  char cw[2];
} _c2s;

union {
  unsigned int iw;
  char cw[4];
} _c2i;


void do_alrm(int);
void schreibe(const char *);
int cfg_dev(int,int);
int cmd_l(int,int,char *);
int cmd_s(int,int,short,short);
int cmd_e(int,short);
int cmd_p(int);
int cmd_c(int);
int cmd_v(int,short);
int cmd_a(int,int);
int cmd_m(int,int);

#include "vgag-wave.h"
struct wave_device * wdptr;

static FILE * _open_midi(const char *);
static FILE * _open_mp3(const char *);


static FILE * _open_midi(const char * fname) {
  char pipbuf[2048];
  FILE * fp;
  int i1;
  i1=snprintf(pipbuf,sizeof(pipbuf),"%s -Or%c -s%lu -o - %s 2> /dev/null", \
    VGAG_MIDI_TO_WAVE,wdef[0].modus==0UL?'M':(mno==1?'M':'S'), \
    wdef[0].smpfq/fct,fname);
  if (i1==-1) {
    strcpy(errmsg,"Midi path too long");
    return(NULL);
  }
  if ((fp=popen(pipbuf,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot play midi file \"%s\"",fname);
    return(NULL);
  }
  return(fp);
} /* Ende _open_midi */


static FILE * _open_mp3(const char * fname) {
  char pipbuf[2048];
  FILE * fp;
  int i1;
  long smfq;
  char smfqc[64];
  smfq=wdef[0].smpfq/fct;
#if 0
  if (smfq<=11025+(11025/2)) {strcpy(smfqc,"-4");}
  else if (smfq<=22050+(22050/2)) {strcpy(smfqc,"-2");}
  else {*smfqc='\0';}
#endif
  snprintf(smfqc,sizeof(smfqc),"-r %ld",smfq);
  i1=snprintf(pipbuf,sizeof(pipbuf),"%s --8bit %s %s -q -s %s 2> /dev/null", \
    VGAG_MP3_TO_WAVE,wdef[0].modus==0UL?"--mono":(mno==1?"--mono":"--stereo"), \
    smfqc,fname);
  if (i1==-1) {
    strcpy(errmsg,"MP3 path too long");
    return(NULL);
  }
  if ((fp=popen(pipbuf,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot play mp3 file \"%s\"",fname);
    return(NULL);
  }
  return(fp);
} /* Ende _open_mp3 */


void do_alrm(int signum) {
  sig_alarm=1;
} /* Ende do_alrm */


int main(int argc,char ** argv) {
/* Reads input from stdin:
**   "<action> <wave number or 0> <arg2> <arg3>"
**   Actions:
**   Load:  "L <wave number> <flag: TYP_IS_-Definition> <filename>"
**   Play:  "S <wave number> <selection of channels> <0 or looping sleep time>/<0 or volume increasing time>"
**   End:   "E 0 <channels> <0 or volume decreasing time>"
**   Pause: "P 0 <channels> <empty>"
**   Cont.: "C 0 <channels> <empty>"
**   CVol.: "V 0 <channels> <volume percent from -10 to 20>"
**   Accel: "A 0 <0=let it or reducing factor of samplefrequence from 1 to 4> <0=let it, 1=only mono, 2=stereo or mono>"
**   MVol.: "M 0 <volume from 0 to 100, or -1> <device: 0=main, 1=pcm>"
** Terminates when closing stdin
*/
  int i1,i2,wvnr,chnn,ssv;
  char buf[1024],* p1,* p2,aktz;
  unsigned char * frg_uc;
  short * frg_si,s1;
  double volx;
  FILE * ffp;
  struct sigaction sa;
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=SIG_DFL;
  sigaction(SIGPIPE,&sa,NULL);
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=do_alrm;
  sigaction(SIGALRM,&sa,NULL);
  quiet=0;

  /* read config file */
  memset(&wave_cf,0,sizeof(wave_cf));
  wave_cf.dfsfq=8000UL;
  wave_cf.dfmod=0UL;
  snprintf(buf,sizeof(buf),"%s/wave.conf",SHARE_PATH);
  if ((ffp=fopen(buf,"r"))!=NULL) {
    while (fgets(buf,sizeof(buf),ffp)!=NULL) {
      if ((p1=strchr(buf,'#'))!=NULL) {*p1='\0';}
      for (p1=buf+strlen(buf)-1;(unsigned char)*p1<=' ';p1--) {
        if (p1<buf) {break;}
      }
      p1[1]='\0';
      if (strncmp(buf,"LIB=",4)==0) {
        snprintf(wave_cf.lib,sizeof(wave_cf.lib),"%s",buf+4);
      } else if (strncmp(buf,"PCM-DEVICE=",11)==0) {
        strncpy(wave_cf.dev_pcm,buf+11,sizeof(wave_cf.dev_pcm)-1);
        wave_cf.dev_pcm[sizeof(wave_cf.dev_pcm)-1]='\0';
      } else if (strncmp(buf,"MIXER-DEVICE=",13)==0) {
        strncpy(wave_cf.dev_mixer,buf+13,sizeof(wave_cf.dev_mixer)-1);
        wave_cf.dev_mixer[sizeof(wave_cf.dev_mixer)-1]='\0';
      } else if (strncmp(buf,"PARA-S=",7)==0) {
        wave_cf.dfsfq=atol(buf+7);
        wave_cf.dfmod=(buf[strlen(buf)-1]=='S'?1UL:0UL);
      } else if (strncmp(buf,"DEBUGFILE=",10)==0) {
        snprintf(wave_cf.debugfile,sizeof(wave_cf.debugfile),"%s",buf+10);
      }
    }
    fclose(ffp);
    if (*wave_cf.debugfile!='\0') {
      wave_cf.debugfd=fopen(wave_cf.debugfile,"w");
    }
  }

  if (argc>=2) {
    for (i1=1;i1<argc;i1++) {
      if (strcmp(argv[i1],"-h")==0) {
        printf("\nSound server for VgaGames.\n");
        printf("It is called directly from VgaGames.\n");
        printf("For testing purposes you can call it from a shell.\n");
        printf("Commands are accepted from stdin.\n");
        printf("A config file wave.conf is read from %s\n",SHARE_PATH);
        printf("\n%s is compiled for:\n",argv[0]);
        for (wdptr=w_dev;wdptr->open_dev!=NULL;wdptr++) {
          printf(" - %s\n",wdptr->libname);
        }
        exit(1);
      }
      if (strcmp(argv[i1],"-q")==0) {
        quiet=1;
      } else if (strncmp(argv[i1],"-s",2)==0) {
        wave_cf.dfsfq=atol(argv[i1]+2);
        wave_cf.dfmod=(argv[i1][strlen(argv[i1])-1]=='S'?1UL:0UL);
      }
    }
  }
  wdef_allo=wdef_used=0;
  for (i1=1;i1<=CN_MAX;i1++) {
    memset(&chan_opt[CN_N(i1)],0,sizeof(chan_opt[CN_N(i1)]));
    volpc[i1]=0;  /* 100% */
  }
  no_sound=1;
  p1=getcwd(buf,sizeof(buf)-1);
  if (p1!=NULL) {
    sprintf(errmsg,"Current path: \"%s\"",p1);
    schreibe(errmsg);
  }

  /* open sound device */
  ssv=-1;
_open_again:
  for (wdptr=w_dev;wdptr->open_dev!=NULL;wdptr++) {
    if ((*wave_cf.lib!='\0') && (strcmp(wdptr->libname,wave_cf.lib)!=0)) {continue;}
    snprintf(errmsg,sizeof(errmsg),"trying sound device \"%s\" ...",wdptr->libname);
    schreibe(errmsg);
    sig_alarm=0;
    if ((frgm=wdptr->open_dev(wdptr))>=0) {
      snprintf(errmsg,sizeof(errmsg)," --> \"%s\" OK",wdptr->libname);
      schreibe(errmsg);
      break;
    }
    schreibe(errmsg);
  }
  sig_alarm=0;
  if (wdptr->open_dev==NULL) {
    if (ssv==-1) {soundserver(&ssv); goto _open_again;}
    strcpy(errmsg,"no sound device can be opened.");
    schreibe(errmsg);
    soundserver(&ssv);
    exit(1);
  }
  sprintf(errmsg,"Fragment size: %u",frgm);
  schreibe(errmsg);
  if (ssv==-1) {ssv=0;}

  /* sound device opened, now become original user */
  setgid(getgid());
  setuid(getuid());

  /* initialize sound device */
  fct=1;
  mno=0;
  wav_block=(WAV_BLOCK/(frgm*24))*(frgm*24);  /* 24=kgV of 1-4,6,8 */
  mid_block=(MID_BLOCK/frgm)*frgm;
  if ((wav_block<=0) || (mid_block<=0)) {
    if (wav_block<=0) {
      sprintf(errmsg,"WAV_BLOCK is too small (%lu but fragment size %u)",(long)WAV_BLOCK,frgm);
      schreibe(errmsg);
    }
    if (mid_block<=0) {
      sprintf(errmsg,"MID_BLOCK is too small (%lu but fragment size %u)",(long)MID_BLOCK,frgm);
      schreibe(errmsg);
    }
    wdptr->close_dev(wdptr);
    soundserver(&ssv);
    exit(1);
  }
  if (mid_block>wav_block) {mid_block=wav_block;}
  if (mid_block>4096) {mid_block=4096;}
  if (((frg_uc=malloc(sizeof(unsigned char)*frgm*4))==NULL) \
  || ((frg_si=malloc(sizeof(short)*frgm))==NULL)) {
    schreibe("Cannot allocate memory");
    wdptr->close_dev(wdptr);
    soundserver(&ssv);
    exit(1);
  }

  /* configure sound device */
  wdef_allo=10;
  if ((wdef=malloc(sizeof(wavedef)*wdef_allo))==NULL) {
    schreibe("Cannot allocate memory");
    wdptr->close_dev(wdptr);
    soundserver(&ssv);
    exit(1);
  }
  memset(&wdef[0],0,sizeof(wdef[0]));
  memset(&mdef,0,sizeof(mdef));
  mdef.typ=0;
  mdef.fname=NULL;
  mdef.bitsmp=8UL;
  mdef.modus=wave_cf.dfmod;
  mdef.smpfq=wave_cf.dfsfq;
  mdef.bytsek=(wave_cf.dfmod+1)*wave_cf.dfsfq;
  mdef.dbeg=mdef.dlen=0;
  mdef.w16to8=0;
  wdef_used=1;
  if (cfg_dev(0,0)<0) {
    schreibe(errmsg);
    wdptr->close_dev(wdptr);
    soundserver(&ssv);
    exit(1);
  }

  fcntl(STDIN_FILENO,F_SETFL,fcntl(STDIN_FILENO,F_GETFL,0)|O_NONBLOCK);
  schreibe(">>> For a short command description see vgag-wave.c,");
  schreibe(">>> at begin of main function.");
  schreibe("OK");

  p1=buf;
  *p1='\0';
  while (1) {
    while ((i1=read(STDIN_FILENO,p1,1))>0) {
      if (*p1=='\n') {break;}
      p1++; if (p1-buf>=(int)sizeof(buf)) {p1--;}
    }
    if ((i1==-1) && (errno!=EAGAIN)) {
      schreibe("Error reading pipe");
      wdptr->close_dev(wdptr);
      soundserver(&ssv);
      exit(1);
    }
    if (i1==0) {break;}
    if (*p1=='\n') {  /* new commando */
      *p1='\0';
      snprintf(errmsg,sizeof(errmsg),"Got command: \"%s\"",buf);
      schreibe(errmsg);
      if (((p1=strchr(buf,' '))!=NULL) && (p1>buf)) {
        aktz=*(p1-1);
        if ((p2=strchr(p1+1,' '))!=NULL) {
          *p2='\0';
          wvnr=atoi(p1);
          p1=p2+1;
          if ((p2=strchr(p1,' '))!=NULL) {
            *p2='\0';
            chnn=atoi(p1);
            p1=p2+1;
            switch(aktz) {
              case 'L': /* load a wave/midi file name */
                        if (cmd_l(wvnr,chnn,p1)<0) {
                          schreibe(errmsg);
                          wdptr->close_dev(wdptr);
                          soundserver(&ssv);
                          exit(1);
                        }
                        break;
              case 'S': /* play a loaded wave/midi file */
                        if ((p2=strchr(p1,'/'))!=NULL) {
                          p2++;
                        } else {p2=p1+strlen(p1);}
                        if (cmd_s(wvnr,chnn,(short)atoi(p1),(short)atoi(p2))<0) {
                          schreibe(errmsg);
                        }
                        break;
              case 'E': /* end playing a loaded wave/midi file */
                        cmd_e(chnn,(short)atoi(p1));
                        break;
              case 'P': /* pause playing a loaded wave/midi file */
                        cmd_p(chnn);
                        break;
              case 'C': /* continue playing a loaded wave/midi file */
                        cmd_c(chnn);
                        break;
              case 'V': /* change volume of channel(s) */
                        if (cmd_v(chnn,(short)atoi(p1))<0) {
                          schreibe(errmsg);
                        }
                        break;
              case 'A': /* acceleration */
                        if (cmd_a(chnn,atoi(p1))<0) {
                          schreibe(errmsg);
                        }
                        break;
              case 'M': /* main volume */
                        if ((i1=cmd_m(chnn,atoi(p1)))<0) {
                          schreibe(errmsg);
                        } else {
                          fprintf(stderr,"%d\n",i1); fflush(stderr);
                        }
                        break;
            }
          }
        }
      }
      p1=buf;
      *p1='\0';
    }
    /* send next fragment to wave device */
    for (i2=0;i2<frgm;i2++) {frg_si[i2]=128;}
    for (i1=1;i1<=CN_MAX;i1++) {
      if (chan_opt[CN_N(i1)].status==1) {  /* channel active */
        if (chan_opt[CN_N(i1)].nlp>0L) {  /* looping channel pauses */
          chan_opt[CN_N(i1)].nlp--;
          continue;
        }
        if (chan_opt[CN_N(i1)].dread==0) {  /* read next data */
          static size_t ranz=0;
          if (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_WAVE) {
            if (chan_opt[CN_N(i1)].dnoch<(size_t)wav_block) {
              ranz=chan_opt[CN_N(i1)].dnoch;
            } else {ranz=wav_block;}
            if (ranz>0) {
              ranz=fread(chan_opt[CN_N(i1)].data,sizeof(unsigned char),ranz,chan_opt[CN_N(i1)].fp);
            }
            if (ranz==0) {  /* wave file playing ended */
              if ((chan_opt[CN_N(i1)].vv>=0) \
              && (chan_opt[CN_N(i1)].loop>0L)) {
                /* looping */
                fseek(chan_opt[CN_N(i1)].fp,wdef[chan_opt[CN_N(i1)].wvnr].dbeg,SEEK_SET);
                chan_opt[CN_N(i1)].dnoch=wdef[chan_opt[CN_N(i1)].wvnr].dlen;
                chan_opt[CN_N(i1)].dread=0;
                chan_opt[CN_N(i1)].nlp=chan_opt[CN_N(i1)].loop;
                chan_opt[CN_N(i1)].vm=100000;
                chan_opt[CN_N(i1)].vv=0;
              } else {
                cmd_e(CN_(i1),0);
                if (chan_opt[CN_N(i1)].n_wvnr>0) {  /* waiting next file */
                  if (cmd_s(chan_opt[CN_N(i1)].n_wvnr, \
                      chan_opt[CN_N(i1)].n_chnn, \
                      chan_opt[CN_N(i1)].n_loopse, \
                      chan_opt[CN_N(i1)].n_volse)<0) {
                    schreibe(errmsg);
                  }
                }
              }
              continue;
            }
            chan_opt[CN_N(i1)].dnoch-=ranz;
            if (ranz%(frgm*24)>0) {  /* 24=kgV of 1-4,6,8 */
              memset(chan_opt[CN_N(i1)].data+ranz,wdef[chan_opt[CN_N(i1)].wvnr].w16to8?0:128,frgm*24-(ranz%(frgm*24)));
              ranz+=(frgm*24-(ranz%(frgm*24)));
            }
          } else if ((wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MIDI) \
                 || (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MP3)) {
            ranz=mid_block;
            ranz=fread(chan_opt[CN_N(i1)].data,sizeof(unsigned char),ranz,chan_opt[CN_N(i1)].fp);
            if (ranz==0) {  /* file playing ended */
              if ((chan_opt[CN_N(i1)].vv>=0) \
              && (chan_opt[CN_N(i1)].loop>0L)) {
                /* looping */
                pclose(chan_opt[CN_N(i1)].fp);
                if (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MIDI) {
                  chan_opt[CN_N(i1)].fp=_open_midi(wdef[chan_opt[CN_N(i1)].wvnr].fname);
                } else if (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MP3) {
                  chan_opt[CN_N(i1)].fp=_open_mp3(wdef[chan_opt[CN_N(i1)].wvnr].fname);
                } else {chan_opt[CN_N(i1)].fp=NULL;}
                if (chan_opt[CN_N(i1)].fp!=NULL) {
                  chan_opt[CN_N(i1)].dnoch=wdef[chan_opt[CN_N(i1)].wvnr].dlen;
                  chan_opt[CN_N(i1)].dread=0;
                  chan_opt[CN_N(i1)].nlp=chan_opt[CN_N(i1)].loop;
                  chan_opt[CN_N(i1)].vm=100000;
                  chan_opt[CN_N(i1)].vv=0;
                } else {
                  cmd_e(CN_(i1),0);
                }
              } else {
                cmd_e(CN_(i1),0);
                if (chan_opt[CN_N(i1)].n_wvnr>0) {  /* waiting next file */
                  if (cmd_s(chan_opt[CN_N(i1)].n_wvnr, \
                      chan_opt[CN_N(i1)].n_chnn, \
                      chan_opt[CN_N(i1)].n_loopse, \
                      chan_opt[CN_N(i1)].n_volse)<0) {
                    schreibe(errmsg);
                  }
                }
              }
              continue;
            }
            if (ranz%frgm>0) {
              memset(chan_opt[CN_N(i1)].data+ranz,128,frgm-(ranz%frgm));
              ranz+=(frgm-(ranz%frgm));
            }
          } else {ranz=0; continue;}
          chan_opt[CN_N(i1)].dread=ranz;
          chan_opt[CN_N(i1)].dptr=chan_opt[CN_N(i1)].data;
        }
        /* evtl. in-/decrease volume */
        if (chan_opt[CN_N(i1)].vv!=0) {
          chan_opt[CN_N(i1)].vm+=chan_opt[CN_N(i1)].vv;
          if (chan_opt[CN_N(i1)].vm>=100000) {
            chan_opt[CN_N(i1)].vm=100000;
            chan_opt[CN_N(i1)].vv=0;
          } else if (chan_opt[CN_N(i1)].vm<=0) {
            chan_opt[CN_N(i1)].vm=0;
            chan_opt[CN_N(i1)].vv=0;
            cmd_e(CN_(i1),0);
            if (chan_opt[CN_N(i1)].n_wvnr>0) {  /* waiting next file */
              if (cmd_s(chan_opt[CN_N(i1)].n_wvnr, \
                  chan_opt[CN_N(i1)].n_chnn, \
                  chan_opt[CN_N(i1)].n_loopse, \
                  chan_opt[CN_N(i1)].n_volse)<0) {
                schreibe(errmsg);
              }
            }
            continue;
          }
        }
        volx=(double)chan_opt[CN_N(i1)].volpc/10.-(1.-(double)chan_opt[CN_N(i1)].vm/100000.)*(1+(double)chan_opt[CN_N(i1)].volpc/10.);
        /* set data to channel */
        if (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_WAVE) {
          if (wdef[chan_opt[CN_N(i1)].wvnr].w16to8) {  /* convert 16LE bit to 8U bit */
            if (wdef[chan_opt[CN_N(i1)].wvnr].modus==0UL) {  /* mono */
              for (i2=0;i2<frgm;i2++) {
                chan_opt[CN_N(i1)].dptr++;
                s1=(short)((char)*chan_opt[CN_N(i1)].dptr);
                s1+=((short)((double)s1*volx));
                frg_si[i2]+=s1;
                chan_opt[CN_N(i1)].dptr+=(fct*2-1);
              }
              chan_opt[CN_N(i1)].dread-=(frgm*fct*2);
            } else {  /* stereo */
              if (mno==0) {
                for (i2=0;i2<frgm;i2++) {
                  chan_opt[CN_N(i1)].dptr++;
                  s1=(short)((char)*chan_opt[CN_N(i1)].dptr++);
                  s1+=((short)((double)s1*volx));
                  frg_si[i2++]+=s1;
                  chan_opt[CN_N(i1)].dptr++;
                  s1=(short)((char)*chan_opt[CN_N(i1)].dptr);
                  s1+=((short)((double)s1*volx));
                  frg_si[i2]+=s1;
                  chan_opt[CN_N(i1)].dptr+=(fct*4-3);
                }
                chan_opt[CN_N(i1)].dread-=(frgm*fct*2);
              } else {  /* force to mono */
                for (i2=0;i2<frgm;i2++) {
                  chan_opt[CN_N(i1)].dptr++;
                  s1=(short)((char)*chan_opt[CN_N(i1)].dptr++);
                  chan_opt[CN_N(i1)].dptr++;
                  s1+=(short)((char)*chan_opt[CN_N(i1)].dptr);
                  s1/=2;
                  s1+=((short)((double)s1*volx));
                  frg_si[i2]+=s1;
                  chan_opt[CN_N(i1)].dptr+=(fct*4-3);
                }
                chan_opt[CN_N(i1)].dread-=(frgm*fct*4);
              }
            }
          } else {  /* 8 bit U */
            if (wdef[chan_opt[CN_N(i1)].wvnr].modus==0UL) {  /* mono */
              for (i2=0;i2<frgm;i2++) {
                s1=(short)(*chan_opt[CN_N(i1)].dptr)-128;
                s1+=((short)((double)s1*volx));
                frg_si[i2]+=s1;
                chan_opt[CN_N(i1)].dptr+=fct;
              }
              chan_opt[CN_N(i1)].dread-=(frgm*fct);
            } else {  /* stereo */
              if (mno==0) {
                for (i2=0;i2<frgm;i2++) {
                  s1=(short)(*chan_opt[CN_N(i1)].dptr++)-128;
                  s1+=((short)((double)s1*volx));
                  frg_si[i2++]+=s1;
                  s1=(short)(*chan_opt[CN_N(i1)].dptr)-128;
                  s1+=((short)((double)s1*volx));
                  frg_si[i2]+=s1;
                  chan_opt[CN_N(i1)].dptr+=(fct*2-1);
                }
                chan_opt[CN_N(i1)].dread-=(frgm*fct);
              } else {  /* force to mono */
                for (i2=0;i2<frgm;i2++) {
                  s1=(short)(*chan_opt[CN_N(i1)].dptr++)-128;
                  s1+=(short)(*chan_opt[CN_N(i1)].dptr)-128;
                  s1/=2;
                  s1+=((short)((double)s1*volx));
                  frg_si[i2]+=s1;
                  chan_opt[CN_N(i1)].dptr+=(fct*2-1);
                }
                chan_opt[CN_N(i1)].dread-=(frgm*fct*2);
              }
            }
          }
        } else {
          for (i2=0;i2<frgm;i2++) {
            s1=(short)(*chan_opt[CN_N(i1)].dptr++)-128;
            s1+=((short)((double)s1*volx));
            frg_si[i2]+=s1;
          }
          chan_opt[CN_N(i1)].dread-=frgm;
        }
      } else {  /* channel not active */
        static unsigned char dptr[2]="\x80";
        for (i2=0;i2<frgm;i2++) {frg_si[i2]+=(short)(*dptr)-128;}
      }
    }
    /* collect channels */
    if (wdptr->simul_stereo && wdptr->simul_16) {
      for (i2=0;i2<frgm;i2++) {
        frg_si[i2]=(frg_si[i2]<0?0:(frg_si[i2]>255?255:frg_si[i2]));
        frg_si[i2]-=128;
#if BYTE_ORDER == BIG_ENDIAN
        frg_uc[i2*4+1]=frg_uc[i2*4+3]=0;
        frg_uc[i2*4+0]=frg_uc[i2*4+2]=(unsigned char)(frg_si[i2]&0xff);
#else
        frg_uc[i2*4+0]=frg_uc[i2*4+2]=0;
        frg_uc[i2*4+1]=frg_uc[i2*4+3]=(unsigned char)(frg_si[i2]&0xff);
#endif
      }
      wdptr->write_dev(wdptr,frg_uc,frgm*4);
    } else if (wdptr->simul_16) {
      for (i2=0;i2<frgm;i2++) {
        frg_si[i2]=(frg_si[i2]<0?0:(frg_si[i2]>255?255:frg_si[i2]));
        frg_si[i2]-=128;
#if BYTE_ORDER == BIG_ENDIAN
        frg_uc[i2*2+1]=0;
        frg_uc[i2*2+0]=(unsigned char)(frg_si[i2]&0xff);
#else
        frg_uc[i2*2+0]=0;
        frg_uc[i2*2+1]=(unsigned char)(frg_si[i2]&0xff);
#endif
      }
      wdptr->write_dev(wdptr,frg_uc,frgm*2);
    } else if (wdptr->simul_stereo) {
      for (i2=0;i2<frgm;i2++) {
        frg_uc[i2*2+0]=frg_uc[i2*2+1]=(unsigned char)(frg_si[i2]<0?0:(frg_si[i2]>255?255:frg_si[i2]));
      }
      wdptr->write_dev(wdptr,frg_uc,frgm*2);
    } else {
      for (i2=0;i2<frgm;i2++) {
        frg_uc[i2]=(unsigned char)(frg_si[i2]<0?0:(frg_si[i2]>255?255:frg_si[i2]));
      }
      wdptr->write_dev(wdptr,frg_uc,frgm);
    }
  }
  wdptr->close_dev(wdptr);
  free(wdef);
  free(frg_si);
  free(frg_uc);
  if (wave_cf.debugfd!=NULL) {fclose(wave_cf.debugfd);}
  soundserver(&ssv);
  exit(0);
} /* Ende main */


void schreibe(const char * errm) {
  if (quiet==0) {printf("%s\n",errm); fflush(stdout);}
  if (wave_cf.debugfd!=NULL) {
    time_t zt=time(NULL);
    struct tm * lzt=localtime(&zt);
    fprintf(wave_cf.debugfd,"%02d:%02d:%02d %s\n",lzt->tm_hour,lzt->tm_min,lzt->tm_sec,errm);
    fflush(wave_cf.debugfd);
  }
} /* Ende schreibe */


int cfg_dev(int wvnr,int flag) {
  static int m_fct=-1,m_mno=-1;
  unsigned long smpfq,modus;
  int i1;
  wavedef * tdef;
  /* test whether anything has changed: wdef[0] are actual settings */
  if (wvnr==0) {tdef=&mdef;} else {tdef=&wdef[wvnr];}
  if ((wdef[0].bitsmp==tdef->bitsmp) \
  && (wdef[0].modus==tdef->modus) \
  && (wdef[0].smpfq==tdef->smpfq) \
  && (wdef[0].bytsek==tdef->bytsek) \
  && (m_fct==fct) && (m_mno==mno)) {
    return(0);
  } else if (flag==1) {return(-1);}
  /* change configuration */
  smpfq=tdef->smpfq/fct;
  modus=(mno==0?tdef->modus:0UL);
  if ((i1=wdptr->config_dev(wdptr,tdef->bitsmp,modus,smpfq,tdef->bytsek))==0) {
    snprintf(errmsg,sizeof(errmsg),"Set samplesize to %lu, %s, speed to %lu",tdef->bitsmp,modus==1UL?"stereo":"mono",smpfq);
    schreibe(errmsg);
  }
  wdef[0].bitsmp=tdef->bitsmp;
  wdef[0].modus=tdef->modus;
  wdef[0].smpfq=tdef->smpfq;
  wdef[0].bytsek=tdef->bytsek;
  m_fct=fct; m_mno=mno;
  return(i1);
} /* Ende cfg_dev */


int cmd_l(int wvnr,int typ,char * file) {
/* load wave file configuration */
  char data[wav_block],* ptr;
  FILE * fp;
  size_t glen,flen;
  long l1;
  unsigned short bsmp;
  if (file==NULL) {strcpy(errmsg,"file name NULL"); return(-1);}
  if ((typ!=TYP_IS_WAVE) && (typ!=TYP_IS_MIDI) && (typ!=TYP_IS_MP3) && (typ!=TYP_IS_EXTENSION)) {
    strcpy(errmsg,"No valid type");
    return(-1);
  }
  if (wdef_used==wdef_allo) {
    wdef_allo+=10;
    if ((wdef=realloc(wdef,sizeof(wavedef)*wdef_allo))==NULL) {
      wdef_allo-=10;
      strcpy(errmsg,"Cannot allocate memory");
      return(-1);
    }
  }
  if (wvnr!=wdef_used) {
    sprintf(errmsg,"Wave number %d is not %d",wvnr,wdef_used);
    return(-1);
  }
  memset(&wdef[wdef_used],0,sizeof(wdef[wdef_used]));
  if ((fp=fopen(file,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot read \"%s\"",file);
    return(-1);
  }
  if ((flen=fread(data,sizeof(char),sizeof(data),fp))==0) {
    snprintf(errmsg,sizeof(errmsg),"Error reading file %s",file);
    fclose(fp);
    return(-1);
  }
  fclose(fp);
  if (typ==TYP_IS_EXTENSION) {
    if ((ptr=strrchr(file,'.'))!=NULL) {
      if (strncasecmp(ptr+1,"wav",3)==0) {
        typ=TYP_IS_WAVE;
      } else if (strncasecmp(ptr+1,"mid",3)==0) {
        typ=TYP_IS_MIDI;
      } else if (strncasecmp(ptr+1,"mp3",3)==0) {
        typ=TYP_IS_MP3;
      }
    }
    if (typ==TYP_IS_EXTENSION) {
      strcpy(errmsg,"No valid type: Extension not recognized");
      return(-1);
    }
  }

  if (typ==TYP_IS_WAVE) {
    /* +++ find configuration in data with length flen +++ */
    glen=flen;
    ptr=data;
    /* "RIFF" */
    if ((flen<8) || (strncmp(ptr,"RIFF",4)!=0)) {
      strcpy(errmsg,"error: no valid wave file: no RIFF");
      return(-1);
    } else {ptr+=8; flen-=8;}
    /* "WAVE" */
    if ((flen<4) || (strncmp(ptr,"WAVE",4)!=0)) {
      strcpy(errmsg,"error: no valid wave file: no WAVE");
      return(-1);
    } else {ptr+=4; flen-=4;}
    /* other, then "fmt " */
    while (flen>=8) {
      if (strncmp(ptr,"fmt ",4)==0) {break;}
      ptr+=4; flen-=4;
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
      ptr+=4; flen-=4;
      ptr+=_c2i.iw; flen-=(size_t)_c2i.iw;
    }
    ptr+=4; flen-=4;
    if (flen<4) {
      strcpy(errmsg,"error: no valid wave file: too short before \"fmt\"");
      return(-1);
    }
    /* blocklength */
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
    ptr+=4; flen-=4;
    if ((_c2i.iw<16) || (flen<(size_t)_c2i.iw)) {
      strcpy(errmsg,"error: no valid wave file: too short at \"fmt\"");
      return(-1);
    }
    l1=_c2i.iw-16;
    /* PCM code = 1 */
#if BYTE_ORDER == BIG_ENDIAN
      _c2s.cw[0]=ptr[1]; _c2s.cw[1]=ptr[0];
#else
      _c2s.cw[0]=ptr[0]; _c2s.cw[1]=ptr[1];
#endif
    if (_c2s.sw!=1) {
      strcpy(errmsg,"error: no valid wave file: PCM code not 1");
      return(-1);
    }
    ptr+=2; flen-=2;
    /* modus mono or stereo */
#if BYTE_ORDER == BIG_ENDIAN
      _c2s.cw[0]=ptr[1]; _c2s.cw[1]=ptr[0];
#else
      _c2s.cw[0]=ptr[0]; _c2s.cw[1]=ptr[1];
#endif
    wdef[wdef_used].modus=(_c2s.sw==2);
    ptr+=2; flen-=2;
    /* sample frequence */
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
    wdef[wdef_used].smpfq=(unsigned long)_c2i.iw;
    ptr+=4; flen-=4;
    /* bytes per second */
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
    wdef[wdef_used].bytsek=(unsigned long)_c2i.iw;
    ptr+=4; flen-=4;
    /* bytes per sample */
#if BYTE_ORDER == BIG_ENDIAN
      _c2s.cw[0]=ptr[1]; _c2s.cw[1]=ptr[0];
#else
      _c2s.cw[0]=ptr[0]; _c2s.cw[1]=ptr[1];
#endif
    bsmp=_c2s.sw;
    ptr+=2; flen-=2;
    /* bits per sample */
#if BYTE_ORDER == BIG_ENDIAN
      _c2s.cw[0]=ptr[1]; _c2s.cw[1]=ptr[0];
#else
      _c2s.cw[0]=ptr[0]; _c2s.cw[1]=ptr[1];
#endif
    if (_c2s.sw==8) {
      wdef[wdef_used].bitsmp=8UL;
      wdef[wdef_used].w16to8=0;
    } else if (_c2s.sw==16) {
      wdef[wdef_used].bitsmp=8UL;
      wdef[wdef_used].w16to8=1;
    } else {
      strcpy(errmsg,"error: no supported wave file: bit sample not 8 or 16");
      return(-1);
    }
    ptr+=2; flen-=2;
    ptr+=l1;
    if ((wdef[wdef_used].smpfq*bsmp)!=wdef[wdef_used].bytsek) {
      strcpy(errmsg,"error: no valid wave file: freq. * byt.sample not equal byt.seconds");
      return(-1);
    }
    /* other, then "data" */
    while (flen>=8) {
      if (strncmp(ptr,"data",4)==0) {break;}
      ptr+=4; flen-=4;
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
      ptr+=4; flen-=4;
      ptr+=_c2i.iw; flen-=(size_t)_c2i.iw;
    }
    ptr+=4; flen-=4;
    if (flen<4) {
      strcpy(errmsg,"error: no valid wave file: too short before \"data\"");
      return(-1);
    }
    /* wave data length */
#if BYTE_ORDER == BIG_ENDIAN
      _c2i.cw[0]=ptr[3]; _c2i.cw[1]=ptr[2]; _c2i.cw[2]=ptr[1]; _c2i.cw[3]=ptr[0];
#else
      _c2i.cw[0]=ptr[0]; _c2i.cw[1]=ptr[1]; _c2i.cw[2]=ptr[2]; _c2i.cw[3]=ptr[3];
#endif
    wdef[wdef_used].dlen=(size_t)_c2i.iw;
    ptr+=4; flen-=4;
    wdef[wdef_used].dbeg=glen-flen;  /* begin wave data */
    wdef[wdef_used].fname=strdup(file);  /* file name */
    wdef[wdef_used].typ=TYP_IS_WAVE;
    memmove(&mdef,&wdef[wdef_used],sizeof(mdef));
    snprintf(errmsg,sizeof(errmsg),"Loading wave %s (%d bit, %lu kHz, %s) ok",wdef[wdef_used].fname,wdef[wdef_used].w16to8?16:8,wdef[wdef_used].smpfq,wdef[wdef_used].modus==0UL?"mono":"stereo");
    schreibe(errmsg);
  } else if ((typ==TYP_IS_MIDI) || (typ==TYP_IS_MP3)) {
    wdef[wdef_used].modus=0UL;  /* must be set later */
    wdef[wdef_used].smpfq=0UL;  /* must be set later */
    wdef[wdef_used].bitsmp=8UL;
    wdef[wdef_used].bytsek=0UL;  /* must be set later */
    wdef[wdef_used].dlen=(size_t)-1;  /* don't know, assume maximum */
    wdef[wdef_used].dbeg=0;  /* begin midi/mp3->wave data, always 0 */
    wdef[wdef_used].fname=strdup(file);  /* file name */
    wdef[wdef_used].typ=typ;
    snprintf(errmsg,sizeof(errmsg),"Loading %s %s done",typ==TYP_IS_MIDI?"midi":"mp3",wdef[wdef_used].fname);
    schreibe(errmsg);
  }

  wdef_used++;
  return(0);
} /* Ende cmd_l */


int cmd_s(int wvnr,int chnn,short loopse,short volse) {
/* start playing wave/midi at a channel */
  int i1,i2,channel;
  if ((wvnr<1) || (wvnr>=wdef_used)) {
    sprintf(errmsg,"Wave number %d not defined",wvnr);
    return(-1);
  }
  if (loopse<0) {loopse=0;}
  /* find a free channel */
  channel=i1=-1;
  for (i2=1;i2<=CN_MAX;i2++) {
    if (chnn&CN_(i2)) {
      i1=i2;
      if (chan_opt[CN_N(i1)].status==0) {channel=CN_N(i1);}
    }
  }
  if (channel==-1) {  /* no free channel */
    if (i1==-1) {
      strcpy(errmsg,"No free channel found");
      return(-1);
    } else {  /* cancel a running channel */
      cmd_e(CN_(i1),volse);
      if (volse>0) {
        chan_opt[CN_N(i1)].n_wvnr=wvnr;
        chan_opt[CN_N(i1)].n_chnn=CN_(i1);
        chan_opt[CN_N(i1)].n_loopse=loopse;
        chan_opt[CN_N(i1)].n_volse=volse;
        return(0);
      } else {
        channel=CN_N(i1);
      }
    }
  }
  memset(&chan_opt[channel],0,sizeof(chan_opt[channel]));
  if (wdef[wvnr].typ==TYP_IS_WAVE) {
    if (no_sound==1) {
      if (cfg_dev(wvnr,0)<0) {return(-1);}
    } else {
      if (cfg_dev(wvnr,1)<0) {
        strcpy(errmsg,"Cannot play wave file - it has another device configuration");
        return(-1);
      }
    }
    if ((chan_opt[channel].fp=fopen(wdef[wvnr].fname,"r"))==NULL) {
      strcpy(errmsg,"Cannot read wave file");
      return(-1);
    }
    chan_opt[channel].wvnr=wvnr;
    fseek(chan_opt[channel].fp,wdef[wvnr].dbeg,SEEK_SET);
    chan_opt[channel].dnoch=wdef[wvnr].dlen;
    chan_opt[channel].dread=0;
    chan_opt[channel].loop=wdef[0].bytsek/fct;
    if ((mno==1) && (wdef[0].modus==1UL)) {chan_opt[channel].loop/=2;}
    chan_opt[channel].loop*=loopse;
    chan_opt[channel].loop/=frgm;
    chan_opt[channel].nlp=0L;
    chan_opt[channel].volpc=volpc[channel];
    chan_opt[channel].status=1;
    chan_opt[channel].n_wvnr=chan_opt[channel].n_chnn=0;
    chan_opt[channel].n_loopse=chan_opt[channel].n_volse=0;
    snprintf(errmsg,sizeof(errmsg),"Playing %s (no. %u) at channel %u",wdef[wvnr].fname,wvnr,channel+1);
    schreibe(errmsg);
    no_sound=0;
  } else if ((wdef[wvnr].typ==TYP_IS_MIDI) || (wdef[wvnr].typ==TYP_IS_MP3)) {
    if (no_sound==1) {
      if (cfg_dev(0,0)<0) {return(-1);}
    }
    if (wdef[wvnr].typ==TYP_IS_MIDI) {
      chan_opt[channel].fp=_open_midi(wdef[wvnr].fname);
    } else if (wdef[wvnr].typ==TYP_IS_MP3) {
      chan_opt[channel].fp=_open_mp3(wdef[wvnr].fname);
    } else {chan_opt[channel].fp=NULL;}
    if (chan_opt[channel].fp==NULL) {return(-1);}
    chan_opt[channel].wvnr=wvnr;
    chan_opt[channel].dnoch=wdef[wvnr].dlen;
    chan_opt[channel].dread=0;
    chan_opt[channel].loop=wdef[0].bytsek/fct;
    if ((mno==1) && (wdef[0].modus==1UL)) {chan_opt[channel].loop/=2;}
    chan_opt[channel].loop*=loopse;
    chan_opt[channel].loop/=frgm;
    chan_opt[channel].nlp=0L;
    chan_opt[channel].volpc=volpc[channel];
    chan_opt[channel].status=1;
    chan_opt[channel].n_wvnr=chan_opt[channel].n_chnn=0;
    chan_opt[channel].n_loopse=chan_opt[channel].n_volse=0;
    snprintf(errmsg,sizeof(errmsg),"Playing %s (no. %u) at channel %u",wdef[wvnr].fname,wvnr,channel+1);
    schreibe(errmsg);
    no_sound=0;
  }
  if (volse>0) {  /* volume increasing time */
    unsigned long l1;
    l1=wdef[0].bytsek/fct;
    if ((mno==1) && (wdef[0].modus==1UL)) {l1/=2;}
    l1*=volse;
    l1/=frgm;
    chan_opt[channel].vm=0;  /* 0% */
    chan_opt[channel].vv=100000/(int)l1;  /* increasing percent */
    if (chan_opt[channel].vv<1) {chan_opt[channel].vv=1;}
  } else {
    chan_opt[channel].vm=100000;  /* 100% */
    chan_opt[channel].vv=0;  /* no increasing percent */
  }
  return(0);
} /* Ende cmd_s */


int cmd_e(int chnn,short volse) {
/* end playing wave/midi at (a) channel(s) */
  int i1,elen;
  if (volse>0) {  /* volume decreasing time */
    unsigned long l1;
    int dpc;
    elen=sprintf(errmsg,"Decreasing for %d sec. at channel ",volse);
    l1=wdef[0].bytsek/fct;
    if ((mno==1) && (wdef[0].modus==1UL)) {l1/=2;}
    l1*=volse;
    l1/=frgm;
    dpc=-100000/(int)l1;  /* decreasing percent */
    if (dpc>-1) {dpc=-1;}
    for (i1=1;i1<=CN_MAX;i1++) {
      if ((chnn&CN_(i1)) && (chan_opt[CN_N(i1)].status>0) \
      && (chan_opt[CN_N(i1)].vv>=0)) {
        chan_opt[CN_N(i1)].vm=100000;  /* 100% */
        chan_opt[CN_N(i1)].vv=dpc;
        elen+=sprintf(errmsg+elen,"%d ",i1);
      }
    }
  } else {  /* now ending */
    elen=sprintf(errmsg,"Ending at channel ");
    for (i1=1;i1<=CN_MAX;i1++) {
      if ((chnn&CN_(i1)) && (chan_opt[CN_N(i1)].status>0)) {
        if (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_WAVE) {
          fclose(chan_opt[CN_N(i1)].fp);
          chan_opt[CN_N(i1)].status=0;
        } else if ((wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MIDI) \
               || (wdef[chan_opt[CN_N(i1)].wvnr].typ==TYP_IS_MP3)) {
          if (chan_opt[CN_N(i1)].fp!=NULL) {
            pclose(chan_opt[CN_N(i1)].fp);
          }
          chan_opt[CN_N(i1)].status=0;
        }
        elen+=sprintf(errmsg+elen,"%d ",i1);
      }
    }
    for (i1=1;i1<=CN_MAX;i1++) {
      if (chan_opt[CN_N(i1)].status!=0) {break;}
    }
    if (i1>CN_MAX) {  /* no sound playing */
      no_sound=1;
    }
  }
  schreibe(errmsg);
  return(0);
} /* Ende cmd_e */


int cmd_p(int chnn) {
/* pause playing wave/midi at (a) channel(s) */
  int i1,elen;
  elen=sprintf(errmsg,"Pausing at channel ");
  for (i1=1;i1<=CN_MAX;i1++) {
    if ((chnn&CN_(i1)) && (chan_opt[CN_N(i1)].status>0)) {
      chan_opt[CN_N(i1)].status=2;
      elen+=sprintf(errmsg+elen,"%d ",i1);
    }
  }
  schreibe(errmsg);
  return(0);
} /* Ende cmd_p */


int cmd_c(int chnn) {
/* continue playing wave/midi at (a) channel(s) */
  int i1,elen;
  elen=sprintf(errmsg,"Continuing at channel ");
  for (i1=1;i1<=CN_MAX;i1++) {
    if ((chnn&CN_(i1)) && (chan_opt[CN_N(i1)].status>0)) {
      chan_opt[CN_N(i1)].status=1;
      elen+=sprintf(errmsg+elen,"%d ",i1);
    }
  }
  schreibe(errmsg);
  return(0);
} /* Ende cmd_c */


int cmd_v(int chnn,short vpc) {
/* change volume of channel(s) */
  int i1,elen;
  if (vpc<-10) {vpc=-10;}
  if (vpc>20) {vpc=20;}
  elen=sprintf(errmsg,"Changing volume to %d percent at channel ",(vpc+10)*10);
  for (i1=1;i1<=CN_MAX;i1++) {
    if (chnn&CN_(i1)) {
      volpc[CN_N(i1)]=chan_opt[CN_N(i1)].volpc=vpc;
      elen+=sprintf(errmsg+elen,"%d ",i1);
    }
  }
  schreibe(errmsg);
  return(0);
} /* Ende cmd_v */


int cmd_a(int rfct,int rmno) {
/* acceleration
** 1.arg: divide all original samplefrequences through 1.arg (0 or 1 to 4)
** 2.arg: convert all to mono or not (0 or 1 to 2)
** Examples:
**   "0 0" = don't change anything
**   "1 0" = samplefrequence/1 and don't change mono/stereo settings
**   "1 1" = samplefrequence/1 and convert all to mono
**   "1 2" = samplefrequence/1 and let mono=mono/stereo=stereo
**   "2 2" = samplefrequence/2 and let mono=mono/stereo=stereo
**   "3 1" = samplefrequence/3 and convert all to mono
**   "4 0" = samplefrequence/4 and don't change mono/stereo settings
** When using this function, all playing sounds are ended.
*/
  if (rfct<0) {rfct=0;}
  if (rfct>4) {rfct=4;}
  if (rmno<0) {rmno=0;}
  if (rmno>2) {rmno=2;}
  if (rfct>0) {fct=rfct;}
  if (rmno>0) {
    if (rmno==1) {mno=1;}
    if (rmno==2) {mno=0;}
  }
  if ((rfct!=0) && (rmno!=0)) {
    cmd_e(CN_ALL,0);
    snprintf(errmsg,sizeof(errmsg),"Setting samplefreq. to 1/%d and mono settings to %s",rfct,rmno==1?"mono":"mono/stereo");
  } else if ((rfct!=0) && (rmno==0)) {
    cmd_e(CN_ALL,0);
    snprintf(errmsg,sizeof(errmsg),"Setting samplefreq. to 1/%d",rfct);
  } else if ((rfct==0) && (rmno!=0)) {
    cmd_e(CN_ALL,0);
    snprintf(errmsg,sizeof(errmsg),"Setting mono settings to %s",rmno==1?"mono":"mono/stereo");
  } else {
    strcpy(errmsg,"Acceleration: no change");
  }
  schreibe(errmsg);
  return(0);
} /* Ende cmd_a */


int cmd_m(int volm,int flag) {
/* set main or pcm volume
** 1.arg: 0-100 or <0 to get only actual volume
** 2.arg: 0=main volume or 1=pcm volume (wave)
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  return(wdptr->mixer_dev(wdptr,volm,flag));
} /* Ende cmd_m */
