/* *****************************************************************
   vgagames: Version 1.10
   Copyright (C) 2000-2005 Kurt Nienhaus <vgagames@vgagames.de>

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
   ***************************************************************** */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <math.h>
#include "pfad.h"
#include "_t_of_sel.h"
#include "font8x8.h"
#define RGB256_FILE   SHARE_PATH "/rgb256.dat"
#define GREY256_FILE  SHARE_PATH "/grey256.dat"
#include "vgagames.h"
#include "vgagames_wave.h"


/* +++ general +++ */
#ifndef M_PI
  #define M_PI  3.14159265358979323846
#endif
#ifndef PI
  #define PI  M_PI
#endif
#ifndef PI2
  #define PI2  (M_PI + M_PI)
#endif
char errmsg[2048]="";
int _rgb_black,_rgb_white,_rgb_dark;
static unsigned char * deffont=NULL;
static int deffont_w=8,deffont_h=8;
static int col_grey=1;
grafik * backpuffer=NULL;  /* backbuffer */


/* +++ key and mouse +++ */
int _keyi=0;
int _keym[KEYS_INDEX];
int _keyc[KEYS_INDEX]={};
int _maus_x,_maus_y,_maus_left,_maus_right,_maus_middle;


/* +++ colors +++ */
typedef struct {
  unsigned char r,g,b;
} farbkonv;
farbkonv fx[256]={};
#include "colordef.h"


/* +++ additional fonts +++ */
#include "font10x17.h"


/* +++ function declarations +++ */
/* general functions */
int wait_time(int);
int color_index(int,int);
int load_colormap(const char *);
void set_font(unsigned char *,int,int);
grafik * create_grafik(int,int);
grafik * load_grafik(const char *);
grafik * text_to_grafik(int,int,int,int,const char *);
int save_grafik_as_ppm(grafik *,const char *);
int save_grafik_as_ppm3(grafik *,const char *);
void free_grafik(grafik *);
void copy_grafik(grafik *,int,int,grafik *,int,int,int,int,int);
grafik * rotate_grafik(grafik *,grafik *,int,int *,int *,int *,int *);
grafik * zoom_grafik(grafik *,grafik *,float,float,int *,int *,int *,int *);
int get_pixel(grafik *,int,int);
void draw_text(grafik *,int,int,int,const char *,int,int);
void draw_pixel(grafik *,int,int,int);
void draw_line(grafik *,int,int,int,int,int);
void draw_circle(grafik *,int,int,int,int,int);
void draw_fillbox(grafik *,int,int,int,int,int);
film * create_film(int);
int add_bild_to_film(film *,film_bild *,int);
unsigned long copy_film(grafik *,film *,void *);
void reset_film(film *,void *,int);
void free_film(film *);
sprite * create_sprite(void);
sprite * load_sprite(const char *);
sprite * rotate_sprite(sprite *,int);
sprite * zoom_sprite(sprite *,float,float);
int add_grafik_to_sprite(sprite *,grafik *,int,int,int,int,int);
grafik * next_sprite(sprite *,int *,int *);
grafik * last_sprite(sprite *,int *,int *);
int reset_sprite(sprite *);
void free_sprite(sprite *);

/* graphic library dependend functions */
int open_window(const char *,int);
void flush_window(void);
void close_window(void);
int reopen_window(int);
void get_keys(void);
void clearstate(void);
int mouse_speed(int);
void brightness(int);

/* internal functions */
static FILE * _vga_grafik(const char *);
static int _konv_map(unsigned char[],FILE *);
static unsigned char _best_color(farbkonv *,int);
static void _draw_fontXxY(grafik *,int,int,int,const char *,int,unsigned char *,int *,int *,int,int);
static int _save_grf_ppm(grafik *,const char *,int);
static grafik * _get_sprite(sprite *,int *,int *,int);



/* +++ general functions +++ */

int wait_time(int m_sec) {
/* waits rest of m_sec millisecs since last call
** return value: time in millisecs since last call or -1=error
*/
  static struct timeval tv;
  static struct timezone tz;
  static int vs=0,ms,ws;
  m_sec-=TIME_OF_SELECT;
  if (m_sec<0) {m_sec=0;}
  if (gettimeofday(&tv,&tz)!=0) {strcpy(errmsg,"Get time error!"); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  if (ws<m_sec+TIME_OF_SELECT) {
    if (ws<m_sec) {tv.tv_usec=(m_sec-ws)*1000;} else {tv.tv_usec=1;}
    tv.tv_sec=0;
    select(0,NULL,NULL,NULL,&tv);
  }
  if (gettimeofday(&tv,&tz)!=0) {strcpy(errmsg,"Get time error!"); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  vs=ms;
  return(ws);
} /* Ende wait_time */


int color_index(int cl_nr,int prz) {
/* gives back best color index of a color name
** 1.arg: color name (CL_-definition)
** 2.arg: color brightness percent: value from 0 to 100
** return value: best color index
*/
  farbkonv fw;
  if (cl_nr<1) {return(_rgb_black);}
  else if (cl_nr>COLOR_MAX) {return(RGB_WHITE);}
  if (prz<0) {prz=0;}
  else if (prz>100) {prz=100;}
  if (col_grey==1) {  /* color */
    fw.r=coldef[cl_nr-1].r*prz/100;
    fw.g=coldef[cl_nr-1].g*prz/100;
    fw.b=coldef[cl_nr-1].b*prz/100;
  } else {  /* grey */
    fw.r=(coldef[cl_nr-1].r+coldef[cl_nr-1].g+coldef[cl_nr-1].b)/3*prz/100;
    fw.g=fw.b=fw.r;
  }
  return((int)_best_color(&fw,RGB_FULL));
} /* Ende color_index */


int load_colormap(const char * clm) {
/* loads a colormap
** 1.arg: path to colormap or RGB256 or GREY256 or NULL=previous
** return value: 0=OK or -1=error
*/
  static char * colm=NULL,colv=0;
  farbkonv fv[2];
  unsigned char fiw[256],ifiw;
  FILE * fp;
  int i1,i2;
  unsigned int r,g,b,wfb,bfb;
  char buf[256];
  /* open file and load colors */
  if (clm==NULL) {
    clm=colm;
    if (clm==NULL) {
      if (colv==1) {clm=GREY256_FILE;} else {clm=RGB256_FILE;}
    }
  } else {
    if (colm!=NULL) {free(colm);}
    if (strcmp(clm,RGB256)==0) {colm=NULL; colv=0; clm=RGB256_FILE;}
    else if (strcmp(clm,GREY256)==0) {colm=NULL; colv=1; clm=GREY256_FILE;}
    else {colm=strdup(clm);}
  }
  memset(fx,0,sizeof(fx));
  if ((fp=fopen(clm,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Can't open RGB file: \"%s\"",clm);
    if (colm!=NULL) {free(colm); colm=NULL;}
    return(-1);
  }
  i2=0;
  while (fgets(buf,sizeof(buf)-1,fp)!=NULL) {
    if ((*buf<'0') || (*buf>'9')) {continue;}
    sscanf(buf,"%d=%d %d %d",&i1,&r,&g,&b);
    if ((i1>255) || (i1<0)) {continue;}
    r&=0x3f; g&=0x3f; b&=0x3f;
    fx[i1].r=(unsigned char)r;
    fx[i1].g=(unsigned char)g;
    fx[i1].b=(unsigned char)b;
    if ((r!=g) || (r!=b)) {i2++;}
  }
  fclose(fp);
  if (i2>0) {col_grey=1;} else {col_grey=2;}
  /* find best white and black pixel */
  _rgb_white=_rgb_black=0;
  wfb=0;
  bfb=64;  /* 6 bits for color values */
  fv[0].r=fv[0].g=fv[0].b=64;
  fv[1].r=fv[1].g=fv[1].b=0;
  memset(fiw,0,sizeof(fiw)); ifiw=0;
  for (i1=0;i1<256;i1++) {
    r=fx[i1].r; g=fx[i1].g; b=fx[i1].b;
    if ((r>=wfb) && (g>=wfb) && (b>=wfb)) {  /* best white pixel */
      if (r+g+b>(unsigned int)(fv[1].r+fv[1].g+fv[1].b)) {
        _rgb_white=i1; wfb=(r<g?(r<b?r:b):(g<b?g:b));
        fv[1].r=r; fv[1].g=g; fv[1].b=b;
      }
    }
    if ((r<=bfb) && (g<=bfb) && (b<=bfb)) {  /* best black pixel */
      if (r+g+b<(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {
        _rgb_black=i1; bfb=(r>g?(r>b?r:b):(g>b?g:b));
        fv[0].r=r; fv[0].g=g; fv[0].b=b;
        ifiw=0;
      } else if (r+g+b==(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {  /* same black pixel */
        fiw[ifiw++]=(unsigned char)i1;
      }
    }
  }
  for (i1=0;i1<ifiw;i1++) {  /* make same black pixels lighter */
    if (fx[(int)fiw[i1]].r<63) {fx[(int)fiw[i1]].r++;}
    else if (fx[(int)fiw[i1]].g<63) {fx[(int)fiw[i1]].g++;}
    else if (fx[(int)fiw[i1]].b<63) {fx[(int)fiw[i1]].b++;}
  }
  /* find best dark pixel */
  _rgb_dark=0;
  bfb=64;  /* 6 bits for color values */
  fv[0].r=fv[0].g=fv[0].b=64;
  for (i1=0;i1<256;i1++) {
    if (i1==_rgb_black) {continue;}
    r=fx[i1].r; g=fx[i1].g; b=fx[i1].b;
    if ((r<=bfb) && (g<=bfb) && (b<=bfb)) {  /* best dark pixel */
      if (r+g+b<(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {
        _rgb_dark=i1; bfb=(r>g?(r>b?r:b):(g>b?g:b));
        fv[0].r=r; fv[0].g=g; fv[0].b=b;
      }
    }
  }
  brightness(99);
  return(0);
} /* Ende load_colormap */


void set_font(unsigned char * fnt,int fntw,int fnth) {
/* sets font used by draw_text()
** 1.arg: pointer to font
** 2.+3.arg: font size (width,height) in pixels
*/
  deffont=fnt;
  deffont_w=fntw;
  deffont_h=fnth;
  if (deffont==NULL) {fntw=fnth=8;}  /* default font */
} /* Ende set_font */


grafik * create_grafik(int w,int h) {
/* create a graphic
** 1.+2.arg: width and height of the graphic
** return value: pointer to graphic
**               or NULL = error
*/
  grafik * grf;
  if ((w<=0) || (h<=0)) {strcpy(errmsg,"Wrong width or height of graphic."); return(NULL);}
  if ((grf=malloc(sizeof(grafik)))==NULL) {strcpy(errmsg,"Can't allocate memory!"); return(NULL);}
  if ((grf->pxm=malloc(sizeof(unsigned char)*(w*h)))==NULL) {strcpy(errmsg,"Can't allocate memory!"); return(NULL);}
  memset(grf->pxm,_rgb_black,w*h);
  grf->width=w; grf->height=h;
  return(grf);
} /* Ende create_grafik */


grafik * load_grafik(const char * filename) {
/* load a graphic from a file,
** see _vga_grafik() in "vgag-bitmap.h" for supported formats.
** 1.arg: filename
** return value: pointer to graphic
**               or NULL = error
*/
  int w,h;
  unsigned char kv[256],buf[8];
  grafik * grf;
  FILE * ffp;

  /* convert graphic and allocate memory */
  if ((ffp=_vga_grafik(filename))==NULL) {return(NULL);}
  if (fread(buf,1,5,ffp)!=5) {snprintf(errmsg,sizeof(errmsg),"Error reading vga file of \"%s\"!",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  w=atoi(buf);
  if (fread(buf,1,5,ffp)!=5) {snprintf(errmsg,sizeof(errmsg),"Error reading vga file of \"%s\"!",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  h=atoi(buf);
  if ((grf=create_grafik(w,h))==NULL) {fclose(ffp); return(NULL);}

  /* read and convert colormap */
  if (_konv_map(kv,ffp)<0) {fclose(ffp); free_grafik(grf); return(NULL);}

  /* read and convert pixels */
  fread(grf->pxm,sizeof(unsigned char),grf->width*grf->height,ffp);
  fclose(ffp);
  w=grf->width*grf->height;
  for (h=0;h<w;h++) {grf->pxm[h]=kv[(int)grf->pxm[h]];}

  return(grf);
} /* Ende load_grafik */


grafik * text_to_grafik(int fc,int bc,int zpx,int ausr,const char * text) {
/* create a graphic containing text. Newlines are used for line breaks.
** 1.arg: text color
** 2.arg: background color
** 3.arg: pixel between lines
** 4.arg: 0=left adjusted or 1=centered or 2=right adjusted
** 5.arg: text
** return value: pointer to graphic
**               or NULL = error
*/
  int i1,i2,zlm,spm;
  grafik * grf;
  const char * p1,* p2;
  if (text==NULL) {strcpy(errmsg,"text_to_grafik: text pointer NULL"); return(NULL);}
  _draw_fontXxY(NULL,0,0,0,"",0,deffont,&deffont_w,&deffont_h,0,0);
  if (deffont_h+zpx<=0) {zpx=-deffont_h+1;}
  zlm=spm=0;
  for (p1=text,p2=strpbrk(p1,"\n\\");;p2=strpbrk(p2,"\n\\")) {
    if (p2==NULL) {p2=p1+strlen(p1);}
    if ((p2[0]=='\\') && (p2[1]!='n')) {p2++; continue;}
    i1=p2-p1;
    if (i1>zlm) {zlm=i1;}
    spm++;
    if (*p2=='\0') {break;}
    if (*p2=='\\') {p2++;}
    p1=p2+1; p2=p1;
  }
  if ((grf=create_grafik(zlm*deffont_w,spm*(deffont_h+zpx)-zpx))==NULL) {return(NULL);}
  spm=0;
  for (p1=text,p2=strpbrk(p1,"\n\\");;p2=strpbrk(p2,"\n\\")) {
    if (p2==NULL) {p2=p1+strlen(p1);}
    if ((p2[0]=='\\') && (p2[1]!='n')) {p2++; continue;}
    i1=p2-p1;
    if (ausr==1) {
      i2=((zlm-i1)/2)*deffont_w;
    } else if (ausr==2) {
      i2=(zlm-i1)*deffont_w;
    } else {i2=0;}
    _draw_fontXxY(grf,fc,i2,spm*(deffont_h+zpx),p1,i1,deffont,&deffont_w,&deffont_h,RGB_FULL,bc);
    spm++;
    if (*p2=='\0') {break;}
    if (*p2=='\\') {p2++;}
    p1=p2+1; p2=p1;
  }
  return(grf);
} /* Ende text_to_grafik */


int save_grafik_as_ppm(grafik * grf,const char * filename) {
/* save graphic box as ppm file (P6) - e.g. for screenshot
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.arg: filename to save ppm graphic
** return value: 0=OK or -1=error
*/
  return(_save_grf_ppm(grf,filename,6));
} /* Ende save_grafik_as_ppm */


int save_grafik_as_ppm3(grafik * grf,const char * filename) {
/* save graphic box as ppm file (P3) - e.g. for screenshot
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.arg: filename to save ppm graphic
** return value: 0=OK or -1=error
*/
  return(_save_grf_ppm(grf,filename,3));
} /* Ende save_grafik_as_ppm3 */


void free_grafik(grafik * grf) {
/* free and destroy a graphic
** 1.arg: pointer to graphic, created with create_grafik() or load_grafik()
*/
  if (grf==NULL) {return;}
  free(grf->pxm);
  free(grf);
} /* Ende free_grafik */


void copy_grafik(grafik * dgrf,int xd,int yd,grafik * sgrf,int xs,int ys,int ws,int hs,int trsp) {
/* copy a (part of a) graphic into another one
** 1.arg: pointer to destination graphic or NULL=backbuffer
** 2.+3.arg: destination coordinates
** 4.arg: pointer to source graphic or NULL=backbuffer
** 5.+6.arg: source coordinates
** 7.+8.arg: width and height to copy from source coordinates of source graphic
**           if width=0 or height=0, then the whole width or height is used
** 9.arg: RGB_FULL for all copying
**        RGB_TRANS for transparent copying
*/
  int w,h;
  unsigned char * dptr,* sptr;
  if (dgrf==sgrf) {return;}
  if (dgrf==NULL) {dgrf=backpuffer;}
  if (sgrf==NULL) {sgrf=backpuffer;}
  if ((xs>=sgrf->width) || (ys>=sgrf->height)) {return;}
  if ((xd>=dgrf->width) || (yd>=dgrf->height)) {return;}
  if ((ws<1) || (xs+ws>sgrf->width)) {ws=sgrf->width-xs;}
  if ((hs<1) || (ys+hs>sgrf->height)) {hs=sgrf->height-ys;}
  if (xd+ws>dgrf->width) {ws=dgrf->width-xd;}
  if (yd+hs>dgrf->height) {hs=dgrf->height-yd;}
  if (xs<0) {ws+=xs; xs=0;}
  if (ys<0) {hs+=ys; ys=0;}
  if (xd<0) {ws+=xd; xs-=xd; xd=0;}
  if (yd<0) {hs+=yd; ys-=yd; yd=0;}
  for (h=0;h<hs;h++) {
    dptr=dgrf->pxm+(yd+h)*dgrf->width+xd;
    sptr=sgrf->pxm+(ys+h)*sgrf->width+xs;
    for (w=0;w<ws;w++) {
      if ((trsp==RGB_FULL) || (*sptr!=_rgb_black)) {*dptr=*sptr;}
      dptr++; sptr++;
    }
  }
} /* Ende copy_grafik */


grafik * rotate_grafik(grafik * dgrf,grafik * sgrf,int deg,int * nx,int * ny,int * nw,int * nh) {
/* rotate graphic at certain degrees
** 1.arg: pointer to destination graphic
**        in here is put the rotated source graphic
** 2.arg: pointer to source graphic
** 3.arg: degrees to rotate clockwise (-360 to 360)
** 4.+5.arg: value arguments: x,y start position to rotate of source graphic
**           and return arguments:
**             address for relative x/y coordinate to put destination
**             graphic out to backbuffer/graphic:
**               add x to your origin x coordinate
**               add y to your origin y coordinate
** 6.+7.arg: value arguments: width and height to rotate of source graphic
**                            or 0=whole width/height
**           and return arguments:
**             address for width and height of the size of the rotated graphic
**             in the destination graphic
** return value: pointer to 1.arg
**               or NULL = error
*/
  int sxm,sym,dxm,dym,i0,i1,i2,i3,i4,ix,iy,jx,jy,tx,ty;
  float r0,a0,d0,degpi;
  if ((sgrf==NULL) || (dgrf==NULL)) {strcpy(errmsg,"you can't use backbuffer with rotate_grafik!"); return(NULL);}
  if (dgrf==sgrf) {strcpy(errmsg,"both graphics are the same in rotate_grafik!"); return(NULL);}
  if ((nx==NULL) || (ny==NULL) || (nw==NULL) || (nh==NULL)) {strcpy(errmsg,"arguments NULL"); return(NULL);}
  CLEAR_BOX(dgrf,_rgb_black);
  if ((*nx>=sgrf->width) || (*ny>=sgrf->height)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if ((*nw<1) || (*nx+*nw>sgrf->width)) {*nw=sgrf->width-*nx;}
  if ((*nh<1) || (*ny+*nh>sgrf->height)) {*nh=sgrf->height-*ny;}
  if (*nx<0) {*nw+=*nx; *nx=0;}
  if (*ny<0) {*nh+=*ny; *ny=0;}
  if ((*nw<0) || (*nh<0)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  deg%=360;
  deg+=360;
  deg%=360;
  degpi=(float)deg/360.*PI2;  /* angle in Rad */
  /* source middle */
  sxm=*nw/2;
  sym=*nh/2;
  /* destination middle: find the pixel which is farest away from middle */
  /* point 0/0 */
  r0=sqrt(sxm*sxm+sym*sym);  /* distance to middle */
  a0=asin(sym/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i1=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i2=(int)d0;                /* y distance from middle */
  /* point 0/(*nh)-1 */
  i3=*nh-1-sym;
  a0=asin(-i3/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i3=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i4=(int)d0;                /* y distance from middle */
  dxm=(i1>i3?i1:i3);  /* calculated destination middle x */
  dym=(i2>i4?i2:i4);  /* calculated destination middle y */
  i1=dgrf->width/2;
  if (dxm>i1) {dxm=i1;}  /* real destination middle and distance x */
  i1=dgrf->height/2;
  if (dym>i1) {dym=i1;}  /* real destination middle and distance y */
  /* put middle */
  dgrf->pxm[dym*dgrf->width+dxm]=sgrf->pxm[(*ny+sym)*sgrf->width+*nx+sxm];
  /* put rest
  ** related to source middle (0/0) take every loop (variable i0) 4 lines:
  **  - upper source pixel line (1.loop: -1/-1 to  1/-1
  **                             2.loop: -2/-2 to  2/-2
  **                             ... )
  **  - lower source pixel line (1.loop: -1/1  to  1/1
  **                             2.loop: -2/2  to  2/2
  **                             ... )
  **  - left source pixel line  (1.loop: -1/-1 to -1/1
  **                             2.loop: -2/-2 to -2/2
  **                             ... )
  **  - right source pixel line (1.loop:  1/-1 to  1/1
  **                             2.loop:  2/-2 to  2/2
  **                             ... )
  */
  i0=1;  /* distance from source middle */
  do {
    i1=0;
    if (sxm<i0) {i2=-sxm;} else {i2=-i0;}  /* left */
    if (*nw-1-sxm<i0) {i3=*nw-1-sxm;} else {i3=i0;}  /* right */
    if (i0<=sym) {  /* upper source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4]==_rgb_black) {continue;}
        r0=sqrt(i4*i4+i0*i0);
        a0=asin(i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        }
      }
      i1=1;
    }
    if (i0<=*nh-1-sym) {  /* lower source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4]==_rgb_black) {continue;}
        r0=sqrt(i4*i4+i0*i0);
        a0=asin(-i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        }
      }
      i1=1;
    }
    if (sym<i0) {i2=-sym;} else {i2=-i0;}  /* up */
    if (*nh-1-sym<i0) {i3=*nh-1-sym;} else {i3=i0;}  /* down */
    if (i0<=sxm) {  /* left source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0]==_rgb_black) {continue;}
        r0=sqrt(i0*i0+i4*i4);
        a0=asin(-i4/r0);
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        }
      }
      i1=1;
    }
    if (i0<=*nw-1-sxm) {  /* right source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0]==_rgb_black) {continue;}
        r0=sqrt(i0*i0+i4*i4);
        a0=asin(-i4/r0);
        a0=-a0+PI;  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        }
      }
      i1=1;
    }
    i0++;
  } while (i1==1);
  *nx+=sxm-dxm;
  *ny+=sym-dym;
  *nw=dxm*2+1;
  if (*nw>dgrf->width) {*nw=dgrf->width;}
  *nh=dym*2+1;
  if (*nh>dgrf->height) {*nh=dgrf->height;}
  return(dgrf);
} /* Ende rotate_grafik */


grafik * zoom_grafik(grafik * dgrf,grafik * sgrf,float xmul,float ymul,int * nx,int * ny,int * nw,int * nh) {
/* zoom a graphic greater or smaller
** 1.arg: pointer to destination graphic
**        in here is put the zoomed source graphic
** 2.arg: pointer to source graphic
** 3.arg: horizontal zooming (x position)
** 4.arg: vertical zooming (y position)
** 5.+6.arg: value arguments: x,y start position to zoom of source graphic
**           and return arguments:
**             address for relative x/y coordinate to put destination
**             graphic out to backbuffer/graphic:
**               add x to your origin x coordinate
**               add y to your origin y coordinate
** 7.+8.arg: value arguments: width and height to zoom of source graphic
**                            or 0=whole width/height
**           and return arguments:
**             address for width and height of the size of the zoomed graphic
**             in the destination graphic
** return value: pointer to 1.arg
**               or NULL = error
*/
  const int tw=1000;
  int sxm,sym,dxm,dym,txm,tym,sw,i_z,i_a,i_n,vp,bp,hp,hw,i1,i2,ii;
  int fb[256],brk;
  grafik * tgrf;
  if ((sgrf==NULL) || (dgrf==NULL)) {strcpy(errmsg,"you can't use backbuffer with zoom_grafik!"); return(NULL);}
  if (dgrf==sgrf) {strcpy(errmsg,"both graphics are the same in zoom_grafik!"); return(NULL);}
  if ((nx==NULL) || (ny==NULL) || (nw==NULL) || (nh==NULL)) {strcpy(errmsg,"arguments NULL"); return(NULL);}
  CLEAR_BOX(dgrf,_rgb_black);
  if ((*nx>=sgrf->width) || (*ny>=sgrf->height)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if ((*nw<1) || (*nx+*nw>sgrf->width)) {*nw=sgrf->width-*nx;}
  if ((*nh<1) || (*ny+*nh>sgrf->height)) {*nh=sgrf->height-*ny;}
  if (*nx<0) {*nw+=*nx; *nx=0;}
  if (*ny<0) {*nh+=*ny; *ny=0;}
  if ((*nw<0) || (*nh<0)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if (xmul<.01) {xmul=.01;}
  if (ymul<.01) {ymul=.01;}

  /* source middle of selected part */
  sxm=*nw/2;
  sym=*nh/2;
  /* destination middle */
  i1=sxm;
  if (sxm*2<*nw) {i1++;}
  i1=(int)((float)i1*xmul)*2;
  if (sxm*2<*nw) {i1--;}
  dxm=i1/2;
  if (dxm>dgrf->width/2) {dxm=dgrf->width/2;}
  i1=sym;
  if (sym*2<*nh) {i1++;}
  i1=(int)((float)i1*ymul)*2;
  if (sym*2<*nh) {i1--;}
  dym=i1/2;
  if (dym>dgrf->height/2) {dym=dgrf->height/2;}
  /* temp middle */
  if (sxm*2<*nw) {i1=(dxm+1)*2-1;} else {i1=dxm*2;}
  if (i1<1) {i1=1;}
  i2=*nh;
  if (i2<1) {i2=1;}
  if ((tgrf=create_grafik(i1,i2))==NULL) {return(NULL);}
  txm=tgrf->width/2;
  tym=tgrf->height/2;
  /* x direction */
  if ((sw=(int)((float)tw/xmul))<=0) {sw=1;}
  /* zoom x direction right from source to temp */
  brk=0;
  for (i_z=0;i_z<*nh;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    i_a=sxm; i_n=txm;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=*nw) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a+vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=*nw) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a+vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<tgrf->width) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n++]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom x direction left from source to temp */
  brk=0;
  for (i_z=0;i_z<*nh;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    if (sxm*2==*nw) {i_a=sxm-1;} else {i_a=sxm;}
    if (txm*2==tgrf->width) {i_n=txm-1;} else {i_n=txm;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a-vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a-vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n--]=hp; fb[hp]-=sw;}
    }
  }
  /* y direction */
  if ((sw=(int)((float)tw/ymul))<=0) {sw=1;}
  /* zoom y direction down from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    i_a=tym; i_n=dym;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<dgrf->height) {dgrf->pxm[(i_n++)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom y direction up from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    if (tym*2==tgrf->height) {i_a=tym-1;} else {i_a=tym;}
    if (dym*2==dgrf->height) {i_n=dym-1;} else {i_n=dym;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {dgrf->pxm[(i_n--)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  free_grafik(tgrf);
  *nx+=sxm-dxm;
  *ny+=sym-dym;
  *nw=dxm*2+1;
  if (*nw>dgrf->width) {*nw=dgrf->width;}
  *nh=dym*2+1;
  if (*nh>dgrf->height) {*nh=dgrf->height;}
  return(dgrf);
} /* Ende zoom_grafik */


int get_pixel(grafik * grf,int x,int y) {
/* get pixel color of one pixel
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: pixel coordinates X,Y
** return value: color index
*/
  if (grf==NULL) {grf=backpuffer;}
  if ((x<0) || (x>=grf->width) || (y<0) || (y>=grf->height)) {return(_rgb_black);}
  return((int)grf->pxm[y*grf->width+x]);
} /* Ende get_pixel */


void draw_text(grafik * graf,int c,int x,int y,const char * s,int n,int trsp) {
/* draws text
** 1.arg: pointer to grafik or virtual context or NULL=backbuffer
** 2.arg: color index
** 3.+4.arg: target coordinates
** 5.+6.arg: string und string length
** 7.arg: RGB_FULL for full text
**        RGB_TRANS for transparent text
*/
  _draw_fontXxY(graf,c,x,y,s,n,deffont,&deffont_w,&deffont_h,trsp,_rgb_black);
} /* Ende draw_text */


void draw_pixel(grafik * grf,int x,int y,int cl) {
/* draw pixel
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: pixel coordinates X,Y
** 4.arg: color index
*/
  if (grf==NULL) {grf=backpuffer;}
  if ((x<0) || (x>=grf->width) || (y<0) || (y>=grf->height)) {return;}
  grf->pxm[y*grf->width+x]=(unsigned char)cl;
} /* Ende draw_pixel */


void draw_line(grafik * grf,int x1,int y1,int x2,int y2,int cl) {
/* draw line
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: begin of target coordinates X,Y
** 4.+5.arg: end of target coordinates X,Y
** 6.arg: color index
*/
  float f1,f2;
  int i1,i2,xa,xe,ya,ye;
  if (grf==NULL) {grf=backpuffer;}
  xa=x2-x1; if (xa<0) {xa=-xa;}
  ya=y2-y1; if (ya<0) {ya=-ya;}
  if ((xa==0) && (ya==0)) {draw_pixel(grf,x1,y1,cl); return;}
  if (xa>=ya) {
    if (x2<x1) {xa=x2; xe=x1; ya=y2; ye=y1;} else {xa=x1; xe=x2; ya=y1; ye=y2;}
    f1=(float)(ye-ya)/(float)(xe-xa);
    f2=0.;
    for (i1=xa;i1<=xe;i1++) {
      i2=ya+(int)f2;
      if ((i1>=0) && (i1<grf->width) && (i2>=0) && (i2<grf->height)) {
        grf->pxm[i2*grf->width+i1]=(unsigned char)cl;
      }
      f2+=f1;
    }
  } else {
    if (y2<y1) {ya=y2; ye=y1; xa=x2; xe=x1;} else {ya=y1; ye=y2; xa=x1; xe=x2;}
    f1=(float)(xe-xa)/(float)(ye-ya);
    f2=0.;
    for (i1=ya;i1<=ye;i1++) {
      i2=xa+(int)f2;
      if ((i2>=0) && (i2<grf->width) && (i1>=0) && (i1<grf->height)) {
        grf->pxm[i1*grf->width+i2]=(unsigned char)cl;
      }
      f2+=f1;
    }
  }
} /* Ende draw_line */


void draw_circle(grafik * grf,int x,int y,int r,int cl,int fill) {
/* draws circle, centered at x,y
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: target coordinates - middle of circle
** 4.arg: radius
** 5.arg: color index
** 6.arg: 0=not fill, 1=fill it
*/
#define MALE_PIX_   if ((i1>=0) && (i1<grf->width) && (i2>=0) && (i2<grf->height)) { \
                      grf->pxm[i2*grf->width+i1]=(unsigned char)cl; \
                    }
#define MALE_LINE_   if (i1<0) {i3+=i1; i1=0;} \
                     if (i1+i3>grf->width) {i3=grf->width-i1;} \
                     if ((i2>=0) && (i2<grf->height) && (i3>0)) { \
                       memset(grf->pxm+i2*grf->width+i1,cl,i3); \
                     }
  int i1,i2,i3,dx,dy,dd;
  dx=r; dy=0; dd=1-r;
  if (grf==NULL) {grf=backpuffer;}
  if (r<1) {draw_pixel(grf,x,y,cl); return;}
  if (fill==0) {
    i1=x-dx; i2=y; MALE_PIX_;
    i1=x+dx; i2=y; MALE_PIX_;
    i1=x; i2=y-dx; MALE_PIX_;
    i1=x; i2=y+dx; MALE_PIX_;
  } else {
    i1=x; i2=y-dx; MALE_PIX_;
    i1=x; i2=y+dx; MALE_PIX_;
    i1=x-dx; i2=y; i3=2*dx+1; MALE_LINE_;
  }
  while (dy<dx) {
    if (dd<0) {dd+=dy*2+3;} else {dd+=dy*2-dx*2+5; dx--;}
    dy++;
    if (fill==0) {
      i1=x-dx; i2=y-dy; MALE_PIX_;
      i1=x-dx; i2=y+dy; MALE_PIX_;
      i1=x+dx; i2=y-dy; MALE_PIX_;
      i1=x+dx; i2=y+dy; MALE_PIX_;
      i1=x-dy; i2=y-dx; MALE_PIX_;
      i1=x-dy; i2=y+dx; MALE_PIX_;
      i1=x+dy; i2=y-dx; MALE_PIX_;
      i1=x+dy; i2=y+dx; MALE_PIX_;
    } else {
      i1=x-dx; i2=y-dy; i3=2*dx+1; MALE_LINE_;
      i1=x-dx; i2=y+dy; i3=2*dx+1; MALE_LINE_;
      i1=x-dy; i2=y-dx; i3=2*dy+1; MALE_LINE_;
      i1=x-dy; i2=y+dx; i3=2*dy+1; MALE_LINE_;
    }
  }
#undef MALE_LINE_
#undef MALE_PIX_
} /* Ende draw_circle */


void draw_fillbox(grafik * grf,int x,int y,int w,int h,int cl) {
/* draw filled rectangle
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: target coordinates
** 4.+5.arg: width and height
**           if width=0 or height=0, then the whole width or height is used
** 6.arg: color index
*/
  int i1;
  if (grf==NULL) {grf=backpuffer;}
  if ((x>=grf->width) || (y>=grf->height)) {return;}
  if ((w<1) || (x+w>grf->width)) {w=grf->width-x;}
  if ((h<1) || (y+h>grf->height)) {h=grf->height-y;}
  if (x<0) {w+=x; x=0;}
  if (y<0) {h+=y; y=0;}
  for (i1=0;i1<h;i1++) {
    memset(grf->pxm+(y+i1)*grf->width+x,cl,w);
  }
} /* Ende draw_fillbox */


film * create_film(int anz) {
/* create a new film
** 1.arg: number of graphical sequences
** return value: pointer to created film or NULL=error
*/
  film * flm;
  int i1;
  if (anz<=0) {strcpy(errmsg,"create_film: wrong parameter"); return(NULL);}
  if ((flm=malloc(sizeof(film)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->func=NULL;
  flm->step=0UL;
  flm->id=0;
  flm->x=flm->y=0;
  if ((flm->_pixold=malloc(sizeof(grafik *)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->_pixold[0]=NULL;
  if ((flm->_pixnew=malloc(sizeof(grafik *)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->_pixnew[0]=NULL;
  if ((flm->aktiv=malloc(sizeof(int)*(anz+1)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->aktiv[0]=0;
  if ((flm->bildnr=malloc(sizeof(int)*(anz+1)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->bildnr[0]=0;
  if ((flm->bild=malloc(sizeof(film_bild **)*(anz+2)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->bild[0]=NULL;
  for (i1=1;i1<=anz;i1++) {
    if ((flm->bild[i1]=malloc(sizeof(film_bild *)*2))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
    flm->bild[i1][0]=NULL;
    flm->bild[i1][1]=NULL;
    flm->aktiv[i1]=FILM_ACT_OK;
    flm->bildnr[i1]=0;
  }
  flm->bild[anz+1]=NULL;
  return(flm);
} /* Ende create_film */


void free_film(film * flm) {
/* free and destroy a film
** 1.arg: pointer to created film
*/
  int f1,f2;
  if (flm==NULL) {return;}
  free(flm->aktiv);
  free(flm->bildnr);
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    for (f2=1;flm->bild[f1][f2]!=NULL;f2++) {free(flm->bild[f1][f2]);}
    free(flm->bild[f1]);
  }
  free(flm->_pixold);
  for (f1=0;flm->_pixnew[f1]!=NULL;f1++) {free(flm->_pixnew[f1]);}
  free(flm->_pixnew);
  free(flm->bild);
} /* Ende free_film */


int add_bild_to_film(film * flm,film_bild * add_bld,int row) {
/* appends a bild in a film to a certain graphical sequence (row)
** 1.arg: pointer to film
** 2.arg: pointer to filled bild
** 3.arg: graphical sequence number (from 1 to ?, where ? is the argument of
**                                   create_film())
** return value: position of bild in that row (from 1 to ??)
**               or -1=error
*/
  int f1,f2;
  grafik * gmrk;
  if ((flm==NULL) || (add_bld==NULL) || (row<=0)) {strcpy(errmsg,"add_bild_to_film: empty parameter"); return(-1);}
  gmrk=add_bld->pic;
  if (add_bld->pic!=NULL) {  /* with graphic */
    for (f1=0;flm->_pixold[f1]!=NULL;f1++) {  /* copy already allocated? */
      if (flm->_pixold[f1]==(grafik *)1) {continue;}
      if (add_bld->pic==flm->_pixold[f1]) {
        if ((add_bld->pic->width==flm->_pixnew[f1]->width) \
        && (add_bld->pic->height==flm->_pixnew[f1]->height) \
        && (memcmp(add_bld->pic->pxm,flm->_pixnew[f1]->pxm,add_bld->pic->width*add_bld->pic->height)==0)) {
          break;
        } else {
          flm->_pixold[f1]=(grafik *)1;
        }
      }
    }
    if (flm->_pixold[f1]==NULL) {  /* new graphic */
      if ((flm->_pixold=realloc(flm->_pixold,sizeof(grafik *)*(f1+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      flm->_pixold[f1]=add_bld->pic;
      flm->_pixold[f1+1]=NULL;
      if ((flm->_pixnew=realloc(flm->_pixnew,sizeof(grafik *)*(f1+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      if ((flm->_pixnew[f1]=create_grafik(GRAFIK_WIDTH(add_bld->pic),GRAFIK_HEIGHT(add_bld->pic)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      copy_grafik(flm->_pixnew[f1],0,0,add_bld->pic,0,0,0,0,RGB_FULL);
      flm->_pixnew[f1+1]=NULL;
    }
    add_bld->pic=flm->_pixnew[f1];
  }
  for (f1=1;flm->bild[f1]!=NULL;f1++) {  /* find row */
    if (f1==row) {break;}
  }
  if (flm->bild[f1]==NULL) {strcpy(errmsg,"add_bild_to_film: graphical seq. number to great"); return(-1);}
  for (f2=1;flm->bild[f1][f2]!=NULL;f2++) {;}  /* find last bild */
  if ((flm->bild[f1]=realloc(flm->bild[f1],sizeof(film_bild *)*(f2+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
  if ((flm->bild[f1][f2]=malloc(sizeof(film_bild)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
  memmove(flm->bild[f1][f2],add_bld,sizeof(film_bild));
  add_bld->pic=gmrk;
  flm->bild[f1][f2+1]=NULL;
  return(f2);
} /* Ende add_bild_to_film */


void reset_film(film * flm,void * fpt,int fres) {
/* reset a film ready to start from the beginning
** 1.arg: pointer to film
** 2.arg: pointer to data for film function (2.arg) or NULL
** 3.arg: value for film function (3.arg) or 0
*/
  int f1;
  if (flm==NULL) {return;}
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    flm->aktiv[f1]=FILM_ACT_OK;
    flm->bildnr[f1]=0;
  }
  flm->step=0UL;
  if (flm->func!=NULL) {flm->func(flm,fpt,fres);}
} /* Ende reset_film */


unsigned long copy_film(grafik * gto,film * flm,void * fpt) {
/* draws next step of the film to graphic
** 1.arg: graphic to which film is drawn or NULL=backbuffer
** 2.arg: pointer to film
** 3.arg: pointer to data for film function (2.arg) or NULL
**        (the film function - if any - is called with their 3.arg=0)
** return value: step number of film
*/
  int f1;
  film_bild * bld;
  if (flm==NULL) {return(0UL);}
  flm->step++;
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    if (flm->aktiv[f1]!=FILM_ACT_OK) {continue;}
    if (flm->bildnr[f1]<1) {flm->bildnr[f1]=0;}
    bld=flm->bild[f1][flm->bildnr[f1]];
    while ((flm->bildnr[f1]==0) || (bld->count>=bld->count_to)) {
      bld=flm->bild[f1][++flm->bildnr[f1]];
      if (bld!=NULL) {bld->count=0; break;}
      else if (flm->bildnr[f1]==1) {flm->bildnr[f1]=0; break;}
      else {flm->bildnr[f1]=0;}
    }
    if (flm->bildnr[f1]>=1) {bld->count++;}
  }
  if (flm->func!=NULL) {flm->func(flm,fpt,0);}
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    if (flm->aktiv[f1]==FILM_ACT_DEAD) {continue;}
    if (flm->bildnr[f1]<1) {continue;}
    bld=flm->bild[f1][flm->bildnr[f1]];
    if (bld->pic!=NULL) {
      copy_grafik(gto,flm->x+bld->x,flm->y+bld->y,bld->pic,0,0,0,0,bld->ft);
    }
    if ((bld->count==1) && (flm->aktiv[f1]==FILM_ACT_OK)) {
      if (bld->wnr>0) {  /* start wave (once) */
        play_wave(bld->wnr,bld->wch,0,bld->wvol);
      } else if (bld->wnr==-1) {  /* stop wave */
        stop_wave(bld->wch,bld->wvol);
      }
    }
  }
  return(flm->step);
} /* Ende copy_film */


sprite * create_sprite() {
/* create a new sprite, can be filled with add_grafik_to_sprite()
** return value: pointer to sprite
**               or NULL = error
*/
  sprite * spr;
  if ((spr=malloc(sizeof(sprite)))==NULL) {strcpy(errmsg,"Can't allocate memory!"); return(NULL);}
  memset(spr,0,sizeof(sprite));
  return(spr);
} /* Ende create_sprite */


sprite * load_sprite(const char * filename) {
/* load a sprite from a file
** 1.arg: filename of sprite file
** return value: pointer to sprite
**               or NULL = error
** Note: a sprite file is a text file containing:
**         1.line:           "[SPRITE]"
**         following lines:  <graphic file>;<maxrunde>[;ZOOM=<x-zoom>,<y-zoom>][;ROTATE=<degrees>]
**
**       Comment lines begin with `#'.
**       <graphic file>   is loaded from the path, where the sprite file is;
**       <maxrunde>   is an integer how many loops this graphic is shown;
**       ZOOM=<x-zoom>,<y-zoom>   is optional, how to zoom graphic;
**       ROTATE=<degrees>   is optional, how to rotate graphic.
**
**       Example:
**         sprite file is "/usr/local/game/mygame/sprites/test.sprite"
**         contains following lines:
**            [SPRITE]
**            pict1.vga;5
**            ../vga/pict2.vga;17;ROTATE=90
**            /usr/vga/pict3.vga;10;ZOOM=1.5,1.0
**            pict4.vga;3;ZOOM=0.5,0.5;ROTATE=270
**            pict5.vga;0
**
**         --> 1.line is "[SPRITE]"
**             2.line shows graphic "/usr/local/game/mygame/sprites/pict1.vga"
**                    for 5 loops
**             3.line shows graphic "/usr/local/game/mygame/vga/pict2.vga"
**                    rotated at 90 degrees
**                    for 17 loops
**             4.line shows graphic "/usr/vga/pict3.vga"
**                    zoomed 1.5 in x direction and original in y direction
**                    for 10 loops
**             5.line shows graphic "/usr/local/game/mygame/sprites/pict4.vga"
**                    zoomed 0.5 in x direction and 0.5 in y direction
**                    and rotated at 270 degrees
**                    for 3 loops
**             6.line shows graphic "/usr/local/game/mygame/sprites/pict5.vga"
**                    not at all
*/
  FILE * ffp;
  char buf[1024],bfgra[1024],* p1,* p2;
  int zl,maxrunde,vorpfad;
  sprite * spr;
  grafik * grf;
  if (filename==NULL) {strcpy(errmsg,"load_sprite: empty parameter"); return(NULL);}
  if ((ffp=fopen(filename,"r"))==NULL) {snprintf(errmsg,sizeof(errmsg),"Error reading sprite file \"%s\"",filename); return(NULL);}
  if ((p1=strrchr(filename,'/'))!=NULL) {vorpfad=p1-filename+1;} else {vorpfad=0;}
  if ((spr=create_sprite())==NULL) {return(NULL);}
  /* read in each line of sprite file */
  if ((fgets(buf,sizeof(buf),ffp)==NULL) || (strncmp(buf,"[SPRITE]",8)!=0)) {
    snprintf(errmsg,sizeof(errmsg),"load_sprite: \"%s\" seems not to be a vaild sprite file",filename);
    fclose(ffp);
    free_sprite(spr);
    return(NULL);
  }
  zl=1;
  while (fgets(buf,sizeof(buf),ffp)!=NULL) {
    zl++;
    if ((*buf=='#') || ((p1=strchr(buf,';'))==NULL)) {continue;}
    *p1='\0';
    if (*buf!='/') {
      snprintf(bfgra,sizeof(bfgra),"%.*s%s",vorpfad,filename,buf);
    } else {
      snprintf(bfgra,sizeof(bfgra),"%s",buf);
    }
    if ((grf=load_grafik(bfgra))==NULL) {
      snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot load graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
      fclose(ffp);
      free_sprite(spr);
      return(NULL);
    }
    maxrunde=atoi(p1+1);
    /* manipulate graphic if desired */
    if ((p2=strstr(p1+1,"ZOOM="))!=NULL) {  /* zoom graphic */
      grafik * gtmp;
      float xmul,ymul;
      int x,y,w,h;
      xmul=atof(p2+5);
      if ((p2=strchr(p2,','))!=NULL) {ymul=atof(p2+1);} else {ymul=1.;}
      if (xmul<.01) {xmul=.01;}
      if (ymul<.01) {ymul=.01;}
      x=y=w=h=0;
      if (((gtmp=create_grafik((int)((float)grf->width*xmul)+5,(int)((float)grf->height*ymul)+5))==NULL) \
      || (zoom_grafik(gtmp,grf,xmul,ymul,&x,&y,&w,&h)==NULL)) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot zoom graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(grf);
        if (gtmp!=NULL) {free_grafik(gtmp);}
        free_sprite(spr);
        return(NULL);
      }
      free_grafik(grf);
      if ((grf=create_grafik(w,h))==NULL) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot zoom graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(gtmp);
        free_sprite(spr);
        return(NULL);
      }
      copy_grafik(grf,0,0,gtmp,0,0,w,h,RGB_FULL);
      free_grafik(gtmp);
    }
    if ((p2=strstr(p1+1,"ROTATE="))!=NULL) {  /* rotate graphic */
      grafik * gtmp;
      int degree,x,y,w,h;
      degree=atoi(p2+7);
      x=y=w=h=0;
      if (((gtmp=create_grafik(grf->width>grf->height?grf->width:grf->height,grf->height>grf->width?grf->height:grf->width))==NULL) \
      || (rotate_grafik(gtmp,grf,degree,&x,&y,&w,&h)==NULL)) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot rotate graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(grf);
        if (gtmp!=NULL) {free_grafik(gtmp);}
        free_sprite(spr);
        return(NULL);
      }
      free_grafik(grf);
      if ((grf=create_grafik(w,h))==NULL) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot rotate graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(gtmp);
        free_sprite(spr);
        return(NULL);
      }
      copy_grafik(grf,0,0,gtmp,0,0,w,h,RGB_FULL);
      free_grafik(gtmp);
    }
    /* add graphic and maxrunde to sprite */
    if (add_grafik_to_sprite(spr,grf,0,0,0,0,maxrunde)<0) {
      snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot add graphic \"%s\" to sprite at line %d in sprite file \"%s\"",bfgra,zl,filename);
      fclose(ffp);
      free_grafik(grf);
      free_sprite(spr);
      return(NULL);
    }
    free_grafik(grf);
  }
  fclose(ffp);
  return(spr);
} /* Ende load_sprite */


sprite * rotate_sprite(sprite * spr,int deg) {
/* create new sprite from another existing sprite, rotated at certain degrees
** 1.arg: existing sprite
** 2.arg: degrees to rotate clockwise (-360 to 360)
** return value: pointer to new created sprite
**               or NULL = error
*/
  sprite * spr_neu;
  int i1,x,y,w,h;
  grafik * gr1,* gr2;
  if (spr==NULL) {strcpy(errmsg,"rotate_sprite: empty parameter"); return(NULL);}
  if ((spr_neu=create_sprite())==NULL) {return(NULL);}
  reset_sprite(spr);
  w=SPRITE_WIDTH(spr)>SPRITE_HEIGHT(spr)?SPRITE_WIDTH(spr):SPRITE_HEIGHT(spr);
  w++;
  if ((gr2=create_grafik(w,w))==NULL) {return(NULL);}
  for (i1=0;i1<spr->spr_anz;i1++) {
    gr1=spr->spr_ptr[i1].grf;
    w=GRAFIK_WIDTH(gr1)>GRAFIK_HEIGHT(gr1)?GRAFIK_WIDTH(gr1):GRAFIK_HEIGHT(gr1);
    if (w>GRAFIK_WIDTH(gr2)) {
      free_grafik(gr2);
      if ((gr2=create_grafik(w,w))==NULL) {return(NULL);}
    }
    x=y=w=h=0;
    if (rotate_grafik(gr2,gr1,deg,&x,&y,&w,&h)==NULL) {return(NULL);}
    if (add_grafik_to_sprite(spr_neu,gr2,0,0,w,h,spr->spr_ptr[i1].maxrunde)<0) {return(NULL);}
  }
  free_grafik(gr2);
  return(spr_neu);
} /* Ende rotate_sprite */


sprite * zoom_sprite(sprite * spr,float xmul,float ymul) {
/* create new sprite from another existing sprite, zoomed greater or smaller
** 1.arg: existing sprite
** 2.arg: horizontal zooming (x position)
** 3.arg: vertical zooming (y position)
** return value: pointer to new created sprite
**               or NULL = error
*/
  sprite * spr_neu;
  int i1,x,y,w,h;
  grafik * gr1,* gr2;
  if (spr==NULL) {strcpy(errmsg,"zoom_sprite: empty parameter"); return(NULL);}
  if ((spr_neu=create_sprite())==NULL) {return(NULL);}
  reset_sprite(spr);
  w=SPRITE_WIDTH(spr)>SPRITE_HEIGHT(spr)?SPRITE_WIDTH(spr):SPRITE_HEIGHT(spr);
  w++;
  if ((gr2=create_grafik(w,w))==NULL) {return(NULL);}
  for (i1=0;i1<spr->spr_anz;i1++) {
    gr1=spr->spr_ptr[i1].grf;
    w=GRAFIK_WIDTH(gr1)>GRAFIK_HEIGHT(gr1)?GRAFIK_WIDTH(gr1):GRAFIK_HEIGHT(gr1);
    if (w>GRAFIK_WIDTH(gr2)) {
      free_grafik(gr2);
      if ((gr2=create_grafik(w,w))==NULL) {return(NULL);}
    }
    x=y=w=h=0;
    if (zoom_grafik(gr2,gr1,xmul,ymul,&x,&y,&w,&h)==NULL) {return(NULL);}
    if (add_grafik_to_sprite(spr_neu,gr2,0,0,w,h,spr->spr_ptr[i1].maxrunde)<0) {return(NULL);}
  }
  free_grafik(gr2);
  return(spr_neu);
} /* Ende zoom_sprite */


int add_grafik_to_sprite(sprite * spr,grafik * grf,int x,int y,int w,int h,int maxrunde) {
/* add a graphic to a sprite
** 1.arg: loaded or created sprite
** 2.arg: loaded or created graphic
** 3.+4.arg: start coordinates of graphic to copy to sprite
** 5.+6.arg: width and height of graphic to copy to sprite or 0,0=whole graphic
** 7.arg: number of loops to show graphic
** return value: >0: position of graphic in sprite
**               -1: error
**
** The size of the first graphic which is added to a sprite is saved
** as original size (width and height). All following graphics, which of course
** can have other width and height, use this original size to correct their
** x,y-coordinates when you call next_sprite().
*/
  if ((spr==NULL) || (grf==NULL)) {strcpy(errmsg,"add_grafik_to_sprite: empty argument"); return(-1);}
  if (maxrunde<0) {maxrunde=0;}
  if (++spr->spr_anz==1) {  /* first graphic */
    if ((spr->spr_ptr=malloc(sizeof(*spr->spr_ptr)))==NULL) {
      strcpy(errmsg,"add_grafik_to_sprite: cannot allocate memory");
      spr->spr_anz=0;
      return(-1);
    }
  } else {
    if ((spr->spr_ptr=realloc(spr->spr_ptr,sizeof(*spr->spr_ptr)*spr->spr_anz))==NULL) {
      strcpy(errmsg,"add_grafik_to_sprite: cannot allocate memory");
      spr->spr_anz=0;
      return(-1);
    }
  }
  spr->spr_ptr[spr->spr_anz-1].maxrunde=maxrunde;
  if ((x>=grf->width) || (y>=grf->height)) {
    if ((spr->spr_ptr[spr->spr_anz-1].grf=create_grafik(1,1))==NULL) {
      strcpy(errmsg,"add_grafik_to_sprite: cannot allocate memory");
      spr->spr_anz--;
      return(-1);
    }
  } else {
    if ((w<1) || (x+w>grf->width)) {w=grf->width-x;}
    if ((h<1) || (y+h>grf->height)) {h=grf->height-y;}
    if (x<0) {w+=x; x=0;}
    if (y<0) {h+=y; y=0;}
    if ((spr->spr_ptr[spr->spr_anz-1].grf=create_grafik(w,h))==NULL) {
      strcpy(errmsg,"add_grafik_to_sprite: cannot allocate memory");
      spr->spr_anz--;
      return(-1);
    }
    copy_grafik(spr->spr_ptr[spr->spr_anz-1].grf,0,0,grf,x,y,w,h,RGB_FULL);
  }
  if (spr->spr_anz==1) {  /* first graphic */
    spr->w_orig=w;
    spr->h_orig=h;
  }
  return(spr->spr_anz);
} /* Ende add_grafik_to_sprite */


grafik * next_sprite(sprite * spr,int * xkorr,int * ykorr) {
/* get actual graphic of a sprite
** 1.arg: sprite
** 2.+3.arg: return argument:
**             address for relative x/y coordinate to put graphic out:
**               add xkorr to your origin x coordinate
**               add ykorr to your origin y coordinate
** return value: pointer to actual graphic
**               or NULL = no graphic available (sprite empty)
**
** Because the graphics of a sprite can have different sizes, use the
** return arguments xkorr,ykorr to put the graphic out at the same position.
** Example: there are two graphics in a sprite:
**          1. with width and height 9,9
**          2. with width and height 5,5
**          --> the original size is 9,9.
**          When getting the first graphic xkorr,ykorr is 0,0
**          When getting the second graphic xkorr,ykorr is 2,2
**          --> put it out at x+xkorr,y+ykorr
**              e.g. copy_grafik(NULL,x+xkorr,y+ykorr,grf,0,0,0,0,RGB_TRANS)
*/
  return(_get_sprite(spr,xkorr,ykorr,1));
} /* Ende next_sprite */


grafik * last_sprite(sprite * spr,int * xkorr,int * ykorr) {
/* get actual graphic of a sprite again without incrementing the loop counter
** 1.arg: sprite
** 2.+3.arg: return argument: see next_sprite()
** return value: pointer to actual graphic
**               or NULL = no graphic available (sprite empty
**                         or no last graphic, e.g. after reset_sprite())
*/
  return(_get_sprite(spr,xkorr,ykorr,0));
} /* Ende last_sprite */


int reset_sprite(sprite * spr) {
/* reset a sprite
** 1.arg: sprite
** return value: number of loops the whole sprite takes when played once
**               (usefull as counter to play sprite only once)
*/
  int runden,i1;
  if (spr==NULL) {return(0);}
  spr->spr_pos=0;
  spr->runde=0;
  runden=0;
  for (i1=0;i1<spr->spr_anz;i1++) {
    runden+=spr->spr_ptr[i1].maxrunde;
  }
  return(runden);
} /* Ende reset_sprite */


void free_sprite(sprite * spr) {
/* free and destroy a sprite
** 1.arg: sprite
*/
  int i1;
  if (spr==NULL) {return;}
  for (i1=0;i1<spr->spr_anz;i1++) {
    free(spr->spr_ptr[i1].grf);
  }
  if (spr->spr_anz>0) {free(spr->spr_ptr);}
  free(spr);
} /* Ende free_sprite */



/* +++ graphic library dependend functions +++ */

#ifdef GRLIB_VGA
  #include "vgagames_vga.c"  /* svgalib */
#endif

#ifdef GRLIB_VGL
  #include "vgagames_vgl.c"  /* libvgl */
#endif

#ifdef GRLIB_X
  #include "vgagames_x.c"    /* X window */
#endif



/* +++ internal functions +++ */

#include "vgag-bitmap.h"  /* _vga_grafik() */


static int _konv_map(unsigned char kv[],FILE * HDL) {
  int i1;
  farbkonv fw,fv;
  unsigned char bix=0,bfb=64;
  unsigned char fiw[256],ifiw;
  fv.r=fv.g=fv.b=64;
  memset(fiw,0,sizeof(fiw)); ifiw=0;
  for (i1=0;i1<256;i1++) {  /* fw: colormap of graphic */
    if (fscanf(HDL,"%c%c%c",&fw.r,&fw.g,&fw.b)<3) {  /* next value of colormap */
      strcpy(errmsg,"Corrupt graphic file");
      return(-1);
    }
    /* kv[fw]=fx: best color index to colormap of graphic */
    kv[i1]=_best_color(&fw,RGB_TRANS);
    if ((fw.r<=bfb) && (fw.g<=bfb) && (fw.b<=bfb)) {  /* find black pixel */
      if (fw.r+fw.g+fw.b<fv.r+fv.g+fv.b) {
        bix=(unsigned char)i1;
        bfb=(fw.r>fw.g?(fw.r>fw.b?fw.r:fw.b):(fw.g>fw.b?fw.g:fw.b));
        fv.r=fw.r; fv.g=fw.g; fv.b=fw.b;
        ifiw=0;
      } else if (fw.r+fw.g+fw.b==fv.r+fv.g+fv.b) {  /* same black pixel */
        fiw[ifiw++]=(unsigned char)i1;
      }
    }
  }
  kv[(int)bix]=_rgb_black;
  for (i1=0;i1<ifiw;i1++) {  /* make same black pixels also black */
    kv[(int)fiw[i1]]=_rgb_black;
  }
  return(0);
} /* Ende _konv_map */


static unsigned char _best_color(farbkonv * pfw,int trsp) {
  int i1,i2,wr,wg,wb,z0,z2,z3;
  unsigned int wrt;
  unsigned char idx=0;
  wrt=(unsigned int)-1;
  z2=z3=256;
  for (i1=0;i1<256;i1++) {  /* fx: color map of e.g. RGB256_FILE */
    if ((trsp==RGB_TRANS) && (i1==_rgb_black)) {continue;}
    wr=wg=wb=i2=0;
    wr+=pfw->r-fx[i1].r; if (wr<0) {wg-=wr; wb-=wr; i2-=wr; wr=0;}
    wg+=pfw->g-fx[i1].g; if (wg<0) {wb-=wg; i2-=wg; wr-=wg; wg=0;}
    wb+=pfw->b-fx[i1].b; if (wb<0) {i2-=wb; wr-=wb; wg-=wb; wb=0;}
    if (wr>i2) {i2=wr;}
    if (wg>i2) {i2=wg;}
    if (wb>i2) {i2=wb;}
    if ((unsigned int)i2<wrt) {idx=(unsigned char)i1;}
    else if ((unsigned int)i2==wrt) {
      z0=(wr<wg?(wr<wb?wb:wr):(wr<wb?wr:(wb<wg?wg:wb)));
      if (z0>z2) {idx=(unsigned char)i1;}
      else {
        z0=(wr<wg?(wr<wb?wr:wb):(wr<wb?wg:(wb<wg?wb:wg)));
        if (z0>z3) {idx=(unsigned char)i1;}
      }
    }
    if (idx==(unsigned char)i1) {
      z2=(wr<wg?(wr<wb?wb:wr):(wr<wb?wr:(wb<wg?wg:wb)));
      z3=(wr<wg?(wr<wb?wr:wb):(wr<wb?wg:(wb<wg?wb:wg)));
      wrt=i2;
    }
  }
  return(idx);
} /* Ende _best_color */


static void _draw_fontXxY(grafik * graf,int c,int x,int y,const char * s,int n,unsigned char * fontXxY,int * w,int * h,int trsp,int bgc) {
  const int maxwh=64;  /* largest font size in pixels (row and col) */
  static grafik * zeich=NULL;
  unsigned char * font;
  int bt,i1,i2,i3,rv;
  if (zeich==NULL) {
    if ((zeich=create_grafik(maxwh,maxwh))==NULL) {return;}
  }
  if (*h>maxwh) {*h=maxwh;}
  if (*w>maxwh) {*w=maxwh;}
  if (fontXxY==NULL) {fontXxY=font8x8; *w=*h=8;}
  rv=(*w-1)/8+1;  /* bytes per row */
  for (i1=0;i1<n;i1++) {
    bt=0;
    font=fontXxY;
    font+=((unsigned char)s[i1]**h*rv);
    for (i2=0;i2<*h;i2++) {
      for (i3=0;i3<*w;i3++) {
        if (i3%8==0) {bt=*font++;}
        if (bt&(1<<(i3%8))) {
          zeich->pxm[i2*zeich->width+i3]=c;
        } else {
          zeich->pxm[i2*zeich->width+i3]=bgc;
        }
      }
    }
    copy_grafik(graf,x+i1**w,y,zeich,0,0,*w,*h,trsp);
  }
} /* Ende _draw_fontXxY */


static int _save_grf_ppm(grafik * grf,const char * filename,int flag) {
/* save graphic box as ppm file - e.g. for screenshot
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.arg: filename to save ppm graphic
** 3.arg: 3=as P3 or 6=as P6
** return value: 0=OK or -1=error
*/
  FILE * ffp;
  int i1,i2,i3,zlz,gx,gy;
  unsigned char rgb;
  if ((filename!=NULL) && ((ffp=fopen(filename,"w"))!=NULL)) {
    gx=GRAFIK_WIDTH(grf); gy=GRAFIK_HEIGHT(grf);
    if (flag==3) {  /* P3 */
      fprintf(ffp,"P3\n%d %d\n255\n",gx,gy);
      for (i1=0;i1<gy;i1++) {
        zlz=0;
        for (i2=0;i2<gx;i2++) {
          i3=get_pixel(grf,i2,i1);
          fprintf(ffp,"%s%3d %3d %3d",zlz==0?"":"   ",fx[i3].r<<2,fx[i3].g<<2,fx[i3].b<<2);
          if (++zlz==5) {fprintf(ffp,"\n"); zlz=0;}
        }
        if (zlz>0) {fprintf(ffp,"\n");}
      }
    } else if (flag==6) {  /* P6 */
      fprintf(ffp,"P6\n%d %d\n255\n",gx,gy);
      for (i1=0;i1<gy;i1++) {
        for (i2=0;i2<gx;i2++) {
          i3=get_pixel(grf,i2,i1);
          rgb=fx[i3].r<<2;
          fwrite(&rgb,sizeof(unsigned char),1,ffp);
          rgb=fx[i3].g<<2;
          fwrite(&rgb,sizeof(unsigned char),1,ffp);
          rgb=fx[i3].b<<2;
          fwrite(&rgb,sizeof(unsigned char),1,ffp);
        }
      }
    }
    fclose(ffp);
  } else {
    snprintf(errmsg,sizeof(errmsg),"Cannot write file \"%s\"!",filename==NULL?"(null)":filename);
    return(-1);
  }
  return(0);
} /* Ende _save_grf_ppm */


static grafik * _get_sprite(sprite * spr,int * xkorr,int * ykorr,int flag) {
/* get actual graphic of a sprite
** 1.arg: sprite
** 2.+3.arg: return argument: see next_sprite()
** 4.arg: 0=get last loop of graphic again
**        1=get next loop of graphic
** return value: pointer to actual graphic
**               or NULL = no graphic available (sprite empty)
*/
  int w,h;
  if (spr==NULL) {return(NULL);}
  if (spr->spr_anz==0) {return(NULL);}
  if (flag==1) {  /* next loop of graphic */
    int zhl=0;
_cpsp1:
    if (++spr->runde>spr->spr_ptr[spr->spr_pos].maxrunde) {
      if (++spr->spr_pos>=spr->spr_anz) {spr->spr_pos=0; zhl++;}
      spr->runde=0;
      if (zhl==2) {return(NULL);}  /* endless looping sprite */
      goto _cpsp1;
    }
  } else {  /* last loop of graphic */
    if (spr->runde==0) {return(NULL);}
  }
  w=spr->spr_ptr[spr->spr_pos].grf->width;
  h=spr->spr_ptr[spr->spr_pos].grf->height;
  if (xkorr!=NULL) {
    *xkorr=(spr->w_orig-w)/2;
    if (*xkorr*2<spr->w_orig-w) {(*xkorr)++;}
  }
  if (ykorr!=NULL) {
    *ykorr=(spr->h_orig-h)/2;
    if (*ykorr*2<spr->h_orig-h) {(*ykorr)++;}
  }
  return(spr->spr_ptr[spr->spr_pos].grf);
} /* Ende _get_sprite */
