/* Include file for vgag-wave.h - ALSA 0.9 support */

#define ALSA_PCM_NEW_HW_PARAMS_API
#define ALSA_PCM_NEW_SW_PARAMS_API
#include <alsa/asoundlib.h>

/* definitions */
#define ALSA09_PCM_DEV "default"       /* ALSA09 device */
static snd_pcm_t * alsa09_pcm_fd;      /* ALSA09 device handle */
static int alsa09_notconf=1;           /* not yet configured */
static struct {
  int fragm_sz;         /* fragment size */
  int frame_bytes;      /* frame bytes */
  snd_pcm_uframes_t period_size;
  unsigned int periods;
  snd_pcm_uframes_t buffer_size;
} alsa09_dev_cf;
#define ALSA09_MIXER_DEV "default"     /* Mixer device */
static snd_mixer_t * alsa09_mixer_fd;  /* Mixer device handle */
static snd_mixer_elem_t * alsa09_mix_elem[2];

/* number of seconds wait blocking for opening device, or 0=unblocking */
#define ALSA09_ALARM 0


/* declarations */

static int _mixer_callback(snd_mixer_t *,unsigned int,snd_mixer_elem_t *);

int open_alsa09(struct wave_device *);
int config_alsa09(struct wave_device *,unsigned long,unsigned long,unsigned long,unsigned long);
void write_alsa09(struct wave_device *,const unsigned char *,int);
void close_alsa09(struct wave_device *);
int mixer_alsa09(struct wave_device *,int,int);


/* internal functions */

static int _mixer_callback(snd_mixer_t * mixer_fd,unsigned int mask,snd_mixer_elem_t * elem) {
  if (mask&SND_CTL_EVENT_MASK_ADD) {
    unsigned int idx=snd_mixer_selem_get_index(elem);
    if (idx==0) {  /* every first device element */
      if (strcmp(snd_mixer_selem_get_name(elem),"Master")==0) {
        alsa09_mix_elem[0]=elem;
        snd_mixer_selem_set_playback_volume_range(elem,0,100);
      } else if (strcmp(snd_mixer_selem_get_name(elem),"PCM")==0) {
        alsa09_mix_elem[1]=elem;
        snd_mixer_selem_set_playback_volume_range(elem,0,100);
      }
    }
  }
  return(0);
} /* Ende _mixer_callback */


/* functions */

int open_alsa09(struct wave_device * wdptr) {
/* open device and return fragment size or -1=error */
  int err,retry;
  const char * open_dev;
  snd_pcm_sframes_t avail;
  snd_pcm_stream_t stream=SND_PCM_STREAM_PLAYBACK;
  memset(&alsa09_dev_cf,0,sizeof(alsa09_dev_cf));
  if (*(OPEN_DEVICE_PCM)=='\0') {open_dev=ALSA09_PCM_DEV;} else {open_dev=OPEN_DEVICE_PCM;}
  for (retry=5;;retry--) {
#if ALSA09_ALARM == 0
    if ((err=snd_pcm_open(&alsa09_pcm_fd,open_dev,stream,SND_PCM_NONBLOCK))<0) {
      if ((retry>1) && (err==-EBUSY)) {sleep(1); continue;}
      snprintf(errmsg,sizeof(errmsg),"snd_pcm_open \"%s\": %s",open_dev,snd_strerror(err));
      return(-1);
    }
    if ((err=snd_pcm_nonblock(alsa09_pcm_fd,0))<0) {
      snprintf(errmsg,sizeof(errmsg),"snd_pcm_nonblock \"%s\": %s",open_dev,snd_strerror(err));
      snd_pcm_close(alsa09_pcm_fd);
      return(-1);
    }
#else
    alarm(ALSA09_ALARM);
    if ((err=snd_pcm_open(&alsa09_pcm_fd,open_dev,stream,0))<0) {
      alarm(0);
      if ((retry>1) && (err==-EBUSY)) {sleep(1); continue;}
      snprintf(errmsg,sizeof(errmsg),"snd_pcm_open \"%s\": %s",open_dev,snd_strerror(err));
      return(-1);
    }
    alarm(0);
#endif
    break;
  }
  /* to get fragment size call config_alsa09() with rudimentary values */
  alsa09_notconf=1;
  if (config_alsa09(wdptr,8UL,0UL,8000UL,8000UL)<0) {return(-1);}
  avail=snd_pcm_avail_update(alsa09_pcm_fd);
  if ((avail<0) || ((snd_pcm_uframes_t)avail>alsa09_dev_cf.buffer_size)) {avail=alsa09_dev_cf.buffer_size;}
  alsa09_dev_cf.fragm_sz=alsa09_dev_cf.period_size*alsa09_dev_cf.frame_bytes;
  snprintf(errmsg,sizeof(errmsg),"Fragments=%lu, Fragstotal=%d, Fragment-Size=%d, Bytes=%ld",avail/alsa09_dev_cf.period_size,alsa09_dev_cf.periods,alsa09_dev_cf.fragm_sz,avail*alsa09_dev_cf.frame_bytes);
  schreibe(errmsg);
  /* Mixer */
  if (*(OPEN_DEVICE_MIXER)=='\0') {open_dev=ALSA09_MIXER_DEV;} else {open_dev=OPEN_DEVICE_MIXER;}
  if ((err=snd_mixer_open(&alsa09_mixer_fd,0))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_mixer_open: %s",snd_strerror(err));
    snd_pcm_close(alsa09_pcm_fd);
    return(-1);
  }
  if ((err=snd_mixer_attach(alsa09_mixer_fd,open_dev))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_mixer_attach \"%s\": %s",open_dev,snd_strerror(err));
    snd_pcm_close(alsa09_pcm_fd);
    snd_mixer_close(alsa09_mixer_fd);
    return(-1);
  }
  if ((err=snd_mixer_selem_register(alsa09_mixer_fd,NULL,NULL))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_register: %s",snd_strerror(err));
    snd_pcm_close(alsa09_pcm_fd);
    snd_mixer_close(alsa09_mixer_fd);
    return(-1);
  }
  snd_mixer_set_callback(alsa09_mixer_fd,_mixer_callback);
  alsa09_mix_elem[0]=alsa09_mix_elem[1]=NULL;
  if ((err=snd_mixer_load(alsa09_mixer_fd))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_mixer_load: %s",snd_strerror(err));
    snd_pcm_close(alsa09_pcm_fd);
    snd_mixer_close(alsa09_mixer_fd);
    return(-1);
  }
  if ((alsa09_mix_elem[0]==NULL) || (alsa09_mix_elem[1]==NULL)) {
    snprintf(errmsg,sizeof(errmsg),"Mixer volume not found.");
    snd_pcm_close(alsa09_pcm_fd);
    snd_mixer_close(alsa09_mixer_fd);
    return(-1);
  }
  return(alsa09_dev_cf.fragm_sz);
} /* Ende open_alsa09 */


int config_alsa09(struct wave_device * wdptr,unsigned long bitsmp,unsigned long modus,unsigned long smpfq,unsigned long bytsek) {
/* configure device with
**  - bits per sample (always 8)
**  - modus: 1=stereo or 0=mono
**  - sample frequence
**  - bytes per second (not used here)
** returns 0=OK or -1=error
*/
  unsigned int fragments=2;  /* 2 fragments (periods) */
  snd_pcm_uframes_t fragment_size=(1<<9);  /* fragment size 2^9=512 bytes */
  snd_pcm_uframes_t sw_val;
  int err,hw_dir=0;
  unsigned int rate=(unsigned int)smpfq,rrate;
  snd_pcm_hw_params_t * hw_par;
  snd_pcm_sw_params_t * sw_par;
  /* reset device */
  if (alsa09_notconf==0) {
    if (snd_pcm_drop(alsa09_pcm_fd)>=0) {snd_pcm_prepare(alsa09_pcm_fd);}
  } else {alsa09_notconf=0;}
  alsa09_dev_cf.frame_bytes=snd_pcm_format_physical_width(SND_PCM_FORMAT_U8)*(modus+1)/8;

  /* +++ Hardware parameters +++ */
  snd_pcm_hw_params_alloca(&hw_par);
  snd_pcm_hw_params_any(alsa09_pcm_fd,hw_par);
  if ((err=snd_pcm_hw_params_set_access(alsa09_pcm_fd,hw_par,SND_PCM_ACCESS_RW_INTERLEAVED))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_access: %s",snd_strerror(err));
    return(-1);
  }
  /* format 8 bits per sample */
  if ((err=snd_pcm_hw_params_set_format(alsa09_pcm_fd,hw_par,SND_PCM_FORMAT_U8))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_format: %s",snd_strerror(err));
    return(-1);
  }
  /* mono / stereo */
  if ((err=snd_pcm_hw_params_set_channels(alsa09_pcm_fd,hw_par,modus+1))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_channels: %s",snd_strerror(err));
    return(-1);
  }
  /* samplefrequence */
  rrate=rate;
  if ((err=snd_pcm_hw_params_set_rate_near(alsa09_pcm_fd,hw_par,&rate,&hw_dir))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_rate_near: %s",snd_strerror(err));
    return(-1);
  }
  if (((int)rate<(int)(rrate-1000U)) || (rate>rrate+1000U)) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_rate_near: %d Hz requested, %d Hz got",rrate,rate);
    return(-1);
  }
  /* Fragments (periods):
  ** should be 2 or 3 fragments, each 256 or 512 bytes.
  ** The less the more "just in time" sound,
  ** but on slower computers there could be heard clicks
  */
  if ((err=snd_pcm_hw_params_set_periods_integer(alsa09_pcm_fd,hw_par))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_periods_integer: %s",snd_strerror(err));
    return(-1);
  }
  if ((err=snd_pcm_hw_params_set_periods_near(alsa09_pcm_fd,hw_par,&fragments,&hw_dir))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_periods_near: %s",snd_strerror(err));
    return(-1);
  }
  alsa09_dev_cf.periods=fragments;
  fragment_size/=alsa09_dev_cf.frame_bytes;
  if ((err=snd_pcm_hw_params_set_period_size_near(alsa09_pcm_fd,hw_par,&fragment_size,&hw_dir))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params_set_period_size_near: %s",snd_strerror(err));
    return(-1);
  }
  alsa09_dev_cf.period_size=fragment_size;
  /* apply hw params to device */
  if ((err=snd_pcm_hw_params(alsa09_pcm_fd,hw_par))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_hw_params: %s",snd_strerror(err));
    return(-1);
  }
  alsa09_dev_cf.buffer_size=alsa09_dev_cf.periods*alsa09_dev_cf.period_size;

  /* +++ Software parameters +++ */
  snd_pcm_sw_params_alloca(&sw_par);
  snd_pcm_sw_params_current(alsa09_pcm_fd,sw_par);
  sw_val=0;
  if ((err=snd_pcm_sw_params_set_start_threshold(alsa09_pcm_fd,sw_par,sw_val))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_sw_params_set_start_threshold: %s",snd_strerror(err));
    return(-1);
  }
  /* apply sw params to device */
  if ((err=snd_pcm_sw_params(alsa09_pcm_fd,sw_par))<0) {
    snprintf(errmsg,sizeof(errmsg),"snd_pcm_sw_params: %s",snd_strerror(err));
    return(-1);
  }
  return(0);
} /* Ende config_alsa09 */


void write_alsa09(struct wave_device * wdptr,const unsigned char * val,int size) {
/* write samples "val" of size "size" out to device */
  ssize_t erg;
  snd_pcm_uframes_t frames=size/alsa09_dev_cf.frame_bytes;
_wr_again:
  erg=snd_pcm_writei(alsa09_pcm_fd,val,frames);
  if ((erg==-EPIPE) && (snd_pcm_state(alsa09_pcm_fd)==SND_PCM_STATE_XRUN) \
  && ((erg=snd_pcm_prepare(alsa09_pcm_fd))==0)) {goto _wr_again;}
} /* Ende write_alsa09 */


void close_alsa09(struct wave_device * wdptr) {
/* close device */
  snd_mixer_close(alsa09_mixer_fd);
  alsa09_mix_elem[0]=alsa09_mix_elem[1]=NULL;
  if (snd_pcm_state(alsa09_pcm_fd)!=SND_PCM_STATE_OPEN) {snd_pcm_drop(alsa09_pcm_fd);}
  snd_pcm_close(alsa09_pcm_fd);
} /* Ende close_alsa09 */


int mixer_alsa09(struct wave_device * wdptr,int volm,int flag) {
/* set main or pcm volume
** 2.arg: 0-100 or <0 to get only actual volume
** 3.arg: 0=main volume or 1=pcm volume
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  int vola;
  long lvol,rvol;
  int err,sw;
  if (flag!=0) {flag=1;}

  if (!snd_mixer_selem_has_playback_volume(alsa09_mix_elem[flag])) {
    strcpy(errmsg,"Mixer volume not found");
    return(-1);
  }

  /* read volume */
  if (snd_mixer_selem_has_playback_switch(alsa09_mix_elem[flag])) {
    if ((err=snd_mixer_selem_get_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,&sw))<0) {
      snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_switch (%d left): %s",flag,snd_strerror(err));
      return(-1);
    }
    if (sw) {
      if ((err=snd_mixer_selem_get_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,&lvol))<0) {
        snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_volume (%d left): %s",flag,snd_strerror(err));
        return(-1);
      }
    } else {lvol=0L;}
  } else {
    if ((err=snd_mixer_selem_get_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,&lvol))<0) {
      snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_volume (%d left): %s",flag,snd_strerror(err));
      return(-1);
    }
  }
  if (snd_mixer_selem_is_playback_mono(alsa09_mix_elem[flag])) {
    rvol=lvol;
  } else {
    if (snd_mixer_selem_has_playback_switch(alsa09_mix_elem[flag])) {
      if ((err=snd_mixer_selem_get_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_RIGHT,&sw))<0) {
        snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_switch (%d right): %s",flag,snd_strerror(err));
        return(-1);
      }
      if (sw) {
        if ((err=snd_mixer_selem_get_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_RIGHT,&rvol))<0) {
          snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_volume (%d right): %s",flag,snd_strerror(err));
          return(-1);
        }
      } else {rvol=0L;}
    } else {
      if ((err=snd_mixer_selem_get_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_RIGHT,&rvol))<0) {
        snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_get_playback_volume (%d right): %s",flag,snd_strerror(err));
        return(-1);
      }
    }
  }
  vola=(lvol+rvol)/2;
  vola=(vola>100)?100:vola;

  /* set volume */
  if (volm>=0) {
    volm=(volm>100)?100:volm;
    rvol=lvol=(long)volm;
    if ((err=snd_mixer_selem_set_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,lvol))<0) {
      snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_volume (%d left): %s",flag,snd_strerror(err));
      return(-1);
    }
    if (snd_mixer_selem_is_playback_mono(alsa09_mix_elem[flag])) {
      if (snd_mixer_selem_has_playback_switch(alsa09_mix_elem[flag])) {
        if ((err=snd_mixer_selem_set_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,lvol!=0L))<0) {
          snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_switch (%d left): %s",flag,snd_strerror(err));
          return(-1);
        }
      }
    } else {
      if ((err=snd_mixer_selem_set_playback_volume(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_RIGHT,rvol))<0) {
        snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_volume (%d right): %s",flag,snd_strerror(err));
        return(-1);
      }
      if (snd_mixer_selem_has_playback_switch(alsa09_mix_elem[flag])) {
        if (snd_mixer_selem_has_playback_switch_joined(alsa09_mix_elem[flag])) {
          if ((err=snd_mixer_selem_set_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,lvol!=0L || rvol!=0L))<0) {
            snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_switch (%d left+right): %s",flag,snd_strerror(err));
            return(-1);
          }
        } else {
          if ((err=snd_mixer_selem_set_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,lvol!=0L))<0) {
            snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_switch (%d left): %s",flag,snd_strerror(err));
            return(-1);
          }
          if ((err=snd_mixer_selem_set_playback_switch(alsa09_mix_elem[flag],SND_MIXER_SCHN_FRONT_LEFT,rvol!=0L))<0) {
            snprintf(errmsg,sizeof(errmsg),"snd_mixer_selem_set_playback_switch (%d right): %s",flag,snd_strerror(err));
            return(-1);
          }
        }
      }
    }
  }
  return(vola);
} /* Ende mixer_alsa09 */
