/* Include file for vgag-wave.h - OSS support
**
** - functions of "struct wave_device":
**   1. int open_oss(struct wave_device *)
**      open sound device and return fragment size
**      1.arg: pointer to struct wave_device
**   2. int config_oss(struct wave_device *,unsigned long,unsigned long,unsigned long,unsigned long)
**      configure device with given wave values
**      1.arg: pointer to struct wave_device
**      2.arg: samplesize (bits per sample: 8)
**      3.arg: modus (0=mono or 1=stereo)
**      4.arg: sample frequence (speed, e.g. 22050)
**      5.arg: bytes per second (not used here)
**   3. void write_oss(struct wave_device *,const unsigned char *,int)
**      write fragment block out to sound device
**      1.arg: pointer to struct wave_device
**      2.arg: sound data
**      3.arg: fragment size, given from open_oss()
**   4. void close_oss(struct wave_device *)
**      close sound device
**      1.arg: pointer to struct wave_device
**   5. int mixer_oss(struct wave_device *,int,int)
**      set mixer volume
**      1.arg: pointer to struct wave_device
**      2.arg: 0-100 or <0 to get only actual volume
**      3.arg: 0=main volume or 1=pcm volume (wave)
**      return value: volume before call (0-100): ok
**                    or -1=error
**
** - variables of "struct wave_device":
**   1. simul_16 and simul_stereo
**      set both to value=0
**      set simul_16 to value=1
**        if soundcard supports only 16 Bits
**      set simul_stereo to value=1
**        if soundcard supports only stereo
**   2. libname
**      name of sound library, same as in wave.conf
**
*/

#include <sys/ioctl.h>
#include <sys/soundcard.h>

/* definitions */
#define OSS_PCM_DEV "/dev/dsp"      /* OSS device */
static int oss_pcm_fd=-1;           /* OSS file descriptor */
static int oss_notconf=1;           /* OSS device still not configured */
#define OSS_MIXER_DEV "/dev/mixer"  /* Mixer device */
static int oss_mixer_fd=-1;         /* Mixer file descriptor */

/* number of seconds wait blocking for opening device, or 0=unblocking */
#define OSS_ALARM 0


/* declarations */

int open_oss(struct wave_device *);
int config_oss(struct wave_device *,unsigned long,unsigned long,unsigned long,unsigned long);
void write_oss(struct wave_device *,const unsigned char *,int);
void close_oss(struct wave_device *);
int mixer_oss(struct wave_device *,int,int);


/* functions */

int open_oss(struct wave_device * wdptr) {
/* open device and return fragment size or -1=error */
  const char * open_dev;
  audio_buf_info info;
  int i1,ruck,retry;
  if (*(OPEN_DEVICE_PCM)=='\0') {open_dev=OSS_PCM_DEV;} else {open_dev=OPEN_DEVICE_PCM;}
  for (retry=5;;retry--) {
#if OSS_ALARM == 0
    if ((oss_pcm_fd=open(open_dev,O_WRONLY|O_NONBLOCK,0))<0) {
      if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
      sprintf(errmsg,"open %s: %s.",open_dev,strerror(errno));
      return(-1);
    }
    if (fcntl(oss_pcm_fd,F_SETFL,fcntl(oss_pcm_fd,F_GETFL,0)&~O_NONBLOCK)<0) {
      sprintf(errmsg,"fcntl %s: %s.",open_dev,strerror(errno));
      close(oss_pcm_fd);
      return(-1);
    }
#else
    alarm(OSS_ALARM);
    if ((oss_pcm_fd=open(open_dev,O_WRONLY,0))<0) {
      alarm(0);
      if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
      sprintf(errmsg,"open %s: %s.",open_dev,strerror(errno));
      return(-1);
    }
    alarm(0);
#endif
    break;
  }
  /* Fragments:
  ** should be 2 or 3 fragments, each 256 or 512 bytes.
  ** The less the more "just in time" sound,
  ** but on slower computers there could be heard clicks
  */
  i1=0x00020009;  /* 2 fragments (minimum), each has 2^9=512 bytes */
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_SETFRAGMENT,&i1)==-1) {
    sprintf(errmsg,"SETFRAGMENT: ioctl error: %s",strerror(errno));
    close(oss_pcm_fd);
    return(-1);
  }
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_GETOSPACE,&info)==0) {
    sprintf(errmsg,"Fragments=%d, Fragstotal=%d, Fragment-Size=%d, Bytes=%d",info.fragments,info.fragstotal,info.fragsize,info.bytes);
    schreibe(errmsg);
  } else {
    sprintf(errmsg,"GETOSPACE: ioctl error: %s",strerror(errno));
    close(oss_pcm_fd);
    return(-1);
  }
  ruck=info.fragsize;
  /* Mixer */
  if (*(OPEN_DEVICE_MIXER)=='\0') {open_dev=OSS_MIXER_DEV;} else {open_dev=OPEN_DEVICE_MIXER;}
  if ((oss_mixer_fd=open(open_dev,O_RDWR))<0) {
    sprintf(errmsg,"open %s: %s",open_dev,strerror(errno));
    close(oss_pcm_fd);
    return(-1);
  }
  if (ioctl(oss_mixer_fd,SOUND_MIXER_READ_DEVMASK,&i1)==-1) {
    sprintf(errmsg,"SOUND_MIXER_READ_DEVMASK: %s",strerror(errno));
    close(oss_mixer_fd);
    close(oss_pcm_fd);
    return(-1);
  }
  if (((i1&SOUND_MASK_VOLUME)==0) || ((i1&SOUND_MASK_PCM)==0)) {
    strcpy(errmsg,"Mixer volume not found.");
    close(oss_mixer_fd);
    close(oss_pcm_fd);
    return(-1);
  }
  oss_notconf=1;
  return(ruck);
} /* Ende open_oss */


int config_oss(struct wave_device * wdptr,unsigned long bitsmp,unsigned long modus,unsigned long smpfq,unsigned long bytsek) {
/* configure device with
**  - bits per sample (always AFMT_U8)
**  - modus: 1=stereo or 0=mono
**  - sample frequence
**  - bytes per second (not used here)
** returns 0=OK or -1=error
*/
  int i1;
  unsigned long tst,val;
  if (oss_notconf==0) {ioctl(oss_pcm_fd,SNDCTL_DSP_RESET);} else {oss_notconf=0;}
  /* bits per sample */
  tst=val=AFMT_U8;
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_SETFMT,&val)==-1) {
    sprintf(errmsg,"SAMPLESIZE: ioctl error: %s",strerror(errno));
    return(-1);
  } else if (tst!=val) {
    if (val==16) {
      wdptr->simul_16=1;
    } else {
      sprintf(errmsg,"SAMPLESIZE: error: requested %lu and set %lu",tst,val);
      return(-1);
    }
  } else {wdptr->simul_16=0;}
  /* modus */
  val=modus;
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_STEREO,&val)==-1) {
    sprintf(errmsg,"STEREO: ioctl error: %s",strerror(errno));
    return(-1);
  } else if (modus!=val) {
    if (val==1) {
      wdptr->simul_stereo=1;
    } else {
      sprintf(errmsg,"STEREO: error: requested %lu and set %lu",modus,val);
      return(-1);
    }
  } else {wdptr->simul_stereo=0;}
  /* sample frequence */
  val=smpfq;
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_SPEED,&val)==-1) {
    sprintf(errmsg,"SPEED: ioctl error: %s",strerror(errno));
    return(-1);
  } else if (((long)smpfq<(long)(val-1000UL)) || (smpfq>val+1000UL)) {
    sprintf(errmsg,"SPEED: error: requested %lu and set %lu",smpfq,val);
    return(-1);
  }
  /* bytes per second: nothing */

  if (wdptr->simul_stereo || wdptr->simul_16) {
    if (wdptr->simul_16) {
      strcpy(errmsg,"Soundcard seems to support only 16 bits, simulating it ...");
      schreibe(errmsg);
    }
    if (wdptr->simul_stereo) {
      strcpy(errmsg,"Soundcard seems to support only stereo, simulating it ...");
      schreibe(errmsg);
    }
  }
  if (ioctl(oss_pcm_fd,SNDCTL_DSP_GETCAPS,&i1)!=-1) {
    if (i1&DSP_CAP_TRIGGER) {
      i1=PCM_ENABLE_OUTPUT;
      ioctl(oss_pcm_fd,SNDCTL_DSP_SETTRIGGER,&i1);
    } else {ioctl(oss_pcm_fd,SNDCTL_DSP_POST,0);}
  } else {ioctl(oss_pcm_fd,SNDCTL_DSP_POST,0);}
  return(0);
} /* Ende config_oss */


void write_oss(struct wave_device * wdptr,const unsigned char * val,int size) {
/* write samples "val" of size "size" out to device */
  write(oss_pcm_fd,val,size);
} /* Ende write_oss */


void close_oss(struct wave_device * wdptr) {
/* close device */
  close(oss_mixer_fd);
  oss_mixer_fd=-1;
  close(oss_pcm_fd);
  oss_pcm_fd=-1;
} /* Ende close_oss */


int mixer_oss(struct wave_device * wdptr,int volm,int flag) {
/* set main or pcm volume
** 2.arg: 0-100 or <0 to get only actual volume
** 3.arg: 0=main volume or 1=pcm volume
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  int vola,vart;
  if (flag==1) {vart=SOUND_MIXER_READ_PCM;} else {vart=SOUND_MIXER_READ_VOLUME;}
  if (ioctl(oss_mixer_fd,vart,&vola)==-1) {
    sprintf(errmsg,"SOUND_MIXER_READ_...: %s",strerror(errno));
    return(-1);
  }
  /* volume value for ioctl is "right_volume<<8 | left_volume"
  ** We set right and left volume always to the same value
  */
  vart=(vola&0xff); vola=((vola&0xff00)>>8);
  vola=(vart+vola)/2;
  vola=(vola>100)?100:vola;
  if (volm>=0) {
    volm=(volm>100)?100:volm;
    volm|=(volm<<8);
    if (flag==1) {vart=SOUND_MIXER_WRITE_PCM;} else {vart=SOUND_MIXER_WRITE_VOLUME;}
    if (ioctl(oss_mixer_fd,vart,&volm)==-1) {
      sprintf(errmsg,"SOUND_MIXER_WRITE_...: %s",strerror(errno));
      return(-1);
    }
  }
  return(vola);
} /* Ende mixer_oss */
