/* Include file for vgag-wave.h - Sun audio support */

#include <sys/ioctl.h>
#include <sys/audioio.h>

#ifndef AUDIO_ENCODING_ULINEAR_LE
  #define AUDIO_ENCODING_ULINEAR_LE AUDIO_ENCODING_LINEAR8
#endif
#ifndef AUDIO_ENCODING_SLINEAR_LE
  #define AUDIO_ENCODING_SLINEAR_LE AUDIO_ENCODING_LINEAR
#endif


/* definitions */
#define SAUDIO_DEV "/dev/audio"        /* Sun audio device */
static int saudio_fd=-1;               /* Sun audio file descriptor */
static int saudio_notconf=1;           /* Sun audio device still not configured */
static unsigned char gain_wert=0;      /* gain value instead of mixer */
#ifdef C_WAVE__SAUDIOMIXER
  #define SAUDIO_MIXER_DEV "/dev/mixer"  /* Mixer device */
  static int saudio_mixer_fd=-1;         /* Mixer file descriptor */
  mixer_ctrl_t mix_wert;                 /* Mixer value */
#endif

/* number of seconds wait blocking for opening device, or 0=unblocking */
#define SAUDIO_ALARM 0


/* declarations */

int open_saudio(struct wave_device *);
int config_saudio(struct wave_device *,unsigned long,unsigned long,unsigned long,unsigned long);
void write_saudio(struct wave_device *,const unsigned char *,int);
void close_saudio(struct wave_device *);
int mixer_saudio(struct wave_device *,int,int);


/* functions */

int open_saudio(struct wave_device * wdptr) {
/* open device and return fragment size or -1=error */
  const char * open_dev;
  audio_info_t auinfo;
  int ruck,retry;

  if (*(OPEN_DEVICE_PCM)=='\0') {open_dev=SAUDIO_DEV;} else {open_dev=OPEN_DEVICE_PCM;}
  for (retry=5;;retry--) {
#if SAUDIO_ALARM == 0
    if ((saudio_fd=open(open_dev,O_WRONLY|O_NONBLOCK,0))<0) {
      if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
      snprintf(errmsg,sizeof(errmsg),"open %s: %s.",open_dev,strerror(errno));
      return(-1);
    }
    if (fcntl(saudio_fd,F_SETFL,fcntl(saudio_fd,F_GETFL,0)&~O_NONBLOCK)<0) {
      snprintf(errmsg,sizeof(errmsg),"fcntl %s: %s.",open_dev,strerror(errno));
      close(saudio_fd);
      return(-1);
    }
#else
    alarm(SAUDIO_ALARM);
    if ((saudio_fd=open(open_dev,O_WRONLY,0))<0) {
      alarm(0);
      if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
      snprintf(errmsg,sizeof(errmsg),"open %s: %s.",open_dev,strerror(errno));
      return(-1);
    }
    alarm(0);
#endif
    break;
  }
  AUDIO_INITINFO(&auinfo);
#if defined(AUMODE_PLAY) && defined(AUMODE_PLAY_ALL)
  auinfo.mode=AUMODE_PLAY|AUMODE_PLAY_ALL;
#endif
#ifdef C_WAVE__SAUDIOBLKSZ
  /* Fragments:
  ** should be 2 or 3 fragments, each 256 or 512 bytes.
  ** The less the more "just in time" sound,
  ** but on slower computers there could be heard clicks
  */
  auinfo.hiwat=2;  /* 2 fragments (minimum) */
  auinfo.blocksize=(1<<9);  /* each has 2^9=512 bytes */
#endif
  if (ioctl(saudio_fd,AUDIO_SETINFO,&auinfo)==-1) {
    snprintf(errmsg,sizeof(errmsg),"AUDIO_SETINFO: ioctl error: %s",strerror(errno));
    close(saudio_fd);
    return(-1);
  }
  if (ioctl(saudio_fd,AUDIO_GETINFO,&auinfo)==-1) {
    snprintf(errmsg,sizeof(errmsg),"AUDIO_GETINFO: ioctl error: %s",strerror(errno));
    close(saudio_fd);
    return(-1);
  }
#ifdef C_WAVE__SAUDIOBLKSZ
  snprintf(errmsg,sizeof(errmsg),"Fragments=%d, Fragment-Size=%d",auinfo.hiwat,auinfo.blocksize);
  ruck=auinfo.blocksize;
#else
  strcpy(errmsg,"Cannot change Fragment-Size");
  ruck=128;
#endif
  schreibe(errmsg);

  /* Mixer */
  gain_wert=auinfo.play.gain;
  AUDIO_INITINFO(&auinfo);
  if (gain_wert>100) {auinfo.play.gain=20;} else {auinfo.play.gain=200;}
  if ((ioctl(saudio_fd,AUDIO_SETINFO,&auinfo)==-1) \
  || (ioctl(saudio_fd,AUDIO_GETINFO,&auinfo)==-1) \
  || (auinfo.play.gain==gain_wert)) {  /* cannot set volume via gain; use mixer */
#ifdef C_WAVE__SAUDIOMIXER
    mixer_devinfo_t mix_type;
    int i1,anzdev,idx;
    if (*(OPEN_DEVICE_MIXER)=='\0') {open_dev=SAUDIO_MIXER_DEV;} else {open_dev=OPEN_DEVICE_MIXER;}
    if ((saudio_mixer_fd=open(open_dev,O_RDWR))<0) {
      snprintf(errmsg,sizeof(errmsg),"open %s: %s",open_dev,strerror(errno));
      close(saudio_fd);
      return(-1);
    }
    idx=-1;
    for (anzdev=0;;anzdev++) {
      mix_type.index=anzdev;
      if (ioctl(saudio_mixer_fd,AUDIO_MIXER_DEVINFO,&mix_type)<0) {break;}
      if (mix_type.type==AUDIO_MIXER_CLASS) {
        if (strcmp(mix_type.label.name,AudioCoutputs)==0) {idx=mix_type.mixer_class;}
      }
    }
    if (idx==-1) {
      snprintf(errmsg,sizeof(errmsg),"no %s-class found",AudioCoutputs);
      close(saudio_mixer_fd);
      close(saudio_fd);
      return(-1);
    }
    memset(&mix_wert,0,sizeof(mix_wert));
    for (i1=0;i1<anzdev;i1++) {
      mix_type.index=i1;
      if (ioctl(saudio_mixer_fd,AUDIO_MIXER_DEVINFO,&mix_type)<0) {break;}
      if ((mix_type.type==AUDIO_MIXER_VALUE) && (mix_type.mixer_class==idx) \
      && (strcmp(mix_type.label.name,AudioNmaster)==0) && (strcmp(mix_type.un.v.units.name,AudioNvolume)==0)) {
        mix_wert.dev=i1;
        mix_wert.type=mix_type.type;
        mix_wert.un.value.num_channels=2;
        if (ioctl(saudio_mixer_fd,AUDIO_MIXER_READ,&mix_wert)<0) {
          mix_wert.un.value.num_channels=1;
          if (ioctl(saudio_mixer_fd,AUDIO_MIXER_READ,&mix_wert)<0) {
            snprintf(errmsg,sizeof(errmsg),"AUDIO_MIXER_READ: ioctl error: %s",strerror(errno));
            close(saudio_mixer_fd);
            close(saudio_fd);
            return(-1);
          }
        }
      }
    }
    if (mix_wert.un.value.num_channels==0) {
      snprintf(errmsg,sizeof(errmsg),"no %s-%s found",AudioNmaster,AudioNvolume);
      close(saudio_mixer_fd);
      close(saudio_fd);
      return(-1);
    }
#else /* C_WAVE__SAUDIOMIXER */
    strcpy(errmsg,"cannot set volume via audio_info_t.play.gain and no mixer found");
    close(saudio_fd);
    return(-1);
#endif /* C_WAVE__SAUDIOMIXER */
  } else {
#ifdef C_WAVE__SAUDIOMIXER
    saudio_mixer_fd=-1;
#endif
    AUDIO_INITINFO(&auinfo);
    auinfo.play.gain=gain_wert;
    ioctl(saudio_fd,AUDIO_SETINFO,&auinfo);
  }
  saudio_notconf=1;
  return(ruck);
} /* Ende open_saudio */


int config_saudio(struct wave_device * wdptr,unsigned long bitsmp,unsigned long modus,unsigned long smpfq,unsigned long bytsek) {
/* configure device with
**  - bits per sample (always AFMT_U8)
**  - modus: 1=stereo or 0=mono
**  - sample frequence
**  - bytes per second (not used here)
** returns 0=OK or -1=error
*/
  audio_info_t auinfo;
#ifdef AUDIO_FLUSH
  if (saudio_notconf==0) {ioctl(saudio_fd,AUDIO_FLUSH,NULL);} else {saudio_notconf=0;}
#else
  saudio_notconf=0;
#endif

  /* modus */
  modus++;
  AUDIO_INITINFO(&auinfo);
  auinfo.play.channels=(unsigned int)modus;
  ioctl(saudio_fd,AUDIO_SETINFO,&auinfo);
  if (ioctl(saudio_fd,AUDIO_GETINFO,&auinfo)==-1) {
    snprintf(errmsg,sizeof(errmsg),"AUDIO_GETINFO: ioctl error: %s",strerror(errno));
    return(-1);
  }
  if (auinfo.play.channels!=(unsigned int)modus) {
    if (auinfo.play.channels==1) {
      wdptr->simul_stereo=1;
    } else {
      snprintf(errmsg,sizeof(errmsg),"STEREO: error: requested %lu and set %u",modus,auinfo.play.channels);
      return(-1);
    }
  } else {wdptr->simul_stereo=0;}

  /* bits per sample */
  AUDIO_INITINFO(&auinfo);
  auinfo.play.encoding=AUDIO_ENCODING_ULINEAR_LE;
  auinfo.play.precision=8;
  if (ioctl(saudio_fd,AUDIO_SETINFO,&auinfo)==-1) {
    AUDIO_INITINFO(&auinfo);
    auinfo.play.encoding=AUDIO_ENCODING_SLINEAR_LE;
    auinfo.play.precision=16;
    if (ioctl(saudio_fd,AUDIO_SETINFO,&auinfo)==-1) {
      strcpy(errmsg,"SAMPLESIZE: error: cannot set to 8 or 16LE");
      return(-1);
    }
    wdptr->simul_16=1;
  } else {wdptr->simul_16=0;}

  /* sample frequence */
  AUDIO_INITINFO(&auinfo);
  auinfo.play.sample_rate=(unsigned int)smpfq;
  ioctl(saudio_fd,AUDIO_SETINFO,&auinfo);
  if (ioctl(saudio_fd,AUDIO_GETINFO,&auinfo)==-1) {
    snprintf(errmsg,sizeof(errmsg),"AUDIO_GETINFO: ioctl error: %s",strerror(errno));
    return(-1);
  }
  if (((unsigned int)smpfq<(auinfo.play.sample_rate-1000)) || ((unsigned int)smpfq>auinfo.play.sample_rate+1000)) {
    snprintf(errmsg,sizeof(errmsg),"SPEED: error: requested %lu and set %u",smpfq,auinfo.play.sample_rate);
    return(-1);
  }

  /* bytes per second: nothing */

  if (wdptr->simul_stereo || wdptr->simul_16) {
    if (wdptr->simul_16) {
      strcpy(errmsg,"Soundcard seems to support only 16 bits, simulating it ...");
      schreibe(errmsg);
    }
    if (wdptr->simul_stereo) {
      strcpy(errmsg,"Soundcard seems to support only stereo, simulating it ...");
      schreibe(errmsg);
    }
  }
  return(0);
} /* Ende config_saudio */


void write_saudio(struct wave_device * wdptr,const unsigned char * val,int size) {
/* write samples "val" of size "size" out to device */
  write(saudio_fd,val,size);
} /* Ende write_saudio */


void close_saudio(struct wave_device * wdptr) {
/* close device */
#ifdef C_WAVE__SAUDIOMIXER
  if (saudio_mixer_fd>=0) {close(saudio_mixer_fd);}
  saudio_mixer_fd=-1;
#endif
  close(saudio_fd);
  saudio_fd=-1;
} /* Ende close_saudio */


int mixer_saudio(struct wave_device * wdptr,int volm,int flag) {
/* set main or pcm volume (emulating pcm volume)
** 2.arg: 0-100 or <0 to get only actual volume
** 3.arg: 0=main volume or 1=pcm volume
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  static int v_m=-1,v_p=-1;
  int vola,i1;
#ifdef C_WAVE__SAUDIOMIXER
  if (saudio_mixer_fd==-1) {  /* use gain */
#endif
    if (v_m==-1) {v_m=gain_wert*100/256; v_p=100;}
#ifdef C_WAVE__SAUDIOMIXER
  } else {  /* use mixer */
    if (v_m==-1) {v_m=mix_wert.un.value.level[0]*100/256; v_p=100;}
  }
#endif
  if (flag==1) {vola=v_p;} else {vola=v_m;}
  if (volm>=0) {
    if (flag==1) {v_p=volm;} else {v_m=volm;}
    i1=(v_m*3+v_p)/4;  /* 3/4 main und 1/4 pcm */
    i1*=256;
    i1/=100;
    if (i1>255) {i1=255;} else if (i1<0) {i1=0;}
#ifdef C_WAVE__SAUDIOMIXER
    if (saudio_mixer_fd==-1) {  /* use gain */
#else
    {
#endif
      audio_info_t auinfo;
      AUDIO_INITINFO(&auinfo);
      auinfo.play.gain=gain_wert=i1;
      if (ioctl(saudio_fd,AUDIO_SETINFO,&auinfo)==-1) {
        snprintf(errmsg,sizeof(errmsg),"AUDIO_SETINFO (gain): %s",strerror(errno));
        return(-1);
      }
#ifdef C_WAVE__SAUDIOMIXER
    } else {  /* use mixer */
      mix_wert.un.value.level[0]=i1;
      if (mix_wert.un.value.num_channels>1) {mix_wert.un.value.level[1]=mix_wert.un.value.level[0];}
      if (ioctl(saudio_mixer_fd,AUDIO_MIXER_WRITE,&mix_wert)==-1) {
        snprintf(errmsg,sizeof(errmsg),"AUDIO_MIXER_WRITE: %s",strerror(errno));
        return(-1);
      }
#endif
    }
  }
  return(vola);
} /* Ende mixer_saudio */
