/* Rock object */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames2.h>
#include "main.h"
#include "o_rock.h"

extern void iforock_move(struct o_rock *,int);  /* interface call */

static struct o_rock * obj=NULL;  /* object for rocks */
static sprite * expl=NULL;  /* rock-exploding sprite */

/* initializing and destroying functions */
struct o_rock * new_rock(int);   /* create rock objects */
struct o_rock * obj_rock(void);  /* get pointer to objects (interface) */
void free_rock(void);            /* free rock objects */

/* internal functions */
static void new_idx(int);
static int move_idx(int);

/* external functions (called from main program) */
static int set_network(void);  /* set struct pointers to network variables */
static void refresh(void);     /* (master:) called before entering a new loop */
static void move(void);        /* (master:) move rocks */
static void getpos(void);      /* (clients:) get positions of rocks */
static void giveout(void);     /* draw rocks on screen */

/* interface functions (called from an interface) */
static int overlapped(int,struct ovlap *,bitmap *,int,int,int,int);  /* rocks overlap with a bitmap? */
static void change_direction(int,float *,float *);  /* change rock's moving direction */
static void set_dead(int);  /* set a rock exploding */


/* +++ initializing and destroying functions +++ */

struct o_rock * new_rock(int anzahl) {
/* all players: create o_rock-object, return it or NULL=error */
  int idx;

  /* create object */
  if ((obj=calloc(1,sizeof(struct o_rock)))==NULL) {
    fprintf(stderr,"new_rock: calloc: %s\n",strerror(errno));
    return(NULL);
  }

  /* set function pointers */
  obj->set_network=set_network;
  obj->refresh=refresh;
  obj->move=move;
  obj->getpos=getpos;
  obj->giveout=giveout;
  obj->overlapped=overlapped;
  obj->change_direction=change_direction;
  obj->set_dead=set_dead;

  /* create rocks */
  obj->anzahl=anzahl;
  if ((obj->rocks=calloc(anzahl,sizeof(*obj->rocks)))==NULL) {
    fprintf(stderr,"new_rock: calloc: %s\n",strerror(errno));
    return(NULL);
  }

  /* initialize rocks */
  for (idx=0;idx<obj->anzahl;idx++) {
    if ((obj->rocks[idx].bild=vg_sprite_createfromfile("sprites/rock.sprite"))==NULL) {return(NULL);}
    new_idx(idx);
    /* network pointers are set later in set_network() */
    obj->rocks[idx].n_xpos=obj->rocks[idx].n_ypos=obj->rocks[idx].n_status=obj->rocks[idx].n_spritepos=NULL;
  }
  obj->n_snd_hit=NULL;

  /* create network arrays for n_xpos, n_ypos, n_status, n_spritepos.
  ** One element of them belongs to each rock.
  */
  if (vg_nw_setcommon(NWVAR_SHORT,obj->anzahl,"OROCK_XPOS")<1) {return(NULL);}
  if (vg_nw_setcommon(NWVAR_SHORT,obj->anzahl,"OROCK_YPOS")<1) {return(NULL);}
  if (vg_nw_setcommon(NWVAR_SHORT,obj->anzahl,"OROCK_STATUS")<1) {return(NULL);}
  if (vg_nw_setcommon(NWVAR_SHORT,obj->anzahl,"OROCK_SPRITEPOS")<1) {return(NULL);}
  if (vg_nw_setcommon(NWVAR_CHAR,1,"OROCK_SND_HIT")<1) {return(NULL);}

  /* load exploding sprite */
  if ((expl=vg_sprite_createfromfile("sprites/exploding.sprite"))==NULL) {return(NULL);}

  /* load soundfiles */
  vg_sound_attach("sounds/rock_hit.wav","s-rockhit",20);
  return(obj);
} /* end new_rock */


struct o_rock * obj_rock() {
/* interface: returns o_rock-object */
  return(obj);
} /* end obj_rock */


void free_rock() {
/* all players: destroys rocks */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    vg_sprite_free(obj->rocks[idx].bild);
  }
  free(obj->rocks);
  free(obj); obj=NULL;
  vg_sprite_free(expl); expl=NULL;
  vg_sound_stop("s-rockhit",0);
} /* end free_rock */


/* +++ internal functions +++ */

static void new_idx(int idx) {
/* initialize rocks[idx] */
  vg_sprite_reset(obj->rocks[idx].bild);
  obj->rocks[idx].mbp=NULL;
  obj->rocks[idx].xpos=(float)ZUFALL(250)+35;       /* start x-position */
  obj->rocks[idx].ypos=(float)ZUFALL(100)-150;      /* start y-position */
  obj->rocks[idx].xadd=(float)(ZUFALL(41)-20)/10.;  /* moving: add to x */
  obj->rocks[idx].yadd=(float)ZUFALL(15)/10.+.5;    /* moving: add to y */
  obj->rocks[idx].status=0;
  obj->rocks[idx].dreh=ZUFALL(5);  /* rotating velocity */
  obj->rocks[idx].dmom=obj->rocks[idx].dreh;
} /* end new_idx */


static int move_idx(int idx) {
/* move rock, return 1=OK or 0=rock dead */
  /* get bitmap of rock */
  if (++obj->rocks[idx].dmom > obj->rocks[idx].dreh) {
    obj->rocks[idx].dmom=0;
    obj->rocks[idx].mbp=vg_sprite_getnext(obj->rocks[idx].bild);
    if (obj->rocks[idx].mbp==NULL) {  /* end of sprite, call again for new loop */
      obj->rocks[idx].mbp=vg_sprite_getnext(obj->rocks[idx].bild);
    }
  } else {obj->rocks[idx].mbp=vg_sprite_getcurrent(obj->rocks[idx].bild);}

  if (obj->rocks[idx].status) {  /* exploding */
    if (--obj->rocks[idx].status==0) {return(0);}  /* dead */
    return(1);  /* do not check anything */
  }

  /* move rock, checking for overlapping */
  iforock_move(obj,idx);

  /* if rock is out of window, kill it */
  if ((int)obj->rocks[idx].ypos > SC_HEIGHT+vg_bitmap_height(obj->rocks[idx].mbp)/2 \
  || (int)obj->rocks[idx].xpos < -vg_bitmap_width(obj->rocks[idx].mbp)/2 \
  || (int)obj->rocks[idx].xpos > SC_WIDTH+vg_bitmap_width(obj->rocks[idx].mbp)/2) {
    return(0);
  }
  return(1);
} /* end move_idx */


/* +++ external functions +++ */

static int set_network() {
/* all players: set struct pointers to network variables for all rocks,
** return 0=OK or -1=error
*/
  int idx;
  short * hptr;
  if (obj==NULL) {fprintf(stderr,"o_rock.set_network: no object\n"); return(-1);}
  hptr=(short *)vg_nw_getcommon("OROCK_XPOS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->rocks[idx].n_xpos=&hptr[idx];}
  hptr=(short *)vg_nw_getcommon("OROCK_YPOS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->rocks[idx].n_ypos=&hptr[idx];}
  hptr=(short *)vg_nw_getcommon("OROCK_STATUS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->rocks[idx].n_status=&hptr[idx];}
  hptr=(short *)vg_nw_getcommon("OROCK_SPRITEPOS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->rocks[idx].n_spritepos=&hptr[idx];}
  obj->n_snd_hit=(char *)vg_nw_getcommon("OROCK_SND_HIT");
  return(0);
} /* end set_network */


static void refresh() {
/* network-master: called before entering a new loop */
  obj->n_snd_hit[0]=0;  /* reset rock hits */
} /* end refresh */


static void move() {
/* network-master: moving rocks */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    if (move_idx(idx)==0) {new_idx(idx);}  /* rock dead: create new one */
    /* set network pointers */
    obj->rocks[idx].n_xpos[0]=(short)obj->rocks[idx].xpos;
    obj->rocks[idx].n_ypos[0]=(short)obj->rocks[idx].ypos;
    obj->rocks[idx].n_status[0]=(short)obj->rocks[idx].status;
    obj->rocks[idx].n_spritepos[0]=(short)vg_nw_dumpsprite(obj->rocks[idx].bild);
  }
  /* vg_nw_sendcommon() must be called in main-program */
} /* end move */


static void getpos() {
/* network-clients: get positions of rocks */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    obj->rocks[idx].xpos=(int)obj->rocks[idx].n_xpos[0];
    obj->rocks[idx].ypos=(int)obj->rocks[idx].n_ypos[0];
    obj->rocks[idx].status=(int)obj->rocks[idx].n_status[0];
    vg_nw_undumpsprite(obj->rocks[idx].bild,(int)obj->rocks[idx].n_spritepos[0]);
    obj->rocks[idx].mbp=vg_sprite_getcurrent(obj->rocks[idx].bild);
  }
  for (idx=0;idx<(int)obj->n_snd_hit[0];idx++) {  /* play rock-hits */
    vg_sound_play("s-rockhit",0,1);
  }
  obj->n_snd_hit[0]=0;  /* reset */
} /* end getpos */


static void giveout() {
/* all players: giving out rocks */
  int idx;
  bitmap * exbmp,* bmptr;
  if (obj==NULL) {return;}
  exbmp=vg_sprite_getnext(expl);
  if (exbmp==NULL) {exbmp=vg_sprite_getnext(expl);}
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->rocks[idx].status) {bmptr=exbmp;} else {bmptr=obj->rocks[idx].mbp;}
    vg_bitmap_copyto(NULL,(int)obj->rocks[idx].xpos,(int)obj->rocks[idx].ypos,bmptr,0,0,0,0,RGB_TRANS);
  }
} /* end giveout */


/* +++ interface functions +++ */

static int overlapped(int isrock,struct ovlap * ovl,bitmap * bmp,int x1,int y1,int x2,int y2) {
/* check all rocks with passed bitmap for overlapping, return the nearest or -1 */
  int idx,steps,bestidx;
  struct ovlap mvl;
  if (bmp==NULL) {return(-1);}
  steps=bestidx=-1;
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->rocks[idx].status) {continue;}  /* exploding */
    if (bmp==obj->rocks[idx].mbp) {continue;}  /* don't compare with itself */
    if (vg_bitmap_overlap(&mvl,bmp,x1,y1,x2,y2,obj->rocks[idx].mbp,(int)obj->rocks[idx].xpos,(int)obj->rocks[idx].ypos,1)) {
      /* mvl.step=0 is also overlapping, but we ignore it, if checking with another rock */
      if ((!isrock || mvl.step>0) && (steps==-1 || mvl.step<steps)) {
        if (ovl!=NULL) {memmove(ovl,&mvl,sizeof(struct ovlap));}
        steps=mvl.step;
        bestidx=idx;
      }
    }
  }
  return(bestidx);
} /* end overlapped */


static void change_direction(int idx,float * xadd,float * yadd) {
/* exchange xadd and yadd of idx with passed values, return previous */
  float f1;
  if (idx<0 || idx>=obj->anzahl) {return;}
  if (xadd!=NULL) {
    f1=obj->rocks[idx].xadd;
    obj->rocks[idx].xadd=*xadd;
    *xadd=f1;
  }
  if (yadd!=NULL) {
    f1=obj->rocks[idx].yadd;
    obj->rocks[idx].yadd=*yadd;
    *yadd=f1;
  }
} /* end change_direction */


static void set_dead(int idx) {
/* set idx exploding */
  if (idx<0 || idx>=obj->anzahl) {return;}
  obj->rocks[idx].status=20;
  vg_sound_play("s-rockhit",0,1);
  obj->n_snd_hit[0]++;  /* increment */
} /* end set_dead */
