/* Ship object */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames2.h>
#include "main.h"
#include "o_ship.h"

extern void ifoship_move(struct o_ship *,int);  /* interface call */
extern int ifoship_shot(struct o_ship *,int);  /* interface call */

static struct o_ship * obj=NULL;  /* object for ships */
static sprite * fspt=NULL;        /* ship's acceleration */
static sprite * expl=NULL;        /* ship-exploding sprite */
static int myidx=0;

/* initializing and destroying functions */
struct o_ship * new_ship(void);  /* create ship objects */
struct o_ship * obj_ship(void);  /* get pointer to objects (interface) */
void free_ship(void);            /* free ship objects */

/* external functions (called from main program) */
static int set_network(int);    /* set struct pointers to network variables */
static void refresh(void);      /* (master:) called before entering a new loop */
static void move(void);         /* (master:) move ships */
static void getpos(void);       /* (clients:) get positions of ships */
static void giveout(void);      /* draw ships on screen */
static void killship(void);     /* kill your ship */
static int get_credits(void);   /* return credits */

/* interface functions (called from an interface) */
static int overlapped(struct ovlap *,bitmap *,int,int,int,int);  /* ships overlap with a bitmap? */
static void set_dead(int);  /* set a ship dead */
static void inc_credit(int);  /* increment credit */


/* +++ initializing and destroying functions +++ */

struct o_ship * new_ship() {
/* all players: create o_ship-object, return it or NULL=error */
  bitmap * bmp;

  /* create object */
  if ((obj=calloc(1,sizeof(struct o_ship)))==NULL) {
    fprintf(stderr,"new_ship: calloc: %s\n",strerror(errno));
    return(NULL);
  }
  obj->anzahl=0;
  obj->ships=NULL;

  /* set function pointers */
  obj->set_network=set_network;
  obj->refresh=refresh;
  obj->move=move;
  obj->getpos=getpos;
  obj->giveout=giveout;
  obj->killship=killship;
  obj->get_credits=get_credits;
  obj->overlapped=overlapped;
  obj->set_dead=set_dead;
  obj->inc_credit=inc_credit;

  /* create player-data network variables for n_-variables */
  if (vg_nw_setvar(NWVAR_USHORT,1,"OSHIP_XYPOS")<1) {return(NULL);}
  if (vg_nw_setvar(NWVAR_UCHAR,1,"OSHIP_STATUS")<1) {return(NULL);}
  if (vg_nw_setvar(NWVAR_CHAR,1,"OSHIP_ACCEL")<1) {return(NULL);}
  if (vg_nw_setvar(NWVAR_INT,1,"OSHIP_CREDIT")<1) {return(NULL);}
  if (vg_nw_setvar(NWVAR_CHAR,1,"OSHIP_SHIPSHOT")<1) {return(NULL);}
  if (vg_nw_setvar(NWVAR_CHAR,1,"OSHIP_SHIPHIT")<1) {return(NULL);}

  /* create acceleration sprite */
  if ((fspt=vg_sprite_createnew())==NULL) {return(NULL);}
  if ((bmp=vg_bitmap_createnew(5,7))==NULL) {return(NULL);}
  vg_draw_line(bmp,0,0,2,6,vg_color_index(CL_YELLOW,100));
  vg_draw_line(bmp,4,0,2,6,vg_color_index(CL_YELLOW,100));
  vg_draw_fillout(bmp,2,0,vg_color_index(CL_RED,100));
  if (vg_sprite_add(fspt,bmp,2,NULL,0)<0) {return(NULL);}
  vg_draw_line(bmp,0,0,2,6,vg_color_index(CL_RED,100));
  vg_draw_line(bmp,4,0,2,6,vg_color_index(CL_RED,100));
  vg_draw_fillout(bmp,2,0,vg_color_index(CL_YELLOW,100));
  if (vg_sprite_add(fspt,bmp,2,NULL,0)<0) {return(NULL);}
  vg_bitmap_free(bmp);

  /* load exploding sprite */
  if ((expl=vg_sprite_createfromfile("sprites/exploding.sprite"))==NULL) {return(NULL);}

  /* load soundfiles */
  vg_sound_attach("sounds/ship_hit.wav","s-shiphit",100);
  vg_sound_attach("sounds/ship_shot.wav","s-shipshot",30);

  return(obj);
} /* end new_ship */


struct o_ship * obj_ship() {
/* interface: returns o_ship-object */
  return(obj);
} /* end obj_ship */


void free_ship() {
/* all players: destroys ships */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    vg_bitmap_free(obj->ships[idx].bmp);
  }
  if (obj->anzahl>0) {free(obj->ships);}
  free(obj); obj=NULL;
  vg_sprite_free(fspt); fspt=NULL;
  vg_sprite_free(expl); expl=NULL;
  vg_sound_stop("s-shiphit",0);
  vg_sound_stop("s-shipshot",0);
} /* end free_ship */


/* +++ external functions +++ */

static int set_network(int anzahl) {
/* all players: create ships
**              and set struct pointers to network variables for all ships
** 1.arg: number of ships to be created
** return: 0=OK or -1=error
*/
  int idx;
  char buf[512];
  if (obj==NULL) {fprintf(stderr,"o_ship.set_network: no object\n"); return(-1);}

  /* create ships */
  if (anzahl<1 || anzahl>NW_MAXCLIENT) {
    fprintf(stderr,"o_ship.set_network: wrong argument: %d\n",anzahl);
    return(-1);
  }
  obj->anzahl=anzahl;
  if ((obj->ships=calloc(anzahl,sizeof(*obj->ships)))==NULL) {
    fprintf(stderr,"o_ship.set_network: calloc: %s\n",strerror(errno));
    return(-1);
  }

  /* initialize ships */
  for (idx=0;idx<obj->anzahl;idx++) {
    snprintf(buf,sizeof(buf),"bitmaps/ship%d.vga",idx+1);
    if ((obj->ships[idx].bmp=vg_bitmap_createfromfile(buf))==NULL) {return(-1);}
    obj->ships[idx].xpos=SC_WIDTH/(anzahl+1)*(idx+1);
    obj->ships[idx].ypos=SC_HEIGHT-vg_bitmap_height(obj->ships[idx].bmp)/2-10;
    obj->ships[idx].xadd=obj->ships[idx].yadd=0;
    obj->ships[idx].status=1;
    obj->ships[idx].shotpause=0;
    obj->ships[idx].accel=0;
    obj->ships[idx].credit=0;
    obj->ships[idx].plno=idx+1;  /* player-number of the network player */
    if (obj->ships[idx].plno==vg_nw_myplayer()) {myidx=idx;}  /* save own indexnumber */
    /* set network pointers */
    obj->ships[idx].n_xypos=(unsigned short *)vg_nw_getvar("OSHIP_XYPOS",idx+1);
    obj->ships[idx].n_status=(unsigned char *)vg_nw_getvar("OSHIP_STATUS",idx+1);
    obj->ships[idx].n_accel=(char *)vg_nw_getvar("OSHIP_ACCEL",idx+1);
    obj->ships[idx].n_credit=(int *)vg_nw_getvar("OSHIP_CREDIT",idx+1);
    obj->ships[idx].n_shipshot=(char *)vg_nw_getvar("OSHIP_SHIPSHOT",idx+1);
    obj->ships[idx].n_shiphit=(char *)vg_nw_getvar("OSHIP_SHIPHIT",idx+1);
    obj->ships[idx].n_shipshot[0]=0;  /* 0=no sound, 1=sound running, 2=sound paused */
    obj->ships[idx].n_shiphit[0]=0;
  }

  return(0);
} /* end set_network */


static void refresh() {
/* network-master: called before entering a new loop */
  ; /* nothing to do */
} /* end refresh */


static void move() {
/* network-master: moving ships */
  int idx,plno,ix,iy,iw,ih;
  if (obj==NULL) {return;}

  for (idx=0;idx<obj->anzahl;idx++) {
    plno=obj->ships[idx].plno;  /* player-number */
    if (!vg_nw_isalive(plno)) {continue;}  /* dead */

    obj->ships[idx].n_shiphit[0]=0;  /* reset */
    if (obj->ships[idx].status>1) {  /* exploding */
      if (--obj->ships[idx].status==1) {  /* set dead */
        obj->ships[idx].status=0;
        vg_nw_setdead(plno);
      }

    } else {  /* ok */
      /* set xadd and yadd for moving */
      obj->ships[idx].xadd=obj->ships[idx].yadd=0;

      if (plno==vg_nw_myplayer()) {  /* master's player */
        /* vg_key_update() must be done in main-program */
        if (vg_key_pressed(KEY_DCURS,LONGKEY)) {obj->ships[idx].yadd=2;}   /* down */
        if (vg_key_pressed(KEY_UCURS,LONGKEY)) {obj->ships[idx].yadd=-2;}  /* up */
        if (vg_key_pressed(KEY_LCURS,LONGKEY)) {obj->ships[idx].xadd=-2;}  /* left */
        if (vg_key_pressed(KEY_RCURS,LONGKEY)) {obj->ships[idx].xadd=2;}   /* right */

        if (obj->ships[idx].shotpause>0) {obj->ships[idx].shotpause--;}
        if (vg_key_pressed(KEY_LCTRL,LONGKEY)) {  /* shot request */
          if (ifoship_shot(obj,idx)>0) {  /* shot released */
            obj->ships[idx].shotpause=5;
            if (obj->ships[idx].n_shipshot[0]==0) {
              vg_sound_play("s-shipshot",0,0);
            } else if (obj->ships[idx].n_shipshot[0]==2) {vg_sound_catb("s-shipshot");}
            obj->ships[idx].n_shipshot[0]=1;
          }
        } else if (obj->ships[idx].n_shipshot[0]==1) {vg_sound_pate("s-shipshot"); obj->ships[idx].n_shipshot[0]=2;}

      } else if (plno >= vg_nw_virtualplayer()) {  /* virtual player */
        /* move the virtual player synchronously to the master player */
        if (vg_key_pressed(KEY_DCURS,LONGKEY)) {obj->ships[idx].yadd=2;}   /* down */
        if (vg_key_pressed(KEY_UCURS,LONGKEY)) {obj->ships[idx].yadd=-2;}  /* up */
        if (vg_key_pressed(KEY_LCURS,LONGKEY)) {obj->ships[idx].xadd=-2;}  /* left */
        if (vg_key_pressed(KEY_RCURS,LONGKEY)) {obj->ships[idx].xadd=2;}   /* right */

        if (obj->ships[idx].shotpause>0) {obj->ships[idx].shotpause--;}
        if (vg_key_pressed(KEY_LCTRL,LONGKEY)) {  /* shot request */
          if (ifoship_shot(obj,idx)>0) {  /* shot released */
            obj->ships[idx].shotpause=5;
          }
        }

      } else {  /* client's player */
        if (vg_nw_recvkeys(plno)) {  /* client has sent a request */
          if (vg_nw_keypressed(KEY_DCURS,LONGKEY)) {obj->ships[idx].yadd=2;}   /* down */
          if (vg_nw_keypressed(KEY_UCURS,LONGKEY)) {obj->ships[idx].yadd=-2;}  /* up */
          if (vg_nw_keypressed(KEY_LCURS,LONGKEY)) {obj->ships[idx].xadd=-2;}  /* left */
          if (vg_nw_keypressed(KEY_RCURS,LONGKEY)) {obj->ships[idx].xadd=2;}   /* right */
        }

        if (obj->ships[idx].shotpause>0) {obj->ships[idx].shotpause--;}
        if (vg_nw_keypressed(KEY_LCTRL,LONGKEY)) {  /* shot request */
          if (ifoship_shot(obj,idx)>0) {  /* shot released */
            obj->ships[idx].shotpause=5;
            obj->ships[idx].n_shipshot[0]=1;
          }
        } else if (obj->ships[idx].n_shipshot[0]==1) {obj->ships[idx].n_shipshot[0]=2;}
      }

      /* check for screen-boundaries */
      ix=obj->ships[idx].xpos+obj->ships[idx].xadd;
      iy=obj->ships[idx].ypos+obj->ships[idx].yadd;
      iw=vg_bitmap_width(obj->ships[idx].bmp);
      ih=vg_bitmap_height(obj->ships[idx].bmp);
      if (ix<iw/2) {ix=iw/2;}  /* left boundary */
      if (ix>SC_WIDTH-iw/2) {ix=SC_WIDTH-iw/2;}  /* right boundary */
      if (iy<ih/2) {iy=ih/2;}  /* top boundary */
      if (iy>SC_HEIGHT-ih/2) {iy=SC_HEIGHT-ih/2;}  /* bottom boundary */
      obj->ships[idx].xadd=ix-obj->ships[idx].xpos;  /* correct moving x value */
      obj->ships[idx].yadd=iy-obj->ships[idx].ypos;  /* correct moving y value */

      if (obj->ships[idx].yadd<0) {  /* accelerating */
        obj->ships[idx].accel=1;
      } else {obj->ships[idx].accel=0;}

      /* move ship, checking for overlapping */
      ifoship_move(obj,idx);
    }

    /* set rest of network pointers and make network-update for player */
    obj->ships[idx].n_xypos[0]=(unsigned short)(obj->ships[idx].ypos*SC_WIDTH+obj->ships[idx].xpos);
    obj->ships[idx].n_status[0]=(unsigned char)obj->ships[idx].status;
    obj->ships[idx].n_accel[0]=(char)obj->ships[idx].accel;
    obj->ships[idx].n_credit[0]=(int)obj->ships[idx].credit;
    vg_nw_senddata(plno);
  }
} /* end move */


static void getpos() {
/* network-clients: get positions of ships */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    obj->ships[idx].xpos=(int)obj->ships[idx].n_xypos[0]%SC_WIDTH;
    obj->ships[idx].ypos=(int)obj->ships[idx].n_xypos[0]/SC_WIDTH;
    obj->ships[idx].status=(int)obj->ships[idx].n_status[0];
    obj->ships[idx].accel=(int)obj->ships[idx].n_accel[0];
    obj->ships[idx].credit=(int)obj->ships[idx].n_credit[0];
    if (obj->ships[idx].n_shiphit[0]==1) {  /* play sound shiphit */
      vg_sound_play("s-shiphit",0,1);
      obj->ships[idx].n_shiphit[0]=0;
      if (obj->ships[idx].plno==vg_nw_myplayer()) {vg_sound_stop("s-shipshot",0);}
    }
    if (obj->ships[idx].plno==vg_nw_myplayer()) {  /* play/pause sound shipshot */
      static int running=0;
      if (obj->ships[idx].n_shipshot[0]!=running) {
        if (obj->ships[idx].n_shipshot[0]==1) {
          if (!running) {
            vg_sound_play("s-shipshot",0,0);
          } else {
            vg_sound_catb("s-shipshot");
          }
        } else if (obj->ships[idx].n_shipshot[0]==2) {
          vg_sound_pate("s-shipshot");
        }
        running=obj->ships[idx].n_shipshot[0];
      }
    }
  }
} /* end getpos */


static void giveout() {
/* all players: giving out ships */
  int idx,col;
  bitmap * fbmp,* ebmp;
  char buf[32];
  if (obj==NULL) {return;}
  /* get next acceleration sprite-bitmap */
  if ((fbmp=vg_sprite_getnext(fspt))==NULL) {fbmp=vg_sprite_getnext(fspt);}

  /* get next exploding sprite-bitmap */
  if ((ebmp=vg_sprite_getnext(expl))==NULL) {ebmp=vg_sprite_getnext(expl);}

  for (idx=0;idx<obj->anzahl;idx++) {
    snprintf(buf,sizeof(buf),"%d",obj->ships[idx].credit);
    col=vg_color_index(CL_WHITE,(obj->ships[idx].plno==vg_nw_myplayer()?100:50));  /* make credits of own player bold */
    vg_draw_text(NULL,col,SC_WIDTH/(obj->anzahl+1)*(idx+1),SC_HEIGHT-10,buf,NULL,RGB_TRANS);
    if (!vg_nw_isalive(idx+1)) {continue;}
    if (obj->ships[idx].status==1) {
      vg_bitmap_copyto(NULL,obj->ships[idx].xpos,obj->ships[idx].ypos,obj->ships[idx].bmp,0,0,0,0,RGB_TRANS);
      if (obj->ships[idx].accel) {  /* accelerating */
        int xpos=obj->ships[idx].xpos;
        int ypos=obj->ships[idx].ypos+vg_bitmap_height(obj->ships[idx].bmp)/2+vg_bitmap_height(fbmp)/2+2;
        vg_bitmap_copyto(NULL,xpos,ypos,fbmp,0,0,0,0,RGB_TRANS);
      }
    } else if (obj->ships[idx].status>1) {  /* exploding sprite */
      vg_bitmap_copyto(NULL,obj->ships[idx].xpos,obj->ships[idx].ypos,ebmp,0,0,0,0,RGB_TRANS);
    }
  }
} /* end giveout */


static void killship() {
/* all players: kill your ship immediately */
  vg_sound_stop("s-shipshot",0);
  vg_nw_setdead(vg_nw_myplayer());  /* set dead to inform network-server */
} /* end killship */


static int get_credits() {
/* return credits */
  return(obj->ships[myidx].credit);
} /* end get_credits */


/* +++ interface functions +++ */

static int overlapped(struct ovlap * ovl,bitmap * bmp,int x1,int y1,int x2,int y2) {
/* check all ships with passed bitmap for overlapping, return the nearest or -1 */
  int idx,steps,bestidx;
  struct ovlap mvl;
  if (bmp==NULL) {return(-1);}
  steps=bestidx=-1;
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->ships[idx].status!=1 || !vg_nw_isalive(idx+1)) {continue;}  /* exploding or dead */
    if (bmp==obj->ships[idx].bmp) {continue;}  /* don't compare with itself */
    if (vg_bitmap_overlap(&mvl,bmp,x1,y1,x2,y2,obj->ships[idx].bmp,obj->ships[idx].xpos,obj->ships[idx].ypos,1)) {
      if (steps==-1 || mvl.step<steps) {
        if (ovl!=NULL) {memmove(ovl,&mvl,sizeof(struct ovlap));}
        steps=mvl.step;
        bestidx=idx;
      }
    }
  }
  return(bestidx);
} /* end overlapped */


static void set_dead(int idx) {
/* set idx exploding */
  if (idx<0 || idx>=obj->anzahl) {return;}
  obj->ships[idx].status=80;
  vg_sound_play("s-shiphit",0,1);
  obj->ships[idx].n_shiphit[0]=1;
  if (obj->ships[idx].plno==vg_nw_myplayer()) {vg_sound_stop("s-shipshot",0);}
} /* end set_dead */


static void inc_credit(int idx) {
/* increment credit for a player */
  if (idx<0 || idx>=obj->anzahl) {return;}
  obj->ships[idx].credit++;
} /* end inc_credit */
