/* Shot object */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <vgagames2.h>
#include "main.h"
#include "o_shot.h"

extern void ifoshot_move(struct o_shot *,int);  /* interface call */

static struct o_shot * obj=NULL;  /* object for shots */
static int maxships=0;

/* initializing and destroying functions */
struct o_shot * new_shot(int);   /* create shot objects */
struct o_shot * obj_shot(void);  /* get pointer to objects (interface) */
void free_shot(void);            /* free shot objects */

/* external functions (called from main program) */
static int set_network(int);       /* set struct pointers to network variables */
static void refresh(void);         /* (master:) called before entering a new loop */
static void move(void);            /* (master:) move shots */
static void getpos(void);          /* (clients:) get positions of shots */
static void giveout(void);         /* draw shots on screen */
static int activate(int,int,int);  /* (ships:) make a shot active */

/* interface functions (called from an interface) */
static int overlapped(struct ovlap *,bitmap *,int,int,int,int);  /* shots overlap with a bitmap? */
static void set_inactive(int);  /* set a shot inactive */
static int get_ship(int);  /* return ship's number */


/* +++ initializing and destroying functions +++ */

struct o_shot * new_shot(int anzahl) {
/* all players: create o_shot-object, return it or NULL=error */
  int idx;

  /* create object */
  if ((obj=calloc(1,sizeof(struct o_shot)))==NULL) {
    fprintf(stderr,"new_shot: calloc: %s\n",strerror(errno));
    return(NULL);
  }

  /* set function pointers */
  obj->set_network=set_network;
  obj->refresh=refresh;
  obj->move=move;
  obj->getpos=getpos;
  obj->giveout=giveout;
  obj->activate=activate;
  obj->overlapped=overlapped;
  obj->set_inactive=set_inactive;
  obj->get_ship=get_ship;

  /* create shots */
  obj->anzahl=anzahl;
  if ((obj->shots=calloc(anzahl,sizeof(*obj->shots)))==NULL) {
    fprintf(stderr,"new_shot: calloc: %s\n",strerror(errno));
    return(NULL);
  }

  /* initialize shots */
  for (idx=0;idx<obj->anzahl;idx++) {
    if ((obj->shots[idx].bmp=vg_bitmap_createnew(1,5))==NULL) {return(NULL);}
    vg_draw_line(obj->shots[idx].bmp,0,0,0,4,vg_color_index(CL_ORANGE,100));
    obj->shots[idx].xpos=obj->shots[idx].ypos=0;
    obj->shots[idx].yadd=-5;
    obj->shots[idx].status=0;
    /* network pointers are set later in set_network() */
    obj->shots[idx].n_xypos=NULL;
    obj->shots[idx].n_status=NULL;
  }

  /* create network arrays for n_xypos, n_status.
  ** One element of them belongs to each shot.
  */
  if (vg_nw_setcommon(NWVAR_USHORT,obj->anzahl,"OSHOT_XYPOS")<1) {return(NULL);}
  if (vg_nw_setcommon(NWVAR_UCHAR,obj->anzahl,"OSHOT_STATUS")<1) {return(NULL);}

  maxships=0;
  return(obj);
} /* end new_shot */


struct o_shot * obj_shot() {
/* interface: returns o_shot-object */
  return(obj);
} /* end obj_shot */


void free_shot() {
/* all players: destroys shots */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    vg_bitmap_free(obj->shots[idx].bmp);
  }
  free(obj->shots);
  free(obj); obj=NULL;
  maxships=0;
} /* end free_shot */


/* +++ external functions +++ */

static int set_network(int ships) {
/* all players: set struct pointers to network variables for all shots,
** return 0=OK or -1=error
*/
  int idx;
  unsigned short * hptr;
  unsigned char * cptr;
  if (obj==NULL) {fprintf(stderr,"o_shot.set_network: no object\n"); return(-1);}
  hptr=(unsigned short *)vg_nw_getcommon("OSHOT_XYPOS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->shots[idx].n_xypos=&hptr[idx];}
  cptr=(unsigned char *)vg_nw_getcommon("OSHOT_STATUS");
  for (idx=0;idx<obj->anzahl;idx++) {obj->shots[idx].n_status=&cptr[idx];}
  maxships=ships;
  return(0);
} /* end set_network */


static void refresh() {
/* network-master: called before entering a new loop */
  ; /* nothing to do */
} /* end refresh */


static void move() {
/* network-master: moving shots */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->shots[idx].status>0) {  /* active */
      /* move shot, checking for overlapping */
      ifoshot_move(obj,idx);
      /* if shot is out of window, kill it */
      if (obj->shots[idx].ypos < -vg_bitmap_height(obj->shots[idx].bmp)/2) {
        obj->shots[idx].status=0;
      }
    }
    /* set network pointers */
    obj->shots[idx].n_xypos[0]=(unsigned short)(obj->shots[idx].ypos*SC_WIDTH+obj->shots[idx].xpos);
    obj->shots[idx].n_status[0]=(unsigned char)obj->shots[idx].status;
  }
  /* vg_nw_sendcommon() must be called in main-program */
} /* end move */


static void getpos() {
/* network-clients: get positions of shots */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    obj->shots[idx].xpos=(int)obj->shots[idx].n_xypos[0]%SC_WIDTH;
    obj->shots[idx].ypos=(int)obj->shots[idx].n_xypos[0]/SC_WIDTH;
    obj->shots[idx].status=(int)obj->shots[idx].n_status[0];
  }
} /* end getpos */


static void giveout() {
/* all players: giving out shots */
  int idx;
  if (obj==NULL) {return;}
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->shots[idx].status>0) {
      vg_bitmap_copyto(NULL,obj->shots[idx].xpos,obj->shots[idx].ypos,obj->shots[idx].bmp,0,0,0,0,RGB_TRANS);
    }
  }
} /* end giveout */


static int activate(int shipidx,int xpos,int ypos) {
/* ships: activate shot
** 1.arg: ship's indexnumber
** return: 1=activated, 0=not activated
*/
  int idx,anz,frei;
  if (shipidx<0) {return(0);}
  shipidx++;  /* prevent indexnumber 0 */
  anz=0; frei=-1;
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->shots[idx].status==shipidx) {anz++;}
    else if (obj->shots[idx].status==0) {frei=idx;}
  }
  if (frei>=0 && anz < obj->anzahl/maxships) {
    obj->shots[frei].status=shipidx;
    obj->shots[frei].xpos=xpos;
    obj->shots[frei].ypos=ypos;
    /* set network pointers */
    obj->shots[frei].n_xypos[0]=(unsigned short)(obj->shots[frei].ypos*SC_WIDTH+obj->shots[frei].xpos);
    obj->shots[frei].n_status[0]=(unsigned char)obj->shots[frei].status;
    /* vg_nw_sendcommon() must be called in main-program */
    return(1);
  }
  return(0);
} /* end activate */


/* +++ interface functions +++ */

static int overlapped(struct ovlap * ovl,bitmap * bmp,int x1,int y1,int x2,int y2) {
/* check all shots with passed bitmap for overlapping, return the nearest or -1 */
  int idx,steps,bestidx;
  struct ovlap mvl;
  if (bmp==NULL) {return(-1);}
  steps=bestidx=-1;
  for (idx=0;idx<obj->anzahl;idx++) {
    if (obj->shots[idx].status==0) {continue;}  /* inactive */
    if (bmp==obj->shots[idx].bmp) {continue;}  /* don't compare with itself */
    if (vg_bitmap_overlap(&mvl,bmp,x1,y1,x2,y2,obj->shots[idx].bmp,obj->shots[idx].xpos,obj->shots[idx].ypos,1)) {
      if (steps==-1 || mvl.step<steps) {
        if (ovl!=NULL) {memmove(ovl,&mvl,sizeof(struct ovlap));}
        steps=mvl.step;
        bestidx=idx;
      }
    }
  }
  return(bestidx);
} /* end overlapped */


static void set_inactive(int idx) {
/* set idx inactive */
  if (idx<0 || idx>=obj->anzahl) {return;}
  obj->shots[idx].status=0;
} /* end set_inactive */


static int get_ship(int idx) {
/* return ship's indexnumber */
  if (idx<0 || idx>=obj->anzahl) {return(-1);}
  return(obj->shots[idx].status-1);
} /* end get_ship */
