/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* bitmap functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <math.h>
#include "config.h"
#include "bitmap.h"
#include "color.h"
#include "font.h"
#include "utf8.h"
#include "vga_conv.h"

#ifndef M_PI
# define M_PI  3.14159265358979323846
#endif
#ifndef PI
# define PI  M_PI
#endif
#ifndef PI2
# define PI2  (M_PI + M_PI)
#endif

bitmap * backbuffer=NULL;

bitmap * vg_bitmap_createnew(int,int);
bitmap * vg_bitmap_createfromfile(const char *);
bitmap * vg_bitmap_createfromtext(int,int,int,int,const char *,const char *);
bitmap * vg_bitmap_duplicate(const bitmap *);
int vg_bitmap_width(const bitmap *);
int vg_bitmap_height(const bitmap *);
int vg_bitmap_getpixel(const bitmap *,int,int);
int vg_bitmap_save(const bitmap *,const char *,int);
void vg_bitmap_clear(bitmap *,int);
void vg_bitmap_copyto(bitmap *,int,int,const bitmap *,int,int,int,int,int);
bitmap * vg_bitmap_rotate(const bitmap *,int);
bitmap * vg_bitmap_zoom(const bitmap *,double,double);
bitmap * vg_bitmap_mirror(const bitmap *,int);
void vg_bitmap_free(bitmap *);
int vg_bitmap_overlap(struct ovlap *,const bitmap *,int,int,int,int,const bitmap *,int,int,int);
static int overlapping(const bitmap *,int,int,const bitmap *,int,int,int);


/* +++ functions +++ */


bitmap * vg_bitmap_createnew(int w,int h) {
/* create a new empty bitmap
** 1.+2.arg: width and height of the bitmap
** return: bitmap
**         or NULL = error
*/
  bitmap * grf;
  if ((w<=0) || (h<=0)) {fprintf(stderr,"vg_bitmap_createnew: invalid argument (less equal 0).\n"); return(NULL);}
  if (w*h/h!=w) {fprintf(stderr,"vg_bitmap_createnew: invalid argument (overflow).\n"); return(NULL);}
  if ((grf=malloc(sizeof(bitmap)))==NULL) {fprintf(stderr,"vg_bitmap_createnew: malloc: %s.\n",strerror(errno)); return(NULL);}
  if ((grf->pxm=malloc(sizeof(unsigned char)*(w*h)))==NULL) {fprintf(stderr,"vg_bitmap_createnew: malloc: %s.\n",strerror(errno)); return(NULL);}
  memset(grf->pxm,RGB_BLACK,w*h);
  grf->width=w; grf->height=h;
  return(grf);
} /* Ende vg_bitmap_createnew */


bitmap * vg_bitmap_createfromfile(const char * filename) {
/* load a bitmap from a file, has to be the vgagames format
** 1.arg: filename with relative path to game-directory
** return: bitmap
**         or NULL = error
*/
  int w,h;
  unsigned char kv[256];
  char buf[1024];
  bitmap * grf;
  FILE * ffp;

  /* read bitmap and allocate memory */
  if (filename==NULL) {fprintf(stderr,"vg_bitmap_createfromfile: invalid argument (NULL).\n"); return(NULL);}
  if (*filename!='/') {
    snprintf(buf,sizeof(buf),"%s/%s",*_cwdir2=='\0'?CWDIR:_cwdir2,filename);
  } else {snprintf(buf,sizeof(buf),"%s",filename);}
  if ((ffp=fopen(buf,"r"))==NULL) {fprintf(stderr,"vg_bitmap_createfromfile: cannot open file \"%s\".\n",filename); return(NULL);}
  if ((ffp=convtovga(ffp))==NULL) {fprintf(stderr,"vg_bitmap_createfromfile: error converting file \"%s\".\n",filename); return(NULL);}
  if (fread(buf,sizeof(char),5,ffp)!=5) {fprintf(stderr,"vg_bitmap_createfromfile: error reading file \"%s\".\n",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  w=atoi(buf);
  if (fread(buf,sizeof(char),5,ffp)!=5) {fprintf(stderr,"vg_bitmap_createfromfile: error reading file \"%s\".\n",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  h=atoi(buf);
  if ((grf=vg_bitmap_createnew(w,h))==NULL) {fprintf(stderr,"vg_bitmap_createfromfile: error calling vg_bitmap_createnew(%d,%d).\n",w,h); fclose(ffp); return(NULL);}

  /* read and convert colormap */
  if (konv_map(kv,ffp)<0) {fprintf(stderr,"vg_bitmap_createfromfile: error calling konv_map.\n"); fclose(ffp); vg_bitmap_free(grf); return(NULL);}

  /* read and convert pixels */
  fread(grf->pxm,sizeof(unsigned char),grf->width*grf->height,ffp);
  fclose(ffp);
  w=grf->width*grf->height;
  for (h=0;h<w;h++) {grf->pxm[h]=kv[(int)grf->pxm[h]];}

  return(grf);
} /* Ende vg_bitmap_createfromfile */


bitmap * vg_bitmap_createfromtext(int fc,int bc,int zpx,int ausr,const char * text,const char * fontXxY) {
/* create a bitmap containing text. Newlines or "\n" serve for line breaks.
** 1.arg: text color
** 2.arg: background color
** 3.arg: pixel between lines
** 4.arg: 0=left adjusted or 1=centered or 2=right adjusted
** 5.arg: text
** 6.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** return: bitmap
**         or NULL = error
*/
  int i1,i2,zlm,spm;
  int font_w,font_h;
  bitmap * grf;
  const char * p1,* p2;
  if (text==NULL) {fprintf(stderr,"vg_bitmap_createfromtext: invalid argument (NULL).\n"); return(NULL);}
  font_w=vg_font_width(fontXxY);
  font_h=vg_font_height(fontXxY);
  if (font_h+zpx<=0) {zpx=-font_h+1;}  /* at least 1 pixel down each line */
  zlm=spm=0;
  text=utf8text(text);
  for (p1=text,p2=strpbrk(p1,"\n\\");;p2=strpbrk(p2,"\n\\")) {
    if (p2==NULL) {p2=p1+strlen(p1);}
    if ((p2[0]=='\\') && (p2[1]!='n')) {p2++; continue;}
    i1=p2-p1;
    if (i1>zlm) {zlm=i1;}
    spm++;
    if (*p2=='\0') {break;}
    if (*p2=='\\') {p2++;}
    p1=p2+1; p2=p1;
  }
  if ((grf=vg_bitmap_createnew(zlm*font_w,spm*(font_h+zpx)-zpx))==NULL) {return(NULL);}
  spm=0;
  for (p1=text,p2=strpbrk(p1,"\n\\");;p2=strpbrk(p2,"\n\\")) {
    if (p2==NULL) {p2=p1+strlen(p1);}
    if ((p2[0]=='\\') && (p2[1]!='n')) {p2++; continue;}
    i1=p2-p1;
    if (ausr==1) {  /* centered */
      i2=(zlm-i1)*font_w/2;
    } else if (ausr==2) {  /* right */
      i2=(zlm-i1)*font_w;
    } else {i2=0;}  /* left */
    draw_fontXxY(grf,fc,i2,spm*(font_h+zpx),p1,i1,fontXxY,RGB_FULL,bc);
    spm++;
    if (*p2=='\0') {break;}
    if (*p2=='\\') {p2++;}
    p1=p2+1; p2=p1;
  }
  return(grf);
} /* Ende vg_bitmap_createfromtext */


bitmap * vg_bitmap_duplicate(const bitmap * sgrf) {
/* duplicate a bitmap, creating a new one
** 1.arg: source bitmap or NULL=backbuffer
** return: duplicated bitmap
**         or NULL = error
*/
  bitmap * dgrf;
  if (sgrf==NULL) {sgrf=backbuffer;}
  if ((dgrf=vg_bitmap_createnew(sgrf->width,sgrf->height))==NULL) {fprintf(stderr,"vg_bitmap_duplicate: error calling vg_bitmap_createnew.\n"); return(NULL);}
  memmove(dgrf->pxm,sgrf->pxm,dgrf->width*dgrf->height);
  return(dgrf);
} /* Ende vg_bitmap_duplicate */


int vg_bitmap_width(const bitmap * grf) {
/* return width of bitmap
** 1.arg: bitmap or NULL=backbuffer
** return: width of bitmap
*/
  if (grf==NULL) {grf=backbuffer;}
  return(grf->width);
} /* Ende vg_bitmap_width */


int vg_bitmap_height(const bitmap * grf) {
/* return height of bitmap
** 1.arg: bitmap or NULL=backbuffer
** return: height of bitmap
*/
  if (grf==NULL) {grf=backbuffer;}
  return(grf->height);
} /* Ende vg_bitmap_height */


int vg_bitmap_getpixel(const bitmap * grf,int x,int y) {
/* get color index of one pixel
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: pixel coordinates x/y
** return: color index
*/
  if (grf==NULL) {grf=backbuffer;}
  if ((x<0) || (x>=grf->width) || (y<0) || (y>=grf->height)) {return(RGB_BLACK);}
  return((int)grf->pxm[y*grf->width+x]);
} /* Ende vg_bitmap_getpixel */


int vg_bitmap_save(const bitmap * grf,const char * filename,int flag) {
/* save bitmap as vgagames- or ppm-file
** 1.arg: bitmap or NULL=backbuffer
** 2.arg: filename to save with relative path to game-directory or absolute
** 3.arg: VGAFORMAT_VGA: for vgagames format
**        VGAFORMAT_BMP: for windows bmp format
**        VGAFORMAT_PPM3: for plain ppm format (P3)
**        VGAFORMAT_PPM6: for raw ppm format (P6)
** return: 0=OK or -1=error
*/
  FILE * ffp,* tfp;
  int i1,i2,gx,gy;
  unsigned char rgb;
  if (filename==NULL) {fprintf(stderr,"vg_bitmap_save: invalid argument (NULL).\n"); return(-1);}
  if (*filename!='/') {
    char buf[1024];
    snprintf(buf,sizeof(buf),"%s/%s",CWDIR,filename);
    unlink(buf);
    if ((ffp=fopen(buf,"w"))==NULL) {fprintf(stderr,"vg_bitmap_save: cannot write file \"%s\": %s.\n",buf,strerror(errno)); return(-1);}
  } else {
    unlink(filename);
    if ((ffp=fopen(filename,"w"))==NULL) {fprintf(stderr,"vg_bitmap_save: cannot write file \"%s\": %s.\n",filename,strerror(errno)); return(-1);}
  }
  gx=vg_bitmap_width(grf); gy=vg_bitmap_height(grf);

  if ((tfp=tmpfile())==NULL) {fprintf(stderr,"vg_bitmap_save: tmpfile: %s.\n",strerror(errno)); fclose(ffp); return(-1);}
  fprintf(tfp,"VG%05d%05d",gx,gy);
  for (i1=0;i1<256;i1++) {  /* colormap */
    fprintf(tfp,"%c%c%c",fcpix[i1].r,fcpix[i1].g,fcpix[i1].b);
  }
  for (i1=0;i1<gy;i1++) {
    for (i2=0;i2<gx;i2++) {
      rgb=(unsigned char)vg_bitmap_getpixel(grf,i2,i1);
      fwrite(&rgb,sizeof(unsigned char),1,tfp);
    }
  }
  fflush(tfp);
  rewind(tfp);
  if (convfromvga(tfp,ffp,flag)<0) {
    fprintf(stderr,"vg_bitmap_save: cannot convert file \"%s\".\n",filename); fclose(ffp); return(-1);
  }
  fclose(ffp);
  return(0);
} /* Ende vg_bitmap_save */


void vg_bitmap_clear(bitmap * grf,int cl) {
/* clear a bitmap with a given color
** 1.arg: bitmap or NULL=backbuffer
** 2.arg: color index
*/
  if (grf==NULL) {grf=backbuffer;}
  memset(grf->pxm,cl,grf->width*grf->height);
} /* Ende vg_bitmap_clear */


void vg_bitmap_copyto(bitmap * dgrf,int xd,int yd,const bitmap * sgrf,int xs,int ys,int ws,int hs,int trsp) {
/* copy a (part of a) bitmap into another one
** 1.arg: destination bitmap or NULL=backbuffer
** 2.+3.arg: destination coordinates: center of where to copy source bitmap
** 4.arg: source bitmap or NULL=backbuffer
** 5.+6.arg: source coordinates: upper left corner of source bitmap
** 7.+8.arg: width and height to copy from source coordinates of source bitmap
**           if width=0 or height=0, then the whole width or height is used
** 9.arg: RGB_FULL for full copying
**        RGB_TRANS for transparent copying
*/
  int w,h;
  unsigned char * dptr,* sptr;
  if (dgrf==sgrf) {return;}
  if (dgrf==NULL) {dgrf=backbuffer;}
  if (sgrf==NULL) {sgrf=backbuffer;}

  if ((xs>=sgrf->width) || (ys>=sgrf->height)) {return;}
  if (ws<1) {ws=sgrf->width-xs;}
  if (hs<1) {hs=sgrf->height-ys;}
  xd-=ws/2; yd-=hs/2;  /* from center to upper left corner */

  if (xs+ws>sgrf->width) {ws=sgrf->width-xs;}
  if (ys+hs>sgrf->height) {hs=sgrf->height-ys;}
  if (xs<0) {ws+=xs; xs=0;}
  if (ys<0) {hs+=ys; ys=0;}
  if ((ws<=0) || (hs<=0)) {return;}

  if ((xd>=dgrf->width) || (yd>=dgrf->height)) {return;}
  if (xd+ws>dgrf->width) {ws=dgrf->width-xd;}
  if (yd+hs>dgrf->height) {hs=dgrf->height-yd;}
  if (xd<0) {ws+=xd; xs-=xd; xd=0;}
  if (yd<0) {hs+=yd; ys-=yd; yd=0;}
  if ((ws<=0) || (hs<=0)) {return;}

  for (h=0;h<hs;h++) {
    dptr=dgrf->pxm+(yd+h)*dgrf->width+xd;
    sptr=sgrf->pxm+(ys+h)*sgrf->width+xs;
    for (w=0;w<ws;w++) {
      if ((trsp==RGB_FULL) || (*sptr!=RGB_BLACK)) {*dptr=*sptr;}
      dptr++; sptr++;
    }
  }
} /* Ende vg_bitmap_copyto */


bitmap * vg_bitmap_rotate(const bitmap * sgrf,int deg) {
/* rotate bitmap
** 1.arg: bitmap or NULL=backbuffer (will not be modified)
** 2.arg: degrees to rotate clockwise (-360 to 360)
** return: pointer to static rotated bitmap
**         or NULL = error
*/
  static bitmap * dgrf=NULL;
  int sxm,sym,dxm,dym,i0,i1,i2,i3,i4,ix,iy,jx,jy,tx,ty;
  float r0,a0,d0,degpi;
  if (sgrf==NULL) {sgrf=backbuffer;}
  if (dgrf!=NULL) {vg_bitmap_free(dgrf); dgrf=NULL;}
  deg%=360; deg+=360; deg%=360;
  degpi=(float)deg/360.*PI2;  /* angle in Rad */

  /* source middle: sxm,sym */
  sxm=sgrf->width/2;
  sym=sgrf->height/2;

  /* destination middle: sxm,sym: find the pixel which is farest away from middle */
  /* pixel 0/0 */
  r0=sqrt((double)(sxm*sxm+sym*sym));  /* distance to middle */
  a0=asin(sym/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i1=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i2=(int)d0;                /* y distance from middle */
  /* pixel 0/(sgrf->height)-1 */
  i3=sgrf->height-1-sym;
  a0=asin(-i3/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i3=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i4=(int)d0;                /* y distance from middle */
  dxm=(i1>i3?i1:i3);  /* calculated destination middle x */
  dym=(i2>i4?i2:i4);  /* calculated destination middle y */
  if ((dgrf=vg_bitmap_createnew(dxm*2+1,dym*2+1))==NULL) {fprintf(stderr,"vg_bitmap_rotate: error calling vg_bitmap_createnew.\n"); return(NULL);}

  /* put middle */
  dgrf->pxm[dym*dgrf->width+dxm]=sgrf->pxm[sym*sgrf->width+sxm];

  /* put rest
  ** related to source middle (0/0) take every loop (variable i0) 4 lines:
  **  - upper source pixel line (1.loop: -1/-1 to  1/-1
  **                             2.loop: -2/-2 to  2/-2
  **                             ... )
  **  - lower source pixel line (1.loop: -1/1  to  1/1
  **                             2.loop: -2/2  to  2/2
  **                             ... )
  **  - left source pixel line  (1.loop: -1/-1 to -1/1
  **                             2.loop: -2/-2 to -2/2
  **                             ... )
  **  - right source pixel line (1.loop:  1/-1 to  1/1
  **                             2.loop:  2/-2 to  2/2
  **                             ... )
  */
  i0=1;  /* distance from source middle */
  do {
    i1=0;
    if (sxm<i0) {i2=-sxm;} else {i2=-i0;}  /* left */
    if (sgrf->width-1-sxm<i0) {i3=sgrf->width-1-sxm;} else {i3=i0;}  /* right */
    if (i0<=sym) {  /* upper source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(sym-i0)*sgrf->width+sxm+i4]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i4*i4+i0*i0));
        a0=asin(i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source pixel in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(sym-i0)*sgrf->width+sxm+i4];
        /* if destination pixel would be near between two pixels
           draw both to avoid black holes if possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(sym-i0)*sgrf->width+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(sym-i0)*sgrf->width+sxm+i4];
        }
      }
      i1=1;
    }
    if (i0<=sgrf->height-1-sym) {  /* lower source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(sym+i0)*sgrf->width+sxm+i4]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i4*i4+i0*i0));
        a0=asin(-i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source pixel in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(sym+i0)*sgrf->width+sxm+i4];
        /* if destination pixel would be near between two pixels
           draw both to avoid black holes if possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(sym+i0)*sgrf->width+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(sym+i0)*sgrf->width+sxm+i4];
        }
      }
      i1=1;
    }
    if (sym<i0) {i2=-sym;} else {i2=-i0;}  /* up */
    if (sgrf->height-1-sym<i0) {i3=sgrf->height-1-sym;} else {i3=i0;}  /* down */
    if (i0<=sxm) {  /* left source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(sym+i4)*sgrf->width+sxm-i0]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i0*i0+i4*i4));
        a0=asin(-i4/r0);
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(sym+i4)*sgrf->width+sxm-i0];
        /* if destination pixel would be near between two pixels
           draw both to avoid black holes if possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(sym+i4)*sgrf->width+sxm-i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(sym+i4)*sgrf->width+sxm-i0];
        }
      }
      i1=1;
    }
    if (i0<=sgrf->width-1-sxm) {  /* right source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(sym+i4)*sgrf->width+sxm+i0]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i0*i0+i4*i4));
        a0=asin(-i4/r0);
        a0=-a0+PI;  /* source pixel in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(sym+i4)*sgrf->width+sxm+i0];
        /* if destination pixel would be near between two pixels
           draw both to avoid black holes if possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(sym+i4)*sgrf->width+sxm+i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(sym+i4)*sgrf->width+sxm+i0];
        }
      }
      i1=1;
    }
    i0++;
  } while (i1==1);
  return(dgrf);
} /* Ende vg_bitmap_rotate */


bitmap * vg_bitmap_zoom(const bitmap * sgrf,double xmul,double ymul) {
/* zoom a bitmap greater or smaller
** 1.arg: bitmap or NULL=backbuffer (will not be modified)
** 2.arg: horizontal zooming (x axis), where 1.0 is original size
** 3.arg: vertical zooming (y axis), where 1.0 is original size
** return: pointer to static zoomed bitmap
**         or NULL = error
*/
  const int tw=1000;
  static bitmap * dgrf=NULL;
  int sxm,sym,dxm,dym,txm,tym,sw,i_z,i_a,i_n,vp,bp,hp,hw,i1,i2,ii;
  int fb[256],brk;
  bitmap * tgrf;
  if (sgrf==NULL) {sgrf=backbuffer;}
  if (dgrf!=NULL) {vg_bitmap_free(dgrf); dgrf=NULL;}
  if (xmul<.01) {xmul=.01;}
  if (ymul<.01) {ymul=.01;}
  xmul+=.001; ymul+=.001;

  /* source middle: sxm,sym */
  sxm=sgrf->width/2;
  sym=sgrf->height/2;

  /* destination middle: dxm,dym */
  i1=sxm;
  if (sxm*2<sgrf->width) {i1++;}
  i1=(int)((double)i1*xmul)*2;
  if (sxm*2<sgrf->width) {i1--;}
  dxm=i1/2;
  i1=sym;
  if (sym*2<sgrf->height) {i1++;}
  i1=(int)((double)i1*ymul)*2;
  if (sym*2<sgrf->height) {i1--;}
  dym=i1/2;

  /* create bitmaps */
  if (sxm*2<sgrf->width) {i1=(dxm+1)*2-1;} else {i1=dxm*2;}
  if (i1<1) {i1=1;}
  if (sym*2<sgrf->height) {i2=(dym+1)*2-1;} else {i2=dym*2;}
  if (i2<1) {i2=1;}
  if ((dgrf=vg_bitmap_createnew(i1,i2))==NULL) {fprintf(stderr,"vg_bitmap_zoom: error calling vg_bitmap_createnew.\n"); return(NULL);}
  i2=sgrf->height;
  if (i2<1) {i2=1;}
  if ((tgrf=vg_bitmap_createnew(i1,i2))==NULL) {fprintf(stderr,"vg_bitmap_zoom: error calling vg_bitmap_createnew.\n"); return(NULL);}

  /* temp middle: txm,tym */
  txm=tgrf->width/2;
  tym=tgrf->height/2;

  /* x direction */
  if ((sw=(int)((double)tw/xmul))<=0) {sw=1;}
  /* zoom x direction right from source to temp */
  brk=0;
  for (i_z=0;i_z<sgrf->height;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    i_a=sxm; i_n=txm;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=sgrf->width) {brk=1; break;}
        ii=sgrf->pxm[i_z*sgrf->width+i_a+vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=sgrf->width) {brk=1; break;}
        ii=sgrf->pxm[i_z*sgrf->width+i_a+vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<tgrf->width) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n++]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom x direction left from source to temp */
  brk=0;
  for (i_z=0;i_z<sgrf->height;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    if (sxm*2==sgrf->width) {i_a=sxm-1;} else {i_a=sxm;}
    if (txm*2==tgrf->width) {i_n=txm-1;} else {i_n=txm;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[i_z*sgrf->width+i_a-vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[i_z*sgrf->width+i_a-vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n--]=hp; fb[hp]-=sw;}
    }
  }

  /* y direction */
  if ((sw=(int)((double)tw/ymul))<=0) {sw=1;}
  /* zoom y direction down from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    i_a=tym; i_n=dym;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<dgrf->height) {dgrf->pxm[(i_n++)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom y direction up from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    if (tym*2==tgrf->height) {i_a=tym-1;} else {i_a=tym;}
    if (dym*2==dgrf->height) {i_n=dym-1;} else {i_n=dym;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {dgrf->pxm[(i_n--)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  vg_bitmap_free(tgrf);
  return(dgrf);
} /* Ende vg_bitmap_zoom */


bitmap * vg_bitmap_mirror(const bitmap * sgrf,int m_art) {
/* mirror a bitmap vertically or horizontally
** 1.arg: bitmap or NULL=backbuffer (will not be modified)
** 2.arg: MIRROR_VT=vertically or MIRROR_HT=horizontally
** return: pointer to static mirrored bitmap
**         or NULL = error
*/
  static bitmap * dgrf=NULL;
  int i1,i2;
  if (sgrf==NULL) {sgrf=backbuffer;}
  if (dgrf!=NULL) {vg_bitmap_free(dgrf); dgrf=NULL;}
  if ((dgrf=vg_bitmap_createnew(sgrf->width,sgrf->height))==NULL) {fprintf(stderr,"vg_bitmap_mirror: error calling vg_bitmap_createnew.\n"); return(NULL);}

  if (m_art==MIRROR_VT) {  /* vertically */
    for (i1=0;i1<sgrf->height;i1++) {
      for (i2=0;i2<sgrf->width;i2++) {
        dgrf->pxm[i1*dgrf->width+dgrf->width-i2-1]=sgrf->pxm[i1*sgrf->width+i2];
      }
    }
  } else if (m_art==MIRROR_HT) {  /* horizontally */
    for (i1=0;i1<sgrf->width;i1++) {
      for (i2=0;i2<sgrf->height;i2++) {
        dgrf->pxm[(dgrf->height-i2-1)*dgrf->width+i1]=sgrf->pxm[i2*sgrf->width+i1];
      }
    }
  } else {
    fprintf(stderr,"vg_bitmap_mirror: invalid parameter.\n");
    return(NULL);
  }
  return(dgrf);
} /* Ende vg_bitmap_mirror */


void vg_bitmap_free(bitmap * grf) {
/* free a bitmap
** 1.arg: bitmap
*/
  if (grf==NULL) {return;}  /* don't free backbuffer with NULL */
  free(grf->pxm);
  free(grf);
} /* Ende vg_bitmap_free */


int vg_bitmap_overlap(struct ovlap * ovl,const bitmap * grf1,int x1v,int y1v,int x1b,int y1b,const bitmap * grf2,int x2,int y2,int minpix) {
/* checks whether two bitmaps overlap, where the first bitmap is moving
** 1.arg: address of struct ovlap for returning values (or NULL if unimportant)
** 2.arg: first bitmap
** 3.+4.arg: start position (middle) of first bitmap
** 5.+6.arg: end position (middle) of first bitmap
** 7.arg: second bitmap
** 8.+9.arg: position (middle) of second bitmap
** 10.arg: minimum number of overlapping of not-transparent pixels
**         or 0=do not test for pixel-overlapping, only test coordinates
** return: 0=no overlapping (1.arg is left unchanged)
**         1=overlapping found
**
** struct ovlap:
**   int hitside: bitfield of VG_RIGHT, VG_LEFT, VG_TOP, VG_BOTTOM
**                at which side the first bitmap is hit by the second one
**   int x_nohit,y_nohit: last position of first bitmap without touching
**                        the second bitmap, or (-1,-1) if not available
**   int x_hit,y_hit: first position of first bitmap touching second bitmap
**   int steps: number of steps the first bitmap took until touching
**              the second bitmap
*/
  float f1,f2,f3;
  int retw,x0,y0,x9,y9,xa,xe,ya,ye,steps;
  if (grf1==NULL) {grf1=backbuffer;}
  if (grf2==NULL) {grf2=backbuffer;}
  if (grf1==grf2) {return(0);}
  if (minpix<0) {minpix=0;}

  xa=x1b-x1v; if (xa<0) {xa=-xa;}
  ya=y1b-y1v; if (ya<0) {ya=-ya;}
  retw=x0=y0=0;
  x9=y9=-1;
  steps=0;
  if ((xa==0) && (ya==0)) {
    x0=x1v;
    y0=y1v;
    retw=overlapping(grf1,x0,y0,grf2,x2,y2,minpix);
  } else if (xa>=ya) {
    xa=x1v; xe=x1b; ya=y1v; ye=y1b;
    if (xe<xa) {
      f1=(float)(ye-ya)/(float)(xa-xe);
      f2=0.;
      if (ye>ya) {f3=.005;} else if (ye<ya) {f3=-.005;} else {f3=.0;}
      for (x0=xa;x0>=xe;x0--) {
        y0=ya+(int)(f2+f3);
        if ((retw=overlapping(grf1,x0,y0,grf2,x2,y2,minpix))>0) {break;}
        x9=x0; y9=y0; steps++;
        f2+=f1;
      }
    } else {
      f1=(float)(ye-ya)/(float)(xe-xa);
      f2=0.;
      if (ye>ya) {f3=.005;} else if (ye<ya) {f3=-.005;} else {f3=.0;}
      for (x0=xa;x0<=xe;x0++) {
        y0=ya+(int)(f2+f3);
        if ((retw=overlapping(grf1,x0,y0,grf2,x2,y2,minpix))>0) {break;}
        x9=x0; y9=y0; steps++;
        f2+=f1;
      }
    }
  } else {
    ya=y1v; ye=y1b; xa=x1v; xe=x1b;
    if (ye<ya) {
      f1=(float)(xe-xa)/(float)(ya-ye);
      f2=0.;
      if (xe>xa) {f3=.005;} else if (xe<xa) {f3=-.005;} else {f3=.0;}
      for (y0=ya;y0>=ye;y0--) {
        x0=xa+(int)(f2+f3);
        if ((retw=overlapping(grf1,x0,y0,grf2,x2,y2,minpix))>0) {break;}
        x9=x0; y9=y0; steps++;
        f2+=f1;
      }
    } else {
      f1=(float)(xe-xa)/(float)(ye-ya);
      f2=0.;
      if (xe>xa) {f3=.005;} else if (xe<xa) {f3=-.005;} else {f3=.0;}
      for (y0=ya;y0<=ye;y0++) {
        x0=xa+(int)(f2+f3);
        if ((retw=overlapping(grf1,x0,y0,grf2,x2,y2,minpix))>0) {break;}
        x9=x0; y9=y0; steps++;
        f2+=f1;
      }
    }
  }
  if (retw && ovl!=NULL) {
    ovl->hitside=retw;
    ovl->x_nohit=x9;
    ovl->y_nohit=y9;
    ovl->x_hit=x0;
    ovl->y_hit=y0;
    ovl->step=steps;
  }
  return(!!retw);
} /* Ende vg_bitmap_overlap */


static int overlapping(const bitmap * grf1,int x1,int y1,const bitmap * grf2,int x2,int y2,int minpix) {
  int retw,x1s,y1s,x2s,y2s;
  int w0,h0,x0,y0,i1,i2;

  /* check coordinates */
  if (x1+grf1->width/2<x2-grf2->width/2 \
  || x1-grf1->width/2>x2+grf2->width/2 \
  || y1+grf1->height/2<y2-grf2->height/2 \
  || y1-grf1->height/2>y2+grf2->height/2) {return(0);}

  /* set touching sides */
  retw=0;
  x0=x2-x1; if (x0<0) {x0=-x0;}
  y0=y2-y1; if (y0<0) {y0=-y0;}
  if (x0>=y0) {
    if (x2>=x1) {retw|=VG_RIGHT;}
    if (x2<=x1) {retw|=VG_LEFT;}
  }
  if (y0>=x0) {
    if (y2>=y1) {retw|=VG_BOTTOM;}
    if (y2<=y1) {retw|=VG_TOP;}
  }
  if (minpix<=0) {return(retw);}

  /* start pixel of overlapping box of grf1 */
  x1s=x2-grf2->width/2-(x1-grf1->width/2);
  if (x1s<0) {x1s=0;}
  y1s=y2-grf2->height/2-(y1-grf1->height/2);
  if (y1s<0) {y1s=0;}

  /* start pixel of overlapping box of grf2 */
  x2s=x1-grf1->width/2-(x2-grf2->width/2);
  if (x2s<0) {x2s=0;}
  y2s=y1-grf1->height/2-(y2-grf2->height/2);
  if (y2s<0) {y2s=0;}

  /* width and height of overlapping boxes */
  w0=(grf1->width-x1s<grf2->width-x2s?grf1->width-x1s:grf2->width-x2s);
  h0=(grf1->height-y1s<grf2->height-y2s?grf1->height-y1s:grf2->height-y2s);

  /* check every pixel of both bitmaps in this overlapping boxes */
  for (y0=0;y0<h0;y0++) {
    i1=(y1s+y0)*grf1->width+x1s;
    i2=(y2s+y0)*grf2->width+x2s;
    for (x0=0;x0<w0;x0++) {
      if (grf1->pxm[i1+x0]!=RGB_BLACK && grf2->pxm[i2+x0]!=RGB_BLACK) {
        if (--minpix==0) {return(retw);}
      }
    }
  }
  return(0);
} /* Ende overlapping */
