/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* color functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "color.h"
#ifdef HAVE_X11
# include "video_x11.h"
#endif
#ifdef HAVE_SVGALIB
# include "video_svgalib.h"
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
# include "video_vgl.h"
#endif

farbkonv fcpix[256]={};

static farbkonv coldef[]={
  {63,0,0},    /* CL_RED */
  {63,32,0},   /* CL_REDORANGE */
  {63,40,0},   /* CL_ORANGE */
  {63,48,0},   /* CL_YELLOWORANGE */
  {63,63,0},   /* CL_YELLOW */
  {44,63,0},   /* CL_YELLOWGREEN */
  {0,63,0},    /* CL_GREEN */
  {0,63,36},   /* CL_TURQUOISEGREEN */
  {0,63,63},   /* CL_TURQUOISE */
  {0,42,63},   /* CL_TURQUOISEBLUE */
  {0,0,63},    /* CL_BLUE */
  {40,0,63},   /* CL_BLUEVIOLET */
  {63,0,63},   /* CL_VIOLET */
  {63,0,40},   /* CL_REDVIOLET */
  {63,40,40},  /* CL_PINK */
  {63,36,21},  /* CL_BROWN */
  {63,63,63},  /* CL_WHITE */
  {0,0,0}      /* CL_BLACK */
};
static int col_grey=1;

int vg_load_colormap(const char *,char *,size_t);
int vg_color_index(int,int);
int konv_map(unsigned char[],FILE *);
int vg_brightness(int);

static unsigned char _best_color(farbkonv *,int);


static unsigned char _best_color(farbkonv * pfw,int trsp) {
  int i1,i2,wr,wg,wb,z0,z2,z3;
  unsigned int wrt;
  unsigned char idx=0;
  wrt=(unsigned int)-1;
  z2=z3=256;
  for (i1=0;i1<256;i1++) {  /* fcpix: current colormap */
    if ((trsp==RGB_TRANS) && (i1==RGB_BLACK)) {continue;}
    wr=wg=wb=i2=0;
    wr+=pfw->r-fcpix[i1].r; if (wr<0) {wg-=wr; wb-=wr; i2-=wr; wr=0;}
    wg+=pfw->g-fcpix[i1].g; if (wg<0) {wb-=wg; i2-=wg; wr-=wg; wg=0;}
    wb+=pfw->b-fcpix[i1].b; if (wb<0) {i2-=wb; wr-=wb; wg-=wb; wb=0;}
    if (wr>i2) {i2=wr;}
    if (wg>i2) {i2=wg;}
    if (wb>i2) {i2=wb;}
    if ((unsigned int)i2<wrt) {idx=(unsigned char)i1;}
    else if ((unsigned int)i2==wrt) {
      z0=(wr<wg?(wr<wb?wb:wr):(wr<wb?wr:(wb<wg?wg:wb)));
      if (z0>z2) {idx=(unsigned char)i1;}
      else {
        z0=(wr<wg?(wr<wb?wr:wb):(wr<wb?wg:(wb<wg?wb:wg)));
        if (z0>z3) {idx=(unsigned char)i1;}
      }
    }
    if (idx==(unsigned char)i1) {
      z2=(wr<wg?(wr<wb?wb:wr):(wr<wb?wr:(wb<wg?wg:wb)));
      z3=(wr<wg?(wr<wb?wr:wb):(wr<wb?wg:(wb<wg?wb:wg)));
      wrt=i2;
    }
  }
  return(idx);
} /* Ende _best_color */


int konv_map(unsigned char kv[],FILE * ffp) {
/* convert colormap of a bitmap into current colormap
** 1.arg: for returning new pixel color indexes
** 2.arg: filehandle of bitmap file at position of colormap
** return: 0=OK or -1=error
*/
  int i1;
  farbkonv fw,fv;
  unsigned char bix=0,bfb=64;
  unsigned char fiw[256],ifiw;
  fv.r=fv.g=fv.b=64;
  memset(fiw,0,sizeof(fiw)); ifiw=0;
  for (i1=0;i1<256;i1++) {  /* fw: colormap of bitmap */
    if (fscanf(ffp,"%c%c%c",&fw.r,&fw.g,&fw.b)<3) {  /* next value of colormap */
      fprintf(stderr,"konv_map: corrupt bitmap file.\n");
      return(-1);
    }
    /* kv[fw]=fcpix: best color index to colormap of bitmap */
    kv[i1]=_best_color(&fw,RGB_TRANS);
    if ((fw.r<=bfb) && (fw.g<=bfb) && (fw.b<=bfb)) {  /* find black pixel */
      if (fw.r+fw.g+fw.b<fv.r+fv.g+fv.b) {
        bix=(unsigned char)i1;
        bfb=(fw.r>fw.g?(fw.r>fw.b?fw.r:fw.b):(fw.g>fw.b?fw.g:fw.b));
        fv.r=fw.r; fv.g=fw.g; fv.b=fw.b;
        ifiw=0;
      } else if (fw.r+fw.g+fw.b==fv.r+fv.g+fv.b) {  /* same black pixel */
        fiw[ifiw++]=(unsigned char)i1;
      }
    }
  }
  kv[(int)bix]=RGB_BLACK;
  for (i1=0;i1<ifiw;i1++) {  /* make same black pixels also black */
    kv[(int)fiw[i1]]=RGB_BLACK;
  }
  return(0);
} /* Ende konv_map */


int vg_color_index(int cl_nr,int prz) {
/* give back best color index of a color name
** 1.arg: color name (CL_-definition)
** 2.arg: color brightness percent: value from 0 to 100
** return: best color index
*/
  farbkonv fw;
  if (cl_nr<1) {return(RGB_BLACK);}
  else if (cl_nr>COLOR_MAX) {return(RGB_WHITE);}
  if (prz<0) {prz=0;}
  else if (prz>100) {prz=100;}
  if (col_grey==1) {  /* color */
    fw.r=coldef[cl_nr-1].r*prz/100;
    fw.g=coldef[cl_nr-1].g*prz/100;
    fw.b=coldef[cl_nr-1].b*prz/100;
  } else {  /* grey */
    fw.r=(coldef[cl_nr-1].r+coldef[cl_nr-1].g+coldef[cl_nr-1].b)/3*prz/100;
    fw.g=fw.b=fw.r;
  }
  return((int)_best_color(&fw,RGB_FULL));
} /* Ende vg_color_index */


int vg_load_colormap(const char * clm,char * vclm,size_t vlen) {
/* load a colormap from <CWDIR>/share/ or <SHAREDIR>/
** 1.arg: filename of colormap (no path and must end with .clm)
**        (builtin filenames are RGB256 or GREY256)
** 2.arg: address for filename of previous loaded colormap
**        or NULL, if not needed
** 3.arg: sizeof 2.arg
**        or 0, if 2.arg=NULL
** return: 0=OK or -1=error
*/
  static char colm[128]="";
  farbkonv fv[2];
  unsigned char fiw[256],ifiw;
  FILE * ffp;
  int i1,i2;
  unsigned int r,g,b,wfb,bfb;
  char buf[512],vcolm[128];
  const char * kptr;
  strncpy(vcolm,*colm=='\0'?RGB256:colm,sizeof(vcolm)-1); vcolm[sizeof(vcolm)-1]='\0';
  if ((vclm!=NULL) && (vlen>0)) {snprintf(vclm,vlen,"%s",vcolm);}

  /* open file and load colors */
  if (clm==NULL) {fprintf(stderr,"vg_load_colormap: filename of colormap NULL.\n"); return(-1);}
  if ((kptr=strrchr(clm,'/'))!=NULL) {kptr++;} else {kptr=clm;}
  snprintf(colm,sizeof(colm),"%s.clm",kptr);
  i1=strlen(colm);
  if ((i1>8) && (strcmp(colm+i1-8,".clm.clm")==0)) {colm[i1-4]='\0';}
  memset(fcpix,0,sizeof(fcpix));
  snprintf(buf,sizeof(buf),"%s/share/%s",CWDIR,colm);
  if ((ffp=fopen(buf,"r"))==NULL) {
    snprintf(buf,sizeof(buf),"%s/%s",SHAREDIR,colm);
    ffp=fopen(buf,"r");
  }
  if (ffp==NULL) {
    fprintf(stderr,"vg_load_colormap: can't open RGB file: \"%s\".\n",colm);
    strncpy(colm,vcolm,sizeof(colm)-1); colm[sizeof(colm)-1]='\0';
    return(-1);
  }
  i2=0;
  while (fgets(buf,sizeof(buf)-1,ffp)!=NULL) {
    if ((*buf<'0') || (*buf>'9')) {continue;}
    sscanf(buf,"%d=%d %d %d",&i1,&r,&g,&b);
    if ((i1>255) || (i1<0)) {continue;}
    r&=0x3f; g&=0x3f; b&=0x3f;
    fcpix[i1].r=(unsigned char)r;
    fcpix[i1].g=(unsigned char)g;
    fcpix[i1].b=(unsigned char)b;
    if ((r!=g) || (r!=b)) {i2++;}
  }
  fclose(ffp);
  if (i2>0) {col_grey=1;} else {col_grey=2;}
  /* find best white and black pixel */
  RGB_WHITE=RGB_BLACK=0;
  wfb=0;
  bfb=64;  /* 6 bits for color values */
  fv[0].r=fv[0].g=fv[0].b=64;
  fv[1].r=fv[1].g=fv[1].b=0;
  memset(fiw,0,sizeof(fiw)); ifiw=0;
  for (i1=0;i1<256;i1++) {
    r=fcpix[i1].r; g=fcpix[i1].g; b=fcpix[i1].b;
    if ((r>=wfb) && (g>=wfb) && (b>=wfb)) {  /* best white pixel */
      if (r+g+b>(unsigned int)(fv[1].r+fv[1].g+fv[1].b)) {
        RGB_WHITE=i1; wfb=(r<g?(r<b?r:b):(g<b?g:b));
        fv[1].r=r; fv[1].g=g; fv[1].b=b;
      }
    }
    if ((r<=bfb) && (g<=bfb) && (b<=bfb)) {  /* best black pixel */
      if (r+g+b<(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {
        RGB_BLACK=i1; bfb=(r>g?(r>b?r:b):(g>b?g:b));
        fv[0].r=r; fv[0].g=g; fv[0].b=b;
        ifiw=0;
      } else if (r+g+b==(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {  /* same black pixel */
        fiw[ifiw++]=(unsigned char)i1;
      }
    }
  }
  for (i1=0;i1<ifiw;i1++) {  /* make same black pixels lighter */
    if (fcpix[(int)fiw[i1]].r<63) {fcpix[(int)fiw[i1]].r++;}
    else if (fcpix[(int)fiw[i1]].g<63) {fcpix[(int)fiw[i1]].g++;}
    else if (fcpix[(int)fiw[i1]].b<63) {fcpix[(int)fiw[i1]].b++;}
  }
  /* find best dark pixel */
  RGB_DARK=0;
  bfb=64;  /* 6 bits for color values */
  fv[0].r=fv[0].g=fv[0].b=64;
  for (i1=0;i1<256;i1++) {
    if (i1==RGB_BLACK) {continue;}
    r=fcpix[i1].r; g=fcpix[i1].g; b=fcpix[i1].b;
    if ((r<=bfb) && (g<=bfb) && (b<=bfb)) {  /* best dark pixel */
      if (r+g+b<(unsigned int)(fv[0].r+fv[0].g+fv[0].b)) {
        RGB_DARK=i1; bfb=(r>g?(r>b?r:b):(g>b?g:b));
        fv[0].r=r; fv[0].g=g; fv[0].b=b;
      }
    }
  }
  vg_brightness(99);
  return(0);
} /* Ende vg_load_colormap */


int vg_brightness(int vol) {
/* reduces or increases brightness of colormap
** 1.arg: -63 (darkest) to 62 (lightest), 0 is default, or 99=previous
** return: previous brightness value (-63 to 62)
*/
#ifdef HAVE_X11
  if (window_mode.videolib==VIDEOLIB_X11) {
    return(brightness_x11(vol));
  }
#endif
#ifdef HAVE_SVGALIB
  if (window_mode.videolib==VIDEOLIB_SVGALIB) {
    return(brightness_svgalib(vol));
  }
#endif
#if defined(HAVE_VGL4) || defined(HAVE_VGL5)
  if (window_mode.videolib==VIDEOLIB_VGL) {
    return(brightness_vgl(vol));
  }
#endif
  return(0);
} /* Ende vg_brightness */
