/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* nettcp: network support
   define HAVE_ADDRINFO_STRUCT, if you have getaddrinfo()
     or undef HAVE_ADDRINFO_STRUCT, if you have not getaddrinfo().
   define HAVE_SOCKADDR_SA_LEN, if you have sin_len
     or undef HAVE_SOCKADDR_SA_LEN, if you have not sin_len.
   compile if needed with -lsocket -lnsl [-lresolv]
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>
#include "config.h"
#include "vgagames.h"
#include "nw_support.h"
#include "nw_mbcast.h"
#include "nw_rdwr.h"


extern char errmsg[2048];
extern int call_server(const char *,const char *,const char *);
char * netmem;
static int cfd=-1,plynr=-1,plypos=-1,plyvirt=-1,master=0;
static size_t plysz=0;

int start_nettcp(int,int,unsigned short);
void * connect_nettcp(const char *,unsigned short,size_t,int *,int *);
int talk_nettcp(void);
void close_nettcp(void);

static int tcp_connection(const char *,const char *);

static int tcp_connection(const char * host,const char * service) {
/* connect to network server */
  int n1,cfd=0;
  struct addrinfo adi0,* adi1,* adi2;
  int i0;
  memset(&adi0,0,sizeof(struct addrinfo));
#ifdef AF_INET6
  for (i0=2;i0>0;i0--) {
    adi0.ai_family=(i0==2?AF_INET:AF_INET6);
#else
  for (i0=1;i0>0;i0--) {
    adi0.ai_family=AF_INET;
#endif
    adi0.ai_socktype=SOCK_STREAM;
    if ((n1=getaddrinfo(host,service,&adi0,&adi1))!=0) {
      if (i0>1) {continue;}
      fprintf(stderr,"tcp_connection: getaddrinfo: %s%s%s\n",gai_strerror(n1),n1==EAI_SYSTEM?": ":"",n1==EAI_SYSTEM?strerror(errno):"");
      return(-1);
    }
    adi2=adi1;
    do {
      cfd=socket(adi1->ai_family,adi1->ai_socktype,adi1->ai_protocol);
      if (cfd<0) {continue;}
      if (connect(cfd,adi1->ai_addr,adi1->ai_addrlen)==0) {break;}
      close(cfd);
    } while ((adi1=adi1->ai_next)!=NULL);
    if (adi1==NULL) {
      freeaddrinfo(adi2);
      if (i0>1) {continue;}
      fprintf(stderr,"tcp_connection: cannot get socket\n");
      return(-1);
    }
    freeaddrinfo(adi2);
    break;
  }
  return(cfd);
} /* Ende tcp_connection */


int start_nettcp(int anz,int vanz,unsigned short port) {
/* Only called once, from the master player, to start server
** for all players to connect to.
** 1.arg: number of players to wait for connection (from 1 to 8)
** 2.arg: how many virtual players (=guided by computer) are included in 1.arg
** 3.arg: portnumber listening to
** return: 0=ok or -1=error
*/
  char serv[8],anzp[2],vanzp[2];
  pid_t cpid;
  if ((anz<1) || (anz>8) || (vanz<0) || (vanz>=anz)) {strcpy(errmsg,"Wrong number of players"); return(-1);}
  anzp[0]=anz+'0'; anzp[1]='\0';
  vanzp[0]=vanz+'0'; vanzp[1]='\0';
  snprintf(serv,sizeof(serv),"%u",port);
  if ((cpid=fork())==-1) {snprintf(errmsg,sizeof(errmsg),"Fork: %s",strerror(errno)); return(-1);}
  if (cpid==0) {  /* Child */
    if (call_server(anzp,vanzp,serv)<0) {_exit(1);}
    _exit(0);
  }
  master=1;
  plyvirt=vanz;
  sleep(3);
  return(0);
} /* Ende start_nettcp */


void * connect_nettcp(const char * host,unsigned short port,size_t stsz,int * plp,int * pln) {
/* connect to server, waiting until all players are connected
** 1.arg: NULL=get host via broadcast/multicast
**        or hostname or ip of server
** 2.arg: portnumber of server
** 3.arg: size of data field (number of characters) (1 to 128)
** 4.arg: address for player position
** 5.arg: address for number of players
** return: address of network memory or NULL=error
*/
  int n1;
  char serv[8],buf[64],hostp[128+64];
  if ((stsz==0) || (stsz>128)) {strcpy(errmsg,"Wrong data size (must be 1 to 128)"); return(NULL);}
  snprintf(serv,sizeof(serv),"%u",port);

  /* connect to [host]:port */
  cfd=-1;
  if (master) {
    strcpy(hostp,"localhost");
    cfd=tcp_connection(hostp,serv);
  } else if ((host==NULL) || (*host=='\0')) {  /* search server */
    struct ip_mbcast * ipret;
    int ipv=AF_INET;
#ifdef AF_UNSPEC
    ipv=AF_UNSPEC;
#endif
    if (get_hostip(serv,ipv,5,1,&ipret)>0) {
      snprintf(hostp,sizeof(hostp),"%s",ipret->ipnr);
      if (((cfd=tcp_connection(hostp,serv))<0) && (*ipret->iface!='\0')) {
        snprintf(hostp,sizeof(hostp),"%s%%%s",ipret->ipnr,ipret->iface);
        cfd=tcp_connection(hostp,serv);
      }
      free_hostip(ipret);
    }
  } else {
    snprintf(hostp,sizeof(hostp),"%s",host);
    cfd=tcp_connection(hostp,serv);
  }
  if (cfd<0) {
    fprintf(stderr,"Connection to [%s]:%s failed\n",host==NULL?"(broadcast/multicast)":hostp,serv);
    cfd=-1;
    return(NULL);
  }

  printf("Connected to server %s at port %s\n",hostp,serv); fflush(stdout);
  /* set network memory */
  if ((netmem=calloc(1+8,stsz))==NULL) {
    strcpy(errmsg,"Cannot allocate network memory");
    close(cfd); cfd=-1;
    return(NULL);
  }
  netmem[0]=1;
  /* write data size + master identification */
  snprintf(buf,sizeof(buf),"%04u%d",(unsigned int)stsz,master);
  if (writen(cfd,buf,5)==-1) {
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  plysz=stsz;
  /* read number of players and player position */
  if (read(cfd,buf,2)<2) {
    snprintf(errmsg,sizeof(errmsg),"Cannot read from %s (%s)",hostp,serv);
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  plynr=(int)buf[0]-'0';
  plypos=(int)buf[1]-'0';
  if ((plynr<1) || (plynr>8) || (plypos<1) || (plypos>8)) {
    snprintf(errmsg,sizeof(errmsg),"Wrong answer read from %s (%s)",hostp,serv);
    close(cfd); cfd=-1;
    free(netmem);
    return(NULL);
  }
  if (pln!=NULL) {*pln=plynr;}
  if (plp!=NULL) {*plp=plypos;}
  netmem[plypos*plysz]=1;
  if (master) {
    for (n1=1;n1<=plyvirt;n1++) {netmem[(plypos+n1)*plysz]=1;}
  }
  return((void *)netmem);
} /* Ende connect_nettcp */


int talk_nettcp() {
/* sends own data and loads data of the other players
** return: 0=ok or -1=error
*/
  const int rnr=(1+plynr)*plysz;
  ssize_t rr;
  size_t slen;
  char * nmm;
  if (cfd<0) {strcpy(errmsg,"Not connected"); return(-1);}
  if (master) {nmm=netmem; slen=plysz*(1+1+plyvirt);}
  else {nmm=netmem+plypos*plysz; slen=plysz;}
  if (writen(cfd,nmm,slen)==-1) {
    strcpy(errmsg,"Write error to server");
    close(cfd); cfd=-1;
    return(-1);
  }
  if ((rr=readn(cfd,netmem,rnr))!=rnr) {
    strcpy(errmsg,"Read error from server");
    close(cfd); cfd=-1;
    return(-1);
  }
  return(0);
} /* Ende talk_nettcp */


void close_nettcp() {
/* close socket and free network memory */
  if (cfd>=0) {
    close(cfd);
    free(netmem);
  }
  cfd=-1;
} /* Ende close_nettcp */
