/* *****************************************************************
   Copyright (C) 2000-2004 Kurt Nienhaus

   This program is modifiable/redistributable under the terms
   of the GNU General Public Licence as mentioned in vgagames.c.
   ***************************************************************** */

/* sound: handle sound events. Sets SIGPIPE to SIG_IGN! */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames.h"

extern char errmsg[2048];
#define MAX_WAVNR 256
static char wavnr[MAX_WAVNR][16]={};
static int wavidx=0;
static struct {
  unsigned int momzahl;
  int kanal[CN_MAX];
  unsigned int kanzahl[CN_MAX];
} cns={};
#define CN_N(A) (A-1)

int init_sound(long,int);
int load_wave(const char *,int);
void play_wave(int,int,short,short);
void stop_wave(int,short);
void pause_wave(int);
void continue_wave(int);
void volume_wave(int,short);
void accel_wave(short,short);
void end_sound(void);
int set_vol(int,int);



int init_sound(long smpfq,int moso) {
/* start wave server.
** 1.arg: samplefrequence to set
** 2.arg: 0=set mono or 1=set stereo
** return: 0=ok or -1=error
*/
  wavidx=0;
  memset(&cns,0,sizeof(cns));
  return(vg_sound_startserver(smpfq,moso+1,NULL));
} /* Ende init_sound */


int load_wave(const char * wvfile,int emul) {
/* connect a wave file to a number
** 1.arg: wave filename
** 2.arg: emulation type: TYP_IS_WAVE
**        (unused)        TYP_IS_MIDI
**                        TYP_IS_MP3
**                        TYP_IS_EXTENSION
** return value: wave number or 0=error (server exit)
*/
  if (wavidx+1>=MAX_WAVNR) {return(0);}
  wavidx++;
  snprintf(wavnr[wavidx-1],sizeof(wavnr[0]),"s-%d",wavidx);
  vg_sound_attach(wvfile,wavnr[wavidx-1],100);
  return(wavidx);
} /* Ende load_wave */


void play_wave(int wvnr,int chnr,short loop,short voltime) {
/* plays wave file assigned to number
** 1.arg: wave number
** 2.arg: play one of these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 3.arg: 0=play once or >0=play looping with pause of 3.arg seconds
** 4.arg: 0=normal or >0=time in seconds to increase volume to 100%
*/
  int i1,i2,channel,i3;
  unsigned int mx;
  if ((wvnr>=MAX_WAVNR) || (*wavnr[wvnr-1]=='\0')) {return;}
  /* find a free channel */
  channel=i1=-1;
  mx=(unsigned int)-1;
  for (i3=0,i2=1;i2<=CN_MAX;i2++) {
    if (chnr&CN_(i2)) {
      i3++;
      if (cns.kanzahl[CN_N(i2)]<mx) {i1=i2; mx=cns.kanzahl[CN_N(i2)];}
      if (cns.kanal[CN_N(i2)]==0) {channel=CN_N(i2);}
    }
  }
  if (channel==-1) {  /* no free channel */
    if (i1==-1) {return;}
    /* cancel a running channel */
    if (i3==1) {vg_sound_stop(wavnr[cns.kanal[CN_N(i1)]-1],0);}
    channel=CN_N(i1);
  }
  cns.kanal[channel]=wvnr;
  cns.kanzahl[channel]=++cns.momzahl;
  (void)vg_sound_play(wavnr[wvnr-1],(int)voltime,!(int)loop);
} /* Ende play_wave */


void stop_wave(int chnr,short voltime) {
/* stops playing wave file at channel numbers
** 1.arg: stop these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 2.arg: 0=at once or >0=time in seconds to decrease volume to 0%
*/
  int i2;
  for (i2=1;i2<=CN_MAX;i2++) {
    if (chnr&CN_(i2)) {
      if (cns.kanal[CN_N(i2)]>0) {
        vg_sound_stop(wavnr[cns.kanal[CN_N(i2)]-1],(int)voltime);
        cns.kanal[CN_N(i2)]=0;
      }
    }
  }
} /* Ende stop_wave */


void pause_wave(int chnr) {
/* pauses playing wave file at channel numbers
** 1.arg: pause these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
*/
  int i2;
  for (i2=1;i2<=CN_MAX;i2++) {
    if (chnr&CN_(i2)) {
      if (cns.kanal[CN_N(i2)]>0) {vg_sound_paus(wavnr[cns.kanal[CN_N(i2)]-1]);}
    }
  }
} /* Ende pause_wave */


void continue_wave(int chnr) {
/* continues playing wave file at channel numbers
** 1.arg: continue these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
*/
  int i2;
  for (i2=1;i2<=CN_MAX;i2++) {
    if (chnr&CN_(i2)) {
      if (cns.kanal[CN_N(i2)]>0) {vg_sound_cont(wavnr[cns.kanal[CN_N(i2)]-1]);}
    }
  }
} /* Ende continue_wave */


void volume_wave(int chnr,short perc) {
/* changes volume percent at channel numbers
** 1.arg: change these channel numbers
**        (CN_(1) to CN_(CN_MAX) bit or'd resp. CN_ALL for all channels)
** 2.arg: 0 to 30 (where 0=0%, 1=10%, 2=20%, ..., 10=100%, ..., 30=300%)
*/
  int i2;
  perc*=5;
  if (perc>100) {perc=100;}
  for (i2=1;i2<=CN_MAX;i2++) {
    if (chnr&CN_(i2)) {
      if (cns.kanal[CN_N(i2)]>0) {vg_sound_volm(wavnr[cns.kanal[CN_N(i2)]-1],(int)perc);}
    }
  }
} /* Ende volume_wave */


void accel_wave(short smpfq,short moso) {
/* not supported: accelerate playing wave/midi at cost of quality
** 1.arg: 0=don't change samplefrequence factor
**        1-4=divide samplefrequence with this number
** 2.arg: 0=don't change mono/stereo settings
**        1=set all to mono
**        2=set mono files to mono and stereo files to stereo
*/
  return;
} /* Ende accel_wave */


void end_sound() {
/* stops playing and exits server */
  vg_sound_endserver();
} /* Ende end_sound */


int set_vol(int volm,int flag) {
/* set volume to 1.arg
** 1.arg: 0-100 or <0 to get only actual volume
** 2.arg: 0=main volume or 1=pcm volume (wave)
** return: <volume before (>=0)>=ok or -1=error
*/
  static int v_main=-1,v_pcm=-1;
  if (v_main<0) {
    v_main=v_pcm=vg_sound_volm("0",-1);
  }
  if (volm>=0) {
    if (flag==0) {v_main=volm;} else {v_pcm=volm;}
    volm=v_main=v_pcm=(v_main+v_pcm)/2;
  }
  return(vg_sound_volm("0",volm));
} /* Ende set_vol */
