/* *****************************************************************
   vgagames: Version 1.99
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* compat file */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <unistd.h>
#include <math.h>
#include "config.h"
#include "vgagames.h"

#ifndef LITTLE_ENDIAN
# define LITTLE_ENDIAN 1234
#endif
#ifndef BIG_ENDIAN
# define BIG_ENDIAN 4321
#endif
#ifndef BYTE_ORDER
# ifdef ENDIAN_IS_LITTLE
#  define BYTE_ORDER LITTLE_ENDIAN
# else
#  define BYTE_ORDER BIG_ENDIAN
# endif
#endif

#ifndef M_PI
  #define M_PI  3.14159265358979323846
#endif
#ifndef PI
  #define PI  M_PI
#endif
#ifndef PI2
  #define PI2  (M_PI + M_PI)
#endif


char errmsg[2048]="";
int _keyart[KEYS_INDEX];
unsigned char font10x17[]="font10x17";
static char deffont[512]="8x8.font";

typedef struct {
  unsigned char r,g,b;
} farbkonv;
extern int konv_map(unsigned char[],FILE *);
extern bitmap * backbuffer;


/* +++ function declarations +++ */
/* general functions */
int wait_time(int);
int color_index(int,int);
int load_colormap(const char *);
void set_font(unsigned char *,int,int);
grafik * create_grafik(int,int);
grafik * load_grafik(const char *);
grafik * text_to_grafik(int,int,int,int,const char *);
int save_grafik_as_ppm(grafik *,const char *);
int save_grafik_as_ppm3(grafik *,const char *);
void free_grafik(grafik *);
void copy_grafik(grafik *,int,int,grafik *,int,int,int,int,int);
grafik * rotate_grafik(grafik *,grafik *,int,int *,int *,int *,int *);
grafik * zoom_grafik(grafik *,grafik *,float,float,int *,int *,int *,int *);
int get_pixel(grafik *,int,int);
void draw_text(grafik *,int,int,int,const char *,int,int);
void draw_pixel(grafik *,int,int,int);
void draw_line(grafik *,int,int,int,int,int);
void draw_circle(grafik *,int,int,int,int,int);
void draw_fillbox(grafik *,int,int,int,int,int);
film * create_film(int);
int add_bild_to_film(film *,film_bild *,int);
unsigned long copy_film(grafik *,film *,void *);
void reset_film(film *,void *,int);
void free_film(film *);
sprite * create_sprite(void);
sprite * load_sprite(const char *);
sprite * rotate_sprite(sprite *,int);
sprite * zoom_sprite(sprite *,float,float);
int add_grafik_to_sprite(sprite *,grafik *,int,int,int,int,int);
grafik * next_sprite(sprite *,int *,int *);
grafik * last_sprite(sprite *,int *,int *);
int reset_sprite(sprite *);
void free_sprite(sprite *);
int open_window(const char *,int);
void flush_window(void);
void close_window(void);
int reopen_window(int);
void get_keys(void);
void clearstate(void);
int mouse_speed(int);
void brightness(int);

/* internal functions */
static FILE * _bmp2vga(FILE *,const char *);
static FILE * _ppm2vga(FILE *,const char *,int);
static FILE * _vga_grafik(const char *);



/* +++ functions +++ */

int wait_time(int m_sec) {
/* waits rest of m_sec millisecs since last call
** return value: time in millisecs since last call or -1=error
*/
  static struct timeval tv;
  static struct timezone tz;
  static int vs=0,ms,ws;
  m_sec-=TIME_OF_SELECT;
  if (m_sec<0) {m_sec=0;}
  if (gettimeofday(&tv,&tz)!=0) {strcpy(errmsg,"Get time error!"); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  if (ws<m_sec+TIME_OF_SELECT) {
    if (ws<m_sec) {tv.tv_usec=(m_sec-ws)*1000;} else {tv.tv_usec=1;}
    tv.tv_sec=0;
    select(0,NULL,NULL,NULL,&tv);
  }
  if (gettimeofday(&tv,&tz)!=0) {strcpy(errmsg,"Get time error!"); return(-1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ms<vs) {ws=60000+ms-vs;} else {ws=ms-vs;}
  vs=ms;
  return(ws);
} /* Ende wait_time */


int color_index(int cl_nr,int prz) {
/* gives back best color index of a color name
** 1.arg: color name (CL_-definition)
** 2.arg: color brightness percent: value from 0 to 100
** return value: best color index
*/
  return(vg_color_index(cl_nr,prz));
} /* Ende color_index */


int load_colormap(const char * clm) {
/* loads a colormap
** 1.arg: path to colormap or RGB256 or GREY256 or NULL=previous
** return value: 0=OK or -1=error
*/
  static char prevcolm[1024]=RGB256;
  if (clm==NULL) {
    return(vg_load_colormap(prevcolm,NULL,0));
  }
  return(vg_load_colormap(clm,prevcolm,sizeof(prevcolm)));
} /* Ende load_colormap */


void set_font(unsigned char * fnt,int fntw,int fnth) {
/* sets font used by draw_text()
** 1.arg: pointer to font(filename)
** 2.+3.arg: unused (font size (width,height) in pixels)
*/
  if ((fnt==NULL) || (strcmp((const char *)fnt,"font10x17")!=0)) {
    strlcpy(deffont,"8x8.font",sizeof(deffont));
  } else {strlcpy(deffont,"10x17.font",sizeof(deffont));}
} /* Ende set_font */


grafik * create_grafik(int w,int h) {
/* create a graphic
** 1.+2.arg: width and height of the graphic
** return value: pointer to graphic
**               or NULL = error
*/
  return(vg_bitmap_createnew(w,h));
} /* Ende create_grafik */


grafik * load_grafik(const char * filename) {
/* load a graphic from a file
** 1.arg: filename
** return value: pointer to graphic
**               or NULL = error
*/
  int w,h;
  unsigned char kv[256];
  char buf[8];
  grafik * grf;
  FILE * ffp;

  /* convert graphic and allocate memory */
  if ((ffp=_vga_grafik(filename))==NULL) {return(NULL);}
  if (fread(buf,1,5,ffp)!=5) {snprintf(errmsg,sizeof(errmsg),"Error reading vga file of \"%s\"!",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  w=atoi(buf);
  if (fread(buf,1,5,ffp)!=5) {snprintf(errmsg,sizeof(errmsg),"Error reading vga file of \"%s\"!",filename); fclose(ffp); return(NULL);}
  buf[5]='\0';
  h=atoi(buf);
  if ((grf=create_grafik(w,h))==NULL) {fclose(ffp); return(NULL);}

  /* read and convert colormap */
  if (konv_map(kv,ffp)<0) {fclose(ffp); free_grafik(grf); return(NULL);}

  /* read and convert pixels */
  fread(grf->pxm,sizeof(unsigned char),grf->width*grf->height,ffp);
  fclose(ffp);
  w=grf->width*grf->height;
  for (h=0;h<w;h++) {grf->pxm[h]=kv[(int)grf->pxm[h]];}

  return(grf);
} /* Ende load_grafik */


grafik * text_to_grafik(int fc,int bc,int zpx,int ausr,const char * text) {
/* create a graphic containing text. Newlines are used for line breaks.
** 1.arg: text color
** 2.arg: background color
** 3.arg: pixel between lines
** 4.arg: 0=left adjusted or 1=centered or 2=right adjusted
** 5.arg: text
** return value: pointer to graphic
**               or NULL = error
*/
  return(vg_bitmap_createfromtext(fc,bc,zpx,ausr,text,deffont));
} /* Ende text_to_grafik */


int save_grafik_as_ppm(grafik * grf,const char * filename) {
/* save graphic box as ppm file (P6) - e.g. for screenshot
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.arg: filename to save ppm graphic
** return value: 0=OK or -1=error
*/
  return(vg_bitmap_save(grf,filename,VGAFORMAT_PPM6));
} /* Ende save_grafik_as_ppm */


int save_grafik_as_ppm3(grafik * grf,const char * filename) {
/* save graphic box as ppm file (P3) - e.g. for screenshot
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.arg: filename to save ppm graphic
** return value: 0=OK or -1=error
*/
  return(vg_bitmap_save(grf,filename,VGAFORMAT_PPM3));
} /* Ende save_grafik_as_ppm3 */


void free_grafik(grafik * grf) {
/* free and destroy a graphic
** 1.arg: pointer to graphic, created with create_grafik() or load_grafik()
*/
  vg_bitmap_free(grf);
} /* Ende free_grafik */


void copy_grafik(grafik * dgrf,int xd,int yd,grafik * sgrf,int xs,int ys,int ws,int hs,int trsp) {
/* copy a (part of a) graphic into another one
** 1.arg: pointer to destination graphic or NULL=backbuffer
** 2.+3.arg: destination coordinates
** 4.arg: pointer to source graphic or NULL=backbuffer
** 5.+6.arg: source coordinates
** 7.+8.arg: width and height to copy from source coordinates of source graphic
**           if width=0 or height=0, then the whole width or height is used
** 9.arg: RGB_FULL for all copying
**        RGB_TRANS for transparent copying
*/
  int w,h;
  unsigned char * dptr,* sptr;
  if (dgrf==sgrf) {return;}
  if (dgrf==NULL) {dgrf=backbuffer;}
  if (sgrf==NULL) {sgrf=backbuffer;}
  if ((xs>=sgrf->width) || (ys>=sgrf->height)) {return;}
  if ((xd>=dgrf->width) || (yd>=dgrf->height)) {return;}
  if ((ws<1) || (xs+ws>sgrf->width)) {ws=sgrf->width-xs;}
  if ((hs<1) || (ys+hs>sgrf->height)) {hs=sgrf->height-ys;}
  if (xd+ws>dgrf->width) {ws=dgrf->width-xd;}
  if (yd+hs>dgrf->height) {hs=dgrf->height-yd;}
  if (xs<0) {ws+=xs; xs=0;}
  if (ys<0) {hs+=ys; ys=0;}
  if (xd<0) {ws+=xd; xs-=xd; xd=0;}
  if (yd<0) {hs+=yd; ys-=yd; yd=0;}
  for (h=0;h<hs;h++) {
    dptr=dgrf->pxm+(yd+h)*dgrf->width+xd;
    sptr=sgrf->pxm+(ys+h)*sgrf->width+xs;
    for (w=0;w<ws;w++) {
      if ((trsp==RGB_FULL) || (*sptr!=RGB_BLACK)) {*dptr=*sptr;}
      dptr++; sptr++;
    }
  }
} /* Ende copy_grafik */


grafik * rotate_grafik(grafik * dgrf,grafik * sgrf,int deg,int * nx,int * ny,int * nw,int * nh) {
/* rotate graphic at certain degrees
** 1.arg: pointer to destination graphic
**        in here is put the rotated source graphic
** 2.arg: pointer to source graphic
** 3.arg: degrees to rotate clockwise (-360 to 360)
** 4.+5.arg: value arguments: x,y start position to rotate of source graphic
**           and return arguments:
**             address for relative x/y coordinate to put destination
**             graphic out to backbuffer/graphic:
**               add x to your origin x coordinate
**               add y to your origin y coordinate
** 6.+7.arg: value arguments: width and height to rotate of source graphic
**                            or 0=whole width/height
**           and return arguments:
**             address for width and height of the size of the rotated graphic
**             in the destination graphic
** return value: pointer to 1.arg
**               or NULL = error
*/
  int sxm,sym,dxm,dym,i0,i1,i2,i3,i4,ix,iy,jx,jy,tx,ty;
  float r0,a0,d0,degpi;
  if ((sgrf==NULL) || (dgrf==NULL)) {strcpy(errmsg,"you can't use backbuffer with rotate_grafik!"); return(NULL);}
  if (dgrf==sgrf) {strcpy(errmsg,"both graphics are the same in rotate_grafik!"); return(NULL);}
  if ((nx==NULL) || (ny==NULL) || (nw==NULL) || (nh==NULL)) {strcpy(errmsg,"arguments NULL"); return(NULL);}
  CLEAR_BOX(dgrf,RGB_BLACK);
  if ((*nx>=sgrf->width) || (*ny>=sgrf->height)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if ((*nw<1) || (*nx+*nw>sgrf->width)) {*nw=sgrf->width-*nx;}
  if ((*nh<1) || (*ny+*nh>sgrf->height)) {*nh=sgrf->height-*ny;}
  if (*nx<0) {*nw+=*nx; *nx=0;}
  if (*ny<0) {*nh+=*ny; *ny=0;}
  if ((*nw<0) || (*nh<0)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  deg%=360;
  deg+=360;
  deg%=360;
  degpi=(float)deg/360.*PI2;  /* angle in Rad */
  /* source middle */
  sxm=*nw/2;
  sym=*nh/2;
  /* destination middle: find the pixel which is farest away from middle */
  /* point 0/0 */
  r0=sqrt((double)(sxm*sxm+sym*sym));  /* distance to middle */
  a0=asin(sym/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i1=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i2=(int)d0;                /* y distance from middle */
  /* point 0/(*nh)-1 */
  i3=*nh-1-sym;
  a0=asin(-i3/r0);           /* horizontal angle in Rad */
  a0+=degpi;
  if (a0>=PI2) {a0-=PI2;}    /* new horizontal angle in Rad */
  d0=cos(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i3=(int)d0;                /* x distance from middle */
  d0=sin(a0)*r0;
  if (d0>=0.) {d0+=.5;} else {d0=-d0+.5;}
  i4=(int)d0;                /* y distance from middle */
  dxm=(i1>i3?i1:i3);  /* calculated destination middle x */
  dym=(i2>i4?i2:i4);  /* calculated destination middle y */
  i1=dgrf->width/2;
  if (dxm>i1) {dxm=i1;}  /* real destination middle and distance x */
  i1=dgrf->height/2;
  if (dym>i1) {dym=i1;}  /* real destination middle and distance y */
  /* put middle */
  dgrf->pxm[dym*dgrf->width+dxm]=sgrf->pxm[(*ny+sym)*sgrf->width+*nx+sxm];
  /* put rest
  ** related to source middle (0/0) take every loop (variable i0) 4 lines:
  **  - upper source pixel line (1.loop: -1/-1 to  1/-1
  **                             2.loop: -2/-2 to  2/-2
  **                             ... )
  **  - lower source pixel line (1.loop: -1/1  to  1/1
  **                             2.loop: -2/2  to  2/2
  **                             ... )
  **  - left source pixel line  (1.loop: -1/-1 to -1/1
  **                             2.loop: -2/-2 to -2/2
  **                             ... )
  **  - right source pixel line (1.loop:  1/-1 to  1/1
  **                             2.loop:  2/-2 to  2/2
  **                             ... )
  */
  i0=1;  /* distance from source middle */
  do {
    i1=0;
    if (sxm<i0) {i2=-sxm;} else {i2=-i0;}  /* left */
    if (*nw-1-sxm<i0) {i3=*nw-1-sxm;} else {i3=i0;}  /* right */
    if (i0<=sym) {  /* upper source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i4*i4+i0*i0));
        a0=asin(i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym-i0)*sgrf->width+*nx+sxm+i4];
        }
      }
      i1=1;
    }
    if (i0<=*nh-1-sym) {  /* lower source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i4*i4+i0*i0));
        a0=asin(-i0/r0);
        if (i4>0) {a0=-a0+PI;}  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i0)*sgrf->width+*nx+sxm+i4];
        }
      }
      i1=1;
    }
    if (sym<i0) {i2=-sym;} else {i2=-i0;}  /* up */
    if (*nh-1-sym<i0) {i3=*nh-1-sym;} else {i3=i0;}  /* down */
    if (i0<=sxm) {  /* left source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i0*i0+i4*i4));
        a0=asin(-i4/r0);
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm-i0];
        }
      }
      i1=1;
    }
    if (i0<=*nw-1-sxm) {  /* right source pixel line */
      for (i4=i2;i4<=i3;i4++) {
        if (sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0]==RGB_BLACK) {continue;}
        r0=sqrt((double)(i0*i0+i4*i4));
        a0=asin(-i4/r0);
        a0=-a0+PI;  /* source point in right half circle */
        a0+=degpi;
        if (a0>=PI2) {a0-=PI2;}
        /* ix and iy are destination pixel, jx and jy take floating point */
        d0=cos(a0)*r0;
        ix=(int)((float)dxm-d0+.5);
        if ((ix<0) || (ix>=dgrf->width)) {continue;}
        jx=(d0-(int)d0)*10;
        d0=sin(a0)*r0;
        iy=(int)((float)dym-d0+.5);
        if ((iy<0) || (iy>=dgrf->height)) {continue;}
        jy=(d0-(int)d0)*10;
        dgrf->pxm[iy*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        /* if destination pixel would be nearly between two pixels
           draw both to avoid black holes as possible */
        tx=ix; ty=iy;
        if (((jx>=4) && (jx<5)) || ((jx<=-5) && (jx>=-6))) {tx--;}
        else if (((jx<=-4) && (jx>-5)) || ((jx>=5) && (jx<=6))) {tx++;}
        if ((tx!=ix) && (tx>=0) && (tx<dgrf->width)) {
          dgrf->pxm[iy*dgrf->width+tx]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        }
        if (((jy>=4) && (jy<5)) || ((jy<=-5) && (jy>=-6))) {ty--;}
        else if (((jy<=-4) && (jy>-5)) || ((jy>=5) && (jy<=6))) {ty++;}
        if ((ty!=iy) && (ty>=0) && (ty<dgrf->height)) {
          dgrf->pxm[ty*dgrf->width+ix]=sgrf->pxm[(*ny+sym+i4)*sgrf->width+*nx+sxm+i0];
        }
      }
      i1=1;
    }
    i0++;
  } while (i1==1);
  *nx+=sxm-dxm;
  *ny+=sym-dym;
  *nw=dxm*2+1;
  if (*nw>dgrf->width) {*nw=dgrf->width;}
  *nh=dym*2+1;
  if (*nh>dgrf->height) {*nh=dgrf->height;}
  return(dgrf);
} /* Ende rotate_grafik */


grafik * zoom_grafik(grafik * dgrf,grafik * sgrf,float xmul,float ymul,int * nx,int * ny,int * nw,int * nh) {
/* zoom a graphic greater or smaller
** 1.arg: pointer to destination graphic
**        in here is put the zoomed source graphic
** 2.arg: pointer to source graphic
** 3.arg: horizontal zooming (x position)
** 4.arg: vertical zooming (y position)
** 5.+6.arg: value arguments: x,y start position to zoom of source graphic
**           and return arguments:
**             address for relative x/y coordinate to put destination
**             graphic out to backbuffer/graphic:
**               add x to your origin x coordinate
**               add y to your origin y coordinate
** 7.+8.arg: value arguments: width and height to zoom of source graphic
**                            or 0=whole width/height
**           and return arguments:
**             address for width and height of the size of the zoomed graphic
**             in the destination graphic
** return value: pointer to 1.arg
**               or NULL = error
*/
  const int tw=1000;
  int sxm,sym,dxm,dym,txm,tym,sw,i_z,i_a,i_n,vp,bp,hp,hw,i1,i2,ii;
  int fb[256],brk;
  grafik * tgrf;
  if ((sgrf==NULL) || (dgrf==NULL)) {strcpy(errmsg,"you can't use backbuffer with zoom_grafik!"); return(NULL);}
  if (dgrf==sgrf) {strcpy(errmsg,"both graphics are the same in zoom_grafik!"); return(NULL);}
  if ((nx==NULL) || (ny==NULL) || (nw==NULL) || (nh==NULL)) {strcpy(errmsg,"arguments NULL"); return(NULL);}
  CLEAR_BOX(dgrf,RGB_BLACK);
  if ((*nx>=sgrf->width) || (*ny>=sgrf->height)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if ((*nw<1) || (*nx+*nw>sgrf->width)) {*nw=sgrf->width-*nx;}
  if ((*nh<1) || (*ny+*nh>sgrf->height)) {*nh=sgrf->height-*ny;}
  if (*nx<0) {*nw+=*nx; *nx=0;}
  if (*ny<0) {*nh+=*ny; *ny=0;}
  if ((*nw<0) || (*nh<0)) {*nx=*ny=*nw=*nh=0; return(dgrf);}
  if (xmul<.01) {xmul=.01;}
  if (ymul<.01) {ymul=.01;}

  /* source middle of selected part */
  sxm=*nw/2;
  sym=*nh/2;
  /* destination middle */
  i1=sxm;
  if (sxm*2<*nw) {i1++;}
  i1=(int)((float)i1*xmul)*2;
  if (sxm*2<*nw) {i1--;}
  dxm=i1/2;
  if (dxm>dgrf->width/2) {dxm=dgrf->width/2;}
  i1=sym;
  if (sym*2<*nh) {i1++;}
  i1=(int)((float)i1*ymul)*2;
  if (sym*2<*nh) {i1--;}
  dym=i1/2;
  if (dym>dgrf->height/2) {dym=dgrf->height/2;}
  /* temp middle */
  if (sxm*2<*nw) {i1=(dxm+1)*2-1;} else {i1=dxm*2;}
  if (i1<1) {i1=1;}
  i2=*nh;
  if (i2<1) {i2=1;}
  if ((tgrf=create_grafik(i1,i2))==NULL) {return(NULL);}
  txm=tgrf->width/2;
  tym=tgrf->height/2;
  /* x direction */
  if ((sw=(int)((float)tw/xmul))<=0) {sw=1;}
  /* zoom x direction right from source to temp */
  brk=0;
  for (i_z=0;i_z<*nh;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    i_a=sxm; i_n=txm;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=*nw) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a+vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=*nw) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a+vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<tgrf->width) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n++]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom x direction left from source to temp */
  brk=0;
  for (i_z=0;i_z<*nh;i_z++) {
    if ((i_z-sym+tym<0) || (i_z-sym+tym>=tgrf->height)) {continue;}
    if (sxm*2==*nw) {i_a=sxm-1;} else {i_a=sxm;}
    if (txm*2==tgrf->width) {i_n=txm-1;} else {i_n=txm;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a-vp];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=sgrf->pxm[(*ny+i_z)*sgrf->width+*nx+i_a-vp];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {tgrf->pxm[(i_z-sym+tym)*tgrf->width+i_n--]=hp; fb[hp]-=sw;}
    }
  }
  /* y direction */
  if ((sw=(int)((float)tw/ymul))<=0) {sw=1;}
  /* zoom y direction down from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    i_a=tym; i_n=dym;
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a+vp>=tgrf->height) {brk=1; break;}
        ii=tgrf->pxm[(i_a+vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n<dgrf->height) {dgrf->pxm[(i_n++)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  /* zoom y direction up from temp to destination */
  brk=0;
  for (i_z=0;i_z<tgrf->width;i_z++) {
    if ((i_z-txm+dxm<0) || (i_z-txm+dxm>=dgrf->width)) {continue;}
    if (tym*2==tgrf->height) {i_a=tym-1;} else {i_a=tym;}
    if (dym*2==dgrf->height) {i_n=dym-1;} else {i_n=dym;}
    memset(fb,0,sizeof(fb));
    vp=bp=0;
    while (1) {
      i1=sw; i2=0; hp=hw=0;
      while (i1>0) {
        if (i1<tw) {i2=i1;} else {i2=tw;}
        i1-=i2; bp+=i2;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(i2-bp%tw))>hw) {hp=ii; hw=fb[ii];}
        vp=bp/tw;
        if (i_a-vp<0) {brk=1; break;}
        ii=tgrf->pxm[(i_a-vp)*tgrf->width+i_z];
        if ((fb[ii]+=(bp%tw))>hw) {hp=ii; hw=fb[ii];}
      }
      if (brk==1) {brk=0; break;}
      if (i_n>=0) {dgrf->pxm[(i_n--)*dgrf->width+i_z-txm+dxm]=hp; fb[hp]-=sw;}
    }
  }
  free_grafik(tgrf);
  *nx+=sxm-dxm;
  *ny+=sym-dym;
  *nw=dxm*2+1;
  if (*nw>dgrf->width) {*nw=dgrf->width;}
  *nh=dym*2+1;
  if (*nh>dgrf->height) {*nh=dgrf->height;}
  return(dgrf);
} /* Ende zoom_grafik */


int get_pixel(grafik * grf,int x,int y) {
/* get pixel color of one pixel
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: pixel coordinates X,Y
** return value: color index
*/
  return(vg_bitmap_getpixel(grf,x,y));
} /* Ende get_pixel */


void draw_text(grafik * graf,int c,int x,int y,const char * s,int n,int trsp) {
/* draws text
** 1.arg: pointer to grafik or virtual context or NULL=backbuffer
** 2.arg: color index
** 3.+4.arg: target coordinates
** 5.+6.arg: string und string length
** 7.arg: RGB_FULL for full text
**        RGB_TRANS for transparent text
*/
  static char buf[1024]="";
  if (n>=(int)sizeof(buf)) {n=sizeof(buf)-1;}
  strncpy(buf,s,n); buf[n]='\0';
  vg_draw_text(graf,c,x,y,buf,deffont,trsp);
} /* Ende draw_text */


void draw_pixel(grafik * grf,int x,int y,int cl) {
/* draw pixel
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: pixel coordinates X,Y
** 4.arg: color index
*/
  vg_draw_pixel(grf,x,y,cl);
} /* Ende draw_pixel */


void draw_line(grafik * grf,int x1,int y1,int x2,int y2,int cl) {
/* draw line
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: begin of target coordinates X,Y
** 4.+5.arg: end of target coordinates X,Y
** 6.arg: color index
*/
  vg_draw_line(grf,x1,y1,x2,y2,cl);
} /* Ende draw_line */


void draw_circle(grafik * grf,int x,int y,int r,int cl,int fill) {
/* draws circle, centered at x,y
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: target coordinates - middle of circle
** 4.arg: radius
** 5.arg: color index
** 6.arg: 0=not fill, 1=fill it
*/
  vg_draw_circle(grf,x,y,r,cl,fill);
} /* Ende draw_circle */


void draw_fillbox(grafik * grf,int x,int y,int w,int h,int cl) {
/* draw filled rectangle
** 1.arg: pointer to graphic or NULL=backbuffer
** 2.+3.arg: target coordinates
** 4.+5.arg: width and height
**           if width=0 or height=0, then the whole width or height is used
** 6.arg: color index
*/
  vg_draw_box(grf,x,y,w,h,cl,1);
} /* Ende draw_fillbox */


film * create_film(int anz) {
/* create a new film
** 1.arg: number of graphical sequences
** return value: pointer to created film or NULL=error
*/
  film * flm;
  int i1;
  if (anz<=0) {strcpy(errmsg,"create_film: wrong parameter"); return(NULL);}
  if ((flm=malloc(sizeof(film)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->func=NULL;
  flm->step=0UL;
  flm->id=0;
  flm->x=flm->y=0;
  if ((flm->_pixold=malloc(sizeof(grafik *)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->_pixold[0]=NULL;
  if ((flm->_pixnew=malloc(sizeof(grafik *)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->_pixnew[0]=NULL;
  if ((flm->aktiv=malloc(sizeof(int)*(anz+1)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->aktiv[0]=0;
  if ((flm->bildnr=malloc(sizeof(int)*(anz+1)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->bildnr[0]=0;
  if ((flm->bild=malloc(sizeof(film_bild **)*(anz+2)))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
  flm->bild[0]=NULL;
  for (i1=1;i1<=anz;i1++) {
    if ((flm->bild[i1]=malloc(sizeof(film_bild *)*2))==NULL) {strcpy(errmsg,"Cannot create film"); return(NULL);}
    flm->bild[i1][0]=NULL;
    flm->bild[i1][1]=NULL;
    flm->aktiv[i1]=FILM_ACT_OK;
    flm->bildnr[i1]=0;
  }
  flm->bild[anz+1]=NULL;
  return(flm);
} /* Ende create_film */


void free_film(film * flm) {
/* free and destroy a film
** 1.arg: pointer to created film
*/
  int f1,f2;
  if (flm==NULL) {return;}
  free(flm->aktiv);
  free(flm->bildnr);
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    for (f2=1;flm->bild[f1][f2]!=NULL;f2++) {free(flm->bild[f1][f2]);}
    free(flm->bild[f1]);
  }
  free(flm->_pixold);
  for (f1=0;flm->_pixnew[f1]!=NULL;f1++) {free(flm->_pixnew[f1]);}
  free(flm->_pixnew);
  free(flm->bild);
} /* Ende free_film */


int add_bild_to_film(film * flm,film_bild * add_bld,int row) {
/* appends a bild in a film to a certain graphical sequence (row)
** 1.arg: pointer to film
** 2.arg: pointer to filled bild
** 3.arg: graphical sequence number (from 1 to ?, where ? is the argument of
**                                   create_film())
** return value: position of bild in that row (from 1 to ??)
**               or -1=error
*/
  int f1,f2;
  grafik * gmrk;
  if ((flm==NULL) || (add_bld==NULL) || (row<=0)) {strcpy(errmsg,"add_bild_to_film: empty parameter"); return(-1);}
  gmrk=add_bld->pic;
  if (add_bld->pic!=NULL) {  /* with graphic */
    for (f1=0;flm->_pixold[f1]!=NULL;f1++) {  /* copy already allocated? */
      if (flm->_pixold[f1]==(grafik *)1) {continue;}
      if (add_bld->pic==flm->_pixold[f1]) {
        if ((add_bld->pic->width==flm->_pixnew[f1]->width) \
        && (add_bld->pic->height==flm->_pixnew[f1]->height) \
        && (memcmp(add_bld->pic->pxm,flm->_pixnew[f1]->pxm,add_bld->pic->width*add_bld->pic->height)==0)) {
          break;
        } else {
          flm->_pixold[f1]=(grafik *)1;
        }
      }
    }
    if (flm->_pixold[f1]==NULL) {  /* new graphic */
      if ((flm->_pixold=realloc(flm->_pixold,sizeof(grafik *)*(f1+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      flm->_pixold[f1]=add_bld->pic;
      flm->_pixold[f1+1]=NULL;
      if ((flm->_pixnew=realloc(flm->_pixnew,sizeof(grafik *)*(f1+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      if ((flm->_pixnew[f1]=create_grafik(GRAFIK_WIDTH(add_bld->pic),GRAFIK_HEIGHT(add_bld->pic)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
      copy_grafik(flm->_pixnew[f1],0,0,add_bld->pic,0,0,0,0,RGB_FULL);
      flm->_pixnew[f1+1]=NULL;
    }
    add_bld->pic=flm->_pixnew[f1];
  }
  for (f1=1;flm->bild[f1]!=NULL;f1++) {  /* find row */
    if (f1==row) {break;}
  }
  if (flm->bild[f1]==NULL) {strcpy(errmsg,"add_bild_to_film: graphical seq. number to great"); return(-1);}
  for (f2=1;flm->bild[f1][f2]!=NULL;f2++) {;}  /* find last bild */
  if ((flm->bild[f1]=realloc(flm->bild[f1],sizeof(film_bild *)*(f2+2)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
  if ((flm->bild[f1][f2]=malloc(sizeof(film_bild)))==NULL) {strcpy(errmsg,"Cannot add bild to film"); return(-1);}
  memmove(flm->bild[f1][f2],add_bld,sizeof(film_bild));
  add_bld->pic=gmrk;
  flm->bild[f1][f2+1]=NULL;
  return(f2);
} /* Ende add_bild_to_film */


void reset_film(film * flm,void * fpt,int fres) {
/* reset a film ready to start from the beginning
** 1.arg: pointer to film
** 2.arg: pointer to data for film function (2.arg) or NULL
** 3.arg: value for film function (3.arg) or 0
*/
  int f1;
  if (flm==NULL) {return;}
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    flm->aktiv[f1]=FILM_ACT_OK;
    flm->bildnr[f1]=0;
  }
  flm->step=0UL;
  if (flm->func!=NULL) {flm->func(flm,fpt,fres);}
} /* Ende reset_film */


unsigned long copy_film(grafik * gto,film * flm,void * fpt) {
/* draws next step of the film to graphic
** 1.arg: graphic to which film is drawn or NULL=backbuffer
** 2.arg: pointer to film
** 3.arg: pointer to data for film function (2.arg) or NULL
**        (the film function - if any - is called with their 3.arg=0)
** return value: step number of film
*/
  int f1;
  film_bild * bld;
  if (flm==NULL) {return(0UL);}
  flm->step++;
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    if (flm->aktiv[f1]!=FILM_ACT_OK) {continue;}
    if (flm->bildnr[f1]<1) {flm->bildnr[f1]=0;}
    bld=flm->bild[f1][flm->bildnr[f1]];
    while ((flm->bildnr[f1]==0) || (bld->count>=bld->count_to)) {
      bld=flm->bild[f1][++flm->bildnr[f1]];
      if (bld!=NULL) {bld->count=0; break;}
      else if (flm->bildnr[f1]==1) {flm->bildnr[f1]=0; break;}
      else {flm->bildnr[f1]=0;}
    }
    if (flm->bildnr[f1]>=1) {bld->count++;}
  }
  if (flm->func!=NULL) {flm->func(flm,fpt,0);}
  for (f1=1;flm->bild[f1]!=NULL;f1++) {
    if (flm->aktiv[f1]==FILM_ACT_DEAD) {continue;}
    if (flm->bildnr[f1]<1) {continue;}
    bld=flm->bild[f1][flm->bildnr[f1]];
    if (bld->pic!=NULL) {
      copy_grafik(gto,flm->x+bld->x,flm->y+bld->y,bld->pic,0,0,0,0,bld->ft);
    }
    if ((bld->count==1) && (flm->aktiv[f1]==FILM_ACT_OK)) {
      if (bld->wnr>0) {  /* start wave (once) */
        play_wave(bld->wnr,bld->wch,0,bld->wvol);
      } else if (bld->wnr==-1) {  /* stop wave */
        stop_wave(bld->wch,bld->wvol);
      }
    }
  }
  return(flm->step);
} /* Ende copy_film */


sprite * create_sprite() {
/* create a new sprite, can be filled with add_grafik_to_sprite()
** return value: pointer to sprite
**               or NULL = error
*/
  return(vg_sprite_createnew());
} /* Ende create_sprite */


sprite * load_sprite(const char * filename) {
/* load a sprite from a file
** 1.arg: filename of sprite file
** return value: pointer to sprite
**               or NULL = error
** Note: a sprite file is a text file containing:
**         1.line:           "[SPRITE]"
**         following lines:  <graphic file>;<maxrunde>[;ZOOM=<x-zoom>,<y-zoom>][;ROTATE=<degrees>]
**
**       Comment lines begin with `#'.
**       <graphic file>   is loaded from the path, where the sprite file is;
**       <maxrunde>   is an integer how many loops this graphic is shown;
**       ZOOM=<x-zoom>,<y-zoom>   is optional, how to zoom graphic;
**       ROTATE=<degrees>   is optional, how to rotate graphic.
**
**       Example:
**         sprite file is "/usr/local/game/mygame/sprites/test.sprite"
**         contains following lines:
**            [SPRITE]
**            pict1.vga;5
**            ../vga/pict2.vga;17;ROTATE=90
**            /usr/vga/pict3.vga;10;ZOOM=1.5,1.0
**            pict4.vga;3;ZOOM=0.5,0.5;ROTATE=270
**            pict5.vga;0
**
**         --> 1.line is "[SPRITE]"
**             2.line shows graphic "/usr/local/game/mygame/sprites/pict1.vga"
**                    for 5 loops
**             3.line shows graphic "/usr/local/game/mygame/vga/pict2.vga"
**                    rotated at 90 degrees
**                    for 17 loops
**             4.line shows graphic "/usr/vga/pict3.vga"
**                    zoomed 1.5 in x direction and original in y direction
**                    for 10 loops
**             5.line shows graphic "/usr/local/game/mygame/sprites/pict4.vga"
**                    zoomed 0.5 in x direction and 0.5 in y direction
**                    and rotated at 270 degrees
**                    for 3 loops
**             6.line shows graphic "/usr/local/game/mygame/sprites/pict5.vga"
**                    not at all
*/
  FILE * ffp;
  char buf[1024],bfgra[1024],* p1,* p2;
  int zl,maxrunde,vorpfad;
  sprite * spr;
  grafik * grf;
  if (filename==NULL) {strcpy(errmsg,"load_sprite: empty parameter"); return(NULL);}
  if ((ffp=fopen(filename,"r"))==NULL) {snprintf(errmsg,sizeof(errmsg),"Error reading sprite file \"%s\"",filename); return(NULL);}
  if ((p1=strrchr(filename,'/'))!=NULL) {vorpfad=p1-filename+1;} else {vorpfad=0;}
  if ((spr=create_sprite())==NULL) {return(NULL);}
  /* read in each line of sprite file */
  if ((fgets(buf,sizeof(buf),ffp)==NULL) || (strncmp(buf,"[SPRITE]",8)!=0)) {
    snprintf(errmsg,sizeof(errmsg),"load_sprite: \"%s\" seems not to be a vaild sprite file",filename);
    fclose(ffp);
    free_sprite(spr);
    return(NULL);
  }
  zl=1;
  while (fgets(buf,sizeof(buf),ffp)!=NULL) {
    zl++;
    if ((*buf=='#') || ((p1=strchr(buf,';'))==NULL)) {continue;}
    *p1='\0';
    if (*buf!='/') {
      snprintf(bfgra,sizeof(bfgra),"%.*s%s",vorpfad,filename,buf);
    } else {
      snprintf(bfgra,sizeof(bfgra),"%s",buf);
    }
    if ((grf=load_grafik(bfgra))==NULL) {
      snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot load graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
      fclose(ffp);
      free_sprite(spr);
      return(NULL);
    }
    maxrunde=atoi(p1+1);
    /* manipulate graphic if desired */
    if ((p2=strstr(p1+1,"ZOOM="))!=NULL) {  /* zoom graphic */
      grafik * gtmp;
      float xmul,ymul;
      int x,y,w,h;
      xmul=atof(p2+5);
      if ((p2=strchr(p2,','))!=NULL) {ymul=atof(p2+1);} else {ymul=1.;}
      if (xmul<.01) {xmul=.01;}
      if (ymul<.01) {ymul=.01;}
      x=y=w=h=0;
      if (((gtmp=create_grafik((int)((float)grf->width*xmul)+5,(int)((float)grf->height*ymul)+5))==NULL) \
      || (zoom_grafik(gtmp,grf,xmul,ymul,&x,&y,&w,&h)==NULL)) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot zoom graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(grf);
        if (gtmp!=NULL) {free_grafik(gtmp);}
        free_sprite(spr);
        return(NULL);
      }
      free_grafik(grf);
      if ((grf=create_grafik(w,h))==NULL) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot zoom graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(gtmp);
        free_sprite(spr);
        return(NULL);
      }
      copy_grafik(grf,0,0,gtmp,0,0,w,h,RGB_FULL);
      free_grafik(gtmp);
    }
    if ((p2=strstr(p1+1,"ROTATE="))!=NULL) {  /* rotate graphic */
      grafik * gtmp;
      int degree,x,y,w,h;
      degree=atoi(p2+7);
      x=y=w=h=0;
      if (((gtmp=create_grafik(grf->width>grf->height?grf->width:grf->height,grf->height>grf->width?grf->height:grf->width))==NULL) \
      || (rotate_grafik(gtmp,grf,degree,&x,&y,&w,&h)==NULL)) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot rotate graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(grf);
        if (gtmp!=NULL) {free_grafik(gtmp);}
        free_sprite(spr);
        return(NULL);
      }
      free_grafik(grf);
      if ((grf=create_grafik(w,h))==NULL) {
        snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot rotate graphic \"%s\" at line %d in sprite file \"%s\"",bfgra,zl,filename);
        fclose(ffp);
        free_grafik(gtmp);
        free_sprite(spr);
        return(NULL);
      }
      copy_grafik(grf,0,0,gtmp,0,0,w,h,RGB_FULL);
      free_grafik(gtmp);
    }
    /* add graphic and maxrunde to sprite */
    if (add_grafik_to_sprite(spr,grf,0,0,0,0,maxrunde)<0) {
      snprintf(errmsg,sizeof(errmsg),"load_sprite: cannot add graphic \"%s\" to sprite at line %d in sprite file \"%s\"",bfgra,zl,filename);
      fclose(ffp);
      free_grafik(grf);
      free_sprite(spr);
      return(NULL);
    }
    free_grafik(grf);
  }
  fclose(ffp);
  return(spr);
} /* Ende load_sprite */


sprite * rotate_sprite(sprite * spr,int deg) {
/* create new sprite from another existing sprite, rotated at certain degrees
** 1.arg: existing sprite
** 2.arg: degrees to rotate clockwise (-360 to 360)
** return value: pointer to new created sprite
**               or NULL = error
*/
  sprite * spr_neu;
  if ((spr_neu=vg_sprite_duplicate(spr))==NULL) {return(NULL);}
  return(vg_sprite_rotate(spr_neu,deg));
} /* Ende rotate_sprite */


sprite * zoom_sprite(sprite * spr,float xmul,float ymul) {
/* create new sprite from another existing sprite, zoomed greater or smaller
** 1.arg: existing sprite
** 2.arg: horizontal zooming (x position)
** 3.arg: vertical zooming (y position)
** return value: pointer to new created sprite
**               or NULL = error
*/
  sprite * spr_neu;
  if ((spr_neu=vg_sprite_duplicate(spr))==NULL) {return(NULL);}
  return(vg_sprite_zoom(spr_neu,(double)xmul,(double)ymul));
} /* Ende zoom_sprite */


int add_grafik_to_sprite(sprite * spr,grafik * grf,int x,int y,int w,int h,int maxrunde) {
/* add a graphic to a sprite
** 1.arg: loaded or created sprite
** 2.arg: loaded or created graphic
** 3.+4.arg: start coordinates of graphic to copy to sprite
** 5.+6.arg: width and height of graphic to copy to sprite or 0,0=whole graphic
** 7.arg: number of loops to show graphic
** return value: >0: position of graphic in sprite
**               -1: error
**
** The size of the first graphic which is added to a sprite is saved
** as original size (width and height). All following graphics, which of course
** can have other width and height, use this original size to correct their
** x,y-coordinates when you call next_sprite().
*/
  grafik * grf_neu;
  if ((spr==NULL) || (grf==NULL)) {strcpy(errmsg,"add_grafik_to_sprite: empty argument"); return(-1);}
  if ((x>=grf->width) || (y>=grf->height)) {
    if ((grf_neu=create_grafik(1,1))==NULL) {return(-1);}
  } else {
    if ((w<1) || (x+w>grf->width)) {w=grf->width-x;}
    if ((h<1) || (y+h>grf->height)) {h=grf->height-y;}
    if (x<0) {w+=x; x=0;}
    if (y<0) {h+=y; y=0;}
    if ((grf_neu=create_grafik(w,h))==NULL) {return(-1);}
    copy_grafik(grf_neu,0,0,grf,x,y,w,h,RGB_FULL);
  }
  if (vg_sprite_add(spr,grf_neu,maxrunde,NULL,0)<0) {return(-1);}
  return(1);  /* returning position not possible */
} /* Ende add_grafik_to_sprite */


grafik * next_sprite(sprite * spr,int * xkorr,int * ykorr) {
/* get actual graphic of a sprite
** 1.arg: sprite
** 2.+3.arg: return argument:
**             address for relative x/y coordinate to put graphic out:
**               add xkorr to your origin x coordinate
**               add ykorr to your origin y coordinate
** return value: pointer to actual graphic
**               or NULL = no graphic available (sprite empty)
**
** Because the graphics of a sprite can have different sizes, use the
** return arguments xkorr,ykorr to put the graphic out at the same position.
** Example: there are two graphics in a sprite:
**          1. with width and height 9,9
**          2. with width and height 5,5
**          --> the original size is 9,9.
**          When getting the first graphic xkorr,ykorr is 0,0
**          When getting the second graphic xkorr,ykorr is 2,2
**          --> put it out at x+xkorr,y+ykorr
**              e.g. copy_grafik(NULL,x+xkorr,y+ykorr,grf,0,0,0,0,RGB_TRANS)
*/
  grafik * grf;
  int i,j;
  if ((grf=vg_sprite_getnext(spr))==NULL) {
    grf=vg_sprite_getnext(spr);
  }
  if (grf==NULL) {return(NULL);}
  if (xkorr!=NULL) {
    i=GRAFIK_WIDTH(grf);
    j=SPRITE_WIDTH(spr);
    *xkorr=(j-i)/2;
    if (*xkorr*2<j-i) {(*xkorr)++;}
  }
  if (ykorr!=NULL) {
    i=GRAFIK_HEIGHT(grf);
    j=SPRITE_HEIGHT(spr);
    *ykorr=(j-i)/2;
    if (*ykorr*2<j-i) {(*ykorr)++;}
  }
  return(grf);
} /* Ende next_sprite */


grafik * last_sprite(sprite * spr,int * xkorr,int * ykorr) {
/* get actual graphic of a sprite again without incrementing the loop counter
** 1.arg: sprite
** 2.+3.arg: return argument: see next_sprite()
** return value: pointer to actual graphic
**               or NULL = no graphic available (sprite empty
**                         or no last graphic, e.g. after reset_sprite())
*/
  grafik * grf;
  int i,j;
  grf=vg_sprite_getcurrent(spr);
  if (grf==NULL) {return(NULL);}
  if (xkorr!=NULL) {
    i=GRAFIK_WIDTH(grf);
    j=SPRITE_WIDTH(spr);
    *xkorr=(j-i)/2;
    if (*xkorr*2<j-i) {(*xkorr)++;}
  }
  if (ykorr!=NULL) {
    i=GRAFIK_HEIGHT(grf);
    j=SPRITE_HEIGHT(spr);
    *ykorr=(j-i)/2;
    if (*ykorr*2<j-i) {(*ykorr)++;}
  }
  return(grf);
} /* Ende last_sprite */


int reset_sprite(sprite * spr) {
/* reset a sprite
** 1.arg: sprite
** return value: number of loops the whole sprite takes when played once
**               (usefull as counter to play sprite only once)
*/
  int runden,i1;
  if (spr==NULL) {return(0);}
  vg_sprite_reset(spr);
  runden=0;
  for (i1=0;i1<spr->max_elem;i1++) {
    runden+=spr->elem[i1].anz_loop;
  }
  return(runden);
} /* Ende reset_sprite */


void free_sprite(sprite * spr) {
/* free and destroy a sprite
** 1.arg: sprite
*/
  vg_sprite_free(spr);
} /* Ende free_sprite */


int open_window(const char * wnam,int dp) {
/* create window
** 1.arg: name of the window (NULL=same name as before)
** 2.arg: scaling factor
**        0:              get largest window size, no full screen
**        VGAWINDOW_1:    window scaling factor 1  (original size)
**        VGAWINDOW_2:    window scaling factor 2  (4 pixels per pixel)
**        VGAWINDOW_3:    window scaling factor 3  (9 pixels per pixel)
**        VGAWINDOW_FULL: full screen modus
**        VGAWINDOW_NOSWITCH: don't switch to another mode
** return: 0=ok or -1=error
*/
  static int first=1;
  if (first) {vg_init_vgagames("./.",INITVGA_NOFILE,NULL); first=0;}
  return(vg_window_open(wnam,dp,0));
} /* Ende open_window */


void flush_window() {
/* give backbuffer out to screen */
  vg_window_flush();
} /* Ende flush_window */


void close_window() {
/* close window */
  vg_window_close();
} /* Ende window_close */


int reopen_window(int dp) {
/* reopens window for resizing
** 1.arg: see 2.arg in open_window()
** return: see open_window()
*/
  close_window();
  return(open_window(NULL,dp));
} /* Ende reopen_window */


void get_keys() {
/* get pressed or released keys */
  (void)vg_key_update();
} /* Ende get_keys */


void clearstate() {
  vg_key_discard();
} /* Ende clearstate */


int mouse_speed(int index) {
/* speed of mouse, from 1=slow to 3=fast; returns previous speed index */
  return(vg_mouse_speed(index));
} /* Ende mouse_speed */


void brightness(int vol) {
/* reduces or increases brightness of colormap
** (all graphics should be drawn again to take effect of it)
** 1.arg: -63 (darkest) to 62 (lightest), 0 is default, or 99=last brightness
*/
  (void)vg_brightness(vol);
} /* Ende brightness */



/* +++ internal functions +++ */

static FILE * _bmp2vga(FILE * HW,const char * filename) {
/* convert windows bitmap to vgagames format
** 1.arg: filepointer of windows bitmap file located at position 2 (from 0),
**        is closed before returning
** 2.arg: filename
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  union {
    short sw;
    char cw[2];
#if BYTE_ORDER == BIG_ENDIAN
  } _c2s,_c2st;
#else
  } _c2s;
#endif
  union {
    int iw;
    char cw[4];
#if BYTE_ORDER == BIG_ENDIAN
  } _c2i,_c2it;
#else
  } _c2i;
#endif
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  FILE * HX;
  char buf[8];
  int i1,i2,i3,i4,i5,ifl,bwid,bhei,anzfarb,bytedata;
  long offdat,hlen,lw;
  short bpp;
  unsigned char c[4];

  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  /* reading windows bitmap */
  fseek(HW,0x0a,SEEK_SET);  /* 0x0a: offset data */
  fread(&_c2i.iw,4,1,HW);
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  offdat=(long)_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x0e: header size */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  hlen=(long)_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x12: width of bitmap */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bwid=_c2i.iw;
  fread(&_c2i.iw,4,1,HW);  /* 0x16: height of bitmap */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  bhei=_c2i.iw;
  fseek(HW,0x1C,SEEK_SET);  /* 0x1c: bits per pixel at data */
  fread(&_c2s.sw,2,1,HW);
#if BYTE_ORDER == BIG_ENDIAN
  _c2st.cw[0]=_c2s.cw[1]; _c2st.cw[1]=_c2s.cw[0];
  _c2s.sw=_c2st.sw;
#endif
  bpp=_c2s.sw;
  if (bpp>8) {
    snprintf(errmsg,sizeof(errmsg),"Too deep color depth (>256) in file \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  fread(&_c2i.iw,4,1,HW);  /* 0x1e: compression */
#if BYTE_ORDER == BIG_ENDIAN
  _c2it.cw[0]=_c2i.cw[3]; _c2it.cw[1]=_c2i.cw[2]; _c2it.cw[2]=_c2i.cw[1]; _c2it.cw[3]=_c2i.cw[0];
  _c2i.iw=_c2it.iw;
#endif
  lw=(long)_c2i.iw;
  if (lw!=0L) {
    snprintf(errmsg,sizeof(errmsg),"File \"%s\" uses compression: not supported",filename);
    fclose(HW);
    return(NULL);
  }

  anzfarb=(1<<(int)bpp);
  lw=14L+hlen+(long)(anzfarb*4);
  if (offdat!=lw) {
    snprintf(errmsg,sizeof(errmsg),"Corrupt colormap in file \"%s\": %ld!=%ld",filename,offdat,lw);
    fclose(HW);
    return(NULL);
  }

  fseek(HW,0x36,SEEK_SET);  /* 0x36: colormap */
  for (i1=0;i1<anzfarb;i1++) {
    fread(c,1,4,HW);
    fw[i1].r=(c[2]>>2);
    fw[i1].g=(c[1]>>2);
    fw[i1].b=(c[0]>>2);
  }
  lw=(long)(bwid*(int)bpp/32*32);
  if (bwid*(int)bpp%32>0) {lw+=32;}
  lw*=(long)bhei;
  bytedata=(int)(lw/8L);

  /* write vgagames graphic file */
  if ((HX=tmpfile())==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot write vgagames graphic file from \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  strcpy(buf,"VG");
  fwrite(buf,1,2,HX);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  fwrite(&buf,1,5,HX);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  fwrite(&buf,1,5,HX);  /* height */
  for (i1=0;i1<256;i1++) {  /* colormap */
    fwrite(&fw[i1].r,1,1,HX);  /* red byte */
    fwrite(&fw[i1].g,1,1,HX);  /* green byte */
    fwrite(&fw[i1].b,1,1,HX);  /* blue byte */
  }
  c[0]=0;
  for (i1=0;i1<bwid;i1++) {
    for (i2=0;i2<bhei;i2++) {fwrite(c,1,1,HX);}
  }

  fseek(HX,-bwid,SEEK_CUR);
  for (ifl=0,i4=bpp-1,i5=0,i1=0;i1<bytedata;i1+=4) {
    if (fread(c,1,4,HW)<4) {
      snprintf(errmsg,sizeof(errmsg),"Bitmap %s too short (at byte %d)",filename,i1+1);
      fclose(HW);
      fclose(HX);
      return(NULL);
    }
    for (i3=0;i3<4;i3++) {
      if (bpp==8) {
        if (i5++<bwid) {fwrite(&kv[c[i3]],1,1,HX);}
      } else {
        for (i2=7;i2>=0;i2--) {
          ifl|=(!!((unsigned int)c[i3]&(1<<i2))<<i4);
          if (--i4<0) {
            if (i5++<bwid) {fwrite(&kv[ifl],1,1,HX);}
            i4=bpp-1;
            ifl=0;
          }
        }
      }
    }
    if (i5>=bwid) {i5=0; i4=bpp-1; ifl=0; fseek(HX,-bwid*2,SEEK_CUR);}
  }
  fclose(HW);
  fflush(HX);
  fseek(HX,2L,SEEK_SET);
  return(HX);
} /* Ende _bmp2vga */


static FILE * _ppm2vga(FILE * HW,const char * filename,int raw) {
/* convert ppm to vgagames format
** 1.arg: filepointer of ppm file located at position 2 (from 0),
**        is closed before returning
** 2.arg: filename
** 3.arg: 0=true ppm or 1=raw ppm
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  farbkonv fw[256];
  unsigned char kv[256];  /* kv[w]=x */
  FILE * HX;
  char buf[64];
  int ival,i1,i2,i3,i4,bwid=0,bhei=0,maxv=0;
  unsigned char c[3];

  /* Initialization */
  for (i1=0;i1<256;i1++) {
    fw[i1].r=fw[i1].g=fw[i1].b=0;
    kv[i1]=(unsigned char)i1;
  }

  for (i3=0;i3<3;i3++) {
    /* whitespace(s) */
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival>' ') {
        if ((unsigned char)ival=='#') {
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival=='\n') {break;}
          }
          if (ival==EOF) {break;}
        } else {break;}
      }
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }

    /* value */
    i2=0;
    buf[i2++]=(char)ival;
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival<=' ') {break;}
      if ((unsigned char)ival=='#') {break;}
      if (i2==(int)sizeof(buf)) {ival=EOF; break;}
      buf[i2++]=(char)ival;
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }
    buf[i2]='\0';
    if (i3==0) {bwid=atoi(buf);}
    else if (i3==1) {bhei=atoi(buf);}
    else if (i3==2) {maxv=atoi(buf);}
    if ((unsigned char)ival=='#') {
      while ((ival=fgetc(HW))!=EOF) {
        if ((unsigned char)ival=='\n') {break;}
      }
      if (ival==EOF) {
        snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
        fclose(HW);
        return(NULL);
      }
    }
  }
  /* single whitespace just read */

  if (maxv>255) {
    snprintf(errmsg,sizeof(errmsg),"file \"%s\" has too deep color depth (>255)",filename);
    fclose(HW);
    return(NULL);
  }

  if (raw==1) {
    /* whitespace(s) */
    while ((ival=fgetc(HW))!=EOF) {
      if ((unsigned char)ival>' ') {break;}
    }
    if (ival==EOF) {
      snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
      fclose(HW);
      return(NULL);
    }
  }

  /* write vgagames graphic file */
  if ((HX=tmpfile())==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot write vgagames graphic file from \"%s\"",filename);
    fclose(HW);
    return(NULL);
  }
  strcpy(buf,"VG");
  fwrite(buf,1,2,HX);
  snprintf(buf,sizeof(buf),"%05d",bwid);
  fwrite(&buf,1,5,HX);  /* width */
  snprintf(buf,sizeof(buf),"%05d",bhei);
  fwrite(&buf,1,5,HX);  /* height */
  buf[0]=buf[1]=buf[2]=0;
  for (i1=0;i1<256;i1++) {
    fwrite(buf,1,3,HX);  /* room for colormap */
  }

  i4=0;
  for (i1=0;i1<bhei;i1++) {
    for (i2=0;i2<bwid;i2++) {
      if (raw==1) {
        int ii;
        for (ii=0;ii<3;ii++) {
          if (ival==EOF) {
            snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
            fclose(HW);
            return(NULL);
          }
          i3=0;
          buf[i3++]=(char)ival;
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival<=' ') {break;}
            if (i3==(int)sizeof(buf)) {ival=EOF; break;}
            buf[i3++]=(char)ival;
          }
          buf[i3]='\0';
          c[ii]=(unsigned char)atoi(buf);
          /* whitespace(s) */
          while ((ival=fgetc(HW))!=EOF) {
            if ((unsigned char)ival>' ') {break;}
          }
        }
      } else {
        if (fread(c,1,3,HW)<3) {
          snprintf(errmsg,sizeof(errmsg),"file \"%s\" corrupt",filename);
          fclose(HW);
          return(NULL);
        }
      }
      for (i3=0;i3<i4;i3++) {
        if (((c[0]>>2)==fw[i3].r) \
        && ((c[1]>>2)==fw[i3].g) \
        && ((c[2]>>2)==fw[i3].b)) {
          fwrite(&kv[i3],1,1,HX);
          break;
        }
      }
      if (i3==i4) {
        if (++i4==256) {
          snprintf(errmsg,sizeof(errmsg),"Too many colors (>256) in file \"%s\"",filename);
          fclose(HW);
          fclose(HX);
          return(NULL);
        }
        fw[i3].r=(c[0]>>2);
        fw[i3].g=(c[1]>>2);
        fw[i3].b=(c[2]>>2);
        fwrite(&kv[i3],1,1,HX);
      }
    }
  }
  fseek(HX,12L,SEEK_SET);
  for (i1=0;i1<256;i1++) {  /* colormap */
    fwrite(&fw[i1].r,1,1,HX);  /* red byte */
    fwrite(&fw[i1].g,1,1,HX);  /* green byte */
    fwrite(&fw[i1].b,1,1,HX);  /* blue byte */
  }
  fclose(HW);
  fflush(HX);
  fseek(HX,2L,SEEK_SET);
  return(HX);
} /* Ende _ppm2vga */


static FILE * _vga_grafik(const char * filename) {
/* convert graphic file to vgagames format
** 1.arg: filename of graphic file
** return value: filepointer of temporary vgagames graphic file
**               located at position 2 (from 0)
**               or NULL=error
*/
  FILE * HW;
  char buf[8];

  /* read type of graphic file */
  if (filename==NULL) {strcpy(errmsg,"No graphic filename to open"); return(NULL);}
  if ((HW=fopen(filename,"r"))==NULL) {
    snprintf(errmsg,sizeof(errmsg),"Cannot open graphic file \"%s\"",filename);
    return(NULL);
  }
  *buf='\0';
  fread(buf,1,2,HW);
  /* vgagames format */
  if (strncmp(buf,"VG",2)==0) {return(HW);}
  /* windows bitmap format */
  if (strncmp(buf,"BM",2)==0) {return(_bmp2vga(HW,filename));}
  /* ppm format */
  if (strncmp(buf,"P6",2)==0) {return(_ppm2vga(HW,filename,0));}
  if (strncmp(buf,"P3",2)==0) {return(_ppm2vga(HW,filename,1));}
  /* not supported */
  snprintf(errmsg,sizeof(errmsg),"File \"%s\" has no supported graphic format",filename);
  fclose(HW);
  return(NULL);
} /* Ende _vga_grafik */
