#!/bin/sh
#
# config.sh: configure values according to your system into config.h
#

localinst=0
if test "$1" = "-local" || test "$1" = "--local"
then
  localinst=1
elif test "$1" != ""
then
  echo "Configure VgaGames for your system"
  echo "Usage: $0 [-local]"
  echo "Use parameter \"-local\" for local-installation of VgaGames"
  exit 1
fi

# check echo
ECHO="echo"; ECHOSTRIP="\c"
test "`$ECHO \"$ECHOSTRIP\"`" != "" && ECHO="echo -e"
test "`$ECHO \"$ECHOSTRIP\"`" != "" && { ECHO="echo -n"; ECHOSTRIP=""; }


# ***************************************
# ***** Check for system parameters *****
# ***************************************

echo
echo "==> VgaGames2 Tests <=="
echo
echo "==> Testing your system ..."
rm -f config.h config.log vgag2-config _vg2.* Makefile
cp vgag2-config.tmpl _vg2.cf
cp Makefile.tmpl _vg2.mk
echo "#ifndef CONFIG_H_" > config.h
echo "#define CONFIG_H_" >> config.h
echo >> config.h

echo "####################" >> config.log
echo "### System Tests ###" >> config.log
echo "####################" >> config.log
echo >> config.log

if test $localinst -eq 1
then
  sharedir=`(cd ../.. && pwd)`/vgagames
else
  sharedir="/usr/share/vgagames"
fi
echo "SHAREDIR=\"$sharedir\"" >> config.log
echo >> config.log
echo "#define SHAREDIR \"$sharedir\"" >> config.h
rm -f _vg2; cat _vg2.cf | sed -e "s%##SHAREDIR##%$sharedir%" > _vg2; mv -f _vg2 _vg2.cf
rm -f _vg2; cat _vg2.mk | sed -e "s%##SHAREDIR##%$sharedir%" > _vg2; mv -f _vg2 _vg2.mk
rm -f _vg2; cat _vg2.mk | sed -e "s%##LOCALINST##%$localinst%" > _vg2; mv -f _vg2 _vg2.mk


# ++++++++++++++++++
# +++ c-compiler +++
# ++++++++++++++++++

$ECHO "checking for c-compiler:                $ECHOSTRIP"
(echo; echo "*** checking for c-compiler ***") >> config.log

rm -f _xt.c _xt.o _xt
cat > _xt.c <<__EOF__
#include <stdio.h>
int main() {
  return(0);
}
__EOF__
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log
tmp=0
for TCC in $CC gcc cc
do
  (echo; echo "--> Begin compiling error messages: $TCC -c _xt.c") >> config.log
  $TCC -c _xt.c 2>> config.log
  x=$?
  echo "<-- End compiling error messages" >> config.log
  if test $x -eq 0
  then
    echo "$TCC"
    (echo; echo "==> $TCC <==") >> config.log
    rm -f _vg2; cat _vg2.mk | sed -e "s%##CC##%$TCC%" > _vg2; mv -f _vg2 _vg2.mk
    tmp=1
    # Check for "-Wall"
    $TCC -Wall -c _xt.c 2> /dev/null
    x=$?
    if test $x -eq 0
    then
      rm -f _vg2; cat _vg2.mk | sed -e "s%##CFLAGS##%-Wall -O2%" > _vg2; mv -f _vg2 _vg2.mk
    else
      rm -f _vg2; cat _vg2.mk | sed -e "s%##CFLAGS##%-O2%" > _vg2; mv -f _vg2 _vg2.mk
    fi
    break
  fi
done
if test $tmp -eq 0
then
  rm -f _xt.c _xt.o _xt
  echo "- not found (see config.log)"
  echo "==> Error: no c-compiler found: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi
CC=$TCC
CPP="$CC -E"
rm -f _vg2; cat _vg2.mk | sed -e "s%##CPP##%$CPP%" > _vg2; mv -f _vg2 _vg2.mk
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt


# ++++++++++++++++++++++++++++++++
# +++ compiling helper program +++
# ++++++++++++++++++++++++++++++++

rm -f read_link
$CC -DUSE_MAIN read_link.c -o read_link || { echo "Cannot compile read_link.c"; rm -f config.h _vg2.* read_link; exit 1; }


# +++++++++++++++++++++++++++++
# +++ check for kernel name +++
# +++++++++++++++++++++++++++++

erg1=`uname`
test "$erg1" = "SunOS" && erg1="SOLARIS"
(echo "#include <sys/param.h>"; echo "#ifndef $erg1"; echo "# define $erg1"; echo "#endif") >> config.h
(echo; echo "*** Kernel name: $erg1 ***") >> config.log
(echo; echo "####################") >> config.log


# ++++++++++++++++++++
# +++ size of long +++
# ++++++++++++++++++++

$ECHO "Size of long:                           $ECHOSTRIP"
(echo; echo "*** Size of long ***") >> config.log

rm -f _xt.c _xt.o _xt
cat > _xt.c <<__EOF__
#include <stdio.h>
int main() {
  printf("%d\n",sizeof(long));
  return(0);
}
__EOF__
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi

(echo; echo "--> Begin executing ./_xt") >> config.log
erg1=`./_xt 2>> config.log`
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Executing error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi
echo "$erg1 bytes"

if test $erg1 -eq 8
then
  echo "#define LONG8_BIT" >> config.h
else
  echo "/* #undef LONG8_BIT */" >> config.h
fi
(echo; echo "==> $erg1 bytes <==") >> config.log
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt


# ++++++++++++++++++++++++++++++
# +++ check for sys/select.h +++
# ++++++++++++++++++++++++++++++

var_sys_select_h=""
$ECHO "have sys/select.h                       $ECHOSTRIP"
(echo; echo "*** have sys/select.h? ***") >> config.log

rm -f _xt.c _xt.o _xt
cat > _xt.c <<__EOF__
#include <sys/types.h>
#include <sys/select.h>
int main() {
  return(0);
}
__EOF__
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC -c _xt.c") >> config.log
$CC -c _xt.c 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -eq 0
then
  echo "yes"
  (echo; echo "==> yes <==") >> config.log
  echo "#define HAVE_SYS_SELECT_H" >> config.h
  var_sys_select_h="#include <sys/select.h>"
else
  echo "no"
  (echo; echo "==> no <==") >> config.log
  echo "/* #undef HAVE_SYS_SELECT_H */" >> config.h
  var_sys_select_h=""
fi
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt


# ++++++++++++++++++++++++++++++++
# +++ get time select(2) needs +++
# ++++++++++++++++++++++++++++++++

$ECHO "Get time select(2) needs:               $ECHOSTRIP"
(echo; echo "*** Get time select(2) needs ***") >> config.log

rm -f _xt.c _xt.o _xt
cat > _xt.c <<__EOF__
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
$var_sys_select_h
int main() {
  struct timeval tv;
  struct timezone tz;
  int ms,ws,i1;
  if (gettimeofday(&tv,&tz)!=0) {return(1);}
  ms=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  for (i1=0;i1<100;i1++) {
    tv.tv_usec=10000;
    tv.tv_sec=0;
    select(0,NULL,NULL,NULL,&tv);
  }
  if (gettimeofday(&tv,&tz)!=0) {return(1);}
  ws=(tv.tv_sec%60)*1000+tv.tv_usec/1000;
  if (ws<ms) {ws+=60000;}
  ws-=ms;
  ws-=1000;
  ws/=100;
  printf("%d\n",ws);
  return(0);
}
__EOF__
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi

(echo; echo "--> Begin executing ./_xt") >> config.log
erg1=`./_xt 2>> config.log`
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Executing error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi
echo "$erg1 milliseconds"

echo "#define TIME_OF_SELECT $erg1" >> config.h
(echo; echo "==> $erg1 milliseconds <==") >> config.log
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt


# +++++++++++++++++++
# +++ find endian +++
# +++++++++++++++++++

$ECHO "Endian?                                 $ECHOSTRIP"
(echo; echo "*** Endian? ***") >> config.log

rm -f _xt.c _xt.o _xt
cat > _xt.c <<__EOF__
#include <stdio.h>
int main() {
  union {
    short a;
    char b[2];
  } ue;
  ue.a=1;
  if (ue.b[0]==1) {return(0);}
  return(1);
}
__EOF__
(echo; echo "--> Begin test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi

(echo; echo "--> Begin executing ./_xt") >> config.log
./_xt 2>> config.log
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -eq 0
then
  echo "#define ENDIAN_IS_LITTLE" >> config.h
  echo "LITTLE_ENDIAN"
  (echo; echo "==> LITTLE_ENDIAN <==") >> config.log
else
  echo "#define ENDIAN_IS_BIG" >> config.h
  echo "BIG_ENDIAN"
  (echo; echo "==> BIG_ENDIAN <==") >> config.log
fi
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt


# ++++++++++++++++++++++++++++++++++
# +++ check for pipe buffer size +++
# ++++++++++++++++++++++++++++++++++

$ECHO "Testing pipe buffer size:               $ECHOSTRIP"
(echo; echo "*** Testing pipe buffer size ***") >> config.log

rm -f _xt.c _xt.o _xt _xt2.c _xt2.o _xt2
cat > _xt.c <<__EOF__
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <signal.h>
size_t blen;
FILE * pp;
void do_pipe(int signum) {
  signal(SIGPIPE,SIG_IGN);
  printf("%d\n",blen); fflush(stdout);
  pclose(pp);
  exit(0);
}
int main() {
  char buf[128];
  size_t i1;
  memset(buf,'0',sizeof(buf));
  signal(SIGPIPE,do_pipe);
  if ((pp=popen("./_xt2","w"))==NULL) {return(1);}
  blen=0;
  while (1) {
    i1=fwrite(buf,1,sizeof(buf),pp); fflush(pp);
    blen+=i1;
  }
  /* notreached */
  pclose(pp);
  return(0);
}
__EOF__
cat > _xt2.c <<__EOF__
#include <unistd.h>
int main() {
  sleep(2);
  return(0);
}
__EOF__

(echo; echo "--> Begin 1st test file _xt.c:") >> config.log
cat _xt.c >> config.log
echo "<-- End 1st test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC _xt.c -o _xt") >> config.log
$CC _xt.c -o _xt 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi

(echo; echo "--> Begin 2nd test file _xt2.c:") >> config.log
cat _xt2.c >> config.log
echo "<-- End 2nd test file" >> config.log

(echo; echo "--> Begin compiling error messages: $CC _xt2.c -o _xt2") >> config.log
$CC _xt2.c -o _xt2 2>> config.log
x=$?
echo "<-- End compiling error messages" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt _xt2.c _xt2.o _xt2
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi

(echo; echo "--> Begin executing ./_xt with subcall to ./xt2") >> config.log
erg1=`./_xt 2>> config.log`
x=$?
echo "<-- End executing with return status $x" >> config.log
if test $x -ne 0
then
  rm -f _xt.c _xt.o _xt _xt2.c _xt2.o _xt2
  echo "- error (see config.log)"
  echo "==> Compiling error: aborting." >> config.log
  rm -f config.h _vg2.* read_link
  exit 1
fi
echo "$erg1 bytes"

echo "#define PIPE_BUFFERSIZE $erg1" >> config.h
(echo; echo "==> $erg1 bytes <==") >> config.log
(echo; echo "####################") >> config.log
rm -f _xt.c _xt.o _xt _xt2.c _xt2.o _xt2


# +++++++++++++++++++++++++
# +++ System tests done +++
# +++++++++++++++++++++++++

echo "<== system tests done"



# ***************************
# ***** Additional test *****
# ***************************

# +++ support functions +++
(echo "$CC"; echo "$CPP"; echo "$sharedir") | sh ./config-support.inc || { rm -f config.h _vg2.* read_link; exit 1; }

# +++ video libraries +++
(echo "$CC"; echo "$CPP"; echo "$sharedir") | sh ./config-video.inc || { rm -f config.h _vg2.* read_link; exit 1; }

# +++ sound devices +++
(echo "$CC"; echo "$CPP"; echo "$sharedir") | sh ./config-sound.inc || { rm -f config.h _vg2.* read_link; exit 1; }

# +++ network parameters +++
(echo "$CC"; echo "$CPP"; echo "$sharedir") | sh ./config-nw.inc || { rm -f config.h _vg2.* read_link; exit 1; }



# ********************
# ***** All done *****
# ********************

echo >> config.h
echo "#endif" >> config.h
mv -f _vg2.cf vgag2-config
mv -f _vg2.mk Makefile
rm -f read_link
echo
echo "Install into directory: \"$sharedir\""
echo
echo "==> VgaGames2 Tests done (see config.log) <=="
echo
exit 0
