/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* drawing functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames2.h"
#include "bitmap.h"
#include "font.h"
#include "utf8.h"

void vg_draw_text(bitmap *,int,int,int,const char *,const char *,int);
void vg_draw_pixel(bitmap *,int,int,int);
void vg_draw_line(bitmap *,int,int,int,int,int);
void vg_draw_box(bitmap *,int,int,int,int,int,int);
void vg_draw_circle(bitmap *,int,int,int,int,int);
void vg_draw_fillout(bitmap *,int,int,int);


/* +++ functions +++ */

void vg_draw_text(bitmap * grf,int fc,int x,int y,const char * text,const char * fontXxY,int trsp) {
/* draw text string with selected font
** 1.arg: bitmap to draw text into or NULL=backbuffer
** 2.arg: text color
** 3.+4.arg: x/y (upper left) position to draw text
** 5.arg: text string
** 6.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** 7.arg: RGB_FULL for full text
**        RGB_TRANS for transparent text
*/
  text=utf8text(text);
  draw_fontXxY(grf,fc,x,y,text,strlen(text),fontXxY,trsp,RGB_BLACK);
} /* Ende vg_draw_text */


void vg_draw_pixel(bitmap * grf,int x,int y,int cl) {
/* draw pixel
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: pixel coordinates x,y
** 4.arg: color index
*/
  if (grf==NULL) {grf=backbuffer;}
  if ((x<0) || (x>=grf->width) || (y<0) || (y>=grf->height)) {return;}
  grf->pxm[y*grf->width+x]=(unsigned char)cl;
} /* Ende vg_draw_pixel */


void vg_draw_line(bitmap * grf,int x1,int y1,int x2,int y2,int cl) {
/* draw line
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: begin of target coordinates x,y
** 4.+5.arg: end of target coordinates x,y
** 6.arg: color index
*/
  float f1,f2,f3;
  int i1,i2,xa,xe,ya,ye;
  if (grf==NULL) {grf=backbuffer;}
  xa=x2-x1; if (xa<0) {xa=-xa;}
  ya=y2-y1; if (ya<0) {ya=-ya;}
  if ((xa==0) && (ya==0)) {vg_draw_pixel(grf,x1,y1,cl); return;}
  if (xa>=ya) {
    xa=x1; xe=x2; ya=y1; ye=y2;
    if (xe<xa) {
      f1=(float)(ye-ya)/(float)(xa-xe);
      f2=0.;
      if (ye>ya) {f3=.005;} else if (ye<ya) {f3=-.005;} else {f3=.0;}
      for (i1=xa;i1>=xe;i1--) {
        i2=ya+(int)(f2+f3);
        if ((i1>=0) && (i1<grf->width) && (i2>=0) && (i2<grf->height)) {
          grf->pxm[i2*grf->width+i1]=(unsigned char)cl;
        }
        f2+=f1;
      }
    } else {
      f1=(float)(ye-ya)/(float)(xe-xa);
      f2=0.;
      if (ye>ya) {f3=.005;} else if (ye<ya) {f3=-.005;} else {f3=.0;}
      for (i1=xa;i1<=xe;i1++) {
        i2=ya+(int)(f2+f3);
        if ((i1>=0) && (i1<grf->width) && (i2>=0) && (i2<grf->height)) {
          grf->pxm[i2*grf->width+i1]=(unsigned char)cl;
        }
        f2+=f1;
      }
    }
  } else {
    ya=y1; ye=y2; xa=x1; xe=x2;
    if (ye<ya) {
      f1=(float)(xe-xa)/(float)(ya-ye);
      f2=0.;
      if (xe>xa) {f3=.005;} else if (xe<xa) {f3=-.005;} else {f3=.0;}
      for (i1=ya;i1>=ye;i1--) {
        i2=xa+(int)(f2+f3);
        if ((i2>=0) && (i2<grf->width) && (i1>=0) && (i1<grf->height)) {
          grf->pxm[i1*grf->width+i2]=(unsigned char)cl;
        }
        f2+=f1;
      }
    } else {
      f1=(float)(xe-xa)/(float)(ye-ya);
      f2=0.;
      if (xe>xa) {f3=.005;} else if (xe<xa) {f3=-.005;} else {f3=.0;}
      for (i1=ya;i1<=ye;i1++) {
        i2=xa+(int)(f2+f3);
        if ((i2>=0) && (i2<grf->width) && (i1>=0) && (i1<grf->height)) {
          grf->pxm[i1*grf->width+i2]=(unsigned char)cl;
        }
        f2+=f1;
      }
    }
  }
} /* Ende vg_draw_line */


void vg_draw_box(bitmap * grf,int x,int y,int w,int h,int cl,int filled) {
/* draw filled or not filled rectangle
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: target coordinates
** 4.+5.arg: width and height
**           if width=0 or height=0, then the whole width or height is used
** 6.arg: color index
** 7.arg: 0=not filled, 1=filled
*/
  int i1;
  if (grf==NULL) {grf=backbuffer;}
  if ((x>=grf->width) || (y>=grf->height)) {return;}
  if ((w<1) || (x+w>grf->width)) {w=grf->width-x;}
  if ((h<1) || (y+h>grf->height)) {h=grf->height-y;}
  if (x<0) {w+=x; x=0;}
  if (y<0) {h+=y; y=0;}
  if ((w<=0) || (h<=0)) {return;}
  if (filled) {
    for (i1=0;i1<h;i1++) {
      memset(grf->pxm+(y+i1)*grf->width+x,cl,w);
    }
  } else {
    memset(grf->pxm+y*grf->width+x,cl,w);
    memset(grf->pxm+(y+h-1)*grf->width+x,cl,w);
    for (i1=1;i1<h-1;i1++) {
      grf->pxm[(y+i1)*grf->width+x]=(unsigned char)cl;
      grf->pxm[(y+i1)*grf->width+x+w-1]=(unsigned char)cl;
    }
  }
} /* Ende vg_draw_box */


void vg_draw_circle(bitmap * grf,int x,int y,int r,int cl,int filled) {
/* draw circle, centered at x,y
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: target coordinates - middle of circle
** 4.arg: radius
** 5.arg: color index
** 6.arg: 0=not filled, 1=filled
*/
#define MALE_PIX_   if ((i1>=0) && (i1<grf->width) && (i2>=0) && (i2<grf->height)) { \
                      grf->pxm[i2*grf->width+i1]=(unsigned char)cl; \
                    }
#define MALE_LINE_   if (i1<0) {i3+=i1; i1=0;} \
                     if (i1+i3>grf->width) {i3=grf->width-i1;} \
                     if ((i2>=0) && (i2<grf->height) && (i3>0)) { \
                       memset(grf->pxm+i2*grf->width+i1,cl,i3); \
                     }
  int i1,i2,i3,dx,dy,dd;
  dx=r; dy=0; dd=1-r;
  if (grf==NULL) {grf=backbuffer;}
  if (r<1) {vg_draw_pixel(grf,x,y,cl); return;}
  if (filled) {
    i1=x; i2=y-dx; MALE_PIX_;
    i1=x; i2=y+dx; MALE_PIX_;
    i1=x-dx; i2=y; i3=2*dx+1; MALE_LINE_;
  } else {
    i1=x-dx; i2=y; MALE_PIX_;
    i1=x+dx; i2=y; MALE_PIX_;
    i1=x; i2=y-dx; MALE_PIX_;
    i1=x; i2=y+dx; MALE_PIX_;
  }
  while (dy<dx) {
    if (dd<0) {dd+=dy*2+3;} else {dd+=dy*2-dx*2+5; dx--;}
    dy++;
    if (filled) {
      i1=x-dx; i2=y-dy; i3=2*dx+1; MALE_LINE_;
      i1=x-dx; i2=y+dy; i3=2*dx+1; MALE_LINE_;
      i1=x-dy; i2=y-dx; i3=2*dy+1; MALE_LINE_;
      i1=x-dy; i2=y+dx; i3=2*dy+1; MALE_LINE_;
    } else {
      i1=x-dx; i2=y-dy; MALE_PIX_;
      i1=x-dx; i2=y+dy; MALE_PIX_;
      i1=x+dx; i2=y-dy; MALE_PIX_;
      i1=x+dx; i2=y+dy; MALE_PIX_;
      i1=x-dy; i2=y-dx; MALE_PIX_;
      i1=x-dy; i2=y+dx; MALE_PIX_;
      i1=x+dy; i2=y-dx; MALE_PIX_;
      i1=x+dy; i2=y+dx; MALE_PIX_;
    }
  }
#undef MALE_LINE_
#undef MALE_PIX_
} /* Ende vg_draw_circle */


void vg_draw_fillout(bitmap * grf,int x,int y,int cl) {
/* fill area around x,y out with color cl
** (works only with x,y inside the bitmap)
** 1.arg: bitmap or NULL=backbuffer
** 2.+3.arg: start coordinates
** 4.arg: color index
*/
  int zahl,fanz,farbe,mp,mn;
  char * ftest,* ptr;
  if (grf==NULL) {grf=backbuffer;}
  if ((x<0) || (x>=grf->width) || (y<0) || (y>=grf->height)) {return;}
  fanz=grf->width*grf->height;
  if ((ftest=malloc(sizeof(char)*(fanz+1)))==NULL) {fprintf(stderr,"vg_draw_fillout: malloc: %s.\n",strerror(errno)); return;}
  memset(ftest,'0',fanz); ftest[fanz]='\0';
  ftest[y*grf->width+x]='1';
  farbe=grf->pxm[y*grf->width+x];
  zahl=0;
  ptr=ftest;
  while (1) {
    if ((ptr=strchr(ptr,'1'))==NULL) {
      if (zahl==0) {break;}
      zahl=0;
      ptr=ftest;
      continue;
    }
    zahl++;
    mp=ptr-ftest;
    grf->pxm[mp]=cl;
    if ((mn=mp-grf->width)>=0) {  /* upper pixel */
      if ((ftest[mn]=='0') && (grf->pxm[mn]==farbe)) {ftest[mn]='1';}
    }
    if ((mn=mp+grf->width)<fanz) {  /* lower pixel */
      if ((ftest[mn]=='0') && (grf->pxm[mn]==farbe)) {ftest[mn]='1';}
    }
    if ((mn=mp+1)<fanz) {  /* right pixel */
      if ((ftest[mn]=='0') && (grf->pxm[mn]==farbe) && (mn%grf->width>0)) {ftest[mn]='1';}
    }
    if ((mn=mp-1)>=0) {  /* left pixel */
      if ((ftest[mn]=='0') && (grf->pxm[mn]==farbe) && (mn%grf->width<grf->width-1)) {ftest[mn]='1';}
    }
    ftest[mp]='2';
    ptr++;
  }
  free(ftest);
} /* Ende vg_draw_fillout */
