/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* film functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "vgagames2.h"
#include "inthash.h"


struct filmpic {
  char * basex,* basey;
  char * loop;
  char * colmap;
  char * bright;
  char * clearbg;  /* NULL=no clear */
  int anz_bitm;
  struct {  /* bitmaps and sprites */
    int idx;  /* bitmap|sprite: bitm_idx[idx-1]|spt_idx[idx-1] */
    char * x,* y;
    char * trans;
    char * next;  /* NULL=bitmap, "current"/"next"=sprite */
  } * bitm;
  int anz_snd;
  struct {
    int idx;  /* sound: snd_idx[idx-1] */
    char * obloop;  /* NULL=stop, "once"/"loop"=start */
  } * snd;
};
typedef struct {
  bitmap ** bitm_idx;  /* bitmaps, last element=NULL */
  sprite ** spt_idx;  /* sprites, last element=NULL */
  char ** snd_idx;  /* h- names, last element=NULL */
  int anz_elem;
  struct {
    int loop;
    struct filmpic fmp;
    char * vars[10];
  } * elem;
} film;


extern void vg_sound_attach_hidden(const char *,const char *,int);
extern int vg_sound_play_hidden(const char *,int,int);
extern void vg_sound_deta_hidden(const char *);

extern void setze_vars(int,char **);
extern int rechne_formel(const char *,long *);
extern char * ermittle_string(const char *,const char *);

int vg_film_play(const char *,int (*)(int));
void vg_film_position(int *,int *,int *);

static int aex=-1,lex=-1,lop=-1;
static char cwkop[512]="";
static char * strip_white(char *);
static char * next_parm(char **);
static int add_filmpic(film *,const char *,struct ihash **,struct ihash **,struct ihash **);
static film * film_load(const char *,int *);
static void film_free(film *);


/* +++ internal functions +++ */

static char * strip_white(char * buf) {
/* strip comment and leading and following whitespaces */
  char * p1,* p2;
  /* strip comment */
  if ((p1=strchr(buf,'#'))!=NULL) {*p1='\0';}
  /* strip leading and following whitespaces */
  p1=buf+strspn(buf," \f\t\v\r\n");
  if (*p1=='\0') {return(p1);}
  for (p2=p1+strlen(p1)-1;strchr(" \f\t\v\r\n",(int)*p2)!=NULL;p2--) {
    if (p2<p1) {break;}
  }
  p2[1]='\0';
  return(p1);
} /* Ende strip_white */


static char * next_parm(char ** panf) {
/* find next parameter for film-picture */
  static char rbf[1024]="";
  char * pend,* p1;
  size_t s1;
  int ebe;
  if ((panf==NULL) || (*panf==NULL) || (**panf=='\0')) {return(NULL);}
  for (ebe=0,pend=*panf+1;*pend!='\0';pend++) {
    if (**panf!=')') {
      if (*pend=='(') {ebe++;}
      if (ebe==0) {
        if ((*pend==',') || (*pend==')')) {break;}
      } else if (*pend==')') {ebe--;}
    }
  }
  if ((s1=pend-(*panf+1))>=sizeof(rbf)) {fprintf(stderr,"next_parm: argument too long (%lu)\n",(unsigned long)s1); return(NULL);}
  strncpy(rbf,*panf+1,s1); rbf[s1]='\0';
  p1=strip_white(rbf);
  if ((*p1=='\0') && (**panf==')')) {return(NULL);}
  *panf=pend;
  return(p1);
} /* Ende next_parm */


static int add_filmpic(film * flm,const char * filename,struct ihash ** h_bitm,struct ihash ** h_spt,struct ihash ** h_snd) {
/* fill in film-picture */
  char buf[4096],* p1,* p2;
  FILE * ffp;
  int i1,i2;
  if ((flm==NULL) || (filename==NULL) || (h_bitm==NULL) || (h_spt==NULL) || (h_snd==NULL)) {
    fprintf(stderr,"add_filmpic: invalid argument (NULL)\n");
    return(-1);
  }
  /* read film-picture-file */
  if (*filename!='/') {
    snprintf(buf,sizeof(buf),"%s/%s",_cwdir2,filename);
  } else {snprintf(buf,sizeof(buf),"%s",filename);}
  if ((ffp=fopen(buf,"r"))==NULL) {fprintf(stderr,"add_filmpic: cannot open file \"%s\".\n",filename); return(-1);}
  memset(&flm->elem[flm->anz_elem-1].fmp,0,sizeof(struct filmpic));

  while (fgets(buf,sizeof(buf),ffp)!=NULL) {
    p1=strip_white(buf);
    if (*p1=='\0') {continue;}
    if ((p2=strchr(p1,'('))==NULL) {continue;}
    /* base() */
    if (strncmp(p1,"base",4)==0) {
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (base).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.basex=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (base).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.basey=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (base).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* clear() */
    if (strncmp(p1,"clear",5)==0) {
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (clear).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.clearbg=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (clear).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* loop() */
    if (strncmp(p1,"loop",4)==0) {
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (loop).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.loop=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (loop).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* colormap() */
    if (strncmp(p1,"colormap",8)==0) {
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (colormap).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.colmap=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (colormap).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* brightness() */
    if (strncmp(p1,"brightness",10)==0) {
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (brightness).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bright=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (brightness).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* bitmap() */
    if (strncmp(p1,"bitmap",6)==0) {
      i1=++flm->elem[flm->anz_elem-1].fmp.anz_bitm;
      if ((flm->elem[flm->anz_elem-1].fmp.bitm=(i1==1?malloc(sizeof(*flm->elem[0].fmp.bitm)):realloc(flm->elem[flm->anz_elem-1].fmp.bitm,sizeof(*flm->elem[0].fmp.bitm)*i1)))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (bitmap).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].x=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (bitmap).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].y=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (bitmap).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].trans=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].next=NULL;
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (bitmap).\n",filename); fclose(ffp); return(-1);}
      if (inthash_get(h_bitm,p1,&i2)<0) {
        if (inthash_get(h_bitm,"+",&i2)<0) {i2=0;}
        i2++;
        if (inthash_add(h_bitm,"+",i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (bitmap).\n",filename); fclose(ffp); return(-1);}
        if ((flm->bitm_idx=(i2==1?malloc(sizeof(bitmap *)*(i2+1)):realloc(flm->bitm_idx,sizeof(bitmap *)*(i2+1))))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
        if ((flm->bitm_idx[i2-1]=vg_bitmap_createfromfile(p1))==NULL) {fprintf(stderr,"add_filmpic: %s: error calling vg_bitmap_createfromfile(%s) (bitmap).\n",filename,p1); fclose(ffp); return(-1);}
        flm->bitm_idx[i2]=NULL;
        if (inthash_add(h_bitm,p1,i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (bitmap).\n",filename); fclose(ffp); return(-1);}
      }
      flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].idx=i2;
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (bitmap).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* sprite() */
    if (strncmp(p1,"sprite",6)==0) {
      i1=++flm->elem[flm->anz_elem-1].fmp.anz_bitm;
      if ((flm->elem[flm->anz_elem-1].fmp.bitm=(i1==1?malloc(sizeof(*flm->elem[0].fmp.bitm)):realloc(flm->elem[flm->anz_elem-1].fmp.bitm,sizeof(*flm->elem[0].fmp.bitm)*i1)))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].x=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].y=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].trans=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].next=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      if (inthash_get(h_spt,p1,&i2)<0) {
        if (inthash_get(h_spt,"+",&i2)<0) {i2=0;}
        i2++;
        if (inthash_add(h_spt,"+",i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (sprite).\n",filename); fclose(ffp); return(-1);}
        if ((flm->spt_idx=(i2==1?malloc(sizeof(sprite *)*(i2+1)):realloc(flm->spt_idx,sizeof(sprite *)*(i2+1))))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
        if ((flm->spt_idx[i2-1]=vg_sprite_createfromfile(p1))==NULL) {fprintf(stderr,"add_filmpic: %s: error calling vg_sprite_createfromfile(%s) (sprite).\n",filename,p1); fclose(ffp); return(-1);}
        flm->spt_idx[i2]=NULL;
        if (inthash_add(h_spt,p1,i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (sprite).\n",filename); fclose(ffp); return(-1);}
      }
      flm->elem[flm->anz_elem-1].fmp.bitm[i1-1].idx=i2;
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (sprite).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* soundstart() */
    if (strncmp(p1,"soundstart",10)==0) {
      int volpc;
      i1=++flm->elem[flm->anz_elem-1].fmp.anz_snd;
      if ((flm->elem[flm->anz_elem-1].fmp.snd=(i1==1?malloc(sizeof(*flm->elem[0].fmp.snd)):realloc(flm->elem[flm->anz_elem-1].fmp.snd,sizeof(*flm->elem[0].fmp.snd)*i1)))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstart).\n",filename); fclose(ffp); return(-1);}
      if ((flm->elem[flm->anz_elem-1].fmp.snd[i1-1].obloop=strdup(p1))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstart).\n",filename); fclose(ffp); return(-1);}
      volpc=atoi(p1);
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstart).\n",filename); fclose(ffp); return(-1);}
      if (inthash_get(h_snd,p1,&i2)<0) {
        char hdb[16],fsnd[1024];
        if (inthash_get(h_snd,"+",&i2)<0) {i2=0;}
        i2++;
        if (inthash_add(h_snd,"+",i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (soundstart).\n",filename); fclose(ffp); return(-1);}
        if ((flm->snd_idx=(i2==1?malloc(sizeof(char *)*(i2+1)):realloc(flm->snd_idx,sizeof(char *)*(i2+1))))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
        snprintf(hdb,sizeof(hdb),"h-b%d",i2);
        if ((flm->snd_idx[i2-1]=strdup(hdb))==NULL) {fprintf(stderr,"add_filmpic: strdup: %s\n",strerror(errno)); fclose(ffp); return(-1);}
        flm->snd_idx[i2]=NULL;
        if (inthash_add(h_snd,p1,i2)<0) {fprintf(stderr,"add_filmpic: %s: error calling inthash_add (soundstart).\n",filename); fclose(ffp); return(-1);}
        if (*p1!='/') {
          snprintf(fsnd,sizeof(fsnd),"%s/%s",_cwdir2,p1);
          p1=fsnd;
        }
        vg_sound_attach_hidden(p1,hdb,volpc);
      }
      flm->elem[flm->anz_elem-1].fmp.snd[i1-1].idx=i2;
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstart).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
    /* soundstop() */
    if (strncmp(p1,"soundstop",9)==0) {
      i1=++flm->elem[flm->anz_elem-1].fmp.anz_snd;
      if ((flm->elem[flm->anz_elem-1].fmp.snd=(i1==1?malloc(sizeof(*flm->elem[0].fmp.snd)):realloc(flm->elem[flm->anz_elem-1].fmp.snd,sizeof(*flm->elem[0].fmp.snd)*i1)))==NULL) {fprintf(stderr,"add_filmpic: malloc/realloc: %s: %s.\n",filename,strerror(errno)); fclose(ffp); return(-1);}
      flm->elem[flm->anz_elem-1].fmp.snd[i1-1].obloop=NULL;
      if ((p1=next_parm(&p2))==NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstop).\n",filename); fclose(ffp); return(-1);}
      if (inthash_get(h_snd,p1,&i2)<0) {fprintf(stderr,"add_filmpic: %s: sound \"%s\" not running (soundstop).\n",filename,p1); fclose(ffp); return(-1);}
      flm->elem[flm->anz_elem-1].fmp.snd[i1-1].idx=i2;
      inthash_del(h_snd,p1);
      if (next_parm(&p2)!=NULL) {fprintf(stderr,"add_filmpic: %s: line corrupt (soundstop).\n",filename); fclose(ffp); return(-1);}
      continue;
    }
  }
  fclose(ffp);
  return(0);
} /* Ende add_filmpic */


static film * film_load(const char * filename,int * msec) {
/* load a film from file
** 1.arg: filename of film with relative path to game-directory
** return: film or NULL=error
*/
  film * flm;
  struct ihash ** h_bitm,** h_spt,** h_snd;
  char buf[4096],* p1,* p2;
  FILE * ffp;
  int i1;

  strlcpy(cwkop,_cwdir2,sizeof(cwkop));
  flm=NULL;
  h_bitm=h_spt=h_snd=NULL;
  /* read film-file */
  if (filename==NULL || msec==NULL) {fprintf(stderr,"film_load: invalid argument (NULL).\n"); return(NULL);}
  if (*filename!='/') {
    snprintf(buf,sizeof(buf),"%s/%s",*_cwdir2=='\0'?CWDIR:_cwdir2,filename);
  } else {snprintf(buf,sizeof(buf),"%s",filename);}
  strlcpy(_cwdir2,buf,sizeof(_cwdir2));
  if ((p1=strrchr(_cwdir2,'/'))!=NULL) {*p1='\0';} else {strlcpy(_cwdir2,".",sizeof(_cwdir2));}
  if ((ffp=fopen(buf,"r"))==NULL) {fprintf(stderr,"film_load: cannot open file \"%s\".\n",filename); goto _flld_bad;}
  if (fgets(buf,sizeof(buf),ffp)==NULL) {fprintf(stderr,"film_load: empty file \"%s\".\n",filename); goto _flld_bad;}
  if (strncmp(buf,"[FILM ",6)!=0) {fprintf(stderr,"film_load: file seems not to be a film file: \"%s\".\n",filename); goto _flld_bad;}
  if ((*msec=atoi(buf+6))<=0) {*msec=50;}

  if ((h_bitm=inthash_new(127))==NULL) {goto _flld_bad;}
  if ((h_spt=inthash_new(127))==NULL) {goto _flld_bad;}
  if ((h_snd=inthash_new(31))==NULL) {goto _flld_bad;}
  if ((flm=calloc(1,sizeof(film)))==NULL) {fprintf(stderr,"film_load: calloc: %s\n",strerror(errno)); goto _flld_bad;}

  while (fgets(buf,sizeof(buf),ffp)!=NULL) {
    p1=strip_white(buf);
    if (*p1=='\0') {continue;}
    flm->anz_elem++;
    flm->elem=(flm->anz_elem==1?malloc(sizeof(*flm->elem)):realloc(flm->elem,sizeof(*flm->elem)*flm->anz_elem));
    if (flm->elem==NULL) {fprintf(stderr,"film_load: malloc/realloc: %s\n",strerror(errno)); goto _flld_bad;}
    /* read loop of picture-file */
    if ((p2=strpbrk(p1," \t"))==NULL) {fprintf(stderr,"film_load: line corrupt: \"%s\"\n",p1); goto _flld_bad;}
    *p2++='\0';
    if ((flm->elem[flm->anz_elem-1].loop=atoi(p1))<=0) {flm->anz_elem--; continue;}
    p2+=strspn(p2," \t");
    /* read filename of picture-file */
    p1=p2;
    if ((p2=strpbrk(p1," \t"))!=NULL) {*p2++='\0';}
    if (add_filmpic(flm,p1,h_bitm,h_spt,h_snd)<0) {goto _flld_bad;}
    memset(flm->elem[flm->anz_elem-1].vars,0,sizeof(flm->elem[0].vars));
    if (p2==NULL) {continue;}
    p2+=strspn(p2," \t");
    /* read arguments of picture-file */
    for (i1=1;i1<=9;i1++) {
      p1=p2;
      if ((p2=strpbrk(p1," \t"))!=NULL) {*p2++='\0';}
      if ((flm->elem[flm->anz_elem-1].vars[i1-1]=strdup(p1))==NULL) {fprintf(stderr,"film_load: strdup: %s\n",strerror(errno)); goto _flld_bad;}
      if (p2==NULL) {i1++; break;}
      p2+=strspn(p2," \t");
    }
    flm->elem[flm->anz_elem-1].vars[i1-1]=NULL;
  }
  fclose(ffp);
  inthash_free(h_bitm);
  inthash_free(h_spt);
  inthash_free(h_snd);
  return(flm);

_flld_bad:
  if (ffp!=NULL) {fclose(ffp);}
  if (flm!=NULL) {free(flm);}
  if (h_bitm!=NULL) {inthash_free(h_bitm);}
  if (h_spt!=NULL) {inthash_free(h_spt);}
  if (h_snd!=NULL) {inthash_free(h_snd);}
  return(NULL);
} /* Ende film_load */


static void film_free(film * flm) {
/* free a film */
  if (flm==NULL) {return;}
  if (flm->bitm_idx!=NULL) {
    bitmap ** pbmp;
    for (pbmp=flm->bitm_idx;*pbmp!=NULL;pbmp++) {
      vg_bitmap_free(*pbmp);
    }
    free(flm->bitm_idx);
  }
  if (flm->spt_idx!=NULL) {
    sprite ** pspt;
    for (pspt=flm->spt_idx;*pspt!=NULL;pspt++) {
      vg_sprite_free(*pspt);
    }
    free(flm->spt_idx);
  }
  if (flm->snd_idx!=NULL) {
    char ** psnd;
    for (psnd=flm->snd_idx;*psnd!=NULL;psnd++) {
      if (**psnd!='\0') {vg_sound_deta_hidden(*psnd);}
      free(*psnd);
    }
    free(flm->snd_idx);
  }
  if (flm->anz_elem>0) {
    struct filmpic * fmp;
    int aex,idx;
    for (aex=0;aex<flm->anz_elem;aex++) {
      for (idx=0;idx<10;idx++) {
        if (flm->elem[aex].vars[idx]==NULL) {break;}
        free(flm->elem[aex].vars[idx]);
      }
      fmp=&flm->elem[aex].fmp;
      if (fmp->basex!=NULL) {free(fmp->basex);}
      if (fmp->basey!=NULL) {free(fmp->basey);}
      if (fmp->loop!=NULL) {free(fmp->loop);}
      if (fmp->colmap!=NULL) {free(fmp->colmap);}
      if (fmp->bright!=NULL) {free(fmp->bright);}
      if (fmp->clearbg!=NULL) {free(fmp->clearbg);}
      for (idx=0;idx<fmp->anz_bitm;idx++) {
        if (fmp->bitm[idx].x!=NULL) {free(fmp->bitm[idx].x);}
        if (fmp->bitm[idx].y!=NULL) {free(fmp->bitm[idx].y);}
        if (fmp->bitm[idx].trans!=NULL) {free(fmp->bitm[idx].trans);}
        if (fmp->bitm[idx].next!=NULL) {free(fmp->bitm[idx].next);}
      }
      for (idx=0;idx<fmp->anz_snd;idx++) {
        if (fmp->snd[idx].obloop!=NULL) {free(fmp->snd[idx].obloop);}
      }
    }
    free(flm->elem);
  }
  free(flm);
} /* Ende film_free */


/* +++ functions +++ */

int vg_film_play(const char * filename,int (*funk)(int)) {
/* play a film
** 1.arg: filename of film with relative path to game-directory
** 2.arg: a function, which is called each loop,
**        its parameter is the <milliseconds for each loop> from "[FILM ?]"
**        its return value must be: 0=ok or 1=stop film,
**        its use is to call vg_window_flush(), vg_wait_time()
**        and to process keystrokes.
**        The 2.arg may be NULL, which is the same as:
**          int funk(int milliseconds) {
**            vg_window_flush();
**            vg_wait_time(milliseconds);
**            return(0);
**          }
** return value: 0=OK or -1=error
**
** film-file format:
**   "[FILM " <milliseconds for each loop> "]"
**   [#]<number of repeats> <picture-file> [<1. argument> [... <9. argument>]]
**   ...
**   [#]<number of repeats> <picture-file> [<1. argument> [... <9. argument>]]
**
** picture-file format:
**   # arguments may be used with $1 to $9
**   # <number of repeats> is $0
**
**   # +++ once: +++
**
**   # base coordinates at x/y
**   base(<x>,<y>)
**
**   # how many loops for this picture-file
**   loop(<number of loops>)
**
**   # load colormap
**   colormap() <colormap-file>
**
**   # brightness from -62 to 63
**   brightness(<number>)
**
**   # clear screen with background color number
**   clear(<background color number>)
**
**   # +++ once or more: +++
**
**   # show bitmap at x/y, full or transparent
**   # <bitmap-file> is relative to picture-file, not to game-directory
**   bitmap(<x>,<y>,<full|trans>) <bitmap-file>
**
**   # show sprite-bitmap at x/y, full or transparent
**   # with vg_sprite_getcurrent() or vg_sprite_getnext()
**   # <sprite-file> is relative to picture-file, not to game-directory
**   sprite(<x>,<y>,<full|trans>,<current|next>) <sprite-file>
**
**   # start/stop sound
**   # play it once or in an endless loop
**   # <sound-file> is relative to picture-file, not to game-directory
**   soundstart(<once|loop>,<volumepercent 0-200>) <sound-file>
**   # stop sound
**   soundstop() <sound-file>
*/
  film * flm;
  struct filmpic * fmp;
  int msec,lopmax,idx;
  int bs_x,bs_y,b_x,b_y,b_trans,b_ob;
  long l1;
  char vclm[128],* ptr;
  int vbr;
  bitmap * bmp;
  sprite * spt;
  *vclm='\0'; vbr=99;
  if ((flm=film_load(filename,&msec))==NULL) {fprintf(stderr,"vg_film_play: error calling film_load()\n"); return(-1);}
  for (aex=0;aex<flm->anz_elem;aex++) {
    for (lex=1;lex<=flm->elem[aex].loop;lex++) {
      setze_vars(lex,lex==1?flm->elem[aex].vars:NULL);
      fmp=&flm->elem[aex].fmp;
      /* base x/y */
      if ((fmp->basex!=NULL) && (*fmp->basex!='\0')) {
        if (rechne_formel(fmp->basex,&l1)<0) {goto p_bad;}
        bs_x=(int)l1;
      } else {bs_x=0;}
      if ((fmp->basey!=NULL) && (*fmp->basey!='\0')) {
        if (rechne_formel(fmp->basey,&l1)<0) {goto p_bad;}
        bs_y=(int)l1;
      } else {bs_y=0;}
      /* loop */
      if ((fmp->loop!=NULL) && (*fmp->loop!='\0')) {
        if (rechne_formel(fmp->loop,&l1)<0) {goto p_bad;}
        lopmax=(int)l1;
      } else {lopmax=1;}
      /* colormap */
      if ((fmp->colmap!=NULL) && (*fmp->colmap!='\0')) {
        if ((ptr=ermittle_string(fmp->colmap,NULL))==NULL) {goto p_bad;}
        if (vg_load_colormap(ptr,vclm,*vclm!='\0'?0:sizeof(vclm))<0) {goto p_bad;}
      }
      /* brightness */
      if ((fmp->bright!=NULL) && (*fmp->bright!='\0')) {
        if (rechne_formel(fmp->bright,&l1)<0) {goto p_bad;}
        if (vbr!=99) {vg_brightness((int)l1);} else {vbr=vg_brightness((int)l1);}
      }
      /* for each loop */
      for (lop=1;lop<=lopmax;lop++) {
        /* clear */
        if ((fmp->clearbg!=NULL) && (*fmp->clearbg!='\0')) {
          if (rechne_formel(fmp->clearbg,&l1)<0) {goto p_bad;}
          vg_bitmap_clear(NULL,(int)l1);
        }
        /* bitmaps and sprites */
        for (idx=0;idx<fmp->anz_bitm;idx++) {
          /* x */
          if ((fmp->bitm[idx].x!=NULL) && (*fmp->bitm[idx].x!='\0')) {
            if (rechne_formel(fmp->bitm[idx].x,&l1)<0) {goto p_bad;}
            b_x=(int)l1;
          } else {b_x=0;}
          /* y */
          if ((fmp->bitm[idx].y!=NULL) && (*fmp->bitm[idx].y!='\0')) {
            if (rechne_formel(fmp->bitm[idx].y,&l1)<0) {goto p_bad;}
            b_y=(int)l1;
          } else {b_y=0;}
          /* full|trans */
          if ((fmp->bitm[idx].trans!=NULL) && (*fmp->bitm[idx].trans!='\0')) {
            if ((ptr=ermittle_string(fmp->bitm[idx].trans,NULL))==NULL) {goto p_bad;}
            b_trans=(strcmp(ptr,"trans")==0?RGB_TRANS:RGB_FULL);
          } else {b_trans=RGB_FULL;}
          /* current|next|<empty> */
          if ((fmp->bitm[idx].next!=NULL) && (*fmp->bitm[idx].next!='\0')) {
            spt=flm->spt_idx[fmp->bitm[idx].idx-1];
            if ((ptr=ermittle_string(fmp->bitm[idx].next,NULL))==NULL) {goto p_bad;}
            if (strcmp(ptr,"current")==0) {
              bmp=vg_sprite_getcurrent(spt);
            } else if ((bmp=vg_sprite_getnext(spt))==NULL) {bmp=vg_sprite_getnext(spt);}
          } else {
            bmp=flm->bitm_idx[fmp->bitm[idx].idx-1];
          }
          if (bmp!=NULL) {vg_bitmap_copyto(NULL,bs_x+b_x,bs_y+b_y,bmp,0,0,0,0,b_trans);}
        }
        /* sound */
        if (lop==1) {
          char * sptr;
          for (idx=0;idx<fmp->anz_snd;idx++) {
            sptr=flm->snd_idx[fmp->snd[idx].idx-1];
            /* once|loop|<empty> */
            if ((fmp->snd[idx].obloop!=NULL) && (*fmp->snd[idx].obloop!='\0')) {
              if ((ptr=ermittle_string(fmp->snd[idx].obloop,NULL))==NULL) {goto p_bad;}
              b_ob=(strcmp(ptr,"loop")==0?0:1);
              (void)vg_sound_play_hidden(sptr,0,b_ob);
            } else {
              vg_sound_deta_hidden(sptr);
              *sptr='\0';
            }
          }
        }
        /* call wait-function */
        if (funk!=NULL) {
          if (funk(msec)!=0) {goto p_stop;}
        } else {vg_window_flush(); vg_wait_time(msec);}
      }
    }
  }
p_stop:
  film_free(flm);
  if (*vclm!='\0') {(void)vg_load_colormap(vclm,NULL,0);}
  if (vbr!=99) {vg_brightness(vbr);}
  vg_bitmap_clear(NULL,RGB_BLACK);
  vg_window_flush();
  aex=lex=lop=-1;
  strlcpy(_cwdir2,cwkop,sizeof(_cwdir2));
  return(0);
p_bad:
  film_free(flm);
  if (*vclm!='\0') {(void)vg_load_colormap(vclm,NULL,0);}
  if (vbr!=99) {vg_brightness(vbr);}
  vg_bitmap_clear(NULL,RGB_BLACK);
  vg_window_flush();
  fprintf(stderr,"vg_film_play: error playing \"%s\"\n",filename);
  aex=lex=lop=-1;
  strlcpy(_cwdir2,cwkop,sizeof(_cwdir2));
  return(-1);
} /* Ende vg_film_play */


void vg_film_position(int * laex,int * llex,int * llop) {
/* get current position of a playing film,
** to call from an each-filmloop-function
** 1.arg: address for returning line of film-file
** 2.arg: address for returning current number-of-repeats
** 3.arg: address for returning current loop-number of picture-file
*/
  if (laex!=NULL) {*laex=aex+1;}
  if (llex!=NULL) {*llex=lex;}
  if (llop!=NULL) {*llop=lop;}
} /* Ende vg_film_position */
