/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* font functions */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include "config.h"
#include "font.h"
#include "bitmap.h"
#include "inthash.h"

static int maxwh_font=64;  /* largest font size in pixels (row and col) */

static unsigned char * load_font(const char *,int *,int *);

void draw_fontXxY(bitmap *,int,int,int,const char *,int,const char *,int,int);
int vg_font_width(const char *);
int vg_font_height(const char *);


static unsigned char * load_font(const char * fontname,int * w,int * h) {
/* load font file
** 1.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** 2.+3.arg: address for width and height of font
** return: font data or NULL=error
*/
  static struct ihash ** h_font=NULL;
  static struct {
    int w,h;
    unsigned char * data;
  } * f_font=NULL;
  static int f_size=0;
  int i1;
  if ((h_font==NULL) && ((h_font=inthash_new(127))==NULL)) {fprintf(stderr,"load_font: error calling inthash_new.\n"); return(NULL);}
  if (fontname==NULL) {fontname="8x8.font";}
  if (inthash_get(h_font,fontname,&i1)<0) {  /* load font file */
    char buf[512],endg[16];
    const char * kptr;
    FILE * ffp;
    int w1,h1,whges,i2;
    unsigned int iw;
    w1=h1=0;
    if ((kptr=strchr(fontname,'x'))!=NULL) {
      w1=atoi(fontname);
      h1=atoi(kptr+1);
    }
    if ((w1<=0) || (h1<=0)) {
      fprintf(stderr,"load_font: invalid font filename \"%s\".\n",fontname);
      return(NULL);
    }
    if ((w1>maxwh_font) || (h1>maxwh_font)) {
      fprintf(stderr,"load_font: font too big (> %dx%d): \"%s\".\n",maxwh_font,maxwh_font,fontname);
      return(NULL);
    }
    i1=strlen(fontname);
    if ((i1>5) && (strcmp(fontname+i1-5,".font")==0)) {*endg='\0';} else {snprintf(endg,sizeof(endg),".font");}
    snprintf(buf,sizeof(buf),"%s/share/%s%s",CWDIR,fontname,endg);
    if ((ffp=fopen(buf,"r"))==NULL) {
      snprintf(buf,sizeof(buf),"%s/%s%s",SHAREDIR,fontname,endg);
      ffp=fopen(buf,"r");
    }
    if (ffp==NULL) {
      fprintf(stderr,"load_font: can't open font file: \"%s%s\".\n",fontname,endg);
      return(NULL);
    }
    if (f_size==0) {
      f_font=malloc(sizeof(*f_font));
    } else {
      f_font=realloc(f_font,sizeof(*f_font)*(f_size+1));
    }
    if (f_font==NULL) {fprintf(stderr,"load_font: malloc/realloc: %s.\n",strerror(errno)); fclose(ffp); return(NULL);}
    whges=((w1+7)/8)*h1*256;
    if ((f_font[f_size].data=calloc(whges,sizeof(unsigned char)))==NULL) {fprintf(stderr,"load_font: calloc: %s.\n",strerror(errno)); fclose(ffp); return(NULL);}
    f_font[f_size].w=w1; f_font[f_size].h=h1;
    f_size++;
    if (inthash_add(h_font,fontname,f_size)<0) {fprintf(stderr,"load_font: error calling inthash_add.\n"); fclose(ffp); return(NULL);}
    i2=0;
    while (1) {
      i1=fscanf(ffp,"%x ,",&iw);
      if (i1==EOF) {break;}
      if (i1<1) {
        fgets(buf,sizeof(buf),ffp);
      } else {
        if (++i2>whges) {break;}
        f_font[f_size-1].data[i2-1]=(unsigned char)iw;
      }
    }
    fclose(ffp);
    i1=f_size;
  }
  if (w!=NULL) {*w=f_font[i1-1].w;}
  if (h!=NULL) {*h=f_font[i1-1].h;}
  return(f_font[i1-1].data);
} /* Ende load_font */


void draw_fontXxY(bitmap * grf,int fgc,int x,int y,const char * text,int tlen,const char * fontXxY,int trsp,int bgc) {
/* draw text string with selected font
** 1.arg: bitmap to draw text into or NULL=backbuffer
** 2.arg: text color
** 3.+4.arg: x/y (upper left) position to draw text
** 5.arg: text string
** 6.arg: string length of 5.arg
** 7.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** 8.arg: RGB_FULL or RGB_TRANS
** 9.arg: background color
*/
  static bitmap * zeich=NULL;
  unsigned char * font,* fontptr;
  int bt,i1,i2,i3,rv;
  int w,h;
  if (zeich==NULL) {
    if ((zeich=vg_bitmap_createnew(maxwh_font,maxwh_font))==NULL) {return;}
  }
  if ((fontptr=load_font(fontXxY,&w,&h))==NULL) {fprintf(stderr,"draw_fontXxY: error calling load_font.\n"); return;}
  if (h>maxwh_font) {h=maxwh_font;}
  if (w>maxwh_font) {w=maxwh_font;}
  rv=(w-1)/8+1;  /* bytes per row */
  for (i1=0;i1<tlen;i1++) {
    bt=0;
    font=fontptr;
    font+=((unsigned char)text[i1]*h*rv);
    for (i2=0;i2<h;i2++) {
      for (i3=0;i3<w;i3++) {
        if (i3%8==0) {bt=*font++;}
        if (bt&(1<<(i3%8))) {
          zeich->pxm[i2*zeich->width+i3]=fgc;
        } else {
          zeich->pxm[i2*zeich->width+i3]=bgc;
        }
      }
    }
    vg_bitmap_copyto(grf,x+i1*w+w/2,y+h/2,zeich,0,0,w,h,trsp);
  }
} /* Ende draw_fontXxY */


int vg_font_width(const char * fontXxY) {
/* return font width of 1.arg
** 1.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** return: its width
*/
  int w;
  if (load_font(fontXxY,&w,NULL)==NULL) {fprintf(stderr,"vg_font_width: error calling load_font.\n"); return(0);}
  return(w);
} /* Ende vg_font_width */


int vg_font_height(const char * fontXxY) {
/* return font height of 1.arg
** 1.arg: font filename to load from <CWDIR>/share/ or <SHAREDIR>/
**        (no path and must look like "<number>x<number>[_<text>].font",
**         e.g. 8x8_default.font or 16x10.font)
**        or NULL=default font: 8x8.font
** return: its height
*/
  int h;
  if (load_font(fontXxY,NULL,&h)==NULL) {fprintf(stderr,"vg_font_height: error calling load_font.\n"); return(0);}
  return(h);
} /* Ende vg_font_height */
