/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* formel.c: Formelberechnung */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>


static int maxtiefe=4;  /* hoechster Wert "tiefe" in next_opform() */

void setze_vars(int,char **);
int rechne_formel(const char *,long *);
char * ermittle_string(const char *,const char *);

static char ** vars=NULL,loop[32]="";

static char * expandvar(const char *,const char *);
static int einer_ops(long *,const char *,int);
static int berechne_wertform(const char *,const char *,long *);
static int verbinde_werteform(long *,long,const char *,int);
static const char * next_opform(const char *,const char *,int,int *);
static int rechne_formel_rek(const char *,const char *,int,long *);
static char * gib_string(const char *,const char *);


/* +++ statische Funktionen +++ */

static char * expandvar(const char * panf,const char * pend) {
  static char retb[4]="";
  *retb='\0';
  if (panf>pend) {return(retb);}
  if (((unsigned char)*panf<'0') || ((unsigned char)*panf>'9')) {retb[0]='$'; retb[1]=*panf; retb[2]='\0'; return(retb);}
  if (*panf=='0') {return(loop);}
  return(vars[*panf-'1']!=NULL?vars[*panf-'1']:"");
} /* Ende expandvar */


static int einer_ops(long * dwert,const char * ops,int iops) {
/* wendet uebergebene Einer-Operatoren auf 1.Arg an */
  if ((dwert==NULL) || (ops==NULL)) {fprintf(stderr,"einer_ops: parameter=NULL"); return(-1);}

  for (;iops>0;iops--) {
    switch(ops[iops-1]) {
      case '-': *dwert=-*dwert; break;
      case '+': break;
      case '!': *dwert=!(*dwert); break;
      case '~': *dwert=~(long)*dwert; break;
      default: fprintf(stderr,"einer_ops: unknown operator: \"%c\"",ops[iops-1]); return(-1);
    }
  }
  return(0);
} /* Ende einer_ops */


static int berechne_wertform(const char * panf,const char * pend,long * dwert) {
/* berechnet Wert, nun ohne Operatoren in dieser Ebene */
  const char * porg;
  char ops[128];
  int iops;
  if ((panf==NULL) || (pend==NULL) || (dwert==NULL)) {fprintf(stderr,"berechne_wertform: parameter=NULL"); return(-1);}

  iops=0;
  pend--;  /* Ende inklusive machen */
  for (;panf<=pend;panf++) {
    if (isspace((int)*panf)) {continue;}
    if (strchr("-+!~",(int)*panf)!=NULL) {  /* Einer-Operator */
      ops[iops]=*panf;
      if (++iops>(int)sizeof(ops)) {fprintf(stderr,"berechne_wertform: iops overflow in \"%s\"",panf); return(-1);}
      continue;
    }
    break;
  }
  if (panf>pend) {fprintf(stderr,"berechne_wertform: no valid value before \"%s\"",panf); return(-1);}
  porg=panf;
  if (*panf=='(') {  /* tiefere Ebene */
    for (;pend>panf;pend--) {
      if (!isspace((int)*pend)) {break;}
    }
    if ((pend==panf) || (*pend!=')')) {fprintf(stderr,"berechne_wertform: missing ')' in \"%.*s\"",(int)(pend-porg+1),porg); return(-1);}
    if (rechne_formel_rek(panf+1,pend,1,dwert)<0) {return(-1);}
    if (einer_ops(dwert,ops,iops)<0) {return(-1);}
    return(0);
  }
  if ((*panf=='$') && (panf<pend)) {  /* Variable */
    char * varptr=expandvar(panf+1,pend);
    *dwert=atol(varptr);
    if (einer_ops(dwert,ops,iops)<0) {return(-1);}
    return(0);
  }
  if ((isdigit((int)*panf)) || (*panf=='.')) {  /* Zahl */
    for (;panf<=pend;panf++) {
      if ((!isdigit((int)*panf)) && (*panf!='.')) {break;}
    }
    for (;panf<=pend;panf++) {
      if (!isspace((int)*panf)) {break;}
    }
    if (panf<=pend) {fprintf(stderr,"berechne_wertform: invalid \"%c\" in \"%.*s\"",*panf,(int)(pend-porg+1),porg); return(-1);}
    *dwert=atol(porg);
    if (einer_ops(dwert,ops,iops)<0) {return(-1);}
    return(0);
  }
  fprintf(stderr,"berechne_wertform: invalid \"%c\" in \"%.*s\"",*panf,(int)(pend-porg+1),porg);
  return(-1);
} /* Ende berechne_wertform */


static int verbinde_werteform(long * gesamt,long wert,const char * operator,int oplen) {
/* fuegt 2.Arg an 1.Arg an gemaess Operator */
  if (gesamt==NULL) {fprintf(stderr,"verbinde_werteform: gesamt=NULL"); return(-1);}

  if ((operator==NULL) || (oplen<1)) {*gesamt=wert; return(0);}

  /* gemaess Operator verbinden, Operatoren wie in next_opform() */
  if (oplen==1) {  /* Operatoren der Laenge 1 */
    if (*operator=='+') {         /* Strich: + */
      *gesamt=(*gesamt+wert);
    } else if (*operator=='-') {  /* Strich: - */
      *gesamt=(*gesamt-wert);
    } else if (*operator=='*') {  /* Punkt: * */
      *gesamt=(*gesamt*wert);
    } else if (*operator=='/') {  /* Punkt: / */
      *gesamt=(*gesamt/wert);
    } else if (*operator=='%') {  /* Punkt: % */
      *gesamt=(*gesamt%wert);
    } else if (*operator=='&') {  /* bitweise: & */
      *gesamt=(*gesamt&wert);
    } else if (*operator=='|') {  /* bitweise: | */
      *gesamt=(*gesamt|wert);
    } else if (*operator=='^') {  /* bitweise: ^ */
      *gesamt=(*gesamt^wert);
    } else {fprintf(stderr,"verbinde_werteform: unknown operator \"%c\"",*operator); return(-1);}
  } else if (oplen==2) {  /* Operatoren der Laenge 2 */
    if (strncmp(operator,"<<",oplen)==0) {  /* shift: << */
      *gesamt=(*gesamt<<wert);
    } else if (strncmp(operator,">>",oplen)==0) {  /* shift: >> */
      *gesamt=(*gesamt>>wert);
    } else {fprintf(stderr,"verbinde_werteform: unknown operator \"%c%c\"",operator[0],operator[1]); return(-1);}
  } else {fprintf(stderr,"verbinde_werteform: unknown operator length %d",oplen); return(-1);}
  return(0);
} /* Ende verbinde_werteform */


static const char * next_opform(const char * panf,const char * pend,int tiefe,int * oplen) {
/* gibt Pointer auf naechsten Operator zurueck gemaess tiefe oder pend=Ende oder NULL=Fehler */
  const char * p1;
  int klambene=0;
  if (oplen!=NULL) {*oplen=0;}
  if ((panf==NULL) || (pend==NULL)) {return(pend);}

  if (tiefe==1) {  /* bitweise: & | ^ */
    if (oplen!=NULL) {*oplen=1;}
    for (p1=panf;p1<pend;p1++) {
      if (*p1=='$') {p1++; continue;}
      if (*p1=='(') {klambene++;} else if (*p1==')') {klambene--;}
      if (klambene!=0) {continue;}
      if ((*p1=='&') || (*p1=='|') || (*p1=='^')) {return(p1);}
    }
  } else if (tiefe==2) {  /* shift: << >> */
    if (oplen!=NULL) {*oplen=2;}
    for (p1=panf;p1<pend-1;p1++) {
      if (*p1=='$') {p1++; continue;}
      if (*p1=='(') {klambene++;} else if (*p1==')') {klambene--;}
      if (klambene!=0) {continue;}
      if ((*p1=='<') && (p1[1]=='<')) {return(p1);}
      if ((*p1=='>') && (p1[1]=='>')) {return(p1);}
    }
  } else if (tiefe==3) {  /* Strich: + - */
    const char * p2;
    if (oplen!=NULL) {*oplen=1;}
    for (p1=panf;p1<pend;p1++) {
      if (*p1=='$') {p1++; continue;}
      if (*p1=='(') {klambene++;} else if (*p1==')') {klambene--;}
      if (klambene!=0) {continue;}
      if ((*p1=='+') || (*p1=='-')) {
        for (p2=p1-1;p2>=panf;p2--) {  /* pruefen, ob Vorzeichen */
          if (isspace((int)*p2)) {continue;}
          if ((isdigit((int)*p2)) || (*p2=='.') || (*p2==')') \
          || (isalnum((int)*p2)) || (*p2=='_') || (*p2==']') || (*p2=='}')) {return(p1);}
          break;
        }
      }
    }
  } else if (tiefe==4) {  /* Punkt: * / % */
    if (oplen!=NULL) {*oplen=1;}
    for (p1=panf;p1<pend;p1++) {
      if (*p1=='$') {p1++; continue;}
      if (*p1=='(') {klambene++;} else if (*p1==')') {klambene--;}
      if (klambene!=0) {continue;}
      if ((*p1=='*') || (*p1=='/') || (*p1=='%')) {return(p1);}
    }
  }

  if (oplen!=NULL) {*oplen=0;}
  return(pend);
} /* Ende next_opform */


static int rechne_formel_rek(const char * panf,const char * pend,int tiefe,long * dwert) {
/* berechnet rekursiv Wert von Formel 1.Arg - 2.Arg(ausschliesslich) */
  const char * pnop1,* pnop2;
  int oplen1,oplen2;
  long zwert;
  if ((panf==NULL) || (pend==NULL) || (dwert==NULL)) {fprintf(stderr,"rechne_formel_rek: formel/dwert=NULL"); return(-1);}

  /* pruefen, ob maximale Operatortiefe erreicht ist */
  if (tiefe>maxtiefe) {
    if (berechne_wertform(panf,pend,dwert)<0) {return(-1);}
    return(0);
  }

  /* naechste Operatoren uebergebener OpTiefe suchen */
  pnop1=NULL; oplen1=0;
  do {
    if ((pnop2=next_opform(panf,pend,tiefe,&oplen2))==NULL) {return(-1);}
    if (rechne_formel_rek(panf,pnop2,tiefe+1,&zwert)<0) {return(-1);}
    if (verbinde_werteform(dwert,zwert,pnop1,oplen1)<0) {return(-1);}
    panf=pnop2+oplen2;
    pnop1=pnop2; oplen1=oplen2;
  } while (panf<pend);
  return(0);
} /* Ende rechne_formel_rek */


static char * gib_string(const char * panf,const char * pend) {
/* setzt String aus 1.-2.Arg(exkl.) in statischen String */
  static char retb[1024]="";
  int pos,i0;
  const char * manf;
  if ((panf==NULL) || (pend==NULL)) {fprintf(stderr,"gib_string: parameter=NULL"); return(NULL);}
  pos=0;
  for (manf=panf;panf<pend;panf++) {
    if (*panf=='$') {  /* Variable */
      char * varptr=expandvar(panf+1,pend);
      i0=(int)(panf-manf);
      if (pos+i0>=(int)sizeof(retb)) {fprintf(stderr,"gib_string: variable too big"); return(NULL);}
      strncpy(retb+pos,manf,i0); pos+=i0;
      i0=strlen(varptr);
      if (pos+i0>=(int)sizeof(retb)) {fprintf(stderr,"gib_string: variable too big"); return(NULL);}
      strncpy(retb+pos,varptr,i0); pos+=i0;
      manf=panf+1;
      continue;
    }
  }
  i0=(int)(panf-manf);
  if (pos+i0>=(int)sizeof(retb)) {fprintf(stderr,"gib_string: variable too big"); return(NULL);}
  strncpy(retb+pos,manf,i0); pos+=i0;
  retb[pos]='\0';
  return(retb);
} /* Ende gib_string */


/* +++ Funktionen +++ */

void setze_vars(int lp,char ** vs) {
/* set variables */
  int i1;
  snprintf(loop,sizeof(loop),"%d",lp);
  if (vs!=NULL) {
    vars=vs;
    for (i1=0;i1<9;i1++) {
      if (vars[i1]==NULL) {break;}
    }
    for (;i1<9;i1++) {vars[i1]=NULL;}
  }
} /* Ende setze_vars */


int rechne_formel(const char * frml,long * dwert) {
/* berechnet Formel
** 1.Arg: Formel, kann enthalten
**        - (Bruch)Zahlen (Trenner Punkt)
**        - Operatoren: bitweise & | ^, shift << >>, Strich + -, Punkt * / %
**        - runde Klammern
**        - Einer-Operatoren: Vorzeichen + -, Negation !, Einer-Komplement ~
**        - Variablen: "$<0-9>"
** 2.Arg: Adresse fuer Rueckgabe berechneter Wert
** Rueckgabe: 0=OK oder -1=Fehler
*/
  if ((frml==NULL) || (dwert==NULL)) {fprintf(stderr,"rechne_formel: formel or return value = NULL"); return(-1);}
  if (rechne_formel_rek(frml,frml+strlen(frml),1,dwert)<0) {return(-1);}
  return(0);
} /* Ende rechne_formel */


char * ermittle_string(const char * panf,const char * pend) {
/* ermittelt String aus 1.-2.Arg und setzt in 3.Arg
** 1.Arg: Beginnpointer
** 2.Arg: Endepointer (ausschliesslich) oder NULL=bis Ende
** Rueckgabe: String (statisch)
*/
  if (panf==NULL) {fprintf(stderr,"ermittle_string: parameter=NULL"); return(NULL);}
  if (pend==NULL) {pend=panf+strlen(panf);}
  return(gib_string(panf,pend));
} /* Ende ermittle_string */
