/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* simple hash with char * as key and integer as value */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <math.h>
#include "inthash.h"

int primzahl(int,int);
struct ihash ** inthash_new(int);
int inthash_add(struct ihash **,const char *,int);
int inthash_get(struct ihash **,const char *,int *);
int inthash_del(struct ihash **,const char *);
void inthash_free(struct ihash **);
char * case_upper(const char *);
char * case_lower(const char *);

static char * set_case(const char *,int);


int primzahl(int primz,int flag) {
/* gives back the next primary number since primz
** flag: -1=search back or 1=search forward
*/
  int d1,wurz;
  int rt,i1;
  i1=0;
  if (primz==0) {return(1);}
  if ((rt=flag)!=-1) {rt=1;}
  while (primz>2) {
    if (primz%2==0) {primz+=rt; continue;}
    wurz=(int)(sqrt((double)primz)+1);
    d1=3;
    i1=1;
    while (d1<=wurz) {
      if (primz%d1==0) {i1=0; break;}
      d1+=2;
    }
    if (i1!=0) {break;}
    primz+=rt;
  }
  return(primz);
} /* Ende primzahl */


struct ihash ** inthash_new(int primz) {
/* create a new int-hash
** 1.arg: primary number
** return: pointer to int-hash
*/
  struct ihash ** inthash;
  if (primz<1) {primz=11;}
  if ((inthash=calloc(primz+1,sizeof(struct ihash *)))==NULL) {fprintf(stderr,"inthash_new: calloc: %s\n",strerror(errno)); return(NULL);}
  if ((inthash[0]=malloc(sizeof(struct ihash)))==NULL) {fprintf(stderr,"inthash_new: malloc: %s\n",strerror(errno)); return(NULL);}
  inthash[0][0].val=primz;  /* number of items in list */
  return(inthash);
} /* Ende inthash_new */


int inthash_add(struct ihash ** inthash,const char * key,int val) {
/* add an item to an existing int-hash
** 1.arg: int-hash
** 2.arg: key
** 3.arg: value
** return: 0=OK or -1=error
*/
  int primz,indx,fidx;
  const char * kptr;
  if ((inthash==NULL) || (key==NULL)) {fprintf(stderr,"inthash_add: argument error\n"); return(-1);}
  primz=inthash[0][0].val;
  indx=0;
  for (kptr=key;*kptr!='\0';kptr++) {
    indx=((indx*256)+(unsigned char)kptr[0])%primz;  /* Horner-Schema */
  }
  indx++;
  if (inthash[indx]==NULL) {
    if ((inthash[indx]=malloc(sizeof(struct ihash)*2))==NULL) {fprintf(stderr,"inthash_add: malloc: %s\n",strerror(errno)); return(-1);}
    inthash[indx][0].val=2;  /* number of field items */
    fidx=1;
  } else {
    for (fidx=1;fidx<inthash[indx][0].val;fidx++) {
      if (strcmp(inthash[indx][fidx].key,key)==0) {free(inthash[indx][fidx].key); break;}
    }
    if (fidx==inthash[indx][0].val) {
      inthash[indx][0].val++;  /* number of field items */
      if ((inthash[indx]=realloc(inthash[indx],sizeof(struct ihash)*inthash[indx][0].val))==NULL) {fprintf(stderr,"inthash_add: realloc: %s\n",strerror(errno)); return(-1);}
    }
  }
  if ((inthash[indx][fidx].key=strdup(key))==NULL) {fprintf(stderr,"inthash_add: strdup: %s\n",strerror(errno)); return(-1);}
  inthash[indx][fidx].val=val;
  return(0);
} /* Ende inthash_add */


int inthash_get(struct ihash ** inthash,const char * key,int * val) {
/* get an item of an int-hash
** 1.arg: int-hash
** 2.arg: key
** 3.arg: address for value
** return: 0=OK or -1=key not found
*/
  int primz,indx,fidx;
  const char * kptr;
  if ((inthash==NULL) || (key==NULL)) {fprintf(stderr,"inthash_get: argument error\n"); return(-1);}
  primz=inthash[0][0].val;
  indx=0;
  for (kptr=key;*kptr!='\0';kptr++) {
    indx=((indx*256)+(unsigned char)kptr[0])%primz;  /* Horner-Schema */
  }
  indx++;
  if (inthash[indx]==NULL) {return(-1);}
  for (fidx=1;fidx<inthash[indx][0].val;fidx++) {
    if (strcmp(inthash[indx][fidx].key,key)==0) {
      if (val!=NULL) {*val=inthash[indx][fidx].val;}
      break;
    }
  }
  if (fidx==inthash[indx][0].val) {return(-1);}
  return(0);
} /* Ende inthash_get */


int inthash_del(struct ihash ** inthash,const char * key) {
/* delete an item of an int-hash
** 1.arg: int-hash
** 2.arg: key to delete
** return: 0=OK or -1=key not found
*/
  int primz,indx,fidx,f2;
  const char * kptr;
  if ((inthash==NULL) || (key==NULL)) {fprintf(stderr,"inthash_del: argument error\n"); return(-1);}
  primz=inthash[0][0].val;
  indx=0;
  for (kptr=key;*kptr!='\0';kptr++) {
    indx=((indx*256)+(unsigned char)kptr[0])%primz;  /* Horner-Schema */
  }
  indx++;
  if (inthash[indx]==NULL) {return(-1);}
  for (fidx=1;fidx<inthash[indx][0].val;fidx++) {
    if (strcmp(inthash[indx][fidx].key,key)==0) {break;}
  }
  if (fidx==inthash[indx][0].val) {return(-1);}
  free(inthash[indx][fidx].key);
  for (f2=fidx+1;f2<inthash[indx][0].val;f2++) {
    inthash[indx][f2-1].key=inthash[indx][f2].key;
    inthash[indx][f2-1].val=inthash[indx][f2].val;
  }
  inthash[indx][0].val--;
  return(0);
} /* Ende inthash_del */


void inthash_free(struct ihash ** inthash) {
/* free an int-hash
** 1.arg: int-hash
*/
  int primz,indx,fidx;
  if (inthash==NULL) {return;}
  primz=inthash[0][0].val;
  for (indx=1;indx<=primz;indx++) {
    if (inthash[indx]==NULL) {continue;}
    for (fidx=1;fidx<inthash[indx][0].val;fidx++) {
      free(inthash[indx][fidx].key);
    }
    free(inthash[indx]);
  }
  free(inthash);
} /* Ende inthash_free */


char * case_upper(const char * string) {
/* set 1.arg to upper case
** 1.arg: string
** return: static string of 1.arg set to upper case
*/
  return(set_case(string,1));
} /* Ende case_upper */


char * case_lower(const char * string) {
/* set 1.arg to lower case
** 1.arg: string
** return: static string of 1.arg set to lower case
*/
  return(set_case(string,2));
} /* Ende case_lower */


static char * set_case(const char * string,int flag) {
/* returns static string of 1.arg set to upper/lower case */
  static char * rval=NULL;
  static int rvalsize=0;
  int slen,i1;
  if (string==NULL) {return(NULL);}
  if (rval==NULL) {
    if ((rval=malloc(sizeof(char)*128))==NULL) {fprintf(stderr,"set_case(%d): malloc: %s\n",flag,strerror(errno)); return(NULL);}
    rvalsize=128;
  }
  slen=strlen(string);
  if (slen>=rvalsize) {
    if ((rval=realloc(rval,sizeof(char)*(slen+1)))==NULL) {fprintf(stderr,"set_case(%d): realloc: %s\n",flag,strerror(errno)); return(NULL);}
    rvalsize=slen+1;
  }
  if (flag==1) {
    for (i1=0;i1<slen;i1++) {rval[i1]=toupper((int)string[i1]);}
  } else {
    for (i1=0;i1<slen;i1++) {rval[i1]=tolower((int)string[i1]);}
  }
  rval[slen]='\0';
  return(rval);
} /* Ende set_case */
