/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* network client */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "config.h"
#include "vgagames2.h"
#include "nw_support.h"
#include "nw_mbcast.h"
#include "nw_rdwr.h"
#include "nw_glob.h"


void set_master(void);
int vg_nw_connect(int,const char *,const char *,const char *);
int vg_nw_waitforconnects(void);
void vg_nw_close(void);
int vg_nw_myplayer(void);
int vg_nw_maxplayer(void);
int vg_nw_virtualplayer(void);
int vg_nw_isalive(int);
int vg_nw_recvkeys(int);
int vg_nw_keypressed(int,int);
int vg_nw_mousepressed(int,int);
int vg_nw_mousex(void);
int vg_nw_mousey(void);
void vg_nw_senddata(int);
void vg_nw_sendcommon(void);
void vg_nw_sendinit(void);
int vg_nw_recvinit(void);
void vg_nw_setdead(int);
int vg_nw_setvar(int,int,const char *);
void * vg_nw_getvar(const char *,int);
int vg_nw_setcommon(int,int,const char *);
void * vg_nw_getcommon(const char *);
void vg_nw_dumppacket(void);
int vg_nw_isopen(void);
int vg_nw_update(unsigned char *);

int lposvar=0,lposcomm=0;

static void do_alrm(int);
static int get_connection(int,const char *,const char *,int);
static int len_of_type(int);
static void endian_swap(int);
static int send_recv(void);

static volatile sig_atomic_t sig_alrm=0;
static int connfd=-9,connprot=0;
static char connhost[512]="",connserv[128]="";
static int master=0;
static int myplayer=0,maxplayer=0,virtplayer=0,pl_size=0,comm_beg=0,comm_len=0,reserved=0;
static char palive[NW_MAXCLIENT]="";
static struct {  /* network packet to variables */
  char paket[NW_MAXPAKET];  /* network packet */
  int anzcom,anzvar;        /* number of variables */
  struct {                  /* variables of network packet */
    int pos,size,type;
    char * name;
  } * com,* var;
  unsigned char ckeys[NW_KEYSIZE];  /* keystrokes of actual client */
} nwvar={"",0,0,NULL,NULL,""};

#define DATMAX ((NW_MAXPAKET-(1+NW_MAXCLIENT+3)/4*4)/4*4)   /* maximum space for all data (truncated aligned) */


/* +++ internal functions +++ */

static void do_alrm(int signum) {
  sig_alrm=1;
} /* Ende do_alrm */


static int get_connection(int protocol,const char * host,const char * service,int szch) {
/* connect to network server */
  int n1,cfd=0;
  struct addrinfo adi0,* adi1,* adi2;
  int i0;
  struct sigaction sa,osa;
  if (szch==0) {
    memset(&sa,0,sizeof(sa));
    sa.sa_handler=do_alrm;
    sigaction(SIGALRM,&sa,&osa);
    sig_alrm=0;
  }
  memset(&adi0,0,sizeof(struct addrinfo));
#ifdef AF_INET6
  for (i0=2;i0>0;i0--) {
    adi0.ai_family=(i0==2?AF_INET:AF_INET6);
#else
  for (i0=1;i0>0;i0--) {
    adi0.ai_family=AF_INET;
#endif
    adi0.ai_socktype=(protocol==PROTO_TCP?SOCK_STREAM:SOCK_DGRAM);
    if ((n1=getaddrinfo(host,service,&adi0,&adi1))!=0) {
      if (i0>1) {continue;}
      fprintf(stderr,"get_connection: getaddrinfo: %s%s%s\n",gai_strerror(n1),n1==EAI_SYSTEM?": ":"",n1==EAI_SYSTEM?strerror(errno):"");
      if (szch==0) {sigaction(SIGALRM,&osa,NULL);}
      return(-1);
    }
    adi2=adi1;
    do {
      cfd=socket(adi1->ai_family,adi1->ai_socktype,adi1->ai_protocol);
      if (cfd<0) {continue;}
      if (szch==0) {alarm(5);}
      if (connect(cfd,adi1->ai_addr,adi1->ai_addrlen)==0) {alarm(0); break;}
      alarm(0);
      close(cfd);
    } while ((adi1=adi1->ai_next)!=NULL);
    if (adi1==NULL) {
      freeaddrinfo(adi2);
      if (i0>1) {continue;}
      fprintf(stderr,"get_connection: cannot get socket\n");
      if (szch==0) {sigaction(SIGALRM,&osa,NULL);}
      return(-1);
    }
    freeaddrinfo(adi2);
    /* write first packet */
    if (writen(cfd,szch?"1":"0",1)!=(ssize_t)1) {
      if (i0>1) {close(cfd); continue;}
      fprintf(stderr,"get_connection: writen first packet: %s\n",strerror(errno));
      close(cfd);
      if (szch==0) {sigaction(SIGALRM,&osa,NULL);}
      return(-1);
    }
    break;
  }
  if (szch==0) {sigaction(SIGALRM,&osa,NULL);}
  return(cfd);
} /* Ende get_connection */


static int len_of_type(int type) {
/* give back number of bytes of type */
  switch(type) {
    case NWVAR_CHAR:
    case NWVAR_UCHAR:  return(1);
    case NWVAR_SHORT:
    case NWVAR_USHORT: return(2);
    case NWVAR_INT:
    case NWVAR_UINT:   return(4);
  }
  return(0);
} /* Ende len_of_type */


static void endian_swap(int vnr) {
/* set byte order of network variables */
  int i1,i2;
  char * ptr;
  if ((vnr<0) || (vnr>maxplayer)) {return;}
  if (vnr==0) {  /* common */
    for (i1=0;i1<nwvar.anzcom;i1++) {
      ptr=nwvar.paket+comm_beg+nwvar.com[i1].pos;
      switch(len_of_type(nwvar.com[i1].type)) {
        case 2:
          for (i2=0;i2<nwvar.com[i1].size;i2++,ptr+=2) {
            *(unsigned short *)ptr=htons(*(unsigned short *)ptr);
          }
          break;
        case 4:
          for (i2=0;i2<nwvar.com[i1].size;i2++,ptr+=4) {
            *(unsigned int *)ptr=htonl(*(unsigned int *)ptr);
          }
          break;
      }
    }
  } else {  /* player-data */
    for (i1=0;i1<nwvar.anzvar;i1++) {
      ptr=nwvar.paket+reserved+(vnr-1)*pl_size+nwvar.var[i1].pos;
      switch(len_of_type(nwvar.var[i1].type)) {
        case 2:
          for (i2=0;i2<nwvar.var[i1].size;i2++,ptr+=2) {
            *(unsigned short *)ptr=htons(*(unsigned short *)ptr);
          }
          break;
        case 4:
          for (i2=0;i2<nwvar.var[i1].size;i2++,ptr+=4) {
            *(unsigned int *)ptr=htonl(*(unsigned int *)ptr);
          }
          break;
      }
    }
  }
} /* Ende endian_swap */


static int send_recv() {
  int i1,erg;
  ssize_t anz;

  if (writen(connfd,nwvar.paket,sizeof(nwvar.paket))!=(ssize_t)sizeof(nwvar.paket)) {
    fprintf(stderr,"send_recv: writen: %s\n",strerror(errno));
    close(connfd); connfd=-1;
    return(-1);
  }
  erg=0;
  while ((anz=readn(connfd,nwvar.paket,sizeof(nwvar.paket)))==(ssize_t)sizeof(nwvar.paket)) {
    erg=1;
    if (master || connprot==PROTO_TCP) {break;}
  }
  if (anz==(ssize_t)-1) {
    if (errno!=EAGAIN) {
      fprintf(stderr,"Connection to network-server closed\n");
      close(connfd); connfd=-1;
      return(-1);
    }
  } else if (anz!=(ssize_t)sizeof(nwvar.paket)) {
    if (anz>0) {
      fprintf(stderr,"send_recv: readn: short read (%d bytes of %d)\n",(int)anz,(int)sizeof(nwvar.paket));
    }
    close(connfd); connfd=-1;
    return(-1);
  }
  if (erg) {  /* got a packet */
    if (!master) {
      endian_swap(0);
      for (i1=1;i1<=maxplayer;i1++) {palive[i1-1]=nwvar.paket[i1]; endian_swap(i1);}
    }
  }
  return(erg);
} /* Ende send_recv */


/* +++ functions +++ */

void set_master() {
  master=1;
} /* Ende set_master */


int vg_nw_connect(int protocol,const char * host,const char * service,const char * servcast) {
/* connect to a network server
** 1.arg: Protocol: PROTO_TCP=tcp or PROTO_UDP=udp
** 2.arg: hostname/ip or NULL=search server via broadcast/multicast
** 3.arg: Portnumber of network server
** 4.arg: Portnumber for broadcast/multicast server
**        or NULL if hostname/ip is given
** return: 0=OK or -1=error
*/
  if ((protocol!=PROTO_TCP) && (protocol!=PROTO_UDP)) {
    fprintf(stderr,"vg_nw_connect: protocol %d not specified\n",protocol);
    return(-1);
  }
  if ((service==NULL) || (*service=='\0')) {
    fprintf(stderr,"vg_nw_connect: no service specified\n");
    return(-1);
  }
  if (((host==NULL) || (*host=='\0')) && ((servcast==NULL) || (*servcast=='\0'))) {
    fprintf(stderr,"vg_nw_connect: no host and no broadcast/multicast port specified\n");
    return(-1);
  }
  /* connect to [host]:service */
  connfd=-1;
  if ((host==NULL) || (*host=='\0')) {  /* search server */
    struct ip_mbcast * ipret;
    int ipv=AF_INET;
#ifdef AF_UNSPEC
    ipv=AF_UNSPEC;
#endif
    if (get_hostip(servcast,ipv,5,1,&ipret)>0) {
      snprintf(connhost,sizeof(connhost),"%s",ipret->ipnr);
      snprintf(connserv,sizeof(connserv),"%s",service);
      if (((connfd=get_connection(protocol,connhost,connserv,1))<0) && (*ipret->iface!='\0')) {
        snprintf(connhost,sizeof(connhost),"%s%%%s",ipret->ipnr,ipret->iface);
        connfd=get_connection(protocol,connhost,connserv,1);
      }
      free_hostip(ipret);
    }
  } else {
    snprintf(connhost,sizeof(connhost),"%s",host);
    snprintf(connserv,sizeof(connserv),"%s",service);
    connfd=get_connection(protocol,connhost,connserv,1);
  }
  if (connfd<0) {
    fprintf(stderr,"vg_nw_connect: %s-connection to [%s]:%s failed\n",protocol==PROTO_TCP?"tcp":"udp",host==NULL?"(broadcast/multicast)":host,service);
    connfd=-1;
    return(-1);
  }
  printf("vg_nw_connect: %s-connection to %s:%s established\n",protocol==PROTO_TCP?"tcp":"udp",connhost,connserv);
  fflush(stdout);
  memset(palive,0,sizeof(palive));
  connprot=protocol;
  return(0);
} /* Ende vg_nw_connect */


int vg_nw_waitforconnects() {
/* after your own connect, wait for all others and get player infos
** If you are master, you can stop waiting with key SPACE (only in graphic mode)
** return: 0=OK or -1=error
*/
  int i1;
  char buf[128];
  if (connfd<0) {
    fprintf(stderr,"vg_nw_waitforconnects: not connected\n");
    return(-1);
  }
  fcntl(connfd,F_SETFL,fcntl(connfd,F_GETFL,0)&~O_NONBLOCK);
  /* read: player number, number of players, begin of virt. players */
  if (master) {
    int cmk=connfd;
    fcntl(connfd,F_SETFL,fcntl(connfd,F_GETFL,0)|O_NONBLOCK);
    block_sysmenu=1;
    while (read(connfd,buf,sizeof(buf))<0) {  /* readline klappt nicht mit UDP */
      if (errno!=EAGAIN) {
        fprintf(stderr,"vg_nw_waitforconnects: error getting answer via read: read error: %s\n",strerror(errno));
        close(connfd); connfd=-1;
        block_sysmenu=0;
        return(-1);
      }
      connfd=-9; (void)vg_key_update(); connfd=cmk;
      if (vg_key_pressed(KEY_SPACE,SHORTKEY)) {  /* stop waiting */
        int cfd;
        cfd=get_connection(connprot,connhost,connserv,0);
        if (cfd>=0) {close(cfd);}
        fcntl(connfd,F_SETFL,fcntl(connfd,F_GETFL,0)&~O_NONBLOCK);
        if (read(connfd,buf,sizeof(buf))<0) {  /* readline klappt nicht mit UDP */
          fprintf(stderr,"vg_nw_waitforconnects: error getting answer via read: read error: %s\n",strerror(errno));
          close(connfd); connfd=-1;
          block_sysmenu=0;
          return(-1);
        }
        break;
      }
    }
    block_sysmenu=0;
    fcntl(connfd,F_SETFL,fcntl(connfd,F_GETFL,0)&~O_NONBLOCK);
  } else {
    if (read(connfd,buf,sizeof(buf))<0) {  /* readline klappt nicht mit UDP */
      fprintf(stderr,"vg_nw_waitforconnects: error getting answer via read: read error: %s\n",strerror(errno));
      close(connfd); connfd=-1;
      return(-1);
    }
  }
  if (sscanf(buf,"%d %d %d %d",&myplayer,&maxplayer,&virtplayer,&pl_size)!=4) {
    fprintf(stderr,"vg_nw_waitforconnects: error getting answer via read: short answer: \"%s\"\n",buf);
    close(connfd); connfd=-1;
    return(-1);
  }
  reserved=(1+maxplayer+3)/4*4;  /* aligned start of player-data */
  comm_beg=reserved+maxplayer*pl_size;  /* begin of common block */
  comm_len=(NW_MAXPAKET-comm_beg)/4*4;  /* aligned (truncated) length of common block */
  if (myplayer<1 || myplayer>NW_MAXCLIENT || maxplayer<1 || maxplayer>NW_MAXCLIENT || virtplayer<2 || virtplayer>NW_MAXCLIENT+1 || pl_size<0 || pl_size%4!=0 || comm_len<0) {
    fprintf(stderr,"vg_nw_waitforconnects: error getting answer via read: answer has invalid values: %d %d %d %d\n",myplayer,maxplayer,virtplayer,pl_size);
    close(connfd); connfd=-1;
    return(-1);
  }
  if (connprot==PROTO_UDP) {fcntl(connfd,F_SETFL,fcntl(connfd,F_GETFL,0)|O_NONBLOCK);}
  memset(palive,0,sizeof(palive));
  for (i1=0;i1<virtplayer-1;i1++) {palive[i1]=1;}  /* set isalive */
  for (;i1<maxplayer;i1++) {palive[i1]=2;}  /* set virtual alive */
  return(0);
} /* Ende vg_nw_waitforconnects */


void vg_nw_close() {
  if (connfd<0) {return;}
  if (master) {
    nwvar.paket[0]=3;
    writen(connfd,nwvar.paket,sizeof(nwvar.paket));
  } else {
    palive[myplayer-1]=3;
    (void)vg_nw_update(NULL);
  }
  if (connfd>=0) {close(connfd); connfd=-1;}
  myplayer=maxplayer=virtplayer=pl_size=comm_beg=comm_len=0;
  lposvar=lposcomm=0;
  memset(&nwvar,0,sizeof(nwvar));
} /* Ende vg_nw_close */


int vg_nw_myplayer() {
  return(myplayer);
} /* Ende vg_nw_myplayer */


int vg_nw_maxplayer() {
  return(maxplayer);
} /* Ende vg_nw_maxplayer */


int vg_nw_virtualplayer() {
  return(virtplayer);
} /* Ende vg_nw_virtualplayer */


int vg_nw_isalive(int vnr) {
  if ((vnr<1) || (vnr>maxplayer)) {return(0);}
  if ((palive[vnr-1]==1) || (palive[vnr-1]==2)) {return(1);}
  return(0);
} /* Ende vg_nw_isalive */


int vg_nw_recvkeys(int vnr) {
/* master function,
** get keystrokes of a client,
** return whether client has sent a keystroke request (0=no, 1=yes)
*/
  if (!master || vnr<1 || vnr>=virtplayer || vnr==myplayer) {return(0);}
  getlock(1,vnr);
  memmove(nwvar.ckeys,memptr+NW_MAXPAKET+(vnr-1)*NW_KEYSIZE,NW_KEYSIZE);
  memptr[NW_MAXPAKET+(vnr-1)*NW_KEYSIZE]=0;
  freelock(vnr);
  if (nwvar.ckeys[0]==3) {palive[vnr-1]=0; nwvar.ckeys[0]=0;}
  if (nwvar.ckeys[0]==1) {return(1);}
  return(0);
} /* Ende vg_nw_recvkeys */


int vg_nw_keypressed(int key,int art) {
/* master function,
** whether a key of the actual client is pressed
** 1.arg: key-define (KEY_0, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
** A call to this function is valid only if vg_nw_recvkeys() returned 1
*/
  int k_press,k_mody,k_flush;
  if (!master) {return(0);}
  if ((key<0) || (key>=KEYS_INDEX)) {return(0);}
  k_press=!!(nwvar.ckeys[1+key]&1);
  k_mody=!!(nwvar.ckeys[1+key]&2);
  k_flush=!!(nwvar.ckeys[1+key]&4);
  if (k_flush) {
    if (k_press&k_mody) {k_flush=0;} else {return(0);}
  }
  if (art==LONGKEY) {
    return(k_press);
  } else if (art==SHORTKEY) {
    return(k_press&k_mody);
  }
  return(0);
} /* Ende vg_nw_keypressed */


int vg_nw_mousepressed(int bnr,int art) {
/* master function,
** whether a mousebutton of the actual client is pressed
** 1.arg: mousebutton-define (MOUSE_LEFT, ...)
** 2.arg: LONGKEY or SHORTKEY
** return: 0=not pressed, 1=pressed
** A call to this function is valid only if vg_nw_recvkeys() returned 1
*/
  int idx,m_press,m_mody,m_flush;
  if (!master) {return(0);}
  idx=1+KEYS_INDEX;
  switch(bnr) {
    case MOUSE_RIGHT:
      idx++;
    case MOUSE_MIDDLE:
      idx++;
    case MOUSE_LEFT:
      m_press=!!(nwvar.ckeys[idx]&1);
      m_mody=!!(nwvar.ckeys[idx]&2);
      m_flush=!!(nwvar.ckeys[idx]&4);
      if (m_flush) {
        if (m_press&m_mody) {m_flush=0;} else {return(0);}
      }
      if (art==LONGKEY) {
        return(m_press);
      } else if (art==SHORTKEY) {
        return(m_press&m_mody);
      }
  }
  return(0);
} /* Ende vg_nw_mousepressed */


int vg_nw_mousex() {
/* master function,
** return actual clients mouse_x position
** A call to this function is valid only if vg_nw_recvkeys() returned 1
*/
  int idx;
  idx=1+KEYS_INDEX+3;
  if (nwvar.ckeys[idx]==255) {return(-1);}
  return(nwvar.ckeys[idx]*256+nwvar.ckeys[idx+1]);
} /* Ende vg_nw_mousex */


int vg_nw_mousey() {
/* master function,
** return actual clients mouse_y position
** A call to this function is valid only if vg_nw_recvkeys() returned 1
*/
  int idx;
  idx=1+KEYS_INDEX+3+2;
  if (nwvar.ckeys[idx]==255) {return(-1);}
  return(nwvar.ckeys[idx]);
} /* Ende vg_nw_mousey */


void vg_nw_senddata(int vnr) {
/* master function,
** send data of a player
*/
  if (!master || vnr<1 || vnr>maxplayer) {return;}
  endian_swap(vnr);
  getlock(1,0);
  memptr[vnr]=palive[vnr-1];
  memmove(memptr+reserved+(vnr-1)*pl_size,nwvar.paket+reserved+(vnr-1)*pl_size,pl_size);
  freelock(0);
  endian_swap(vnr);
} /* Ende vg_nw_senddata */


void vg_nw_sendcommon() {
/* master function,
** send common-block
*/
  if (!master) {return;}
  endian_swap(0);
  getlock(1,0);
  memmove(memptr+comm_beg,nwvar.paket+comm_beg,comm_len);
  freelock(0);
  endian_swap(0);
} /* Ende vg_nw_sendcommon */


void vg_nw_sendinit() {
/* master function,
** send initial values for players and common-block
*/
  int vnr;
  if (!master) {return;}
  getlock(1,0);
  for (vnr=1;vnr<=maxplayer;vnr++) {
    endian_swap(vnr);
    memptr[vnr]=palive[vnr-1];
    memmove(memptr+reserved+(vnr-1)*pl_size,nwvar.paket+reserved+(vnr-1)*pl_size,pl_size);
    endian_swap(vnr);
  }
  endian_swap(0);
  memmove(memptr+comm_beg,nwvar.paket+comm_beg,comm_len);
  endian_swap(0);
  memptr[0]=1;
  freelock(0);
} /* Ende vg_nw_sendinit */


int vg_nw_recvinit() {
/* client function,
** waiting for initial master packet with vg_nw_sendinit()
** return: 0=OK or -1=error (and connection closed)
*/
  if (master) {return(0);}
  for (;;) {
    if (vg_nw_update(NULL)<0) {return(-1);}
    if (nwvar.paket[0]==0) {  /* still waiting for first master packet */
      struct timeval tv;
      tv.tv_sec=0; tv.tv_usec=50000;
      select(0,NULL,NULL,NULL,&tv);
    } else {break;}
  }
  return(0);
} /* Ende vg_nw_recvinit */


void vg_nw_setdead(int vnr) {
  if ((vnr<1) || (vnr>maxplayer)) {return;}
  palive[vnr-1]=3;
  if (master) {
    vg_nw_senddata(vnr);
  } else {
    (void)vg_nw_update(NULL);
  }
} /* Ende vg_nw_setdead */


int vg_nw_setvar(int type,int size,const char * name) {
/* define a network variable in the player-data
** 1.arg: NWVAR_*
** 2.arg: number of elements
** 3.arg: name of variable
** return:  1=OK
**          0=no space left
**         -1=error
*/
  static union {
    char c[DATMAX];
    short h[DATMAX/2];
    int i[DATMAX/4];
  } ub={};
  int pos,len,i1,i2,datmax;
  if ((size<1) || (name==NULL) || (*name=='\0')) {fprintf(stderr,"vg_nw_setvar: invalid argument\n"); return(-1);}
  if ((len=len_of_type(type))<1) {fprintf(stderr,"vg_nw_setvar: invalid argument \"type\": %d\n",type); return(-1);}
  if (nwvar.anzvar==0) {memset(ub.c,0,sizeof(ub.c)); lposvar=0;}
  datmax=DATMAX;
  switch(len) {
    case 1: for (pos=0;pos<datmax;pos++) {
              if (ub.c[pos]==0) {
                for (i2=size,i1=pos;i1<datmax;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.c[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.c[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax) {fprintf(stderr,"vg_nw_setvar: no space left.\n"); return(0);}
            break;
    case 2: for (pos=0;pos<datmax/2;pos++) {
              if (ub.h[pos]==0) {
                for (i2=size,i1=pos;i1<datmax/2;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.h[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.h[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax/2) {fprintf(stderr,"vg_nw_setvar: no space left.\n"); return(0);}
            break;
    case 4: for (pos=0;pos<datmax/4;pos++) {
              if (ub.i[pos]==0) {
                for (i2=size,i1=pos;i1<datmax/4;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.i[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.i[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax/4) {fprintf(stderr,"vg_nw_setvar: no space left.\n"); return(0);}
            break;
    default: fprintf(stderr,"vg_nw_setvar: invalid type length.\n"); return(-1);
  }
  if (lposvar<(pos+size)*len) {lposvar=(pos+size)*len;}
  if ((lposvar+3)/4*4+(lposcomm+3)/4*4>datmax) {  /* all data aligned */
    fprintf(stderr,"vg_nw_setvar: no space left.\n");
    return(0);
  }
  nwvar.anzvar++;
  if ((nwvar.var=(nwvar.anzvar==1?malloc(sizeof(*nwvar.var)):realloc(nwvar.var,sizeof(*nwvar.var)*nwvar.anzvar)))==NULL) {
    fprintf(stderr,"vg_nw_setvar: malloc/realloc: %s\n",strerror(errno));
    nwvar.anzvar=0;
    return(-1);
  }
  nwvar.var[nwvar.anzvar-1].pos=pos*len;
  nwvar.var[nwvar.anzvar-1].size=size;
  nwvar.var[nwvar.anzvar-1].type=type;
  if ((nwvar.var[nwvar.anzvar-1].name=strdup(name))==NULL) {
    fprintf(stderr,"vg_nw_setvar: strdup: %s\n",strerror(errno));
    nwvar.anzvar--;
    return(-1);
  }
  return(1);
} /* Ende vg_nw_setvar */


void * vg_nw_getvar(const char * name,int vnr) {
/* return a pointer to a network variable in the player-data
** 1.arg: name of variable
** 2.arg: player number (1 up to maxplayer)
** return: pointer to variable or NULL=name not found;
**         has to be casted, e.g. (short *)vg_nw_getvar(...)
*/
  int i1;
  if ((vnr<1) || (vnr>maxplayer)) {return(NULL);}
  for (i1=0;i1<nwvar.anzvar;i1++) {
    if (strcmp(nwvar.var[i1].name,name)==0) {break;}
  }
  if (i1==nwvar.anzvar) {return(NULL);}
  return(nwvar.paket+reserved+(vnr-1)*pl_size+nwvar.var[i1].pos);
} /* Ende vg_nw_getvar */


int vg_nw_setcommon(int type,int size,const char * name) {
/* define a network variable in the common data
** 1.arg: NWVAR_*
** 2.arg: number of elements
** 3.arg: name of variable
** return:  1=OK
**          0=no space left
**         -1=error
*/
  static union {
    char c[DATMAX];
    short h[DATMAX/2];
    int i[DATMAX/4];
  } ub={};
  int pos,len,i1,i2,datmax;
  if ((size<1) || (name==NULL) || (*name=='\0')) {fprintf(stderr,"vg_nw_setcommon: invalid argument\n"); return(-1);}
  if ((len=len_of_type(type))<1) {fprintf(stderr,"vg_nw_setcommon: invalid argument \"type\": %d\n",type); return(-1);}
  if (nwvar.anzcom==0) {memset(ub.c,0,sizeof(ub.c)); lposcomm=0;}
  datmax=DATMAX;
  switch(len) {
    case 1: for (pos=0;pos<datmax;pos++) {
              if (ub.c[pos]==0) {
                for (i2=size,i1=pos;i1<datmax;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.c[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.c[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax) {fprintf(stderr,"vg_nw_setcommon: no space left.\n"); return(0);}
            break;
    case 2: for (pos=0;pos<datmax/2;pos++) {
              if (ub.h[pos]==0) {
                for (i2=size,i1=pos;i1<datmax/2;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.h[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.h[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax/2) {fprintf(stderr,"vg_nw_setcommon: no space left.\n"); return(0);}
            break;
    case 4: for (pos=0;pos<datmax/4;pos++) {
              if (ub.i[pos]==0) {
                for (i2=size,i1=pos;i1<datmax/4;i1++,i2--) {
                  if (i2==0) {break;}
                  if (ub.i[i1]!=0) {break;}
                }
                if (i2==0) {  /* found */
                  for (;i2<size;i2++) {ub.i[pos+i2]=-1;}
                  break;
                }
                pos=i1-1;
              }
            }
            if (pos==datmax/4) {fprintf(stderr,"vg_nw_setcommon: no space left.\n"); return(0);}
            break;
    default: fprintf(stderr,"vg_nw_setcommon: invalid type length.\n"); return(-1);
  }
  if (lposcomm<(pos+size)*len) {lposcomm=(pos+size)*len;}
  if ((lposcomm+3)/4*4+(lposvar+3)/4*4>datmax) {  /* all data aligned */
    fprintf(stderr,"vg_nw_setcommon: no space left.\n");
    return(0);
  }
  nwvar.anzcom++;
  if ((nwvar.com=(nwvar.anzcom==1?malloc(sizeof(*nwvar.com)):realloc(nwvar.com,sizeof(*nwvar.com)*nwvar.anzcom)))==NULL) {
    fprintf(stderr,"vg_nw_setcommon: malloc/realloc: %s\n",strerror(errno));
    nwvar.anzcom=0;
    return(-1);
  }
  nwvar.com[nwvar.anzcom-1].pos=pos*len;
  nwvar.com[nwvar.anzcom-1].size=size;
  nwvar.com[nwvar.anzcom-1].type=type;
  if ((nwvar.com[nwvar.anzcom-1].name=strdup(name))==NULL) {
    fprintf(stderr,"vg_nw_setcommon: strdup: %s\n",strerror(errno));
    nwvar.anzcom--;
    return(-1);
  }
  return(1);
} /* Ende vg_nw_setcommon */


void * vg_nw_getcommon(const char * name) {
/* return a pointer to a network variable in the common data
** 1.arg: name of variable
** return: pointer to variable or NULL=name not found;
**         has to be casted, e.g. (short *)vg_nw_getcommon(...)
*/
  int i1;
  for (i1=0;i1<nwvar.anzcom;i1++) {
    if (strcmp(nwvar.com[i1].name,name)==0) {break;}
  }
  if (i1==nwvar.anzcom) {return(NULL);}
  return(nwvar.paket+comm_beg+nwvar.com[i1].pos);
} /* Ende vg_nw_getcommon */


void vg_nw_dumppacket() {
/* for debugging: dumps network variables to stdout,
** do not call it before calling vg_nw_setplayer()
*/
  int i1,len;
  char * type;

  printf("Player-data:\n");
  for (i1=0;i1<nwvar.anzvar;i1++) {
    switch(nwvar.var[i1].type) {
      case 1: type="char"; len=1; break;
      case 2: type="unsigned char"; len=1; break;
      case 3: type="short"; len=2; break;
      case 4: type="unsigned short"; len=2; break;
      case 5: type="int"; len=4; break;
      case 6: type="unsigned int"; len=4; break;
      default: type="(unknown)"; len=0;
    }
    printf("  %s: %s[%d] begins at %d, ends at %d\n",nwvar.var[i1].name,type,nwvar.var[i1].size,nwvar.var[i1].pos,nwvar.var[i1].pos+nwvar.var[i1].size*len-1);
  }

  printf("Common-block:\n");
  for (i1=0;i1<nwvar.anzcom;i1++) {
    switch(nwvar.com[i1].type) {
      case 1: type="char"; len=1; break;
      case 2: type="unsigned char"; len=1; break;
      case 3: type="short"; len=2; break;
      case 4: type="unsigned short"; len=2; break;
      case 5: type="int"; len=4; break;
      case 6: type="unsigned int"; len=4; break;
      default: type="(unknown)"; len=0;
    }
    printf("  %s: %s[%d] begins at %d, ends at %d\n",nwvar.com[i1].name,type,nwvar.com[i1].size,nwvar.com[i1].pos,nwvar.com[i1].pos+nwvar.com[i1].size*len-1);
  }
} /* Ende vg_nw_dumppacket */


int vg_nw_isopen() {
  if (connfd<0) {return(0);}
  return(1);
} /* Ende vg_nw_isopen */


int vg_nw_update(unsigned char * ktn) {
/* send network data to server and receive update
** return: 0=OK or -1=error (and connection closed)
*/
  static char mpakt[NW_MAXPAKET]="";
  int erg;
  if (connfd==-9) {return(0);}   /* never connected */
  if (connfd==-1) {return(-1);}  /* no longer connected */
  if (master) {
    if (connprot==PROTO_TCP) {
      memmove(mpakt,nwvar.paket,NW_MAXPAKET);
      if ((erg=send_recv())<0) {return(-1);}
      if (erg==0) {memmove(nwvar.paket,mpakt,NW_MAXPAKET);}
    }
  } else {
    memmove(mpakt,nwvar.paket,NW_MAXPAKET);
    nwvar.paket[0]=0;
    nwvar.paket[myplayer]=palive[myplayer-1];
    if (ktn!=NULL) {memmove(nwvar.paket+reserved,ktn,NW_KEYSIZE);}
    if ((erg=send_recv())<0) {return(-1);}
    if (erg==0) {memmove(nwvar.paket,mpakt,NW_MAXPAKET);}
  }
  return(0);
} /* Ende vg_nw_update */
