/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   from: UNIX Network Programming Vol. 1, 2nd Ed.
         W. Richard Stevens
         LIMITS OF LIABILITY AND DISCLAIMER OF WARRANTY
         The author and publisher of the book "UNIX Network Programming" have
         used their best efforts in preparing this software.  These efforts
         include the development, research, and testing of the theories and
         programs to determine their effectiveness.  The author and publisher
         make no warranty of any kind, express or implied, with regard to
         these programs or the documentation contained in the book. The author
         and publisher shall not be liable in any event for incidental or
         consequential damages in connection with, or arising out of, the
         furnishing, performance, or use of these programs.
   and modified.
   Get network interfaces for IPv4
   ***************************************************************** */

/* usage example (only for family=AF_INET):
  #include "nw_ifi_get.h"
  struct ifi_info * ifi,* ifihead;
  int doaliases=1;  // doaliases=<0|1>
  if ((ifihead=get_ifi_info(doaliases))==NULL) {exit(1);}  // error
  for (ifi=ifihead;ifi!=NULL;ifi=ifi->ifi_next) {
    printf("%s: <", ifi->ifi_name);  // interface name
    if (ifi->ifi_flags&IFF_UP) {printf("UP ");}  // interface is up
    if (ifi->ifi_flags&IFF_BROADCAST) {printf("BCAST ");}  // supports broadcast
    if (ifi->ifi_flags&IFF_MULTICAST) {printf("MCAST ");}  // supports multicast
    if (ifi->ifi_flags&IFF_LOOPBACK) {printf("LOOP ");}  // loopback interface
    if (ifi->ifi_flags&IFF_POINTOPOINT) {printf("P2P ");}  // PPP interface
    printf(">\n");
    if (ifi->ifi_addr!=NULL) {
      char buf[16];
      struct sockaddr_in * sin=(struct sockaddr_in *)ifi->ifi_addr;
      if (inet_ntop(AF_INET,&sin->sin_addr,buf,sizeof(buf))!=NULL) {
        if (ifi->ifi_myflags&IFI_ALIAS) {printf("Alias: ");}
        printf("%s\n",buf);
      }
    }
    if (ifi->ifi_brdaddr!=NULL) {
      char buf[16];
      struct sockaddr_in * sin=(struct sockaddr_in *)ifi->ifi_brdaddr;
      if (inet_ntop(AF_INET,&sin->sin_addr,buf,sizeof(buf))!=NULL) {
        printf("Broadcast: %s\n",buf);
      }
    }
    if (ifi->ifi_dstaddr!=NULL) {
      char buf[16];
      struct sockaddr_in * sin=(struct sockaddr_in *)ifi->ifi_dstaddr;
      if (inet_ntop(AF_INET,&sin->sin_addr,buf,sizeof(buf))!=NULL) {
        printf("Destination: %s\n",buf);
      }
    }
    if (ifi->ifi_netmask!=NULL) {
      char buf[16];
      struct sockaddr_in * sin=(struct sockaddr_in *)ifi->ifi_netmask;
      if (inet_ntop(AF_INET,&sin->sin_addr,buf,sizeof(buf))!=NULL) {
        printf("Netmask: %s\n",buf);
      }
    }
    printf("\n");
  }
  free_ifi_info(ifihead);
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <errno.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include "config.h"
#ifdef HAVE_SYS_SOCKIO_H
# include <sys/sockio.h>
#endif
#include "nw_support.h"
#include "nw_ifi_get.h"

struct ifi_info * get_ifi_info(int);
void free_ifi_info(struct ifi_info *);


struct ifi_info * get_ifi_info(int doaliases) {
  struct ifi_info * ifi,* ifihead,** ifipnext;
  int sockfd,len,lastlen,flags,myflags;
  char * ptr,* buf,lastname[IFNAMSIZ],* cptr;
  struct ifconf ifc;
  struct ifreq * ifr=NULL;
  struct sockaddr_in * sinptr;

  if ((sockfd=socket(AF_INET,SOCK_DGRAM,0))<0) {
    fprintf(stderr,"get_ifi_info: socket: %s\n",strerror(errno));
    return(NULL);
  }

  /* get buffer size */
  lastlen=0;
  len=100*sizeof(struct ifreq);  /* initial buffer size guess */
  for (;;) {
    if ((buf=malloc(len))==NULL) {
      fprintf(stderr,"get_ifi_info: malloc: %s\n",strerror(errno));
      close(sockfd);
      return(NULL);
    }
    ifc.ifc_len=len;
    ifc.ifc_buf=buf;
    if (ioctl(sockfd,SIOCGIFCONF,&ifc)==-1) {
      if ((errno!=EINVAL) || (lastlen!=0)) {
        fprintf(stderr,"get_ifi_info: ioctl SIOCGIFCONF: %s\n",strerror(errno));
        close(sockfd);
        return(NULL);
      }
    } else {
      if (ifc.ifc_len==lastlen) {break;}  /* success, len has not changed */
      lastlen=ifc.ifc_len;
    }
    len+=10*sizeof(struct ifreq);  /* increment */
    free(buf);
  }

  ifihead=NULL;
  ifipnext=&ifihead;
  lastname[0]=0;

  for (ptr=buf;ptr<buf+ifc.ifc_len;) {  /* all found interfaces */
#ifdef HAVE_SOCKADDR_SA_LEN
    /* length of each (struct ifreq *)ptr is individual,
    ** so copying to a allocated ifr will always force it aligned
    */
    len=IFNAMSIZ+((struct ifreq *)ptr)->ifr_addr.sa_len;
    if (len<(int)sizeof(struct ifreq)) {len=sizeof(struct ifreq);}
    if ((ifr=(ifr==NULL?malloc(len):realloc(ifr,len)))==NULL) {
      fprintf(stderr,"get_ifi_info: malloc: %s\n",strerror(errno));
      ifihead=NULL;
      goto _ende;
    }
    memmove(ifr,ptr,len);
    ptr+=len;  /* for next one in buffer */
#else
    ifr=(struct ifreq *)ptr;
    switch(ifr->ifr_addr.sa_family) {
# ifdef AF_INET6
      case AF_INET6:
        len=(int)sizeof(struct sockaddr_in6);
        break;
# endif
      case AF_INET:
      default:
        len=(int)sizeof(struct sockaddr);
        break;
    }
    if ((len+=IFNAMSIZ)<(int)sizeof(struct ifreq)) {len=sizeof(struct ifreq);}
    ptr+=len;  /* for next one in buffer */
#endif /* HAVE_SOCKADDR_SA_LEN */
    if (ifr->ifr_addr.sa_family!=AF_INET) {continue;}  /* ignore all but AF_INET */

    /* check for alias */
    myflags=0;
    if ((cptr=strchr(ifr->ifr_name,':'))!=NULL) {*cptr='\0';}
    if (strncmp(lastname,ifr->ifr_name,IFNAMSIZ)==0) {
      if (doaliases==0) {continue;}  /* already processed this interface */
      myflags=IFI_ALIAS;
    }
    memmove(lastname,ifr->ifr_name,IFNAMSIZ);
    if (cptr!=NULL) {*cptr=':';}
#ifdef NW_DEBUG
    fprintf(stderr,"Interface found: \"%s\"\n",ifr->ifr_name);
#endif

    /* create new element of struct ifi_info */
    if ((ifi=calloc(1,sizeof(struct ifi_info)))==NULL) {
      fprintf(stderr,"get_ifi_info: calloc: %s\n",strerror(errno));
      ifihead=NULL;
      goto _ende;
    }
    *ifipnext=ifi;  /* prev points to this new one */
    ifipnext=&ifi->ifi_next;  /* pointer to next one goes here */

    /* fill in address, interface name and myflags */
    sinptr=(struct sockaddr_in *)&ifr->ifr_addr;
    if ((ifi->ifi_addr=calloc(1,sizeof(struct sockaddr_in)))==NULL) {
      fprintf(stderr,"get_ifi_info: calloc: %s\n",strerror(errno));
      ifihead=NULL;
      goto _ende;
    }
    memmove(ifi->ifi_addr,sinptr,sizeof(struct sockaddr_in));
    memmove(ifi->ifi_name,ifr->ifr_name,IFNAMSIZ); ifi->ifi_name[IFNAMSIZ-1]='\0';
    ifi->ifi_myflags=myflags;  /* IFI_* values */

    /* get and fill in flags */
    if (ioctl(sockfd,SIOCGIFFLAGS,ifr)==-1) {
      fprintf(stderr,"get_ifi_info: ioctl SIOCGIFFLAGS: %s\n",strerror(errno));
      ifihead=NULL;
      goto _ende;
    }
    flags=ifr->ifr_flags;
    ifi->ifi_flags=flags;  /* IFF_* values */

    /* fill in broadcast address */
#ifdef SIOCGIFBRDADDR
    if (flags&IFF_BROADCAST) {
      if (ioctl(sockfd,SIOCGIFBRDADDR,ifr)==-1) {
        fprintf(stderr,"get_ifi_info: ioctl SIOCGIFBRDADDR: %s\n",strerror(errno));
        ifihead=NULL;
        goto _ende;
      }
      sinptr=(struct sockaddr_in *)&ifr->ifr_broadaddr;
      if ((ifi->ifi_brdaddr=calloc(1,sizeof(struct sockaddr_in)))==NULL) {
        fprintf(stderr,"get_ifi_info: calloc: %s\n",strerror(errno));
        ifihead=NULL;
        goto _ende;
      }
      memmove(ifi->ifi_brdaddr,sinptr,sizeof(struct sockaddr_in));
    }
#endif

    /* fill in destination address */
#ifdef SIOCGIFDSTADDR
    if (flags&IFF_POINTOPOINT) {
      if (ioctl(sockfd,SIOCGIFDSTADDR,ifr)==-1) {
        fprintf(stderr,"get_ifi_info: ioctl SIOCGIFDSTADDR: %s\n",strerror(errno));
        ifihead=NULL;
        goto _ende;
      }
      sinptr=(struct sockaddr_in *)&ifr->ifr_dstaddr;
      if ((ifi->ifi_dstaddr=calloc(1,sizeof(struct sockaddr_in)))==NULL) {
        fprintf(stderr,"get_ifi_info: calloc: %s\n",strerror(errno));
        ifihead=NULL;
        goto _ende;
      }
      memmove(ifi->ifi_dstaddr,sinptr,sizeof(struct sockaddr_in));
    }
#endif

    /* fill in network mask */
#ifdef SIOCGIFNETMASK
    if (ioctl(sockfd,SIOCGIFNETMASK,ifr)>=0) {
      sinptr=(struct sockaddr_in *)&ifr->ifr_addr;
      if ((ifi->ifi_netmask=calloc(1,sizeof(struct sockaddr_in)))==NULL) {
        fprintf(stderr,"get_ifi_info: calloc: %s\n",strerror(errno));
        ifihead=NULL;
        goto _ende;
      }
      memmove(ifi->ifi_netmask,sinptr,sizeof(struct sockaddr_in));
    }
#endif
  }

_ende:
  free(buf);
#ifdef HAVE_SOCKADDR_SA_LEN
  if (ifr!=NULL) {free(ifr);}
#endif
  close(sockfd);
  return(ifihead);  /* pointer to first structure in linked list */
} /* Ende get_ifi_info */


void free_ifi_info(struct ifi_info * ifihead) {
  struct ifi_info * ifi,* ifinext;
  for (ifi=ifihead;ifi!=NULL;ifi=ifinext) {
    if (ifi->ifi_addr!=NULL) {free(ifi->ifi_addr);}
    if (ifi->ifi_brdaddr!=NULL) {free(ifi->ifi_brdaddr);}
    if (ifi->ifi_dstaddr!=NULL) {free(ifi->ifi_dstaddr);}
    if (ifi->ifi_netmask!=NULL) {free(ifi->ifi_netmask);}
    ifinext=ifi->ifi_next;
    free(ifi);
  }
} /* Ende free_ifi_info */
