/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* getting ip via broadcast or multicast */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <net/if.h>  /* (if_nameindex), IFF_* */
#include "config.h"
#include "nw_mbcast.h"
#include "nw_support.h"
#include "nw_ifi_get.h"
#ifdef HAVE_SYS_SELECT_H
# include <sys/select.h>
#endif

#ifndef HAVE_SOCKLEN_T
  typedef int socklen_t;
#endif

#ifdef AF_INET6
# ifndef IPV6_ADD_MEMBERSHIP
#   define IPV6_ADD_MEMBERSHIP IPV6_JOIN_GROUP
# endif
#endif

struct bc_nameindex {
  int idx;
  char bcaddr[16];
  char iface[32];
};

int mbcast_server(const char *,int,int,int);
int get_hostip(const char *,int,int,int,struct ip_mbcast **);
void free_hostip(struct ip_mbcast *);

static int get_hostip_bc(struct bc_nameindex *,const char *,int,int,struct ip_mbcast **);
static int serv_bcast(const char *,const char *,int,int);
#ifdef AF_INET6
static int get_hostip_mc(struct if_nameindex *,const char *,const char *,int,int,struct ip_mbcast **);
static int serv_mcast(const char *,const char *,int,int);
#endif


/* +++ Functions +++ */

void free_hostip(struct ip_mbcast * ipret) {
/* free ip-number struct
** 1.arg: ip-number struct
*/
  struct ip_mbcast * ipm;
  if (ipret==NULL) {return;}
  for (ipm=NULL;ipret!=NULL;ipret=ipm) {
    ipm=ipret->link;
    free(ipret);
  }
} /* Ende free_hostip */


int get_hostip(const char * service,int ipv,int altm,int ipmax,struct ip_mbcast ** ipret) {
/* get ip-number(s) for a listening port via broadcast/multicast
** 1.arg: portnumber of listening port
** 2.arg: 4=ipv4
**        6=ipv6
**        0=both
** 3.arg: timeout for receiving no data
** 4.arg: up to how many ip-numbers to return (0=all)
** 5.arg: address for returning ip-number struct (last element=NULL)
** return:  1=ip-number(s) found
**          0=no ip-number found
**         -1=error
*/
  struct ip_mbcast ** ipm;
  int n1;
  if ((service==NULL) || (ipret==NULL)) {
    fprintf(stderr,"get_hostip: parameter error\n");
    return(-1);
  }
  if (ipmax<0) {ipmax=0;}
  *ipret=NULL;
  ipm=ipret;

  /* try broadcast */
  if ((ipv==4) || (ipv==0)) {
    const int ifbcmax=10;
    struct bc_nameindex * ifbc;
    int ifbcanz=0;
    struct ifi_info * ifi,* ifihead;
    char ipnr[16];
    if ((ifbc=malloc(sizeof(*ifbc)*(ifbcmax+2)))==NULL) {
      fprintf(stderr,"get_hostip: malloc: %s\n",strerror(errno));
      return(-1);
    }
    if ((ifihead=get_ifi_info(1))!=NULL) {
      struct sockaddr_in * sin;
      for (ifi=ifihead;ifi!=NULL;ifi=ifi->ifi_next) {
        if ((ifi->ifi_flags&IFF_UP)==0) {continue;}  /* interface down */
        if ((ifi->ifi_flags&IFF_BROADCAST)==0) {continue;}  /* no broadcast */
        if (ifi->ifi_flags&IFF_LOOPBACK) {continue;}  /* loopback interface */
        if ((ifi->ifi_addr==NULL) || (ifi->ifi_brdaddr==NULL)) {continue;}
        sin=(struct sockaddr_in *)ifi->ifi_brdaddr;
        if (inet_ntop(AF_INET,&sin->sin_addr,ipnr,sizeof(ipnr))!=NULL) {
          for (n1=0;n1<ifbcanz;n1++) {
            if (strcmp(ifbc[n1].bcaddr,ipnr)==0) {break;}
          }
          if ((n1==ifbcanz) && (ifbcanz<ifbcmax)) {
            snprintf(ifbc[ifbcanz].bcaddr,sizeof(ifbc[0].bcaddr),"%s",ipnr);
            snprintf(ifbc[ifbcanz].iface,sizeof(ifbc[0].iface),"%s",ifi->ifi_name);
            ifbc[ifbcanz].idx=ifbcanz+1;
            ifbcanz++;
          }
        }
      }
      free_ifi_info(ifihead);
    }
    snprintf(ifbc[ifbcanz].bcaddr,sizeof(ifbc[0].bcaddr),"255.255.255.255");
    *ifbc[ifbcanz].iface='\0';
    ifbc[ifbcanz].idx=ifbcanz+1;
    ifbcanz++;
    ifbc[ifbcanz].idx=0;
    n1=get_hostip_bc(ifbc,service,altm,ipmax,ipret);
    free(ifbc);
    if (n1<0) {fprintf(stderr,"error get_hostip\n"); return(-1);}
    if (ipmax>0) {
      ipmax-=n1;
      if (ipmax==0) {return(1);}
    }
    for (;*ipret!=NULL;ipret=&(*ipret)->link) {;}
  }

#ifdef AF_INET6
  /* try multicast */
  if ((ipv==6) || (ipv==0)) {
    struct if_nameindex * ifmc=NULL;
    if ((ifmc=if_nameindex())!=NULL) {
      n1=get_hostip_mc(ifmc,"ff02::1",service,altm,ipmax,ipret);
      if_freenameindex(ifmc);
      if (n1<0) {fprintf(stderr,"error get_hostip\n"); return(-1);}
      if (ipmax>0) {
        ipmax-=n1;
        if (ipmax==0) {return(1);}
      }
      for (;*ipret!=NULL;ipret=&(*ipret)->link) {;}
    }
  }
#endif
  if (ipm==ipret) {return(0);}
  return(1);
} /* Ende get_hostip */


int mbcast_server(const char * service,int alts,int ipv,int fk) {
/* start broadcast/multicast server
** 1.arg: portnumber of server
** 2.arg: timeout after 2.arg seconds without request or 0=no timeout
** 3.arg: 4=ipv4
**        6=ipv6
** 4.arg: 0=no fork, 1=fork
** return: 0=OK or -1=error
*/
  pid_t cpid;
  int i1,pd[2];

#ifndef AF_INET6
  if (ipv==6) {
    fprintf(stderr,"mbcast_server: ipv6 not available\n");
    return(-1);
  }
#endif
  if (fk) {
    if (pipe(pd)<0) {
      fprintf(stderr,"mbcast_server: pipe: %s\n",strerror(errno));
      return(-1);
    }
    if ((cpid=fork())==(pid_t)-1) {
      fprintf(stderr,"mbcast_server: fork: %s\n",strerror(errno));
      close(pd[0]); close(pd[1]);
      return(-1);
    }
    if (cpid!=0) {  /* parent */
      close(pd[0]);
      return(0);
    }
    close(pd[1]);
    for (i1=0;i1<100;i1++) {
      if ((i1==STDIN_FILENO) || (i1==STDOUT_FILENO) || (i1==STDERR_FILENO) || (i1==pd[0])) {continue;}
      close(i1);
    }
  } else {pd[0]=-1;}
#ifdef AF_INET6
  if (ipv==6) {
    serv_mcast("0::0",service,alts,pd[0]);
  }
#endif
  if (ipv==4) {
    serv_bcast("0.0.0.0",service,alts,pd[0]);
  }
  if (fk) {
    close(pd[0]);
    _exit(0);
  }
  return(0);
} /* Ende mbcast_server */


static int get_hostip_bc(struct bc_nameindex * ifbc,const char * service,int altm,int ipmax,struct ip_mbcast ** ipret) {
/* get ip-number(s) via broadcast
** 1.arg: list of broadcast ip-numbers
** 2.arg: portnumber of broadcast server
** 3.arg: timeout for receiving no data
** 4.arg: up to how many ip-numbers to return (0=all)
** 5.arg: address for returning ip-number struct (last element=NULL)
** return: >=0: OK (number of ips) or -1=error
*/
  const int on=1;
  int * sockfd,i1,i2,anz,bcanz,fdmax,ipanz;
  char buf[2],ipnr[16];
  socklen_t * salen,slen;
  struct sockaddr * servaddr;
  struct sockaddr_in * sain;
  struct addrinfo hints,* res,* ressave;
  fd_set oset,aset;
  struct timeval tv;

  if ((ifbc==NULL) || (service==NULL) || (ipret==NULL)) {
    fprintf(stderr,"get_hostip_bc: parameter error\n");
    return(-1);
  }
  *ipret=NULL;
  ipanz=0;
  if (ipmax<0) {return(0);}

  if (((sockfd=malloc(sizeof(int)))==NULL) || ((salen=malloc(sizeof(socklen_t)))==NULL)) {
    fprintf(stderr,"get_hostip_bc: malloc: %s\n",strerror(errno));
    return(-1);
  }
  slen=0; servaddr=NULL;
  for (bcanz=0;ifbc[bcanz].idx>0;bcanz++) {
    /* open socket */
    memset(&hints,0,sizeof(struct addrinfo));
    hints.ai_family=AF_INET;
    hints.ai_socktype=SOCK_DGRAM;
    if ((anz=getaddrinfo(ifbc[bcanz].bcaddr,service,&hints,&res))!=0) {continue;}
    ressave=res;
    if (((sockfd=realloc(sockfd,sizeof(int)*(bcanz+1)))==NULL) || ((salen=realloc(salen,sizeof(socklen_t)*(bcanz+1)))==NULL)) {
      fprintf(stderr,"get_hostip_bc: realloc: %s\n",strerror(errno));
      for (;bcanz>=0;bcanz--) {
        if (sockfd[bcanz]>-1) {close(sockfd[bcanz]);}
      }
      freeaddrinfo(ressave);
      return(-1);
    }
    do {
      if (res->ai_family!=AF_INET) {continue;}
      sockfd[bcanz]=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
      if (sockfd[bcanz]>=0) {break;}
    } while ((res=res->ai_next)!=NULL);
    if (res==NULL) {
      freeaddrinfo(ressave);
      sockfd[bcanz]=-1;
      continue;
    }
    salen[bcanz]=res->ai_addrlen;
    if (slen<salen[bcanz]) {
      if ((servaddr=(slen==0?malloc(salen[bcanz]):realloc(servaddr,salen[bcanz])))==NULL) {
        fprintf(stderr,"get_hostip_bc: malloc: %s\n",strerror(errno));
        for (;bcanz>=0;bcanz--) {
          if (sockfd[bcanz]>-1) {close(sockfd[bcanz]);}
        }
        freeaddrinfo(ressave);
        free(sockfd); free(salen);
        return(-1);
      }
      slen=salen[bcanz];
    }
    memmove(servaddr,res->ai_addr,salen[bcanz]);
    freeaddrinfo(ressave);
    /* prepare for getting server ip */
    if (setsockopt(sockfd[bcanz],SOL_SOCKET,SO_BROADCAST,&on,(socklen_t)sizeof(on))<0) {
      close(sockfd[bcanz]);
      sockfd[bcanz]=-1;
      continue;
    }
    /* send broadcast */
    buf[0]='\n'; buf[1]='\0';
    if (sendto(sockfd[bcanz],buf,1,0,servaddr,salen[bcanz])<0) {
      close(sockfd[bcanz]);
      sockfd[bcanz]=-1;
      continue;
    }
  }

  /* get answer */
  FD_ZERO(&oset);
  for (fdmax=-1,i1=0;i1<bcanz;i1++) {
    if (sockfd[i1]>=0) {
      FD_SET(sockfd[i1],&oset);
      if (sockfd[i1]>fdmax) {fdmax=sockfd[i1];}
#ifdef NW_DEBUG
      fprintf(stderr,"Broadcast to \"%s%%%s\"\n",ifbc[i1].bcaddr,ifbc[i1].iface);
#endif
    }
  }
  while (fdmax>=0) {
    aset=oset;
    tv.tv_sec=(altm<=0?5:altm); tv.tv_usec=0;
    anz=select(fdmax+1,&aset,NULL,NULL,&tv);
    if (anz<0) {
      fprintf(stderr,"get_hostip_bc: select: %s\n",strerror(errno));
      ipanz=-1;
      break;
    } else if (anz==0) {break;}  /* timeout */
    for (i1=0;i1<bcanz;i1++) {
      if ((sockfd[i1]>=0) && (FD_ISSET(sockfd[i1],&aset))) {
        slen=salen[i1];
        anz=recvfrom(sockfd[i1],buf,1,0,servaddr,&slen);
        sain=(struct sockaddr_in *)servaddr;
        if ((anz<0) || (inet_ntop(AF_INET,&sain->sin_addr,ipnr,sizeof(ipnr))==NULL)) {
          FD_CLR(sockfd[i1],&oset);
          close(sockfd[i1]); sockfd[i1]=-1;
          for (fdmax=-1,i2=0;i2<bcanz;i2++) {
            if (sockfd[i2]>fdmax) {fdmax=sockfd[i2];}
          }
          continue;
        }
        if (ipanz==0) {
          *ipret=malloc(sizeof(struct ip_mbcast));
        } else {
          (*ipret)->link=malloc(sizeof(struct ip_mbcast));
          ipret=&(*ipret)->link;
        }
        if (*ipret==NULL) {
          fprintf(stderr,"get_hostip_bc: malloc: %s\n",strerror(errno));
          ipanz=-1; break;
        }
        snprintf((*ipret)->ipnr,sizeof((*ipret)->ipnr),"%s",ipnr);
        snprintf((*ipret)->iface,sizeof((*ipret)->iface),"%s",ifbc[i1].iface);
        (*ipret)->ipv=4;
        (*ipret)->link=NULL;
        if (++ipanz==ipmax) {break;}
      }
    }
    if (i1<bcanz) {break;}
  }
  for (i1=0;i1<bcanz;i1++) {
    if (sockfd[i1]>=0) {close(sockfd[i1]);}
  }
  free(sockfd); free(salen); if (servaddr!=NULL) {free(servaddr);}
  return(ipanz);
} /* Ende get_hostip_bc */


static int serv_bcast(const char * ipnr,const char * service,int alts,int pd) {
/* broadcast server
** 1.arg: listen ip
** 2.arg: listen port
** 3.arg: timeout or 0
** 4.arg: reading pipe descriptor
*/
  const int on=1;
  int sockfd,anz;
  char buf[2];
  socklen_t addrlen,alen;
  struct sockaddr * cliaddr;
  struct addrinfo hints,* res,* ressave;
  struct timeval tv;
  fd_set oset,aset;

  if ((ipnr==NULL) || (service==NULL) || (alts<0)) {
    fprintf(stderr,"serv_bcast: parameter error\n");
    return(-1);
  }

  /* open broadcast server socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_flags=AI_PASSIVE;
  hints.ai_family=AF_INET;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(ipnr,service,&hints,&res))!=0) {
    fprintf(stderr,"serv_bcast: getaddrinfo: %s%s%s\n",gai_strerror(anz),anz==EAI_SYSTEM?": ":"",anz==EAI_SYSTEM?strerror(errno):"");
    return(-1);
  }
  ressave=res;
  do {
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd<0) {continue;}
    if (setsockopt(sockfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on))<0) {
      fprintf(stderr,"serv_bcast: setsockopt: %s\n",strerror(errno));
      freeaddrinfo(ressave);
      close(sockfd);
      return(-1);
    }
    if (bind(sockfd,res->ai_addr,res->ai_addrlen)==0) {break;}
    close(sockfd);
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    fprintf(stderr,"serv_bcast: cannot get socket\n");
    freeaddrinfo(ressave);
    return(-1);
  }
  addrlen=res->ai_addrlen;
  freeaddrinfo(ressave);

  /* prepare to receive data */
  if ((cliaddr=malloc(addrlen))==NULL) {
    fprintf(stderr,"serv_bcast: malloc: %s\n",strerror(errno));
    close(sockfd);
    return(-1);
  }

  /* wait for datagrams */
  FD_ZERO(&oset);
  FD_SET(sockfd,&oset);
  if (pd>=0) {FD_SET(pd,&oset);}
  while (1) {
    aset=oset;
    tv.tv_sec=alts; tv.tv_usec=0;
    anz=select((sockfd>pd?sockfd:pd)+1,&aset,NULL,NULL,alts==0?NULL:&tv);
    if (anz<0) {
      fprintf(stderr,"serv_bcast: select: %s\n",strerror(errno));
      continue;
    } else if (anz==0) {break;}  /* timeout */
    if ((pd>=0) && (FD_ISSET(pd,&aset))) {
      if (read(pd,buf,1)==0) {break;}  /* parent exits */
    }
    if (FD_ISSET(sockfd,&aset)) {
      alen=addrlen;
      anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,cliaddr,&alen);
      if (anz<0) {
        fprintf(stderr,"serv_bcast: recvfrom: %s\n",strerror(errno));
      } else {
        buf[0]='\n'; buf[1]='\0';
        if (sendto(sockfd,buf,1,0,cliaddr,alen)<0) {
          fprintf(stderr,"serv_bcast: sendto: %s\n",strerror(errno));
        }
      }
    }
  }
  close(sockfd);
  free(cliaddr);
  return(0);
} /* Ende serv_bcast */


#ifdef AF_INET6
static int get_hostip_mc(struct if_nameindex * ifmc,const char * mcadd,const char * service,int altm,int ipmax,struct ip_mbcast ** ipret) {
/* get ip-number(s) via multicast
** 1.arg: interfaces
** 2.arg: multicast ip
** 3.arg: portnumber of multicast server
** 4.arg: timeout for receiving no data
** 5.arg: up to how many ip-numbers to return (0=all)
** 6.arg: address for returning ip-number struct (last element=NULL)
** return: >=0: OK (number of ips) or -1=error
*/
  int * sockfd,i1,i2,anz,bcanz,fdmax,ipanz;
  char buf[2],ipnr[64];
  socklen_t * salen,slen;
  struct sockaddr * servaddr;
  struct sockaddr_in6 * sain;
  struct addrinfo hints,* res,* ressave;
  fd_set oset,aset;
  struct timeval tv;

  if ((ifmc==NULL) || (mcadd==NULL) || (service==NULL) || (ipret==NULL)) {
    fprintf(stderr,"get_hostip_mc: parameter error\n");
    return(-1);
  }
  *ipret=NULL;
  ipanz=0;
  if (ipmax<0) {return(0);}

  if (((sockfd=malloc(sizeof(int)))==NULL) || ((salen=malloc(sizeof(socklen_t)))==NULL)) {
    fprintf(stderr,"get_hostip_mc: malloc: %s\n",strerror(errno));
    return(-1);
  }
  slen=0; servaddr=NULL;
  for (bcanz=0;ifmc[bcanz].if_index>0;bcanz++) {
    /* open socket */
    memset(&hints,0,sizeof(struct addrinfo));
    hints.ai_family=AF_INET6;
    hints.ai_socktype=SOCK_DGRAM;
    if ((anz=getaddrinfo(mcadd,service,&hints,&res))!=0) {continue;}
    ressave=res;
    if (((sockfd=realloc(sockfd,sizeof(int)*(bcanz+1)))==NULL) || ((salen=realloc(salen,sizeof(socklen_t)*(bcanz+1)))==NULL)) {
      fprintf(stderr,"get_hostip_mc: realloc: %s\n",strerror(errno));
      for (;bcanz>=0;bcanz--) {
        if (sockfd[bcanz]>-1) {close(sockfd[bcanz]);}
      }
      freeaddrinfo(ressave);
      return(-1);
    }
    do {
      if (res->ai_family!=AF_INET6) {continue;}
      sockfd[bcanz]=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
      if (sockfd[bcanz]>=0) {break;}
    } while ((res=res->ai_next)!=NULL);
    if (res==NULL) {
      freeaddrinfo(ressave);
      sockfd[bcanz]=-1;
      continue;
    }
    salen[bcanz]=res->ai_addrlen;
    if (slen<salen[bcanz]) {
      if ((servaddr=(slen==0?malloc(salen[bcanz]):realloc(servaddr,salen[bcanz])))==NULL) {
        fprintf(stderr,"get_hostip_mc: malloc: %s\n",strerror(errno));
        for (;bcanz>=0;bcanz--) {
          if (sockfd[bcanz]>-1) {close(sockfd[bcanz]);}
        }
        freeaddrinfo(ressave);
        free(sockfd); free(salen);
        return(-1);
      }
      slen=salen[bcanz];
    }
    memmove(servaddr,res->ai_addr,salen[bcanz]);
    freeaddrinfo(ressave);
    /* prepare for getting server ip */
    anz=ifmc[bcanz].if_index;
    if (setsockopt(sockfd[bcanz],IPPROTO_IPV6,IPV6_MULTICAST_IF,&anz,(socklen_t)sizeof(anz))<0) {
      close(sockfd[bcanz]);
      sockfd[bcanz]=-1;
      continue;
    }
    /* send multicast */
    buf[0]='\n'; buf[1]='\0';
    if (sendto(sockfd[bcanz],buf,1,0,servaddr,salen[bcanz])<0) {
      close(sockfd[bcanz]);
      sockfd[bcanz]=-1;
      continue;
    }
  }

  /* get answer */
  FD_ZERO(&oset);
  for (fdmax=-1,i1=0;i1<bcanz;i1++) {
    if (sockfd[i1]>=0) {
      FD_SET(sockfd[i1],&oset);
      if (sockfd[i1]>fdmax) {fdmax=sockfd[i1];}
    }
  }
  while (fdmax>=0) {
    aset=oset;
    tv.tv_sec=(altm<=0?5:altm); tv.tv_usec=0;
    anz=select(fdmax+1,&aset,NULL,NULL,&tv);
    if (anz<0) {
      fprintf(stderr,"get_hostip_mc: select: %s\n",strerror(errno));
      ipanz=-1;
      break;
    } else if (anz==0) {break;}  /* timeout */
    for (i1=0;i1<bcanz;i1++) {
      if ((sockfd[i1]>=0) && (FD_ISSET(sockfd[i1],&aset))) {
        slen=salen[i1];
        anz=recvfrom(sockfd[i1],buf,1,0,servaddr,&slen);
        sain=(struct sockaddr_in6 *)servaddr;
        if ((anz<0) || (inet_ntop(AF_INET6,&sain->sin6_addr,ipnr,sizeof(ipnr))==NULL)) {
          FD_CLR(sockfd[i1],&oset);
          close(sockfd[i1]); sockfd[i1]=-1;
          for (fdmax=-1,i2=0;i2<bcanz;i2++) {
            if (sockfd[i2]>fdmax) {fdmax=sockfd[i2];}
          }
          continue;
        }
        if (ipanz==0) {
          *ipret=malloc(sizeof(struct ip_mbcast));
        } else {
          (*ipret)->link=malloc(sizeof(struct ip_mbcast));
          ipret=&(*ipret)->link;
        }
        if (*ipret==NULL) {
          fprintf(stderr,"get_hostip_mc: malloc: %s\n",strerror(errno));
          ipanz=-1; break;
        }
        snprintf((*ipret)->ipnr,sizeof((*ipret)->ipnr),"%s",ipnr);
        snprintf((*ipret)->iface,sizeof((*ipret)->iface),"%s",ifmc[i1].if_name==NULL?"":ifmc[i1].if_name);
        (*ipret)->ipv=6;
        (*ipret)->link=NULL;
        if (++ipanz==ipmax) {break;}
      }
    }
    if (i1<bcanz) {break;}
  }
  for (i1=0;i1<bcanz;i1++) {
    if (sockfd[i1]>=0) {close(sockfd[i1]);}
  }
  free(sockfd); free(salen); if (servaddr!=NULL) {free(servaddr);}
  return(ipanz);
} /* Ende get_hostip_mc */


static int serv_mcast(const char * ipnr,const char * service,int alts,int pd) {
/* multicast server (ipv6)
** 1.arg: listen ip
** 2.arg: listen port
** 3.arg: timeout or 0
** 4.arg: reading pipe descriptor
*/
  const int on=1;
  int sockfd=-1,anz;
  char buf[2];
  socklen_t addrlen,alen;
  struct sockaddr * cliaddr;
  struct addrinfo hints,* res,* ressave;
  struct ipv6_mreq mreq6;
  struct if_nameindex * ifa,* ifi;
  struct timeval tv;
  fd_set oset,aset;
  const char mcadd[]="ff02::1";

  if ((ipnr==NULL) || (service==NULL) || (alts<0)) {
    fprintf(stderr,"serv_mcast: parameter error\n");
    return(-1);
  }

  inet_pton(AF_INET6,mcadd,&mreq6.ipv6mr_multiaddr);
  mreq6.ipv6mr_interface=0;

  /* open multicast server socket */
  memset(&hints,0,sizeof(struct addrinfo));
  hints.ai_flags=AI_PASSIVE;
  hints.ai_family=AF_INET6;
  hints.ai_socktype=SOCK_DGRAM;
  if ((anz=getaddrinfo(ipnr,service,&hints,&res))!=0) {
    fprintf(stderr,"serv_mcast: getaddrinfo: %s%s%s\n",gai_strerror(anz),anz==EAI_SYSTEM?": ":"",anz==EAI_SYSTEM?strerror(errno):"");
    return(-1);
  }
  ressave=res;
  do {
    if (res->ai_family!=AF_INET6) {continue;}
    sockfd=socket(res->ai_family,res->ai_socktype,res->ai_protocol);
    if (sockfd<0) {continue;}
    if (setsockopt(sockfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on))<0) {
      fprintf(stderr,"serv_mcast: setsockopt: %s\n",strerror(errno));
      freeaddrinfo(ressave);
      close(sockfd);
      return(-1);
    }
    if (bind(sockfd,res->ai_addr,res->ai_addrlen)==0) {break;}
    close(sockfd);
  } while ((res=res->ai_next)!=NULL);
  if (res==NULL) {
    fprintf(stderr,"serv_mcast: cannot get socket\n");
    freeaddrinfo(ressave);
    return(-1);
  }
  addrlen=res->ai_addrlen;
  freeaddrinfo(ressave);

  /* prepare to receive data */
  if ((cliaddr=malloc(addrlen))==NULL) {
    fprintf(stderr,"serv_mcast: malloc: %s\n",strerror(errno));
    close(sockfd);
    return(-1);
  }
  ifa=if_nameindex();
  printf("Multicast server: adding membership for \"");
  if (ifa!=NULL) {
    ifi=ifa;
    while (ifi->if_index>0) {
      mreq6.ipv6mr_interface=ifi->if_index;
      if (setsockopt(sockfd,IPPROTO_IPV6,IPV6_ADD_MEMBERSHIP,&mreq6,(socklen_t)sizeof(mreq6))==0) {
        printf("%s ",ifi->if_name);
      }
      ifi++;
    }
    if_freenameindex(ifa);
  }
  printf("\"\n");

  /* wait for datagrams */
  FD_ZERO(&oset);
  FD_SET(sockfd,&oset);
  if (pd>=0) {FD_SET(pd,&oset);}
  while (1) {
    aset=oset;
    tv.tv_sec=alts; tv.tv_usec=0;
    anz=select((sockfd>pd?sockfd:pd)+1,&aset,NULL,NULL,alts==0?NULL:&tv);
    if (anz<0) {
      fprintf(stderr,"serv_mcast: select: %s\n",strerror(errno));
      continue;
    } else if (anz==0) {break;}  /* timeout */
    if ((pd>=0) && (FD_ISSET(pd,&aset))) {
      if (read(pd,buf,1)==0) {break;}  /* parent exits */
    }
    if (FD_ISSET(sockfd,&aset)) {
      alen=addrlen;
      anz=recvfrom(sockfd,buf,sizeof(buf)-1,0,cliaddr,&alen);
      if (anz<0) {
        fprintf(stderr,"serv_mcast: recvfrom: %s\n",strerror(errno));
      } else {
        buf[0]='\n'; buf[1]='\0';
        if (sendto(sockfd,buf,1,0,cliaddr,alen)<0) {
          fprintf(stderr,"serv_mcast: sendto: %s\n",strerror(errno));
        }
      }
    }
  }
  close(sockfd);
  free(cliaddr);
  return(0);
} /* Ende serv_mcast */
#endif
