/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* network server */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include <sys/mman.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include "config.h"
#include "vgagames2.h"
#include "nw_support.h"
#include "nw_mbcast.h"
#include "nw_rdwr.h"
#include "nw_glob.h"

#ifndef HAVE_SOCKLEN_T
  typedef int socklen_t;
#endif

#ifndef MAP_FILE
  #define MAP_FILE 0
#endif
#ifndef MAP_FAILED
  #define MAP_FAILED (void *)-1
#endif

extern int lposvar,lposcomm;
extern void set_master(void);

int vg_nw_setplayer(int,int);
int vg_nw_startserver(int,const char *,int,int,const char *,int);

char * memptr=NULL;
int memfd=-1;
static void * memadr=NULL;
static FILE * memffp=NULL;

static volatile sig_atomic_t sig_alrm=0;
static int pl_mincli=0;
static int pl_maxcli=0;
static int pl_size=0;
static size_t mem_len=NW_MAXPAKET+NW_MAXCLIENT*NW_KEYSIZE;

static void do_alrm(int);
static int mmap_create(void);
static int open_server(int,const char *,int,socklen_t *);
static int run_server(int,int,int,socklen_t,int,int);


/* +++ internal functions +++ */

static void do_alrm(int signum) {
  sig_alrm=1;
} /* Ende do_alrm */


static int mmap_create() {
/* create shared memory region */
  unsigned char buf[(int)mem_len];
  if ((memffp=tmpfile())==NULL) {
    fprintf(stderr,"mmap_create: tmpfile: %s\n",strerror(errno));
    return(-1);
  }
  memfd=fileno(memffp);
  memset(buf,0,mem_len);
  if (write(memfd,buf,mem_len)!=(ssize_t)mem_len) {
    fprintf(stderr,"mmap_create: tmpfile write: %s\n",strerror(errno));
    fclose(memffp);
    return(-1);
  }
  memadr=NULL;
  if ((memadr=mmap(NULL,mem_len,PROT_READ|PROT_WRITE,MAP_SHARED|MAP_FILE,memfd,(off_t)0))==MAP_FAILED) {
    fprintf(stderr,"mmap_create: mmap: %s\n",strerror(errno));
    fclose(memffp);
    return(-1);
  }
  memptr=(char *)memadr;
  fflush(memffp);
  return(0);
} /* Ende mmap_create */


static int open_server(int protocol,const char * service,int ipv,socklen_t * addrlen) {
/* open a listening socket, return listening descriptor and address length */
  const int on=1;
  struct addrinfo adi0,* adi1,* adi2;
  int af_ipv,n1,sockfd=0;
  char host[16];
  if ((protocol!=PROTO_TCP) && (protocol!=PROTO_UDP)) {
    fprintf(stderr,"open_server: parameter error: protocol invalid: %d\n",protocol);
    return(-1);
  }
  if ((service==NULL) || (*service=='\0')) {
    fprintf(stderr,"open_server: parameter error: no port defined\n");
    return(-1);
  }
  if ((ipv!=4) && (ipv!=6)) {
    fprintf(stderr,"open_server: parameter error: invalid ipv=%d\n",ipv);
    return(-1);
  }
  af_ipv=AF_INET;
  strlcpy(host,"0.0.0.0",sizeof(host));
#ifndef AF_INET6
  if (ipv==6) {
    fprintf(stderr,"open_server: parameter error: ipv=6 but ipv6 is not available\n");
    return(-1);
  }
#else
  if (ipv==6) {af_ipv=AF_INET6; strlcpy(host,"0::0",sizeof(host));}
#endif
  memset(&adi0,0,sizeof(struct addrinfo));
  adi0.ai_flags=AI_PASSIVE;
  adi0.ai_socktype=(protocol==PROTO_TCP?SOCK_STREAM:SOCK_DGRAM);
  adi0.ai_family=af_ipv;
  if ((n1=getaddrinfo(host,service,&adi0,&adi1))!=0) {
    fprintf(stderr,"open_server: getaddrinfo: %s%s%s\n",gai_strerror(n1),n1==EAI_SYSTEM?": ":"",n1==EAI_SYSTEM?strerror(errno):"");
    return(-1);
  }
  adi2=adi1;
  do {
    sockfd=socket(adi1->ai_family,adi1->ai_socktype,adi1->ai_protocol);
    if (sockfd<0) {continue;}
    if (setsockopt(sockfd,SOL_SOCKET,SO_REUSEADDR,&on,sizeof(on))<0) {
      fprintf(stderr,"open_server: setsockopt: %s\n",strerror(errno));
      freeaddrinfo(adi2);
      close(sockfd);
      return(-1);
    }
    if (bind(sockfd,adi1->ai_addr,adi1->ai_addrlen)==0) {break;}
    close(sockfd);
  } while ((adi1=adi1->ai_next)!=NULL);
  if (adi1==NULL) {
    fprintf(stderr,"open_server: cannot get socket\n");
    freeaddrinfo(adi2);
    return(-1);
  }
  if (protocol==PROTO_TCP) {
    if (listen(sockfd,5)<0) {
      fprintf(stderr,"open_server: listen: %s\n",strerror(errno));
      freeaddrinfo(adi2);
      close(sockfd);
      return(-1);
    }
  }
  if (addrlen!=NULL) {*addrlen=adi1->ai_addrlen;}
  freeaddrinfo(adi2);
  return(sockfd);
} /* Ende open_server */


static int run_server(int protocol,int sockfd,int contimo,socklen_t addrlen,int altm,int ipv) {
/* run network server, at first get all connects, then act with clients */
  pid_t pid;
  struct {
    struct sockaddr * ska;
    int fd[NW_MAXCLIENT];
    socklen_t skalen;
  } tcp_conn;
  struct {
    struct sockaddr ** ska;
    socklen_t skalen;
  } udp_conn;
  int i1,i2,anz,reserved;
  int all_cli,real_cli,living_cli;
  char rpaket[NW_MAXPAKET],playlive[NW_MAXCLIENT];
  struct sigaction sa;
  if ((protocol!=PROTO_TCP) && (protocol!=PROTO_UDP)) {
    fprintf(stderr,"run_server: parameter error: protocol invalid: %d\n",protocol);
    return(-1);
  }
  if (pl_maxcli<1) {
    fprintf(stderr,"run_server: number of players not set (vg_nw_setplayer())\n");
    return(-1);
  }
  if (contimo<5) {contimo=5;}
  if (altm<0) {altm=0;}

  if ((pid=fork())==(pid_t)-1) {
    fprintf(stderr,"run_server: fork: %s\n",strerror(errno));
    return(-1);
  } else if (pid!=0) {return(0);}  /* parent returns */

  for (i1=0;i1<100;i1++) {
    if (i1==STDIN_FILENO || i1==STDOUT_FILENO || i1==STDERR_FILENO || i1==sockfd || i1==memfd) {continue;}
    close(i1);
  }

  /* +++ get all connections +++ */
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=do_alrm;
  sigaction(SIGALRM,&sa,NULL);
  if (protocol==PROTO_TCP) {
    if ((tcp_conn.ska=malloc((size_t)addrlen))==NULL) {
      fprintf(stderr,"run_server (child): malloc: %s\n",strerror(errno));
      close(sockfd);
      munmap(memadr,mem_len);
      fclose(memffp);
      _exit(1);
    }
    for (i1=0;i1<pl_maxcli;i1++) {
      if ((int)contimo<=0) {break;}
      tcp_conn.skalen=addrlen;
      alarm(contimo);
      if ((tcp_conn.fd[i1]=accept(sockfd,tcp_conn.ska,&tcp_conn.skalen))<0) {
        if ((int)(contimo=alarm(0))<=0) {break;}
        i1--;
        continue;
      }
      contimo=alarm(0);
      if (readn(tcp_conn.fd[i1],rpaket,1)!=(ssize_t)1) {close(tcp_conn.fd[i1]); i1--; continue;}
      if (rpaket[0]=='0') {close(tcp_conn.fd[i1]); break;}  /* master-player wants to stop it */
    }
    if ((real_cli=i1)<1) {  /* real_cli = number of real players */
      fprintf(stderr,"run_server (child): got no connection\n");
      close(sockfd);
      munmap(memadr,mem_len);
      fclose(memffp);
      _exit(1);
    }
  } else {  /* PROTO_UDP */
    if ((udp_conn.ska=malloc(sizeof(struct sockaddr *)*(NW_MAXCLIENT+1)))==NULL) {
      fprintf(stderr,"run_server (child): malloc: %s\n",strerror(errno));
      close(sockfd);
      munmap(memadr,mem_len);
      fclose(memffp);
      _exit(1);
    }
    for (i1=0;i1<=NW_MAXCLIENT;i1++) {
      if ((udp_conn.ska[i1]=malloc((size_t)addrlen))==NULL) {
        fprintf(stderr,"run_server (child): malloc: %s\n",strerror(errno));
        close(sockfd);
        munmap(memadr,mem_len);
        fclose(memffp);
        _exit(1);
      }
    }
    for (i1=0;i1<pl_maxcli;i1++) {
      if ((int)contimo<=0) {break;}
      udp_conn.skalen=addrlen;
      alarm(contimo);
      if ((anz=recvfrom(sockfd,rpaket,sizeof(rpaket),0,udp_conn.ska[i1],&udp_conn.skalen))<0) {
        if ((int)(contimo=alarm(0))<=0) {break;}
        i1--;
        continue;
      }
      contimo=alarm(0);
      if ((anz>0) && (rpaket[0]=='0')) {break;}  /* master-player wants to stop it */
    }
    if ((real_cli=i1)<1) {  /* real_cli = number of real players */
      fprintf(stderr,"run_server (child): got no connection\n");
      close(sockfd);
      munmap(memadr,mem_len);
      fclose(memffp);
      _exit(1);
    }
  }

  /* +++ reply to clients: player number, number of players, begin of virt. players, player size +++ */
  memset(playlive,0,sizeof(playlive));
  for (i1=0;i1<real_cli;i1++) {playlive[i1]=1;}  /* set isalive */
  if (pl_mincli<real_cli) {all_cli=real_cli;} else {all_cli=pl_mincli;}
  for (i1=real_cli;i1<all_cli;i1++) {playlive[i1]=2;}  /* set virtual alive */
  reserved=(1+all_cli+3)/4*4;  /* aligned start of player-data */
  if (protocol==PROTO_TCP) {
    for (i1=0;i1<real_cli;i1++) {
      anz=snprintf(rpaket,sizeof(rpaket),"%d %d %d %d\n",i1+1,all_cli,real_cli+1,pl_size);
      writen(tcp_conn.fd[i1],rpaket,anz);
    }
  } else {  /* PROTO_UDP */
    for (i1=0;i1<real_cli;i1++) {
      anz=snprintf(rpaket,sizeof(rpaket),"%d %d %d %d\n",i1+1,all_cli,real_cli+1,pl_size);
      sendto(sockfd,rpaket,anz,0,udp_conn.ska[i1],udp_conn.skalen);
    }
  }

  /* +++ read and send packets +++
  ** master sets memptr:
  **   byte 0: 1
  **   byte 1 to all_cli: 1=player, 2=virtual player, 3=set to dead
  **   (aligning space)
  **   all_cli*pl_size bytes: player-data for all_cli players
  **   (aligning space)
  **   then up to NW_MAXPAKET: common-block
  ** client sends packet:
  **   byte 0: 0
  **   his byte (between 1 and all_cli): 1=ok, 3=dead
  **   from reserved: keystrokes (NW_KEYSIZE)
  ** master reads memptr:
  **   from NW_MAXPAKET NW_MAXCLIENT*NW_KEYSIZE bytes: client health (0=no keys, 1=keys, 3=dead) and keystrokes
  ** client reads packet:
  **   memptr, NW_MAXPAKET bytes
  ** master sends a packet:
  **   if he wants to close all: byte 0: 3
  **   if TCP: byte 0: 1
  ** master reads a packet:
  **   if TCP: memptr, NW_MAXPAKET bytes (data not important)
  */
  living_cli=real_cli;  /* living_cli=number of living real players */

  while (living_cli>0) {

    if (protocol==PROTO_TCP) {  /* PROTO_TCP: synchronized */
      /* read data */
      for (i1=0;i1<real_cli;i1++) {
        if (playlive[i1]!=1) {continue;}  /* client dead */
        if (altm>0) {alarm(altm);}
        if (readn(tcp_conn.fd[i1],rpaket,sizeof(rpaket))!=(ssize_t)sizeof(rpaket)) {
          alarm(0);
          close(tcp_conn.fd[i1]);
          living_cli--;
          playlive[i1]=0;  /* set client dead */
          getlock(0,i1+1); memptr[NW_MAXPAKET+i1*NW_KEYSIZE]=3; freelock(i1+1);
          continue;
        }
        alarm(0);

        /* read in packets */
        if (rpaket[0]) {  /* master */
          if (rpaket[0]==3) {  /* all closed, exit */
            for (i2=0;i2<all_cli;i2++) {close(tcp_conn.fd[i2]); playlive[i2]=0;}
            living_cli=0;
            break;
          }
        } else if (playlive[i1]) {  /* client */
          getlock(0,i1+1);
          if (rpaket[i1+1]==3) {  /* client dead */
            close(tcp_conn.fd[i1]); playlive[i1]=0; living_cli--;
            memptr[NW_MAXPAKET+i1*NW_KEYSIZE]=3;
          } else {
            memmove(memptr+NW_MAXPAKET+i1*NW_KEYSIZE,rpaket+reserved,NW_KEYSIZE);
          }
          freelock(i1+1);
        }
      }
      /* send data */
      getlock(0,0);
      for (i1=0;i1<real_cli;i1++) {
        if (playlive[i1]!=1) {continue;}  /* client dead */
        writen(tcp_conn.fd[i1],memptr,NW_MAXPAKET);
      }
      freelock(0);

    } else {  /* PROTO_UDP: asynchron */
      struct sockaddr_in * sa1;
      struct sockaddr_in * sa2;
#ifdef AF_INET6
      struct sockaddr_in6 * sb1;
      struct sockaddr_in6 * sb2;
#endif
      /* read data */
      udp_conn.skalen=addrlen;
      if (altm>0) {alarm(altm);}
      if ((anz=recvfrom(sockfd,rpaket,sizeof(rpaket),0,udp_conn.ska[NW_MAXCLIENT],&udp_conn.skalen))<0) {alarm(0); break;}
      alarm(0);
      if (anz!=(int)sizeof(rpaket)) {continue;}
      for (i1=0;i1<real_cli;i1++) {  /* check for valid sender address */
        if (playlive[i1]!=1) {continue;}  /* client dead */
#ifdef AF_INET6
        if (ipv==6) {
          sb1=(struct sockaddr_in6 *)udp_conn.ska[NW_MAXCLIENT];
          sb2=(struct sockaddr_in6 *)udp_conn.ska[i1];
          if (memcmp(&sb1->sin6_addr,&sb2->sin6_addr,sizeof(sb1->sin6_addr))==0 \
          && memcmp(&sb1->sin6_port,&sb2->sin6_port,sizeof(sb1->sin6_port))==0) {break;}
        } else
#endif
        {
          sa1=(struct sockaddr_in *)udp_conn.ska[NW_MAXCLIENT];
          sa2=(struct sockaddr_in *)udp_conn.ska[i1];
          if (memcmp(&sa1->sin_addr,&sa2->sin_addr,sizeof(sa1->sin_addr))==0 \
          && memcmp(&sa1->sin_port,&sa2->sin_port,sizeof(sa1->sin_port))==0) {break;}
        }
      }
      if (i1==real_cli) {continue;}  /* unknown address */

      /* read in packets */
      if (rpaket[0]) {  /* master */
        if (rpaket[0]==3) {  /* all closed, exit */
          for (i2=0;i2<all_cli;i2++) {playlive[i2]=0;}
          living_cli=0;
          break;
        }
      } else if (playlive[i1]) {  /* client */
        getlock(0,i1+1);
        if (rpaket[i1+1]==3) {  /* client dead */
          playlive[i1]=0; living_cli--;
          memptr[NW_MAXPAKET+i1*NW_KEYSIZE]=3;
        } else {
          memmove(memptr+NW_MAXPAKET+i1*NW_KEYSIZE,rpaket+reserved,NW_KEYSIZE);
        }
        /* send data */
        freelock(i1+1);
        getlock(0,0);
        sendto(sockfd,memptr,NW_MAXPAKET,0,udp_conn.ska[NW_MAXCLIENT],udp_conn.skalen);
        freelock(0);
      }
    }
  }
  close(sockfd);
  munmap(memadr,mem_len);
  fclose(memffp);
  _exit(0);
  return(-1);  /* prevent compiler warning */
} /* Ende run_server */


/* +++ functions +++ */

int vg_nw_setplayer(int mincli,int maxcli) {
/* must be called by master before calling vg_nw_startserver(),
** to set the wanted number of players and the size of player-data
** 1.arg: minimal number of clients (1 to NW_MAXCLIENT)
** 2.arg: maximal number of clients (1.arg to NW_MAXCLIENT)
** return: real maximal number of clients (0 to 2.arg)
*/
  int reserved,left;
  if (mincli<1) {mincli=1;} else if (mincli>NW_MAXCLIENT) {mincli=NW_MAXCLIENT;}
  if (maxcli<mincli) {maxcli=mincli;} else if (maxcli>NW_MAXCLIENT) {maxcli=NW_MAXCLIENT;}
  lposvar=(lposvar+3)/4*4;  /* align it */
  lposcomm=(lposcomm+3)/4*4;  /* align it */
  reserved=(1+maxcli+3)/4*4;
  left=NW_MAXPAKET-(reserved+lposvar*maxcli+lposcomm);
  if (left<0) {
    left=(-left-1)/lposvar+1;
    maxcli-=left;
    if (mincli>maxcli) {mincli=maxcli;}
    if (maxcli<1) {mincli=maxcli=0;}
    fprintf(stderr,"vg_nw_setplayer: maximal %d players possible\n",maxcli);
  }
  pl_mincli=mincli;
  pl_maxcli=maxcli;
  pl_size=lposvar;
  return(maxcli);
} /* Ende vg_nw_setplayer */


int vg_nw_startserver(int protocol,const char * service,int contimo,int ipv,const char * servcast,int altm) {
/* start network server and connect to it
** 1.arg: Protocol: PROTO_TCP=tcp or PROTO_UDP=udp
** 2.arg: Portnumber for server
** 3.arg: timeout in seconds for waiting for connects
** 4.arg: ip-version: 4=ipv4 or 6=ipv6
** 5.arg: Portnumber for broadcast/multicast server (or NULL)
** 6.arg: timeout in seconds for waiting for packets or 0=no timeout
** return: 0=OK or -1=error
*/
  int sockfd;
  socklen_t addrlen;
  char lho[64];
  if (contimo<5) {contimo=5;}

  /* open server socket */
  if ((sockfd=open_server(protocol,service,ipv,&addrlen))<0) {
    fprintf(stderr,"vg_nw_startserver: error calling open_server\n");
    return(-1);
  }

  if ((servcast!=NULL) && (pl_maxcli>1)) {
    /* start broadcast/multicast server */
    if (mbcast_server(servcast,contimo,ipv,1)<0) {
      fprintf(stderr,"vg_nw_startserver: error calling mbcast_server\n");
      close(sockfd);
      return(-1);
    }
  }

  /* create shared memory region */
  if (mmap_create()<0) {
    fprintf(stderr,"vg_nw_startserver: error calling mmap_create\n");
    close(sockfd);
    return(-1);
  }

  /* run network server */
  if (run_server(protocol,sockfd,contimo,addrlen,altm,ipv)<0) {
    fprintf(stderr,"vg_nw_startserver: error calling run_server\n");
    close(sockfd);
    munmap(memadr,mem_len);
    fclose(memffp);
    return(-1);
  }
  close(sockfd);

  /* connect but don't wait for all other connects */
  if (ipv==4) {strlcpy(lho,"127.0.0.1",sizeof(lho));}
  else if (ipv==6) {strlcpy(lho,"::1",sizeof(lho));}
  else {strlcpy(lho,"localhost",sizeof(lho));}
  vg_nw_connect(protocol,lho,service,servcast);
  set_master();

  return(0);
} /* Ende vg_nw_startserver */
