/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* support functions for networking */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <unistd.h>
#include <errno.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <ctype.h>
#include "config.h"
#include "nw_support.h"


#ifndef HAVE_GETADDRINFO

/* *****************************************************************
   from: UNIX Network Programming Vol. 1, 2nd Ed.
         W. Richard Stevens
         LIMITS OF LIABILITY AND DISCLAIMER OF WARRANTY
         The author and publisher of the book "UNIX Network Programming" have
         used their best efforts in preparing this software.  These efforts
         include the development, research, and testing of the theories and
         programs to determine their effectiveness.  The author and publisher
         make no warranty of any kind, express or implied, with regard to
         these programs or the documentation contained in the book. The author
         and publisher shall not be liable in any event for incidental or
         consequential damages in connection with, or arising out of, the
         furnishing, performance, or use of these programs.
   ***************************************************************** */

int getaddrinfo(const char *,const char *,const struct addrinfo *,struct addrinfo **);
void freeaddrinfo(struct addrinfo *);
char * gai_strerror(int);
int getnameinfo(const struct sockaddr *,socklen_t,char *,size_t,char *,size_t,int);

struct search {
  const char * host;  /* hostname or address string */
  int family;  /* AF_* */
};

static int ga_aistruct(struct addrinfo ***,const struct addrinfo *,const void *,int);
static int ga_echeck(const char *,const char *,int,int,int,int);
static struct addrinfo * ga_clone(struct addrinfo *);
static int ga_nsearch(const char *,const struct addrinfo *,struct search *);
static int ga_port(struct addrinfo *,int,int);
static int ga_serv(struct addrinfo *,const struct addrinfo *,const char *);

#ifndef INADDRSZ
# define INADDRSZ    4
#endif

#ifndef AF_UNSPEC
# define AF_UNSPEC AF_INET
#endif


int getaddrinfo(const char * hostname,const char * servname,const struct addrinfo * hintsp,struct addrinfo ** result) {
  int rc,error;
  char * canon;
  struct addrinfo hints,* aihead,** aipnext;
# define gaderror(e) {error=(e); goto bad;}
  char ** ap;
  struct hostent * hptr;
  struct search search;

  aihead=NULL;
  aipnext=&aihead;
  canon=NULL;
  if (hintsp==NULL) {
    memset(&hints,0,sizeof(hints));
    hints.ai_family=AF_UNSPEC;
  } else {
    hints=*hintsp;
  }
  if ((rc=ga_echeck(hostname,servname,hints.ai_flags,hints.ai_family,hints.ai_socktype,hints.ai_protocol))!=0) {gaderror(rc);}

  if (ga_nsearch(hostname,&hints,&search)==-1) {gaderror(-1);}
  if (isdigit((int)search.host[0])) {
    struct in_addr inaddr;
    if (inet_pton(AF_INET,search.host,&inaddr)>0) {
      if ((hints.ai_family!=AF_UNSPEC) && (hints.ai_family!=AF_INET)) {gaderror(EAI_ADDRFAMILY);}
      if (search.family!=AF_INET) {gaderror(EAI_ADDRFAMILY);}
      rc=ga_aistruct(&aipnext,&hints,&inaddr,AF_INET);
      if (rc!=0) {gaderror(rc);}
    }
  } else {
    hptr=gethostbyname(search.host);
    if (hptr==NULL) {
      switch (h_errno) {
        case HOST_NOT_FOUND:  gaderror(EAI_NONAME);
        case TRY_AGAIN:       gaderror(EAI_AGAIN);
        case NO_RECOVERY:     gaderror(EAI_FAIL);
        case NO_DATA:         gaderror(EAI_NODATA);
        default:              gaderror(EAI_NONAME);
      }
    }
    if ((hints.ai_family!=AF_UNSPEC) && (hints.ai_family!=hptr->h_addrtype)) {gaderror(EAI_ADDRFAMILY);}
    if ((hostname!=NULL) && (hostname[0]!='\0') && (hints.ai_flags&AI_CANONNAME) && (canon==NULL)) {
      if ((canon=strdup(hptr->h_name))==NULL) {gaderror(EAI_MEMORY);}
    }
    for (ap=hptr->h_addr_list;*ap!=NULL;ap++) {
      rc=ga_aistruct(&aipnext,&hints,*ap,hptr->h_addrtype);
      if (rc!=0) {gaderror(rc);}
    }
  }
  if (aihead==NULL) {gaderror(EAI_NONAME);}

  if ((hostname!=NULL) && (hostname[0]!='\0') && (hints.ai_flags&AI_CANONNAME)) {
    if (canon!=NULL) {aihead->ai_canonname=canon;}
    else {
      if ((aihead->ai_canonname=strdup(search.host))==NULL) {gaderror(EAI_MEMORY);}
    }
  }
  if ((servname!=NULL) && (servname[0]!='\0')) {
    if ((rc=ga_serv(aihead,&hints,servname))!=0) {gaderror(rc);}
  }
  *result=aihead;
  return(0);

bad:
  freeaddrinfo(aihead);
  return(error);
} /* Ende getaddrinfo */


void freeaddrinfo(struct addrinfo * aihead) {
  struct addrinfo * ai,* ainext;
  for (ai=aihead;ai!=NULL;ai=ainext) {
    if (ai->ai_addr!=NULL) {free(ai->ai_addr);}
    if (ai->ai_canonname!=NULL) {free(ai->ai_canonname);}
    ainext=ai->ai_next;
    free(ai);
  }
} /* Ende freeaddrinfo */


static int ga_aistruct(struct addrinfo *** paipnext,const struct addrinfo * hintsp,const void * addr,int family) {
  struct addrinfo * ai;
  if ((ai=calloc(1,sizeof(struct addrinfo)))==NULL) {return(EAI_MEMORY);}
  ai->ai_next=NULL;
  ai->ai_canonname=NULL;
  **paipnext=ai;
  *paipnext=&ai->ai_next;
  if ((ai->ai_socktype=hintsp->ai_socktype)==0) {ai->ai_flags|=AI_CLONE;}
  ai->ai_protocol=hintsp->ai_protocol;

  switch((ai->ai_family=family)) {
    case AF_INET: {
      struct sockaddr_in * sinptr;
      if ((sinptr=calloc(1,sizeof(struct sockaddr_in)))==NULL) {return(EAI_MEMORY);}
#ifdef HAVE_SOCKADDR_SA_LEN
      sinptr->sin_len=sizeof(struct sockaddr_in);
#endif
      sinptr->sin_family=AF_INET;
      memcpy(&sinptr->sin_addr,addr,sizeof(struct in_addr));
      ai->ai_addr=(struct sockaddr *)sinptr;
      ai->ai_addrlen=sizeof(struct sockaddr_in);
      break;
    }
  }
  return(0);
} /* Ende ga_aistruct */


static struct addrinfo * ga_clone(struct addrinfo * ai) {
  struct addrinfo * new;
  if ((new=calloc(1,sizeof(struct addrinfo)))==NULL) {return(NULL);}
  new->ai_next=ai->ai_next;
  ai->ai_next=new;
  new->ai_flags=0;
  new->ai_family=ai->ai_family;
  new->ai_socktype=ai->ai_socktype;
  new->ai_protocol=ai->ai_protocol;
  new->ai_canonname=NULL;
  new->ai_addrlen=ai->ai_addrlen;
  if ((new->ai_addr=malloc(ai->ai_addrlen))==NULL) {return(NULL);}
  memcpy(new->ai_addr,ai->ai_addr,ai->ai_addrlen);
  return(new);
} /* Ende ga_clone */


static int ga_echeck(const char * hostname,const char * servname,int flags,int family,int socktype,int protocol) {
  if (flags&~(AI_PASSIVE|AI_CANONNAME)) {return(EAI_BADFLAGS);}
  if ((hostname==NULL) || (hostname[0]=='\0')) {
    if ((servname==NULL) || (servname[0]=='\0')) {return(EAI_NONAME);}
  }
  switch(family) {
    case AF_UNSPEC:
    case AF_INET:
      if ((socktype!=0) && (socktype!=SOCK_STREAM) && (socktype!=SOCK_DGRAM) && (socktype!=SOCK_RAW)) {return(EAI_SOCKTYPE);}
      break;
    default:
      return(EAI_FAMILY);
  }
  return(0);
} /* Ende ga_echeck */


static int ga_nsearch(const char * hostname,const struct addrinfo * hintsp,struct search * search) {
  int nsearch=0;
  if (hostname==NULL) {
    if (hintsp->ai_flags&AI_PASSIVE) {  /* no host and AI_PASSIVE: wildcard bind */
      switch (hintsp->ai_family) {
        case AF_UNSPEC:
        case AF_INET:
          search[0].host="0.0.0.0";
          search[0].family=AF_INET;
          nsearch++;
          break;
      }

    } else {  /* no host and not AI_PASSIVE: connect to local host */
      switch (hintsp->ai_family) {
        case AF_UNSPEC:
        case AF_INET:
          search[0].host="localhost";
          search[0].family=AF_INET;
          nsearch++;
          break;
      }
    }

  } else {  /* host is specified */
    switch (hintsp->ai_family) {
      case AF_UNSPEC:
      case AF_INET:
        search[0].host=hostname;
        search[0].family=AF_INET;
        nsearch++;
        break;
    }
  }
  if (nsearch!=1) {return(-1);}
  return(nsearch);
} /* Ende ga_nsearch */


static int ga_port(struct addrinfo * aihead,int port,int socktype) {
  int nfound=0;
  struct addrinfo * ai;
  for (ai=aihead;ai!=NULL;ai=ai->ai_next) {
    if (ai->ai_flags&AI_CLONE) {
      if (ai->ai_socktype!=0) {
        if ((ai=ga_clone(ai))==NULL) {return(-1);}
      }
    } else if (ai->ai_socktype!=socktype) {continue;}
    ai->ai_socktype=socktype;
    switch (ai->ai_family) {
      case AF_INET:
        ((struct sockaddr_in *)ai->ai_addr)->sin_port=port;
        nfound++;
        break;
    }
  }
  return(nfound);
} /* Ende ga_port */


static int ga_serv(struct addrinfo * aihead,const struct addrinfo * hintsp,const char * serv) {
  int port,rc,nfound;
  struct servent * sptr;
  nfound=0;
  if (isdigit((int)serv[0])) {
    port=htons(atoi(serv));
    if (hintsp->ai_socktype) {
      if ((rc=ga_port(aihead,port,hintsp->ai_socktype))<0) {return(EAI_MEMORY);}
      nfound+=rc;
    } else {
      if ((rc=ga_port(aihead,port,SOCK_STREAM))<0) {return(EAI_MEMORY);}
      nfound+=rc;
      if ((rc=ga_port(aihead,port,SOCK_DGRAM))<0) {return(EAI_MEMORY);}
      nfound+=rc;
    }
  } else {
    if ((hintsp->ai_socktype==0) || (hintsp->ai_socktype==SOCK_STREAM)) {
      if ((sptr=getservbyname((char *)serv,"tcp"))!=NULL) {
        if ((rc=ga_port(aihead,sptr->s_port,SOCK_STREAM))<0) {return(EAI_MEMORY);}
        nfound+=rc;
      }
    }
    if ((hintsp->ai_socktype==0) || (hintsp->ai_socktype==SOCK_DGRAM)) {
      if ((sptr=getservbyname((char *)serv,"udp"))!=NULL) {
        if ((rc=ga_port(aihead,sptr->s_port,SOCK_DGRAM))<0) {return(EAI_MEMORY);}
        nfound+=rc;
      }
    }
  }
  if (nfound==0) {
    if (hintsp->ai_socktype==0) {return(EAI_NONAME);} else {return(EAI_SERVICE);}
  }
  return(0);
} /* Ende ga_serv */


char * gai_strerror(int err) {
  switch(err) {
    case EAI_ADDRFAMILY: return("address family for host not supported");
    case EAI_AGAIN: return("temporary failure in name resolution");
    case EAI_BADFLAGS: return("invalid flags value");
    case EAI_FAIL: return("non-recoverable failure in name resolution");
    case EAI_FAMILY: return("address family not supported");
    case EAI_MEMORY: return("memory allocation failure");
    case EAI_NODATA: return("no address associated with host");
    case EAI_NONAME: return("host nor service provided, or not known");
    case EAI_SERVICE: return("service not supported for socket type");
    case EAI_SOCKTYPE: return("socket type not supported");
    case EAI_SYSTEM: return("system error");
    default: return("unknown getaddrinfo() error");
  }
} /* Ende gai_strerror */


int getnameinfo(const struct sockaddr * sa,socklen_t salen,char * host,size_t hostlen,char * serv,size_t servlen,int flags) {
  struct sockaddr_in * sain;
  if (sa->sa_family!=AF_INET) {return(EAI_FAMILY);}
  sain=(struct sockaddr_in *)sa;
  if (hostlen>0) {
    struct hostent * hptr;
    if (flags&NI_NUMERICHOST) {
      if (inet_ntop(AF_INET,&sain->sin_addr,host,hostlen)==NULL) {return(EAI_SYSTEM);}
    } else {
      hptr=gethostbyaddr((void *)&sain->sin_addr,sizeof(struct in_addr),AF_INET);
      if ((hptr!=NULL) && (hptr->h_name!=NULL)) {
        if (flags&NI_NOFQDN) {
          char * ptr;
          if ((ptr=strchr(hptr->h_name,'.'))!=NULL) {*ptr=0;}
        }
        snprintf(host,hostlen,"%s",hptr->h_name);
      } else {
        if (flags&NI_NAMEREQD) {return(EAI_NONAME);}
        if (inet_ntop(AF_INET,&sain->sin_addr,host,hostlen)==NULL) {return(EAI_SYSTEM);}
      }
    }
  }
  if (servlen>0) {
    struct servent * sptr;
    if (flags&NI_NUMERICSERV) {
      snprintf(serv,servlen,"%d",ntohs(sain->sin_port));
    } else {
      sptr=getservbyport(sain->sin_port,(flags&NI_DGRAM)?"udp":NULL);
      if ((sptr!=NULL) && (sptr->s_name!=NULL)) {
        snprintf(serv,servlen,"%s",sptr->s_name);
      } else {
        snprintf(serv,servlen,"%d",ntohs(sain->sin_port));
      }
    }
  }
  return(0);
} /* Ende getnameinfo */

#endif /* HAVE_GETADDRINFO */


#ifndef HAVE_INETPTON
/* from: Paul Vixie, 1996 */

int inet_pton(int,const char *,void *);
static int _inet_pton4(const char *,void *);

int inet_pton(int af,const char * src,void * dst) {
  if (af==AF_INET) {
    return(_inet_pton4(src,dst));
  }
  errno=EAFNOSUPPORT;
  return(-1);
} /* Ende inet_pton */


static int _inet_pton4(const char * src,void * dst) {
  unsigned int punktanz=0,imrk;
  unsigned char mrk[INADDRSZ],* pmrk;
  char c1;
  memset(mrk,0,INADDRSZ);
  pmrk=mrk;
  while ((c1=*src++)!='\0') {
    switch(c1) {
      case '0':
      case '1':
      case '2':
      case '3':
      case '4':
      case '5':
      case '6':
      case '7':
      case '8':
      case '9': imrk=*pmrk*10+(c1-'0');
                if (imrk>255) {return(0);}
                *pmrk=imrk;
                break;
      case '.': if (++punktanz>=INADDRSZ) {return(0);}
                pmrk++;
                break;
      default: return(0);
    }
  }
  if (dst!=NULL) {memmove(dst,mrk,INADDRSZ);}
  return(1);
} /* Ende _inet_pton4 */

#endif /* HAVE_INETPTON */


#ifndef HAVE_INETNTOP
/* from: Paul Vixie, 1996 */

const char * inet_ntop(int,const void *,char *,size_t);
static const char * _inet_ntop4(const void *,char *,size_t);

const char * inet_ntop(int af,const void * src,char * dst,size_t size) {
  if (af==AF_INET) {
    return(_inet_ntop4(src,dst,size));
  }
  errno=EAFNOSUPPORT;
  return(NULL);
} /* Ende inet_ntop */


static const char * _inet_ntop4(const void * src,char * dst,size_t size) {
  static const char fmt[]="%u.%u.%u.%u";
  char tmp[16];
  unsigned char * csrc;
  csrc=(unsigned char *)src;
  snprintf(tmp,sizeof(tmp),fmt,csrc[0],csrc[1],csrc[2],csrc[3]);
  if (strlen(tmp)>size) {errno=ENOSPC; return(NULL);}
  if (dst!=NULL) {strncpy(dst,tmp,size-1); dst[size-1]='\0';}
  return(dst);
} /* Ende _inet_ntop4 */

#endif /* HAVE_INETNTOP */
