/* *****************************************************************
   VgaGames2
   Copyright (C) 2000-2007 Kurt Nienhaus <vgagames@vgagames.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
   ***************************************************************** */

/* frontend for sound functions. Sets SIGPIPE to SIG_IGN! */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>
#include "config.h"
#include "vgagames2.h"
#include "nw_rdwr.h"

#define NO_INR

static int pdw[2],pdr[2];
static pid_t cpidw=0;

int vg_sound_startserver(int,int,const char *);
void vg_sound_endserver(void);
void vg_sound_attach(const char *,const char *,int);
void vg_sound_attach_hidden(const char *,const char *,int);
void vg_sound_copy_hidden(const char *,const char *);
static void sound_attach_cmd(const char *,const char *,int,int);
int vg_sound_play(const char *,int,int);
int vg_sound_play_hidden(const char *,int,int);
static int sound_play_cmd(const char *,int,int,int);
void vg_sound_deta_hidden(const char *);
void vg_sound_paus(const char *);
void vg_sound_cont(const char *);
void vg_sound_pate(const char *);
void vg_sound_catb(const char *);
void vg_sound_stop(const char *,int);
void vg_sound_gclr(const char *);
void vg_sound_gadd(const char *,const char *);
void vg_sound_gdel(const char *,const char *);
int vg_sound_volm(const char *,int);
void vg_sound_volumename(const char *,const char *);
int sound_frgs(int *,int *,int *,int *);


int vg_sound_startserver(int srate,int channels,const char * output) {
/* start sound server
** 1.arg: sample rate (0=default) (vgag-sound: option -r)
** 2.arg: channels: 0=default, 1=mono, 2=stereo (vgag-sound: option -c)
** 3.arg: verbose output file from CWDIR or NULL/empty (vgag-sound: option -v)
** return: 0=OK or -1=error
*/
  struct sigaction sa;
  if (cpidw!=0) {
    fprintf(stderr,"vg_sound_startserver: sound server already running\n");
    return(-1);
  }
  memset(&sa,0,sizeof(sa));
  sa.sa_handler=SIG_IGN;
  sigaction(SIGPIPE,&sa,NULL);
  if (pipe(pdw)==-1) {
    fprintf(stderr,"vg_sound_startserver: pipe: %s\n",strerror(errno));
    return(-1);
  }
  if (pipe(pdr)==-1) {
    fprintf(stderr,"vg_sound_startserver: pipe: %s\n",strerror(errno));
    close(pdw[0]); close(pdw[1]);
    return(-1);
  }
  if ((cpidw=fork())==(pid_t)-1) {
    fprintf(stderr,"vg_sound_startserver: fork: %s\n",strerror(errno));
    close(pdw[0]); close(pdw[1]);
    close(pdr[0]); close(pdr[1]);
    return(-1);
  }

  if (cpidw==0) {  /* Child */
    char prog[512],** args;
    int argi;
    close(pdw[1]); close(pdr[0]);
    if (pdw[0]!=STDIN_FILENO) {
      if (dup2(pdw[0],STDIN_FILENO)==-1) {
        fprintf(stderr,"vg_sound_startserver (child): dup2: %s\n",strerror(errno));
        exit(1);
      }
      close(pdw[0]);
    }
    if (pdr[1]!=STDOUT_FILENO) {
      if (dup2(pdr[1],STDOUT_FILENO)==-1) {
        fprintf(stderr,"vg_sound_startserver (child): dup2: %s\n",strerror(errno));
        exit(1);
      }
      close(pdr[1]);
    }

    if ((args=malloc(sizeof(char *)*6))==NULL) {
      fprintf(stderr,"vg_sound_startserver (child): malloc: %s\n",strerror(errno));
      exit(1);
    }
    argi=0;
    args[argi]=NULL;
    snprintf(prog,sizeof(prog),"-p%s",CWDIR);
    args[++argi]=strdup(prog);
    if (srate>0) {
      snprintf(prog,sizeof(prog),"-r%d",srate);
      args[++argi]=strdup(prog);
    }
    if (channels>0) {
      snprintf(prog,sizeof(prog),"-c%d",channels);
      args[++argi]=strdup(prog);
    }
    if ((output!=NULL) && (*output!='\0')) {
      snprintf(prog,sizeof(prog),"-v%s",output);
      args[++argi]=strdup(prog);
    }
    args[++argi]=NULL;
    snprintf(prog,sizeof(prog),"%s/bin/vgag-sound",SHAREDIR);
    args[0]=strdup(prog);
    for (argi=0;argi<99;argi++) {  /* close-on-exec */
      if ((argi==STDIN_FILENO)||(argi==STDOUT_FILENO)||(argi==STDERR_FILENO)) {continue;}
      fcntl(argi,F_SETFD,fcntl(argi,F_GETFD,0)|FD_CLOEXEC);
    }
    execv(prog,args);
    fprintf(stderr,"vg_sound_startserver (child): execv \"%s\": %s\n",prog,strerror(errno));
    _exit(1);  /* do not call atexit() */
  }

  close(pdw[0]); close(pdr[1]);
  if (sound_mode.srate>0 || sound_mode.channel>0 || sound_mode.frg_anz>0 || sound_mode.frg_sz>0) {
    sound_frgs(&sound_mode.srate,&sound_mode.channel,&sound_mode.frg_anz,&sound_mode.frg_sz);
  }
  if (sound_mode.vol[0]<=0) {
    sound_mode.vol[0]=vg_sound_volm("0",-1);
  }
  if (sound_mode.vol[0]<=0) {
    sound_mode.vol[0]=vg_sound_volm("0",40);
  } else {vg_sound_volm("0",sound_mode.vol[0]);}
  return(0);
} /* Ende vg_sound_startserver */


void vg_sound_endserver() {
/* exit sound server */
  if (cpidw==0) {return;}
  close(pdw[1]); close(pdr[0]);
  waitpid(cpidw,NULL,WNOHANG);
  cpidw=0;
} /* Ende vg_sound_endserver */


void vg_sound_attach(const char * soundfile,const char * sname,int volpc) {
/* attach a soundfile to a sound-name (vgag-sound: command ATTA)
** 1.arg: path to soundfile (starting from CWDIR)
** 2.arg: sound-name ("s-<max. 13 characters>")
** 3.arg: volumepercent (0-200)
*/
  sound_attach_cmd(soundfile,sname,volpc,0);
} /* Ende vg_sound_attach */


void vg_sound_attach_hidden(const char * soundfile,const char * sname,int volpc) {
/* attach a soundfile to a sound-name for hidden sound-name */
  sound_attach_cmd(soundfile,sname,volpc,1);
} /* Ende vg_sound_attach */


static void sound_attach_cmd(const char * soundfile,const char * sname,int volpc,int flag) {
  char buf[512];
  if ((cpidw==0) || (soundfile==NULL) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_attach \"%s\": sound server has died\n",soundfile);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"%sATTA %s %s %d\n",flag?"_":"",soundfile,sname,volpc);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_attach \"%s\": error writen\n",soundfile);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende sound_attach_cmd */


void vg_sound_copy_hidden(const char * sdupname,const char * sname) {
/* duplicate hidden sound-name */
  char buf[512];
  if ((cpidw==0) || (sdupname==NULL) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_copy_hidden \"%s\": sound server has died\n",sdupname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"_COPY %s %s\n",sdupname,sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_copy_hidden \"%s\": error writen\n",sdupname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_copy_hidden */


int vg_sound_play(const char * sname,int cresc,int loop) {
/* play soundfile referred to by sound-name (vgag-sound: command PLAY)
** 1.arg: sound-name
** 2.arg: crescendo time in seconds or 0
** 3.arg: how often to repeat or 0=infinite
** return: instance-number or 0=error playing file
*/
  return(sound_play_cmd(sname,cresc,loop,0));
} /* Ende vg_sound_play */


int vg_sound_play_hidden(const char * sname,int cresc,int loop) {
/* play soundfile referred to by hidden sound-name */
  return(sound_play_cmd(sname,cresc,loop,1));
} /* Ende vg_sound_play_hidden */


static int sound_play_cmd(const char * sname,int cresc,int loop,int flag) {
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return(0);}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_play \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(0);
  }
  snprintf(buf,sizeof(buf),"%sPLAY %s %d %d\n",flag?"_":"",sname,cresc,loop);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_play \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(0);
  }
  if (flag==0) {
#ifndef NO_INR
    if (readline(pdr[0],buf,sizeof(buf))<=0) {
      fprintf(stderr,"vg_sound_play \"%s\": error readline\n",sname);
      close(pdw[1]); close(pdr[0]); cpidw=0;
      return(0);
    }
#else
    *buf='\0';
#endif
  } else {*buf='\0';}
  return(atoi(buf));
} /* Ende sound_play_cmd */


void vg_sound_deta_hidden(const char * sname) {
/* detach hidden sound-name */
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_deta_hidden \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"_DETA %s\n",sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_deta_hidden \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_deta_hidden */


void vg_sound_paus(const char * sname) {
/* pause a soundfile (vgag-sound: command PAUS)
** 1.arg: sound-name or group-name or "ALL"
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_paus \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"PAUS %s\n",sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_paus \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_paus */


void vg_sound_cont(const char * sname) {
/* continue a paused soundfile (vgag-sound: command CONT)
** 1.arg: sound-name or group-name or "ALL"
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_cont \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"CONT %s\n",sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_cont \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_cont */


void vg_sound_pate(const char * sname) {
/* pause a soundfile at its end (vgag-sound: command PATE)
** 1.arg: sound-name or group-name or "ALL"
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_pate \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"PATE %s\n",sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_pate \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_pate */


void vg_sound_catb(const char * sname) {
/* continue a with PATE paused soundfile (vgag-sound: command CATB)
** 1.arg: sound-name or group-name or "ALL"
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_catb \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"CATB %s\n",sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_catb \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_catb */


void vg_sound_stop(const char * sname,int diminu) {
/* stop a soundfile (vgag-sound: command STOP)
** 1.arg: sound-name or group-name or "ALL"
** 2.arg: diminuendo in seconds
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_stop \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"STOP %s %d\n",sname,diminu);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_stop \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_stop */


void vg_sound_gclr(const char * gname) {
/* clear a group (vgag-sound: command GCLR)
** 1.arg: group-name ("g-<max. 13 characters>")
*/
  char buf[512];
  if ((cpidw==0) || (gname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_gclr \"%s\": sound server has died\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"GCLR %s\n",gname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_gclr \"%s\": error writen\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_gclr */


void vg_sound_gadd(const char * gname,const char * sname) {
/* add a sound-name to a group (vgag-sound: command GADD)
** 1.arg: group-name ("g-<max. 13 characters>")
** 2.arg: sound-name
*/
  char buf[512];
  if ((cpidw==0) || (gname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_gadd \"%s\": sound server has died\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"GADD %s %s\n",gname,sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_gadd \"%s\": error writen\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_gadd */


void vg_sound_gdel(const char * gname,const char * sname) {
/* delete a sound-name from a group (vgag-sound: command GDEL)
** 1.arg: group-name
** 2.arg: sound-name
*/
  char buf[512];
  if ((cpidw==0) || (gname==NULL)) {return;}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_gdel \"%s\": sound server has died\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
  snprintf(buf,sizeof(buf),"GDEL %s %s\n",gname,sname);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_gdel \"%s\": error writen\n",gname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return;
  }
} /* Ende vg_sound_gdel */


int vg_sound_volm(const char * sname,int volume) {
/* change volume and return previous value (vgag-sound: command VOLM)
** 1.arg: sound-name, group-name or "0"=main volume
** 2.arg: new volume (0-100) or -1=no change
** return: previous value (0-100) or -1=error
*/
  char buf[512];
  if ((cpidw==0) || (sname==NULL)) {return(-1);}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"vg_sound_volm \"%s\": sound server has died\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(-1);
  }
  snprintf(buf,sizeof(buf),"VOLM %s %d\n",sname,volume);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"vg_sound_volm \"%s\": error writen\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(-1);
  }
  if (readline(pdr[0],buf,sizeof(buf))<=0) {
    fprintf(stderr,"vg_sound_volm \"%s\": error readline\n",sname);
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(-1);
  }
  return(atoi(buf));
} /* Ende vg_sound_volm */


void vg_sound_volumename(const char * gname,const char * vname) {
/* set volume names
** 1.arg: group-name or sound-name
** 2.arg: description
*/
  if ((gname==NULL) && (vname==NULL)) {volume_name.anz=0; return;}  /* clear all */
  if (volume_name.anz>=VOLUME_MAXNO) {return;}
  if ((gname==NULL) || (*gname=='\0') || (vname==NULL) || (*vname=='\0')) {return;}
  if ((strncmp(gname,"g-",2)!=0) && (strncmp(gname,"s-",2)!=0)) {return;}
  snprintf(volume_name.vname[volume_name.anz],sizeof(volume_name.vname[0]),"%s",vname);
  snprintf(volume_name.gname[volume_name.anz],sizeof(volume_name.gname[0]),"%s",gname);
  if (sound_mode.vol[volume_name.anz+1]>=0) {vg_sound_volm(gname,sound_mode.vol[volume_name.anz+1]);}
  volume_name.anz++;
} /* Ende vg_sound_volumename */


int sound_frgs(int * srate,int * channel,int * frg_anz,int * frg_sz) {
/* change/get samplerate, channel, fragment number and size (vgag-sound: command FRGS) */
  char buf[512];
  if (cpidw==0) {return(0);}
  if (waitpid(cpidw,NULL,WNOHANG)!=0) {
    fprintf(stderr,"sound_frgs: sound server has died\n");
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(0);
  }
  if ((frg_anz==NULL) || (frg_sz==NULL)) {return(1);}
  snprintf(buf,sizeof(buf),"FRGS %d,%d,%d,%d\n",*srate,*channel,*frg_anz,*frg_sz);
  if (writen(pdw[1],buf,strlen(buf))<0) {
    fprintf(stderr,"sound_frgs: error writen\n");
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(0);
  }
  if (readline(pdr[0],buf,sizeof(buf))<=0) {
    fprintf(stderr,"sound_frgs: error readline\n");
    close(pdw[1]); close(pdr[0]); cpidw=0;
    return(0);
  }
  if (sscanf(buf,"%d %d %d %d",srate,channel,frg_anz,frg_sz)!=4) {*srate=*channel=*frg_anz=*frg_sz=0;}
  return(1);
} /* Ende sound_frgs */
