/* Include file for vgag-sound.h - OSS support */

#include <sys/ioctl.h>
#include <sys/soundcard.h>

/* definitions */
static const char * pcm_dev_oss="/dev/dsp";    /* PCM device */
static int pcm_fd_oss=-1;                      /* PCM file descriptor */
static const char * mix_dev_oss="/dev/mixer";  /* Mixer device */
static int mix_fd_oss=-1;                      /* Mixer file descriptor */
static int gain_wert_oss=0;


/* declarations */

static int _config_oss(struct sound_dev *,int,int);

int open_oss(struct sound_dev *,int,int);
void write_oss(struct sound_dev *,const unsigned char *,int);
void close_oss(struct sound_dev *);
int mixer_oss(struct sound_dev *,int);


/* internal functions */

static int _config_oss(struct sound_dev * sdptr,int srate,int channels) {
/* (re)configure pcm device */
  audio_buf_info info;
  int i1,val;
  /* reset device */
  /* ioctl(pcm_fd_oss,SNDCTL_DSP_RESET); */

  if (sdptr->frag_anz==0) {sdptr->frag_anz=fragment_anz;}
  if (sdptr->frag_size==0) {sdptr->frag_size=fragment_size;}

  /* Fragments */
  i1=(sdptr->frag_anz<<16);
  for (val=sdptr->frag_size;val>1;val>>=1) {i1++;}
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_SETFRAGMENT,&i1)==-1) {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_SETFRAGMENT: ioctl error: %s\n",strerror(errno));
    close(pcm_fd_oss);
    close(mix_fd_oss);
    return(-1);
  }
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_GETOSPACE,&info)==0) {
    fprintf(stderr,"  _config_oss: Fragments=%d, Fragstotal=%d, Fragment-Size=%d, Bytes=%d\n",info.fragments,info.fragstotal,info.fragsize,info.bytes);
  } else {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_GETOSPACE: ioctl error: %s\n",strerror(errno));
    close(pcm_fd_oss);
    close(mix_fd_oss);
    return(-1);
  }
  sdptr->frag_anz=info.fragments;
  sdptr->frag_size=info.fragsize;

  /* bits per sample */
#ifdef ENDIAN_IS_BIG
  i1=val=AFMT_S16_BE;
#else
  i1=val=AFMT_S16_LE;
#endif
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_SETFMT,&val)==-1) {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_SETFMT(16LE): ioctl error: %s\n",strerror(errno));
    return(-1);
  } else if (i1!=val) {
    i1=val=AFMT_U8;
    if (ioctl(pcm_fd_oss,SNDCTL_DSP_SETFMT,&val)==-1) {
      fprintf(stderr,"  _config_oss: SNDCTL_DSP_SETFMT(U8): ioctl error: %s\n",strerror(errno));
      return(-1);
    } else if (i1!=val) {
      fprintf(stderr,"  _config_oss: SNDCTL_DSP_SETFMT: error: set to %d\n",val);
      return(-1);
    } else {sdptr->bitsamp=8;}
  } else {sdptr->bitsamp=16;}
  /* channels */
  val=channels-1;
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_STEREO,&val)==-1) {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_STEREO(%d): ioctl error: %s\n",channels,strerror(errno));
    return(-1);
  } else if (val!=channels-1) {
    channels=(channels==2?1:2);
    val=channels-1;
    if (ioctl(pcm_fd_oss,SNDCTL_DSP_STEREO,&val)==-1) {
      fprintf(stderr,"  _config_oss: SNDCTL_DSP_STEREO(%d): ioctl error: %s\n",channels,strerror(errno));
      return(-1);
    } else if (val!=channels-1) {
      fprintf(stderr,"  _config_oss: SNDCTL_DSP_STEREO: error: set to %d\n",val);
      return(-1);
    } else {sdptr->channel=channels;}
  } else {sdptr->channel=channels;}
  /* sample rate */
  val=srate;
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_SPEED,&val)==-1) {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_SPEED: ioctl error: %s\n",strerror(errno));
    return(-1);
  } else if ((srate<val-1000) || (srate>val+1000)) {
    fprintf(stderr,"  _config_oss: SNDCTL_DSP_SPEED: error: %d Hz requested, %d Hz got\n",srate,val);
    return(-1);
  }
  sdptr->srate=srate;
  /* rest */
  if (ioctl(pcm_fd_oss,SNDCTL_DSP_GETCAPS,&i1)!=-1) {
    if (i1&DSP_CAP_TRIGGER) {
      i1=PCM_ENABLE_OUTPUT;
      ioctl(pcm_fd_oss,SNDCTL_DSP_SETTRIGGER,&i1);
    } else {ioctl(pcm_fd_oss,SNDCTL_DSP_POST,0);}
  } else {ioctl(pcm_fd_oss,SNDCTL_DSP_POST,0);}

  return(0);
} /* Ende _config_oss */


/* functions */

int open_oss(struct sound_dev * sdptr,int srate,int channels) {
/* open device
** 2.arg=sample rate
** 3.arg=number of channels (1 or 2)
** return: 0=OK or -1=error
*/
  int i1,retry;

  /* +++ open pcm device +++ */
  for (retry=5;;retry--) {
    if (alarmtime==0) {
      if ((pcm_fd_oss=open(pcm_dev_oss,O_WRONLY|O_NONBLOCK,0))<0) {
        if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
        fprintf(stderr,"  open_oss: open %s: %s\n",pcm_dev_oss,strerror(errno));
        return(-1);
      }
      if (fcntl(pcm_fd_oss,F_SETFL,fcntl(pcm_fd_oss,F_GETFL,0)&~O_NONBLOCK)<0) {
        fprintf(stderr,"  open_oss: fcntl %s: %s\n",pcm_dev_oss,strerror(errno));
        close(pcm_fd_oss);
        return(-1);
      }
    } else {
      alarm(alarmtime);
      if ((pcm_fd_oss=open(pcm_dev_oss,O_WRONLY,0))<0) {
        alarm(0);
        if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
        fprintf(stderr,"  open_oss: open %s: %s\n",pcm_dev_oss,strerror(errno));
        return(-1);
      }
      alarm(0);
    }
    break;
  }

  /* +++ Mixer +++ */
  if ((mix_fd_oss=open(mix_dev_oss,O_RDWR))<0) {
    fprintf(stderr,"  open_oss: open %s: %s\n",mix_dev_oss,strerror(errno));
    close(pcm_fd_oss);
    return(-1);
  }

  if (ioctl(mix_fd_oss,SOUND_MIXER_READ_DEVMASK,&i1)==-1) {
    fprintf(stderr,"  open_oss: SOUND_MIXER_READ_DEVMASK: ioctl error: %s\n",strerror(errno));
    close(mix_fd_oss);
    close(pcm_fd_oss);
    return(-1);
  }
  gain_wert_oss=0;
#ifdef SOUND_MASK_VOLUME
  if (i1&SOUND_MASK_VOLUME) {gain_wert_oss|=1;}
#endif
#ifdef SOUND_MASK_OGAIN
  if (i1&SOUND_MASK_OGAIN) {gain_wert_oss|=2;}
#endif
  if (((i1&SOUND_MASK_PCM)==0) || (gain_wert_oss==0)) {
    fprintf(stderr,"  open_oss: mixer volume not found\n");
    close(mix_fd_oss);
    close(pcm_fd_oss);
    return(-1);
  }

  /* +++ configure pcm device +++ */
  return(_config_oss(sdptr,srate,channels));
} /* Ende open_oss */


void write_oss(struct sound_dev * sdptr,const unsigned char * val,int size) {
/* write samples "val" of size "size" out to device */
  write(pcm_fd_oss,val,size);
} /* Ende write_oss */


void close_oss(struct sound_dev * sdptr) {
/* close device */
  close(mix_fd_oss); mix_fd_oss=-1;
  close(pcm_fd_oss); pcm_fd_oss=-1;
} /* Ende close_oss */


int mixer_oss(struct sound_dev * sdptr,int volm) {
/* set main volume
** 2.arg: 0-100=set volume
**        -1=only get actual volume
**        -2=resume saved values
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  static int fvol=-1,fpcm=-1,fogain=-1;
  int vvol,vpcm,vogain;
  vvol=vpcm=vogain=0;
#ifdef SOUND_MASK_VOLUME
  if ((gain_wert_oss&1) && (ioctl(mix_fd_oss,SOUND_MIXER_READ_VOLUME,&vvol)==-1)) {
    fprintf(stderr,"mixer_oss: SOUND_MIXER_READ_VOLUME: ioctl error: %s\n",strerror(errno));
    return(-1);
  }
#endif
#ifdef SOUND_MASK_OGAIN
  if ((gain_wert_oss&2) && (ioctl(mix_fd_oss,SOUND_MIXER_READ_OGAIN,&vogain)==-1)) {
    fprintf(stderr,"mixer_oss: SOUND_MIXER_READ_OGAIN: ioctl error: %s\n",strerror(errno));
    return(-1);
  }
#endif
  if (ioctl(mix_fd_oss,SOUND_MIXER_READ_PCM,&vpcm)==-1) {
    fprintf(stderr,"mixer_oss: SOUND_MIXER_READ_PCM: ioctl error: %s\n",strerror(errno));
    return(-1);
  }
  if (fvol==-1) {fvol=vvol; fpcm=vpcm; fogain=vogain;}
  /* volume value for ioctl is "right_volume<<8 | left_volume"
  ** We set right and left volume always to the same value
  */
  vvol=(((vvol&0xff00)>>8)+(vvol&0xff))/2;
  if (vvol>100) {vvol=100;}
  vogain=(((vogain&0xff00)>>8)+(vogain&0xff))/2;
  if (vogain>100) {vogain=100;}
  if ((gain_wert_oss&3)==3) {
    vvol=(vvol+vogain)/2;
  } else if (gain_wert_oss&2) {
    vvol=vogain;
  }
  /* set volume */
  if (volm>=0) {
    if (volm>100) {volm=100;}
    volm|=(volm<<8);
    vpcm=100;
    vpcm|=(vpcm<<8);
#ifdef SOUND_MASK_VOLUME
    if ((gain_wert_oss&1) && (ioctl(mix_fd_oss,SOUND_MIXER_WRITE_VOLUME,&volm)==-1)) {
      fprintf(stderr,"mixer_oss: SOUND_MIXER_WRITE_VOLUME: ioctl error: %s\n",strerror(errno));
      return(-1);
    }
#endif
#ifdef SOUND_MASK_OGAIN
    if ((gain_wert_oss&2) && (ioctl(mix_fd_oss,SOUND_MIXER_WRITE_OGAIN,&volm)==-1)) {
      fprintf(stderr,"mixer_oss: SOUND_MIXER_WRITE_OGAIN: ioctl error: %s\n",strerror(errno));
      return(-1);
    }
#endif
    if (ioctl(mix_fd_oss,SOUND_MIXER_WRITE_PCM,&vpcm)==-1) {
      fprintf(stderr,"mixer_oss: SOUND_MIXER_WRITE_PCM: ioctl error: %s\n",strerror(errno));
      return(-1);
    }
  } else if (volm==-2) {
#ifdef SOUND_MASK_VOLUME
    if (gain_wert_oss&1) {ioctl(mix_fd_oss,SOUND_MIXER_WRITE_VOLUME,&fvol);}
#endif
#ifdef SOUND_MASK_OGAIN
    if (gain_wert_oss&2) {ioctl(mix_fd_oss,SOUND_MIXER_WRITE_OGAIN,&fogain);}
#endif
    ioctl(mix_fd_oss,SOUND_MIXER_WRITE_PCM,&fpcm);
  }
  return(vvol);
} /* Ende mixer_oss */
