/* Include file for vgag-sound.h - Sun audio support */

#include <sys/ioctl.h>
#include <sys/audioio.h>

#ifndef AUDIO_ENCODING_ULINEAR_LE
# define AUDIO_ENCODING_ULINEAR_LE AUDIO_ENCODING_LINEAR8
#endif
#ifdef ENDIAN_IS_BIG
# ifndef AUDIO_ENCODING_SLINEAR_BE
#  define AUDIO_ENCODING_SLINEAR_BE AUDIO_ENCODING_LINEAR
# endif
#else
# ifndef AUDIO_ENCODING_SLINEAR_LE
#  define AUDIO_ENCODING_SLINEAR_LE AUDIO_ENCODING_LINEAR
# endif
#endif


/* definitions */
static const char * pcm_dev_saudio="/dev/audio";  /* Sun audio device */
static int pcm_fd_saudio=-1;                      /* Sun audio file descriptor */
static unsigned char gain_wert_saudio=0;          /* gain value instead of mixer */
#ifdef HAVE_SOUND_SAUDIOMIXER
  static const char * mix_dev_saudio="/dev/mixer";  /* Mixer device */
  static int mix_fd_saudio=-1;                      /* Mixer file descriptor */
  mixer_ctrl_t mix_wert_saudio;                     /* Mixer value */
#endif


/* declarations */

static int _config_saudio(struct sound_dev *,int,int);

int open_saudio(struct sound_dev *,int,int);
void write_saudio(struct sound_dev *,const unsigned char *,int);
void close_saudio(struct sound_dev *);
int mixer_saudio(struct sound_dev *,int);


/* internal functions */

static int _config_saudio(struct sound_dev * sdptr,int srate,int channels) {
/* (re)configure pcm device */
  audio_info_t auinfo;
#ifdef AUDIO_FLUSH
  /* reset device */
  /* ioctl(pcm_fd_saudio,AUDIO_FLUSH,NULL); */
#endif

  if (sdptr->frag_anz==0) {sdptr->frag_anz=fragment_anz;}
  if (sdptr->frag_size==0) {sdptr->frag_size=fragment_size;}

  /* Fragments */
  AUDIO_INITINFO(&auinfo);
#if defined(AUMODE_PLAY) && defined(AUMODE_PLAY_ALL)
  auinfo.mode=AUMODE_PLAY|AUMODE_PLAY_ALL;
#endif
#ifdef HAVE_SOUND_SAUDIOBLKSZ
  auinfo.hiwat=sdptr->frag_anz;
  auinfo.blocksize=sdptr->frag_size;
#endif
  if (ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo)==-1) {
    fprintf(stderr,"  _config_saudio: AUDIO_SETINFO: ioctl error: %s\n",strerror(errno));
#ifdef HAVE_SOUND_SAUDIOMIXER
    close(mix_fd_saudio); mix_fd_saudio=-1;
#endif
    close(pcm_fd_saudio); pcm_fd_saudio=-1;
    return(-1);
  }
  if (ioctl(pcm_fd_saudio,AUDIO_GETINFO,&auinfo)==-1) {
    fprintf(stderr,"  _config_saudio: AUDIO_GETINFO: ioctl error: %s\n",strerror(errno));
#ifdef HAVE_SOUND_SAUDIOMIXER
    close(mix_fd_saudio); mix_fd_saudio=-1;
#endif
    close(pcm_fd_saudio); pcm_fd_saudio=-1;
    return(-1);
  }
#ifdef HAVE_SOUND_SAUDIOBLKSZ
  fprintf(stderr,"  _config_saudio: Fragments=%d, Fragment-Size=%d\n",(int)auinfo.hiwat,(int)auinfo.blocksize);
  sdptr->frag_anz=auinfo.hiwat;
  sdptr->frag_size=auinfo.blocksize;
#else
  sdptr->frag_anz=3;
  sdptr->frag_size=1024;
  fprintf(stderr,"  _config_saudio: cannot change Fragment-Size, assuming %d\n",sdptr->frag_size);
#endif

  /* bits per sample */
  AUDIO_INITINFO(&auinfo);
#ifdef ENDIAN_IS_BIG
  auinfo.play.encoding=AUDIO_ENCODING_SLINEAR_BE;
#else
  auinfo.play.encoding=AUDIO_ENCODING_SLINEAR_LE;
#endif
  auinfo.play.precision=16;
  if (ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo)==-1) {
    fprintf(stderr,"  _config_saudio: AUDIO_SETINFO(16LE): ioctl error: %s\n",strerror(errno));
    AUDIO_INITINFO(&auinfo);
    auinfo.play.encoding=AUDIO_ENCODING_ULINEAR_LE;
    auinfo.play.precision=8;
    if (ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo)==-1) {
      fprintf(stderr,"  _config_saudio: AUDIO_SETINFO: ioctl error: %s\n",strerror(errno));
      return(-1);
    } else {sdptr->bitsamp=8;}
  } else {sdptr->bitsamp=16;}
  /* channels */
  AUDIO_INITINFO(&auinfo);
  auinfo.play.channels=(unsigned int)channels;
  ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo);
  if (ioctl(pcm_fd_saudio,AUDIO_GETINFO,&auinfo)==-1) {
    fprintf(stderr,"  _config_saudio: AUDIO_GETINFO: ioctl error: %s\n",strerror(errno));
    return(-1);
  } else if (auinfo.play.channels!=(unsigned int)channels) {
    channels=(channels==2?1:2);
    AUDIO_INITINFO(&auinfo);
    auinfo.play.channels=(unsigned int)channels;
    ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo);
    if (ioctl(pcm_fd_saudio,AUDIO_GETINFO,&auinfo)==-1) {
      fprintf(stderr,"  _config_saudio: AUDIO_GETINFO: ioctl error: %s\n",strerror(errno));
      return(-1);
    } else if (auinfo.play.channels!=(unsigned int)channels) {
      fprintf(stderr,"  _config_saudio: AUDIO_SETINFO: error: set to %d\n",(int)auinfo.play.channels);
      return(-1);
    } else {sdptr->channel=channels;}
  } else {sdptr->channel=channels;}
  /* sample rate */
  AUDIO_INITINFO(&auinfo);
  auinfo.play.sample_rate=(unsigned int)srate;
  ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo);
  if (ioctl(pcm_fd_saudio,AUDIO_GETINFO,&auinfo)==-1) {
    fprintf(stderr,"  _config_saudio: AUDIO_GETINFO: ioctl error: %s\n",strerror(errno));
    return(-1);
  }
  if ((srate<((int)auinfo.play.sample_rate-1000)) || (srate>(int)auinfo.play.sample_rate+1000)) {
    fprintf(stderr,"  _config_saudio: AUDIO_SETINFO: error: %d Hz requested, %d Hz got\n",srate,(int)auinfo.play.sample_rate);
    return(-1);
  }
  sdptr->srate=srate;

  return(0);
} /* Ende _config_saudio */


/* functions */

int open_saudio(struct sound_dev * sdptr,int srate,int channels) {
/* open device
** 2.arg=sample rate
** 3.arg=number of channels (1 or 2)
** return: 0=OK or -1=error
*/
  audio_info_t auinfo;
  int retry;

  /* +++ open pcm device +++ */
  for (retry=5;;retry--) {
    if (alarmtime==0) {
      if ((pcm_fd_saudio=open(pcm_dev_saudio,O_WRONLY|O_NONBLOCK,0))<0) {
        if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
        fprintf(stderr,"  open_saudio: open %s: %s\n",pcm_dev_saudio,strerror(errno));
        return(-1);
      }
      if (fcntl(pcm_fd_saudio,F_SETFL,fcntl(pcm_fd_saudio,F_GETFL,0)&~O_NONBLOCK)<0) {
        fprintf(stderr,"  open_saudio: fcntl %s: %s\n",pcm_dev_saudio,strerror(errno));
        close(pcm_fd_saudio); pcm_fd_saudio=-1;
        return(-1);
      }
    } else {
      alarm(alarmtime);
      if ((pcm_fd_saudio=open(pcm_dev_saudio,O_WRONLY,0))<0) {
        alarm(0);
        if ((retry>1) && (errno==EBUSY)) {sleep(1); continue;}
        fprintf(stderr,"  open_saudio: open %s: %s\n",pcm_dev_saudio,strerror(errno));
        return(-1);
      }
      alarm(0);
    }
    break;
  }

  /* +++ Mixer +++ */
  gain_wert_saudio=auinfo.play.gain;
  AUDIO_INITINFO(&auinfo);
  if (gain_wert_saudio>100) {auinfo.play.gain=20;} else {auinfo.play.gain=200;}
  if ((ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo)==-1) \
  || (ioctl(pcm_fd_saudio,AUDIO_GETINFO,&auinfo)==-1) \
  || (auinfo.play.gain==gain_wert_saudio)) {  /* cannot set volume via gain; use mixer */
#ifdef HAVE_SOUND_SAUDIOMIXER
    mixer_devinfo_t mix_type;
    int i1,anzdev,idx;
    if ((mix_fd_saudio=open(mix_dev_saudio,O_RDWR))<0) {
      fprintf(stderr,"  open_saudio: open %s: %s\n",mix_dev_saudio,strerror(errno));
      close(pcm_fd_saudio); pcm_fd_saudio=-1;
      return(-1);
    }

    idx=-1;
    for (anzdev=0;;anzdev++) {
      mix_type.index=anzdev;
      if (ioctl(mix_fd_saudio,AUDIO_MIXER_DEVINFO,&mix_type)<0) {break;}
      if (mix_type.type==AUDIO_MIXER_CLASS) {
        if (strcmp(mix_type.label.name,AudioCoutputs)==0) {idx=mix_type.mixer_class;}
      }
    }
    if (idx==-1) {
      fprintf(stderr,"  open_saudio: no %s-class found\n",AudioCoutputs);
      close(mix_fd_saudio); mix_fd_saudio=-1;
      close(pcm_fd_saudio); pcm_fd_saudio=-1;
      return(-1);
    }
    memset(&mix_wert_saudio,0,sizeof(mix_wert_saudio));
    for (i1=0;i1<anzdev;i1++) {
      mix_type.index=i1;
      if (ioctl(mix_fd_saudio,AUDIO_MIXER_DEVINFO,&mix_type)<0) {break;}
      if ((mix_type.type==AUDIO_MIXER_VALUE) && (mix_type.mixer_class==idx) \
      && (strcmp(mix_type.label.name,AudioNmaster)==0) && (strcmp(mix_type.un.v.units.name,AudioNvolume)==0)) {
        mix_wert_saudio.dev=i1;
        mix_wert_saudio.type=mix_type.type;
        mix_wert_saudio.un.value.num_channels=2;
        if (ioctl(mix_fd_saudio,AUDIO_MIXER_READ,&mix_wert_saudio)<0) {
          mix_wert_saudio.un.value.num_channels=1;
          if (ioctl(mix_fd_saudio,AUDIO_MIXER_READ,&mix_wert_saudio)<0) {
            fprintf(stderr,"  open_saudio: AUDIO_MIXER_READ: ioctl error: %s\n",strerror(errno));
            close(mix_fd_saudio); mix_fd_saudio=-1;
            close(pcm_fd_saudio); pcm_fd_saudio=-1;
            return(-1);
          }
        }
      }
    }
    if (mix_wert_saudio.un.value.num_channels==0) {
      fprintf(stderr,"  open_saudio: no %s-%s found\n",AudioNmaster,AudioNvolume);
      close(mix_fd_saudio); mix_fd_saudio=-1;
      close(pcm_fd_saudio); pcm_fd_saudio=-1;
      return(-1);
    }
#else /* HAVE_SOUND_SAUDIOMIXER */
    fprintf(stderr,"  open_saudio: cannot set volume via audio_info_t.play.gain and no mixer found\n");
    close(pcm_fd_saudio); pcm_fd_saudio=-1;
    return(-1);
#endif /* HAVE_SOUND_SAUDIOMIXER */
  } else {
#ifdef HAVE_SOUND_SAUDIOMIXER
    mix_fd_saudio=-1;
#endif
    AUDIO_INITINFO(&auinfo);
    auinfo.play.gain=gain_wert_saudio;
    ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo);
  }

  /* +++ configure pcm device +++ */
  return(_config_saudio(sdptr,srate,channels));
} /* Ende open_saudio */


void write_saudio(struct sound_dev * sdptr,const unsigned char * val,int size) {
/* write samples "val" of size "size" out to device */
  write(pcm_fd_saudio,val,size);
} /* Ende write_saudio */


void close_saudio(struct sound_dev * sdptr) {
/* close device */
#ifdef HAVE_SOUND_SAUDIOMIXER
  if (mix_fd_saudio>=0) {close(mix_fd_saudio);}
  mix_fd_saudio=-1;
#endif
  close(pcm_fd_saudio);
  pcm_fd_saudio=-1;
} /* Ende close_saudio */


int mixer_saudio(struct sound_dev * sdptr,int volm) {
/* set main volume
** 2.arg: 0-100=set volume
**        -1=only get actual volume
**        -2=resume saved values
** return: <volume before (0-100)>: ok
**                              -1: error
*/
  static int fvol=-1;
  int vvol;
#ifdef HAVE_SOUND_SAUDIOMIXER
  if (mix_fd_saudio==-1) {  /* use gain */
#endif
    vvol=gain_wert_saudio*100/256;
#ifdef HAVE_SOUND_SAUDIOMIXER
  } else {  /* use mixer */
    vvol=mix_wert_saudio.un.value.level[0]*100/256;
  }
#endif
  if (fvol==-1) {fvol=vvol;}
  /* set volume */
  if (volm>=0) {
    if (volm>100) {volm=100;}
  } else if (volm==-2) {
    volm=fvol;
  }
  if (volm>=0) {
    volm*=256;
    volm/=100;
    if (volm>255) {volm=255;}
#ifdef HAVE_SOUND_SAUDIOMIXER
    if (mix_fd_saudio==-1) {  /* use gain */
#else
    {
#endif
      audio_info_t auinfo;
      AUDIO_INITINFO(&auinfo);
      auinfo.play.gain=gain_wert_saudio=volm;
      if (ioctl(pcm_fd_saudio,AUDIO_SETINFO,&auinfo)==-1) {
        fprintf(stderr,"mixer_saudio: AUDIO_SETINFO (gain): %s\n",strerror(errno));
        return(-1);
      }
#ifdef HAVE_SOUND_SAUDIOMIXER
    } else {  /* use mixer */
      mix_wert_saudio.un.value.level[0]=volm;
      if (mix_wert_saudio.un.value.num_channels>1) {mix_wert_saudio.un.value.level[1]=mix_wert_saudio.un.value.level[0];}
      if (ioctl(mix_fd_saudio,AUDIO_MIXER_WRITE,&mix_wert_saudio)==-1) {
        fprintf(stderr,"mixer_saudio: AUDIO_MIXER_WRITE: %s\n",strerror(errno));
        return(-1);
      }
#endif
    }
  }
  return(vvol);
} /* Ende mixer_saudio */
